﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"
#include <panels/PerformancePanel.h>

#include <nw/ut/ut_Memory.h>
#include <nw/snd.h>
#include <nw/snd/fnd/io/sndfnd_File.h>
#include <nw/snd/fnd/string/sndfnd_Path.h>
#include <resources/GlobalResources.h>

#if defined(NW_PLATFORM_CAFE)
#include <cafe/env.h>
#endif

namespace
{

// ログファイルパス
static const char* LOG_FILEPATH = "PerformanceLog.csv";

// ログ出力のラベル
static const char* PERFORMANCE_LOG_HEADER = "Frame,AxTick,AxBegin,AxEnd,Ax,AxInt,AuxBegin,AuxEnd,Aux,CbBegin,CbEnd,Cb,SndTick,SndBegin,SndEnd,Snd,SndInt,DspBegin,DspEnd,Dsp,Voices";

}

namespace nw {
namespace snd {

const char* PerformancePanel::SOUNDTHREAD_CPU_LABEL     = "CPU  ";
const char* PerformancePanel::SOUNDTHREAD_SND_LABEL     = " SND ";
const char* PerformancePanel::SOUNDTHREAD_SNDCB_LABEL   = "  CB ";
const char* PerformancePanel::SOUNDTHREAD_SNDTH_LABEL   = "  Th.";
const char* PerformancePanel::SOUNDTHREAD_AUX_LABEL     = "  AUX";
const char* PerformancePanel::SOUNDTHREAD_AX_LABEL      = " AX  ";
const char* PerformancePanel::SOUNDTHREAD_DSP_LABEL     = "DSP  ";
const char* PerformancePanel::SOUNDTHREAD_SNDINT_LABEL  = "ThInt";
const char* PerformancePanel::SOUNDTHREAD_AXINT_LABEL   = "CBInt";

PerformancePanel::PerformancePanel() :
m_IsVideoFrameProfile(true),
m_LoggingState(LOGGING_STATE_IDLE),
m_LogBuffer(NULL),
m_Allocator(NULL)
#if !defined(NW_PLATFORM_WIN32) && !defined(NW_USE_NINTENDO_SDK)
, m_PrevAxFrameProcessBegin(0)
, m_PrevNwFrameProcessBegin(0)
, m_PrevAxFrameProcessBeginForLogging(0)
, m_PrevNwFrameProcessBeginForLogging(0)
#endif
#if defined(NW_PLATFORM_CAFE)
, m_FsClient(NULL)
, m_FsCmdBlock(NULL)
, m_HfioVolume(NULL)
#endif
{
    SetMargin(nw::internal::dw::Thickness(0.f));
    SetIsFocusable(true);

    m_Contents.AddItem(&m_Container);
    SetContents(m_Contents);

    m_Container.SetMeasurement(nw::internal::dw::MEASUREMENT_AUTO_HEIGHT);
    m_Container.SetOrientation(nw::internal::dw::VERTICAL);

    nw::internal::dw::UIElementList& containerItems = m_Container.GetContents();

    // パフォーマンスメーターを初期化します。
    for(s32 i=0; i<PERFORMANCE_BAR_COUNT; ++i)
    {
        FixedLabelPeakMeter<LABEL_LENGTH>& performanceMeter = m_PerformanceMeters[i];

        performanceMeter.SetMargin(nw::internal::dw::Thickness(3.f, 1.f));
        performanceMeter.SetMeasurement(nw::internal::dw::MEASUREMENT_AUTO_HEIGHT);
        performanceMeter.SetOrientation(nw::internal::dw::HORIZONTAL);
        performanceMeter.SetMaximumValue(100.f);
        performanceMeter.SetPeakHoldChange(1.f);

        containerItems.AddItem(&performanceMeter);
    }

    m_PerformanceMeters[SOUNDTHREAD_CPU_INDEX].SetLabelText(SOUNDTHREAD_CPU_LABEL);
    m_PerformanceMeters[SOUNDTHREAD_SND_INDEX].SetLabelText(SOUNDTHREAD_SND_LABEL);
    m_PerformanceMeters[SOUNDTHREAD_SNDCB_INDEX].SetLabelText(SOUNDTHREAD_SNDCB_LABEL);
    m_PerformanceMeters[SOUNDTHREAD_SNDTH_INDEX].SetLabelText(SOUNDTHREAD_SNDTH_LABEL);
    m_PerformanceMeters[SOUNDTHREAD_AUX_INDEX].SetLabelText(SOUNDTHREAD_AUX_LABEL);
    m_PerformanceMeters[SOUNDTHREAD_AX_INDEX].SetLabelText(SOUNDTHREAD_AX_LABEL);
    m_PerformanceMeters[SOUNDTHREAD_DSP_INDEX].SetLabelText(SOUNDTHREAD_DSP_LABEL);
    m_PerformanceMeters[SOUNDTHREAD_SNDINT_INDEX].SetLabelText(SOUNDTHREAD_SNDINT_LABEL);
    m_PerformanceMeters[SOUNDTHREAD_AXINT_INDEX].SetLabelText(SOUNDTHREAD_AXINT_LABEL);

    SetLoggingState(LOGGING_STATE_IDLE);

#ifdef NW_PLATFORM_CAFE
    nw::snd::SoundSystem::RegisterProfileReader(m_ProfileReader);
#endif
}

bool PerformancePanel::Initialize(IResourceProvider& resourceProvider)
{
#if defined(NW_PLATFORM_CAFE)
    m_FsClient = reinterpret_cast<FSClient*>(
        resourceProvider.GetResource(GlobalResources::FsClientPath));

    m_FsCmdBlock = reinterpret_cast<FSCmdBlock*>(
        resourceProvider.GetResource(GlobalResources::FsCmdBlockPath));

    m_HfioVolume = reinterpret_cast<const char*>(
        resourceProvider.GetConstResource(GlobalResources::HfioVolumePath));
#endif

#ifdef NW_SOUNDPLAYER_PRIVATE
    m_Allocator = reinterpret_cast<ut::IAllocator*>(
        resourceProvider.GetResource(GlobalResources::AllocatorPath));

    NW_ASSERT_NOT_NULL(m_Allocator);

    m_LogBuffer = m_Allocator->Alloc(LOG_BUFFER_LENGTH);
#endif

    return true;
}

void PerformancePanel::Finalize()
{
#if defined(NW_PLATFORM_CAFE)
    m_FsClient = NULL;
    m_FsCmdBlock = NULL;
#endif

    if(m_Allocator != NULL && m_LogBuffer != NULL)
    {
        m_Allocator->Free(m_LogBuffer);
        m_Allocator = NULL;
        m_LogBuffer = NULL;
    }
}

bool PerformancePanel::IsLogging()
{
    return m_LoggingState != LOGGING_STATE_IDLE;
}

void PerformancePanel::SetLogLamp(Lamp* pLamp)
{
    if(m_pLogLamp == pLamp)
    {
        return;
    }

    if(m_pLogLamp != NULL)
    {
        m_pLogLamp->SetLabelText("");
        m_pLogLamp->SetIsLamped(false);
    }

    m_pLogLamp = pLamp;

    if(m_pLogLamp != NULL)
    {
        m_pLogLamp->SetIsLamped(IsLogging());
        SetLoggingState(m_LoggingState);
    }
}

bool PerformancePanel::OnUpdateFocusedInput(const nw::internal::dw::Inputs& inputs)
{
    if(inputs.GetPad() == NULL)
    {
        return false;
    }

    const nw::dev::Pad& pad = *inputs.GetPad();

    // プロファイリングモードの変更
    if(pad.IsTrig(nw::dev::Pad::MASK_LEFT) ||
        pad.IsTrig(nw::dev::Pad::MASK_RIGHT))
    {
        SetProfileMode(!m_IsVideoFrameProfile);
        return true;
    }

    // パフォーマンスログ出力の開始
    if(pad.IsTrig(nw::dev::Pad::MASK_X))
    {
        StartLogging();
    }

    // パフォーマンスログ出力の終了
    if(pad.IsTrig(nw::dev::Pad::MASK_Y))
    {
        StopLogging();
    }

    return false;
}

void PerformancePanel::OnUpdate(const nw::internal::dw::UIElementTreeContext& context)
{
#if !defined(NW_PLATFORM_WIN32) && !defined(NW_USE_NINTENDO_SDK)

    static nw::snd::SoundProfile profile[nw::snd::internal::MAX_PROFILE_COUNT];

    u32 profileCount = m_ProfileReader.Read(profile, nw::snd::internal::MAX_PROFILE_COUNT);

    // ビデオフレームでの処理負荷を表示
    if ( m_IsVideoFrameProfile )
    {
        UpdateVideoFrameProfiles(profile, profileCount);
    }
    // オーディオフレーム (3ms) 単位での処理負荷を表示
    else
    {
        UpdateAudioFrameProfiles(profile, profileCount);
    }

    s32 loggingTragetIndex = UpdateLoggingState(profile, profileCount);

    if(loggingTragetIndex >= 0)
    {
        LogProfiles(profile, profileCount, static_cast<u32>(loggingTragetIndex));
    }

#endif
}

void PerformancePanel::StartLogging()
{
    SetLoggingState(LOGGING_STATE_PREPARING);
}

void PerformancePanel::StopLogging()
{
    SetLoggingState(LOGGING_STATE_IDLE);
    m_LogWriter.Close();
    m_LogFile.Close();
}

void PerformancePanel::SetLoggingState(LoggingState state)
{
#ifdef NW_SOUNDPLAYER_PRIVATE
    m_LoggingState = state;

    if(m_pLogLamp != NULL)
    {
        switch(state)
        {
        case LOGGING_STATE_IDLE:
            m_pLogLamp->SetIsLamped(false);
            m_pLogLamp->SetLabelText("Not Logging");
            break;

        case LOGGING_STATE_PREPARING:
            m_pLogLamp->SetIsLamped(false);
            m_pLogLamp->SetLabelText("Preparing");
            break;

        case LOGGING_STATE_RUNNING:
            m_pLogLamp->SetIsLamped(true);
            m_pLogLamp->SetLabelText("Logging");
            break;
        }
    }
#endif
}

void PerformancePanel::SetMeter(int index, f32 value)
{
    m_PerformanceMeters[index].SetValue(value * 100.f);
}

#ifdef NW_PLATFORM_CAFE
void PerformancePanel::UpdateVideoFrameProfiles(const nw::snd::SoundProfile* pProfile, u32 profileCount)
{
    NW_NULL_ASSERT(pProfile);

    ProfileData sumProf;

    for ( u32 i = 0; i < profileCount; i++ )
    {
        const nw::snd::SoundProfile& prof = pProfile[i];

        // ビデオフレーム単位で表示
        sumProf.axFrame          += ut::TimeSpan::FromNanoSeconds(OSTicksToNanoseconds(prof.axFrameProcessTick));
        sumProf.axFrameInterval  += prof.GetAxFrameProcessInterval(m_PrevAxFrameProcessBegin);
        sumProf.auxProcessing    += prof.auxProcess.GetSpan();
        sumProf.userCallback     += prof.nwVoiceParamUpdate.GetSpan();
        sumProf.sndFrame         += ut::TimeSpan::FromNanoSeconds(OSTicksToNanoseconds(prof.nwFrameProcessTick));
        sumProf.sndFrameInterval += prof.GetNwFrameProcessInterval(m_PrevNwFrameProcessBegin);
        sumProf.dspFrame         += prof.dspFrameProcess.GetSpan();

        m_PrevAxFrameProcessBegin = prof.axFrameProcess.begin;
        m_PrevNwFrameProcessBegin = prof.nwFrameProcess.begin;
    }

    // 積算値を表示
    {
        const f32 SUM_INTERVAL_USEC =
            static_cast<f32>(nw::snd::SoundSystem::SOUND_THREAD_INTERVAL_USEC * profileCount);

        f32 usec = static_cast<f32>((sumProf.axFrame + sumProf.sndFrame ).GetMicroSeconds());
        SetMeter( SOUNDTHREAD_CPU_INDEX,   usec / SUM_INTERVAL_USEC );

        usec = static_cast<f32>((sumProf.sndFrame + sumProf.userCallback + sumProf.auxProcessing ).GetMicroSeconds());
        SetMeter( SOUNDTHREAD_SND_INDEX,   usec / SUM_INTERVAL_USEC );

        usec = static_cast<f32>(sumProf.userCallback.GetMicroSeconds());
        SetMeter( SOUNDTHREAD_SNDCB_INDEX, usec / SUM_INTERVAL_USEC );

        usec = static_cast<f32>(sumProf.sndFrame.GetMicroSeconds());
        SetMeter( SOUNDTHREAD_SNDTH_INDEX, usec / SUM_INTERVAL_USEC );

        usec = static_cast<f32>(sumProf.auxProcessing.GetMicroSeconds());
        SetMeter( SOUNDTHREAD_AUX_INDEX,   usec / SUM_INTERVAL_USEC);

        usec = static_cast<f32>((sumProf.axFrame - sumProf.userCallback - sumProf.auxProcessing).GetMicroSeconds());
        SetMeter( SOUNDTHREAD_AX_INDEX,    usec / SUM_INTERVAL_USEC);

// HACK : CafeSDK 2.05 では正しい数値が採れないので、無効化しています。
#if 0
        usec = static_cast<f32>(sumProf.sndFrameInterval.GetMicroSeconds());
        SetMeter( SOUNDTHREAD_SNDINT_INDEX,
            ut::Abs( usec - SUM_INTERVAL_USEC ) / SUM_INTERVAL_USEC );
#endif

        usec = static_cast<f32>(sumProf.axFrameInterval.GetMicroSeconds());
        SetMeter( SOUNDTHREAD_AXINT_INDEX,
            ut::Abs( usec - SUM_INTERVAL_USEC ) / SUM_INTERVAL_USEC );

        usec = static_cast<f32>(sumProf.dspFrame.GetMicroSeconds());
        SetMeter( SOUNDTHREAD_DSP_INDEX, usec / SUM_INTERVAL_USEC );
    }
}

void PerformancePanel::UpdateAudioFrameProfiles(const nw::snd::SoundProfile* pProfile, u32 profileCount)
{
    NW_NULL_ASSERT(pProfile);

    for ( u32 i=0; i<profileCount; i++ )
    {
        const nw::snd::SoundProfile& prof = pProfile[i];

        ProfileData sumProf;

        sumProf.axFrame          = ut::TimeSpan::FromNanoSeconds(OSTicksToNanoseconds(prof.axFrameProcessTick));
        sumProf.axFrameInterval  = prof.GetAxFrameProcessInterval(m_PrevAxFrameProcessBegin);
        sumProf.auxProcessing    = prof.auxProcess.GetSpan();
        sumProf.userCallback     = prof.nwVoiceParamUpdate.GetSpan();
        sumProf.sndFrame         = ut::TimeSpan::FromNanoSeconds(OSTicksToNanoseconds(prof.nwFrameProcessTick));
        sumProf.sndFrameInterval = prof.GetNwFrameProcessInterval(m_PrevNwFrameProcessBegin);
        sumProf.dspFrame         = prof.dspFrameProcess.GetSpan();

        m_PrevAxFrameProcessBegin = prof.axFrameProcess.begin;
        m_PrevNwFrameProcessBegin = prof.nwFrameProcess.begin;

        f32 usec = static_cast<f32>(( sumProf.axFrame + sumProf.sndFrame ).GetMicroSeconds());
        SetMeter(
            SOUNDTHREAD_CPU_INDEX,
            usec / nw::snd::SoundSystem::SOUND_THREAD_INTERVAL_USEC
            );

        usec = static_cast<f32>((sumProf.sndFrame + sumProf.userCallback + sumProf.auxProcessing ).GetMicroSeconds());
        SetMeter(
            SOUNDTHREAD_SND_INDEX,
            usec / nw::snd::SoundSystem::SOUND_THREAD_INTERVAL_USEC
            );

        usec = static_cast<f32>(sumProf.userCallback.GetMicroSeconds());
        SetMeter(
            SOUNDTHREAD_SNDCB_INDEX,
            usec / nw::snd::SoundSystem::SOUND_THREAD_INTERVAL_USEC
            );

        usec = static_cast<f32>(sumProf.sndFrame.GetMicroSeconds());
        SetMeter(
            SOUNDTHREAD_SNDTH_INDEX,
            usec / nw::snd::SoundSystem::SOUND_THREAD_INTERVAL_USEC
            );

        usec = static_cast<f32>(sumProf.auxProcessing.GetMicroSeconds());
        SetMeter(
            SOUNDTHREAD_AUX_INDEX,
            usec / nw::snd::SoundSystem::SOUND_THREAD_INTERVAL_USEC
            );

        usec = static_cast<f32>((sumProf.axFrame - sumProf.userCallback - sumProf.auxProcessing).GetMicroSeconds());
        SetMeter(
            SOUNDTHREAD_AX_INDEX,
            usec / nw::snd::SoundSystem::SOUND_THREAD_INTERVAL_USEC
            );

        usec = static_cast<f32>(sumProf.dspFrame.GetMicroSeconds());
        SetMeter(
            SOUNDTHREAD_DSP_INDEX,
            usec / nw::snd::SoundSystem::SOUND_THREAD_INTERVAL_USEC
            );

// HACK : CafeSDK 2.05 では正しい数値が採れないので、無効化しています。
#if 0
        usec = static_cast<f32>(sumProf.sndFrameInterval.GetMicroSeconds());
        SetMeter(
            SOUNDTHREAD_SNDINT_INDEX,
            ut::Abs( usec - nw::snd::SoundSystem::SOUND_THREAD_INTERVAL_USEC ) /
            nw::snd::SoundSystem::SOUND_THREAD_INTERVAL_USEC
            );
#endif

        usec = static_cast<f32>(sumProf.axFrameInterval.GetMicroSeconds());
        SetMeter(
            SOUNDTHREAD_AXINT_INDEX,
            ut::Abs( usec - nw::snd::SoundSystem::SOUND_THREAD_INTERVAL_USEC ) /
            nw::snd::SoundSystem::SOUND_THREAD_INTERVAL_USEC
            );
    }
}

s32 PerformancePanel::UpdateLoggingState(const nw::snd::SoundProfile* pProfile, u32 profileCount)
{
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    return INVALID_INDEX;
#else

    switch(m_LoggingState)
    {
    case LOGGING_STATE_IDLE:
        return INVALID_INDEX;

    case LOGGING_STATE_RUNNING:
        return 0;

    case LOGGING_STATE_PREPARING:
        break;

    default:
        NW_ASSERTMSG(0, "invalid logging state.");
        return INVALID_INDEX;
    }

    for(u32 i=0; i<profileCount; ++i)
    {
        if(pProfile[i].axNumVoices > 0)
        {
            SetLoggingState(LOGGING_STATE_RUNNING);
            return static_cast<s32>(i);
        }
    }

    return INVALID_INDEX;

#endif
}

void PerformancePanel::LogProfiles(const nw::snd::SoundProfile* pProfile, u32 profileCount, u32 startIndex)
{
    NW_NULL_ASSERT(pProfile);

    if(!m_LogFile.IsOpened())
    {
#if defined(NW_PLATFORM_CAFE)
        char saveDir[MAX_LOG_FILE_PATH];
        char hfioSaveDir[MAX_LOG_FILE_PATH];
        char logFilePath[MAX_LOG_FILE_PATH];

        int result = ENVGetEnvironmentVariable("CAFE_SAVE_DIR", saveDir, MAX_LOG_FILE_PATH);
        if(result != 0)
        {
            return;
        }

        internal::fnd::Path::ConvertToPcFilePath(
            saveDir,
            std::strlen(saveDir),
            hfioSaveDir,
            MAX_LOG_FILE_PATH,
            m_HfioVolume,
            std::strlen(m_HfioVolume));

        ut::snprintf(logFilePath, MAX_LOG_FILE_PATH, "%s/%s", hfioSaveDir, LOG_FILEPATH);

        if(internal::fnd::File::Open(
            &m_LogFile,
            m_FsClient,
            m_FsCmdBlock,
            logFilePath,
            internal::fnd::File::ACCESS_MODE_WRITE).IsFailed())
        {
            return;
        }

        m_LogFile.EnableCache(m_LogBuffer, LOG_BUFFER_LENGTH);

        if(m_LogWriter.Open(&m_LogFile).IsFailed())
        {
            m_LogFile.Close();
            return;
        }

        m_LogWriter.EnableWriteFormat(m_LogWriteFormatBuffer, LOG_WRITE_FORMAT_BUFFER_LENGTH);
#else
        return;
#endif

        m_LogCurrentFrame = 0;
        m_LogWriter.WriteLine(PERFORMANCE_LOG_HEADER);
    }

    for ( u32 i=startIndex; i<profileCount; i++ )
    {
        if(profileCount == nw::snd::internal::MAX_PROFILE_COUNT)
        {
            m_LogWriter.WriteLine("[!warning] might drop profiles.");
        }

        const nw::snd::SoundProfile& prof = pProfile[i];

        m_LogWriter.WriteFormat("%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d\r\n",
            m_LogCurrentFrame,
            static_cast<s32>(OSTicksToMicroseconds(prof.axFrameProcessTick)),
            static_cast<s32>(OSTicksToMicroseconds(prof.axFrameProcess.begin)),
            static_cast<s32>(OSTicksToMicroseconds(prof.axFrameProcess.end)),
            static_cast<s32>(prof.axFrameProcess.GetSpan().GetMicroSeconds()),
            static_cast<s32>(prof.GetAxFrameProcessInterval(m_PrevAxFrameProcessBeginForLogging).GetMicroSeconds()),
            static_cast<s32>(OSTicksToMicroseconds(prof.auxProcess.begin)),
            static_cast<s32>(OSTicksToMicroseconds(prof.auxProcess.end)),
            static_cast<s32>(prof.auxProcess.GetSpan().GetMicroSeconds()),
            static_cast<s32>(OSTicksToMicroseconds(prof.nwVoiceParamUpdate.begin)),
            static_cast<s32>(OSTicksToMicroseconds(prof.nwVoiceParamUpdate.end)),
            static_cast<s32>(prof.nwVoiceParamUpdate.GetSpan().GetMicroSeconds()),
            static_cast<s32>(OSTicksToMicroseconds(prof.nwFrameProcessTick)),
            static_cast<s32>(OSTicksToMicroseconds(prof.nwFrameProcess.begin)),
            static_cast<s32>(OSTicksToMicroseconds(prof.nwFrameProcess.end)),
            static_cast<s32>(prof.nwFrameProcess.GetSpan().GetMicroSeconds()),
            static_cast<s32>(prof.GetNwFrameProcessInterval(m_PrevNwFrameProcessBeginForLogging).GetMicroSeconds()),
            static_cast<s32>(OSTicksToMicroseconds(prof.dspFrameProcess.begin)),
            static_cast<s32>(OSTicksToMicroseconds(prof.dspFrameProcess.end)),
            static_cast<s32>(prof.dspFrameProcess.GetSpan().GetMicroSeconds()),
            prof.axNumVoices);
        m_LogCurrentFrame++;

        m_PrevAxFrameProcessBeginForLogging = prof.axFrameProcess.begin;
        m_PrevNwFrameProcessBeginForLogging = prof.nwFrameProcess.begin;
    }
}
#endif

} // snd
} // nw
