﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#include <models/PreviewBank.h>
#include <models/PreviewSoundArchive.h>

namespace nw {
namespace snd {

PreviewBank::PreviewBank() :
m_pPreviewSoundArchive(NULL),
m_BankID(nw::snd::SoundArchive::INVALID_ID),
m_IsLoaded(false)
{
}

void PreviewBank::Initialize(PreviewSoundArchive& previewSoundArchive)
{
    m_pPreviewSoundArchive = &previewSoundArchive;
}

void PreviewBank::Finalize()
{
    // todo:
    // Initialize で、再度初期化する前にリセットが必要な処理（メモリ確保など）が追加されたら
    // こちらに後始末の処理を実装してください。
}


nw::snd::SoundArchive::ItemId PreviewBank::GetBankID() const
{
    return m_BankID;
}

void PreviewBank::SetBankID(nw::snd::SoundArchive::ItemId soundID)
{
    if(m_BankID == soundID)
    {
        return;
    }

    m_BankID = soundID;
}

u32 PreviewBank::GetBankIndex() const
{
    // HACK : インデックスを抽出する正式な方法が見つかれば置き換える
    return nw::snd::internal::Util::GetItemIndex(m_BankID);
}

void PreviewBank::SetBankIndex(u32 index)
{
    NW_ASSERTMSG(m_pPreviewSoundArchive != NULL, "PreviewBank is not initialized.");
    SetBankID(m_pPreviewSoundArchive->GetBankID(index));
}

void PreviewBank::SetLabel(const char* label)
{
    NW_ASSERTMSG(m_pPreviewSoundArchive != NULL, "PreviewBank is not initialized.");
    SetBankID(m_pPreviewSoundArchive->GetBankID(label));
}

const char* PreviewBank::GetLabel() const
{
    //NW_ASSERTMSG(m_pPreviewSoundArchive != NULL, "PreviewBank is not initialized.");

    if (m_pPreviewSoundArchive == NULL)
    {
        return "";
    }

    if(m_BankID == nw::snd::SoundArchive::INVALID_ID)
    {
        return "";
    }
    else
    {
        return m_pPreviewSoundArchive->GetItemLabel(m_BankID);
    }
}

bool PreviewBank::GetIsLoaded() const
{
    return m_IsLoaded;
}

void PreviewBank::SetIsLoaded(bool isLoaded)
{
    m_IsLoaded = isLoaded;
}

void PreviewBank::Update()
{
    // m_BankIDにマッチするデータがあればそれで設定する
    //SetBankIndex(ValidateBankID() ? GetBankIndex() : 0);
    //SetBankIndex(GetBankIndex());
}

void PreviewBank::Next(s32 offset)
{
    if (m_pPreviewSoundArchive == NULL)
    {
        NW_WARNING(false, "PreviewBank is not initialized.\n");
        return;
    }

    s32 bankCount = static_cast<s32>(m_pPreviewSoundArchive->GetBankCount());

    if(bankCount == 0)
    {
        return;
    }

    s32 soundIndex = static_cast<s32>(GetBankIndex()) + offset;

    if(soundIndex >= bankCount)
    {
        soundIndex = 0;
    }
    else if(soundIndex < 0)
    {
        soundIndex = bankCount - 1;
    }

    SetBankIndex(soundIndex);
}


bool PreviewBank::ValidateBankID()
{
    if (m_BankID == nw::snd::SoundArchive::INVALID_ID)
    {
        return false;
    }

    // 登録されているアーカイブに、m_BankIDに一致するデータが含まれるかどうかを調べます。
    if (m_pPreviewSoundArchive == NULL ||
        !m_pPreviewSoundArchive->IsOpened() )
    {
        return false;
    }

    const s32 bankCount = m_pPreviewSoundArchive->GetBankCount();
    if (bankCount < 0)
    {
        return false;
    }

    for (int i = 0; i < bankCount; ++i)
    {
        if (m_pPreviewSoundArchive->GetBankID(i) == m_BankID)
        {
            return true;
        }
    }

    // 見つからなかった
    return false;
}

} // nw::snd
} // nw
