﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#include <models/PreviewAuxBus.h>

namespace nw {
namespace snd {

PreviewAuxBus::PreviewAuxBus() :
m_AuxBus(AUX_BUS_A),
m_pEffects(NULL),
m_EffectCount(0),
m_pBuffer(NULL),
m_BufferSize(0),
m_AssinedSize(0)
{
}

void PreviewAuxBus::Initialize(const InitializeArgs args)
{
    NW_NULL_ASSERT(args.pEffects);
    NW_ASSERT(args.effectCount > 0);

    Finalize();

    m_AuxBus = args.auxBus;
    m_Device = args.device;
    m_pEffects = args.pEffects;
    m_EffectCount = args.effectCount;
    m_pBuffer = args.pBuffer;
    m_BufferSize = args.bufferSize;
}

void PreviewAuxBus::Finalize()
{
    if(m_pEffects != NULL)
    {
        Clear();
    }

    m_pEffects = NULL;
    m_EffectCount = 0;
    m_pBuffer = NULL;
    m_BufferSize = 0;
    m_AssinedSize = 0;
}

PreviewEffectSet* PreviewAuxBus::GetEffect(s32 index)
{
    NW_ASSERTMSG(m_pEffects != NULL, "[PreviewAuxBus] not initialized.\n");

    if(index >= m_EffectCount)
    {
        NW_ASSERTMSG(false, "[PreviewAuxBus] invalid index.\n");
        return NULL;
    }

    return &m_pEffects[index];
}

void PreviewAuxBus::Apply()
{
    NW_ASSERTMSG(m_pEffects != NULL, "[PreviewAuxBus] not initialized.\n");

    Clear();

    u32 assinedSize = 0;

    for(s32 index = 0; index < m_EffectCount; ++index)
    {
        PreviewEffectSet* pEffectSet = GetEffect(index);
        NW_NULL_ASSERT(pEffectSet);

        // 必要なメモリを割り当てます。
        // メモリが足りなくなったら中断します。
        u32 requiredSize = pEffectSet->GetCurrentRequiredMemorySize();

        if(requiredSize == 0)
        {
            continue;
        }

        if(assinedSize + requiredSize > m_BufferSize)
        {
            break;
        }

        NW_NULL_ASSERT(m_pBuffer);

        if(!pEffectSet->AssignBuffer(nw::ut::AddOffsetToPtr(m_pBuffer, assinedSize), requiredSize))
        {
            break;
        }

        assinedSize += requiredSize;

        if(!SoundSystem::AppendEffect(m_AuxBus, pEffectSet->GetCurrent().GetEffect(), m_Device))
        {
            break;
        }
    }

    m_AssinedSize = assinedSize;
}

void PreviewAuxBus::Clear()
{
    NW_ASSERTMSG(m_pEffects != NULL, "[PreviewAuxBus] not initialized.\n");

    SoundSystem::ClearEffect(m_AuxBus, 0, m_Device);
    m_AssinedSize = 0;

    for(s32 index = 0; index < m_EffectCount; ++index)
    {
        PreviewEffectSet* pEffectSet = GetEffect(index);
        NW_NULL_ASSERT(pEffectSet);

        pEffectSet->ReleaseBuffer();
    }
}

} // nw::snd
} // nw
