﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "precompiled.h"
#include "Application.h"

#include <nw/types.h>
#include <nw/ut.h>
#include <nw/gfnd.h>
#include <nw/snd/fnd/string/sndfnd_Path.h>
#include <anim/GfxCoordinator.h>

namespace nw {
namespace snd {

#if defined(NW_USE_NINTENDO_SDK)
const char* Application::FONT_BINARY_FILENAME = "content:/common/fonts/nintendo_NTLG-DB_002_Nw4f.bffnt";
const char Application::SOUND_ARCHIVE_FILEPATH[] = "content:/SoundPlayer/SoundMakerTemporary/PC/temp.bfsar";
#elif defined(NW_PLATFORM_CAFE)
const char* Application::FONT_BINARY_FILENAME = "/vol/content/common/fonts/nintendo_NTLG-DB_002.bffnt";
const char* Application::FONT_SHADER_FILENAME = "/vol/content/common/shaders/font_BuildinShader.gsh";
const char* Application::PRIMITIVE_RENDERER_SHADER_FILENAME = "/vol/content/common/shaders/dev_PrimitiveRenderer.gsh";
const char Application::SOUND_ARCHIVE_FILEPATH[] = "/vol/content/SoundPlayer/SoundMakerTemporary/WiiU/temp.bfsar";
#elif defined(NW_PLATFORM_WIN32)
const char* Application::FONT_BINARY_FILENAME = "%CAFE_CONTENT_DIR%/common/fonts/nintendo_NTLG-DB_002_Nw4f.bffnt";
const char Application::SOUND_ARCHIVE_FILEPATH[] = "/vol/content/SoundPlayer/SoundMakerTemporary/PC/temp.bfsar";
#endif

//---------------------------------------------------------------------------
Application::Application() :
#if defined(NW_ENABLE_SNDEDIT)
m_SoundEditSessionBuffer(NULL),
m_SoundArchiveEditorBuffer(NULL),
#endif
#if defined(NW_ENABLE_SNDCTRL)
m_SoundControlSessionBuffer(NULL),
#endif
#if defined(NW_SND_CONFIG_ENABLE_DEV)
m_pSpyBuffer(NULL),
m_AppFrame(0),
#endif
m_pFontBinary(NULL),
m_pFontShaderBinary(NULL),
m_pPrimitiveRendererShaderBinary(NULL),
m_FontBinarySize(0),
m_FontShaderBinarySize(0),
m_IsAnimSound(true),
m_ConnectionObserver(this, &Application::OnConnected)
{
#if defined(NW_ENABLE_SNDEDIT)
    m_Port[PORT_INDEX_SYNC] = edit::internal::HIO_SNDEDIT_SYNC_CHANNEL;
    m_Port[PORT_INDEX_FUNC] = edit::internal::HIO_SNDEDIT_FUNC_CHANNEL;
#endif
#if defined(NW_ENABLE_SNDCTRL)
    m_Port[PORT_INDEX_CTRL] = edit::internal::HIO_SNDEDIT_CTRL_CHANNEL;
#endif
    m_Port[PORT_INDEX_TOOL] = HIO_SOUNDPLAYER_PASSIVE_CHANNEL;
    m_Port[PORT_INDEX_VIEWER] = HIO_SOUNDPLAYER_ACTIVE_CHANNEL;
    m_Port[PORT_INDEX_PING] = HIO_PING_CHANNEL;
}

//---------------------------------------------------------------------------
bool
Application::IsAvailable() const
{
    // アプリケーションを終了させません。
    return true;
}

//---------------------------------------------------------------------------
void
Application::UpdateInputs(const nw::dev::Pad* pPad, const nw::dev::Mouse* pMouse)
{
#ifdef USE_ANIM_SOUND
    if (m_IsAnimSound)
    {
        GfxCoordinator::GetInstance().UpdateInputs(pPad, pMouse);
    }
#endif

    m_PanelManager.UpdateInputs(pPad, pMouse);
}

//---------------------------------------------------------------------------
void
Application::Update()
{
#if defined(NW_ENABLE_SNDEDIT)
    m_SoundEditSession.Update();
#endif

#if defined(NW_ENABLE_SNDCTRL)
    m_SoundControlSession.Update();
#endif

#if defined(NW_SND_CONFIG_ENABLE_DEV)
    m_SpyController.SetCurrentAppFrame(m_AppFrame);
    nw::snd::spy::NwSndSpy::GetInstance().SendSoundArchivePlayerInfo(m_PreviewSoundArchive.GetSoundArchivePlayer());
    m_AppFrame++;
#endif

#if !defined(NW_SND_EDIT_USE_MCS)
    if(!m_PassiveChannel.IsOpened())
    {
        if(!m_PassiveChannel.Open(GetChannelInfo(HIO_SOUNDPLAYER_PASSIVE_CHANNEL)))
        {
            return;
        }
    }
    if(!m_ActiveChannel.IsOpened())
    {
        if(!m_ActiveChannel.Open(GetChannelInfo(HIO_SOUNDPLAYER_ACTIVE_CHANNEL)))
        {
            return;
        }
    }
    if(!m_PingChannel.IsOpened())
    {
        if(!m_PingChannel.Open(GetChannelInfo(HIO_PING_CHANNEL)))
        {
            return;
        }
    }
#endif

    m_HioManager.Update();

#ifdef USE_ANIM_SOUND
    if (m_IsAnimSound)
    {
        GfxCoordinator::GetInstance().SetCameraEnabled(!m_PanelManager.IsPanelsVisible());
        GfxCoordinator::GetInstance().Calc();
        m_AnimSoundController.Update();
    }
#endif

    m_PanelManager.Update(m_UITreeContext, m_UIRenderer);

    m_PreviewSoundArchive.Update();
    m_PreviewEffectManager.Update();
    m_PreviewEffectManagerDrc.Update();
}

//---------------------------------------------------------------------------
void
Application::Render()
{
#ifdef USE_ANIM_SOUND
    if (m_IsAnimSound)
    {
        GfxCoordinator::GetInstance().Draw();
    }
#endif

    m_UIRenderer.ClearBuffer();
    m_UIRenderer.BeginDraw();
    m_PanelManager.Render(m_UITreeContext, m_UIRenderer);
    m_UIRenderer.EndDraw();
}

PreviewEffectManager& Application::GetPreviewEffectManager()
{
    NW_ASSERT_NOT_NULL(m_PanelManager.CurrentPreviewEffectManager());
    return *m_PanelManager.CurrentPreviewEffectManager();
}

//---------------------------------------------------------------------------
bool
Application::InitializePassiveChannel(nw::ut::IAllocator& allocator)
{
    if(!m_PassiveChannel.Initialize(allocator, CHANNEL_BUFFER_SIZE))
    {
        return false;
    }

    if(!m_PassiveChannel.Open(GetChannelInfo(HIO_SOUNDPLAYER_PASSIVE_CHANNEL)))
    {
        return false;
    }

    m_HioManager.RegisterChannel(m_PassiveChannel);

    return true;
}

//---------------------------------------------------------------------------
bool
Application::InitializeActiveChannel(nw::ut::IAllocator& allocator)
{
    if(!m_ActiveChannel.Initialize(allocator, CHANNEL_BUFFER_SIZE))
    {
        return false;
    }

    if(!m_ActiveChannel.Open(GetChannelInfo(HIO_SOUNDPLAYER_ACTIVE_CHANNEL)))
    {
        return false;
    }

    m_HioManager.RegisterChannel(m_ActiveChannel);

    return true;
}

//----------------------------------------------------------
bool
Application::InitializePingChannel(nw::ut::IAllocator& allocator)
{
    if(!m_PingChannel.Initialize(allocator, PING_CHANNEL_BUFFER_SIZE))
    {
        return false;
    }

    if(!m_PingChannel.Open(GetChannelInfo(HIO_PING_CHANNEL)))
    {
        return false;
    }

    m_PingChannel.RegisterMessageHandler(m_PingHandler);

    m_HioManager.RegisterChannel(m_PingChannel);

    return true;
}

//---------------------------------------------------------------------------
void
Application::InitializeMessageHandlers()
{
    m_OpenSoundArchiveHandler.Initialize(m_PreviewSoundArchive);
    m_CloseSoundArchiveHandler.Initialize(m_PreviewSoundArchive);
    m_StopAllHandler.Initialize(m_PreviewSoundArchive, m_AnimSoundController);
    m_SelectSoundsHandler.Initialize(m_PreviewSoundArchive);
    m_GetSoundInformationsHandler.Initialize(m_PreviewSoundArchive, m_ActiveChannel);
    m_PlaySoundHandler.Initialize(m_PreviewSoundArchive);
    m_PauseSoundHandler.Initialize(m_PreviewSoundArchive);
    m_StopSoundHandler.Initialize(m_PreviewSoundArchive);
    m_SetSoundParametersHandler.Initialize(m_PreviewSoundArchive, m_ActiveChannel);
    m_SetEffectParametersHandler.Initialize(m_ActiveChannel);
    m_Load3DModelHandler.Initialize(m_AnimSoundController);
    m_LoadAnimSoundHandler.Initialize(m_AnimSoundController);
    m_PlayAnimationHandler.Initialize(m_AnimSoundController);
    m_EditAnimSoundEventHandler.Initialize(m_AnimSoundController);
    m_SetAnimationFrameHandler.Initialize(m_AnimSoundController);
    m_UpdateAllEventIDHandler.Initialize(m_AnimSoundController);
    m_SetSequenceVariablesHandler.Initialize(m_PreviewSoundArchive);
    m_GetSequenceVariablesHandler.Initialize(m_PreviewSoundArchive, m_ActiveChannel);

    m_PassiveChannel.RegisterMessageHandler(m_ConnectHandler);
    m_PassiveChannel.RegisterMessageHandler(m_OpenSoundArchiveHandler);
    m_PassiveChannel.RegisterMessageHandler(m_CloseSoundArchiveHandler);
    m_PassiveChannel.RegisterMessageHandler(m_EchoHandler);
    m_PassiveChannel.RegisterMessageHandler(m_StopAllHandler);
    m_PassiveChannel.RegisterMessageHandler(m_SetOutputModeHandler);
    m_PassiveChannel.RegisterMessageHandler(m_SelectSoundsHandler);
    m_PassiveChannel.RegisterMessageHandler(m_GetSoundInformationsHandler);
    m_PassiveChannel.RegisterMessageHandler(m_PlaySoundHandler);
    m_PassiveChannel.RegisterMessageHandler(m_PauseSoundHandler);
    m_PassiveChannel.RegisterMessageHandler(m_StopSoundHandler);
    m_PassiveChannel.RegisterMessageHandler(m_SetSoundParametersHandler);
    m_PassiveChannel.RegisterMessageHandler(m_SetEffectParametersHandler);
    m_PassiveChannel.RegisterMessageHandler(m_Load3DModelHandler);
    m_PassiveChannel.RegisterMessageHandler(m_Load3DAnimationHandler);
    m_PassiveChannel.RegisterMessageHandler(m_LoadAnimSoundHandler);
    m_PassiveChannel.RegisterMessageHandler(m_PlayAnimationHandler);
    m_PassiveChannel.RegisterMessageHandler(m_StopAnimationHandler);
    m_PassiveChannel.RegisterMessageHandler(m_GetAnimationFrameHandler);
    m_PassiveChannel.RegisterMessageHandler(m_SetAnimationFrameHandler);
    m_PassiveChannel.RegisterMessageHandler(m_SetAnimationRateHandler);
    m_PassiveChannel.RegisterMessageHandler(m_SetAnimationLoopHandler);
    m_PassiveChannel.RegisterMessageHandler(m_SetAnimationRangeHandler);
    m_PassiveChannel.RegisterMessageHandler(m_EditAnimSoundEventHandler);
    m_PassiveChannel.RegisterMessageHandler(m_UpdateAllEventIDHandler);
    m_PassiveChannel.RegisterMessageHandler(m_SetSequenceVariablesHandler);
    m_PassiveChannel.RegisterMessageHandler(m_GetSequenceVariablesHandler);

    m_ConnectHandler.GetInvokedEvent() += m_ConnectionObserver;

    m_PreviewSoundArchive.RegisterSoundArchiveOpenedCallback(SoundArchiveOpenedCallback);
}

//---------------------------------------------------------------------------
bool
Application::InitializeFont()
{
    char expandedPath[MAX_FILE_PATH];
    internal::fnd::Path::ExpandEnvironmentVariables(
        expandedPath, MAX_FILE_PATH, FONT_BINARY_FILENAME, strlen(FONT_BINARY_FILENAME));

    m_pFontBinary = LoadFile(expandedPath, FONT_DATA_ALIGNMENT, &m_FontBinarySize);

    if(m_pFontBinary == NULL)
    {
        return false;
    }

    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    {
        m_Font.SetResource(m_pFontBinary);
    }
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();

    return true;
}

//---------------------------------------------------------------------------
bool
Application::InitializeUI()
{
    m_UIRenderer.SetPrimitiveRenderer(nw::dev::PrimitiveRenderer::GetInstance());
    m_UIRenderer.SetTextRenderer(&m_TextRenderer);

    m_PanelManager.Initialize(m_ResourceProvider, m_pAllocator);

    return true;
}

//---------------------------------------------------------------------------
#ifdef NW_SND_CONFIG_ENABLE_DEV
bool
Application::InitializeSpy(nw::snd::spy::SpyController::OpenArg& arg)
{
    size_t size = nw::snd::spy::SpyController::GetRequiredMemorySize(SPY_DATA_BUFFER_LENGTH);
    m_pSpyBuffer = m_pAllocator->Alloc(size);

    m_SpyController.Initialize(m_pSpyBuffer, size, SPY_DATA_BUFFER_LENGTH);
    nw::snd::spy::NwSndSpy::GetInstance().Initialize( &m_SpyController );
    nw::snd::spy::NwSndSpy::GetInstance().AttachSoundArchive( m_PreviewSoundArchive.GetSoundArchive() );

    arg.syncThreadPriority = 17;
    arg.dataThreadPriority = 17;

    return m_SpyController.Open(arg);
}
#endif

//---------------------------------------------------------------------------
void
Application::FinalizePassiveChannel()
{
    m_HioManager.UnregisterChannel(m_PassiveChannel);

    m_PassiveChannel.Close();
    m_PassiveChannel.Finalize();
}

//---------------------------------------------------------------------------
void
Application::FinalizeActiveChannel()
{
    m_HioManager.UnregisterChannel(m_ActiveChannel);

    m_ActiveChannel.Close();
    m_ActiveChannel.Finalize();
}

//----------------------------------------------------------
void
Application::FinalizePingChannel()
{
    m_HioManager.UnregisterChannel(m_PingChannel);

    m_PingChannel.UnregisterMessageHandler(m_PingHandler);

    m_PingChannel.Close();
    m_PingChannel.Finalize();
}

//---------------------------------------------------------------------------
void
Application::FinalizeMessageHandlers()
{
    m_PreviewSoundArchive.RegisterSoundArchiveOpenedCallback(NULL);

    m_ConnectHandler.GetInvokedEvent() -= m_ConnectionObserver;

    m_PassiveChannel.UnregisterMessageHandler(m_ConnectHandler);
    m_PassiveChannel.UnregisterMessageHandler(m_OpenSoundArchiveHandler);
    m_PassiveChannel.UnregisterMessageHandler(m_CloseSoundArchiveHandler);
    m_PassiveChannel.UnregisterMessageHandler(m_EchoHandler);
    m_PassiveChannel.UnregisterMessageHandler(m_StopAllHandler);
    m_PassiveChannel.UnregisterMessageHandler(m_SetOutputModeHandler);
    m_PassiveChannel.UnregisterMessageHandler(m_SelectSoundsHandler);
    m_PassiveChannel.UnregisterMessageHandler(m_GetSoundInformationsHandler);
    m_PassiveChannel.UnregisterMessageHandler(m_PlaySoundHandler);
    m_PassiveChannel.UnregisterMessageHandler(m_PauseSoundHandler);
    m_PassiveChannel.UnregisterMessageHandler(m_StopSoundHandler);
    m_PassiveChannel.UnregisterMessageHandler(m_SetSoundParametersHandler);
    m_PassiveChannel.UnregisterMessageHandler(m_SetEffectParametersHandler);
    m_PassiveChannel.UnregisterMessageHandler(m_Load3DModelHandler);
    m_PassiveChannel.UnregisterMessageHandler(m_Load3DAnimationHandler);
    m_PassiveChannel.UnregisterMessageHandler(m_LoadAnimSoundHandler);
    m_PassiveChannel.UnregisterMessageHandler(m_PlayAnimationHandler);
    m_PassiveChannel.UnregisterMessageHandler(m_StopAnimationHandler);
    m_PassiveChannel.UnregisterMessageHandler(m_GetAnimationFrameHandler);
    m_PassiveChannel.UnregisterMessageHandler(m_SetAnimationFrameHandler);
    m_PassiveChannel.UnregisterMessageHandler(m_SetAnimationRateHandler);
    m_PassiveChannel.UnregisterMessageHandler(m_SetAnimationLoopHandler);
    m_PassiveChannel.UnregisterMessageHandler(m_SetAnimationRangeHandler);
    m_PassiveChannel.UnregisterMessageHandler(m_EditAnimSoundEventHandler);
    m_PassiveChannel.UnregisterMessageHandler(m_UpdateAllEventIDHandler);
    m_PassiveChannel.UnregisterMessageHandler(m_SetSequenceVariablesHandler);
    m_PassiveChannel.UnregisterMessageHandler(m_GetSequenceVariablesHandler);
}

//---------------------------------------------------------------------------
void
Application::FinalizeFont()
{
    if(m_pFontBinary == NULL)
    {
        return;
    }

    m_Font.RemoveResource();

    nw::ut::SafeFree(m_pFontBinary, m_pAllocator);
    m_FontBinarySize = 0;
}

//---------------------------------------------------------------------------
void
Application::FinalizeTextWriter()
{
    m_TextRenderer.Finalize();

    nw::ut::SafeFree(m_pFontShaderBinary, m_pAllocator);
    m_FontShaderBinarySize = 0;
}

//---------------------------------------------------------------------------
void
Application::FinalizePrimitiveRenderer()
{
    nw::dev::PrimitiveRenderer::GetInstance()->Finalize(m_pAllocator);
    nw::ut::SafeFree(m_pPrimitiveRendererShaderBinary, m_pAllocator);
}

//---------------------------------------------------------------------------
void
Application::FinalizeUI()
{
    m_PanelManager.Finalize();
}

//---------------------------------------------------------------------------
void
Application::FinalizeSpy()
{
#ifdef NW_SND_CONFIG_ENABLE_DEV
    m_SpyController.Finalize();
    nw::ut::SafeFree(m_pSpyBuffer, m_pAllocator);
#endif
}

//---------------------------------------------------------------------------
void
Application::UpdateViewport()
{
    static nw::math::MTX44 projectionMatrix;
    static nw::math::MTX34 viewMatrix;

    nw::math::MTX44Ortho(
        &projectionMatrix,
        m_Viewport.left,
        m_Viewport.right,
        m_Viewport.bottom,
        m_Viewport.top,
        0.01f,
        -300.f);

    nw::math::MTX34Identity(&viewMatrix);

    m_UIRenderer.SetProjectionMatrix(projectionMatrix);
    m_UIRenderer.SetViewMatrix(viewMatrix);
}

//---------------------------------------------------------------------------
void
Application::OnConnected()
{
#ifdef USE_ANIM_SOUND
    GfxCoordinator::GetInstance().Unload();
#endif
}

//---------------------------------------------------------------------------
void
Application::SoundArchiveOpenedCallback()
{
    Application::GetInstance().OnSoundArchiveOpened();
}

//---------------------------------------------------------------------------
void
Application::OnSoundArchiveOpened()
{
#if defined(NW_ENABLE_SNDEDIT)
    if (m_pAllocator != NULL)
    {
        m_SoundEditSession.Close();
        m_SoundEditSession.UnregisterSoundArchiveEditor(&m_SoundArchiveEditor);
        FinalizeSoundArchiveEditor();

        InitializeSoundArchiveEditor(m_pAllocator);
        m_SoundEditSession.RegisterSoundArchiveEditor(&m_SoundArchiveEditor);
        m_SoundEditSession.Open();

        nw::snd::spy::NwSndSpy::GetInstance().AttachSoundArchive(m_PreviewSoundArchive.GetSoundArchive());
    }
#endif
}

} // nw::snd
} // nw
