﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_ITEMCONTAINER_H_
#define NW_ITEMCONTAINER_H_

#include <nw/snd/fnd/binary/sndfnd_PrimitiveTypes.h>

namespace nw {
namespace snd {

//-----------------------------------------------------------------------
/* 一括送信用アイテムコンテナ */
typedef struct {
    /* データへのオフセット */
    nw::snd::internal::fnd::BinU32 offsetForData;
    /* アイテム数 */
    nw::snd::internal::fnd::BinU32 itemCount;
    /* 1アイテムのサイズ */
    nw::snd::internal::fnd::BinU32 itemSize;
} ToolSoundItemContainer;

template< typename _ItemType >
class ItemContainerReader
{
    typedef _ItemType   item_type;

public:
    ItemContainerReader(const ToolSoundItemContainer* pContainer)
        : m_pHeader( pContainer ), m_nCurrentPos( 0 )
    {
        NW_NULL_ASSERT( pContainer );

        m_pItems = reinterpret_cast<item_type*>( (unsigned char*)pContainer + pContainer->offsetForData );
    }

    // 次のアイテムを取得します。
    item_type* Next()
    {
        if( m_nCurrentPos >= m_pHeader->itemCount ) { return NULL; }

        item_type* pItem = m_pItems + m_nCurrentPos;
        m_nCurrentPos++;

        return pItem;
    }

    // コンテナに含まれるアイテム数を取得します。
    int GetCount()
    {
        return m_pHeader->itemCount;
    }

    // ポインタをリセットします。
    void Reset()
    {
        m_nCurrentPos = 0;
    }

private:
    // パラメータ
    const ToolSoundItemContainer* m_pHeader;        // コンテナヘッダ
    item_type*                    m_pItems;         // アイテムデータ
    u32                           m_nCurrentPos;    // ポインタ
};


template< typename _ItemType >
class ItemContainerWriter
{
    typedef _ItemType   item_type;

public:
    ItemContainerWriter(ToolSoundItemContainer* pContainer, int nBufferSize)
        : m_pHeader( pContainer ), m_nCurrentPos( 0 ), m_nBufferSize( nBufferSize )
    {
         NW_NULL_ASSERT( pContainer );
         NW_ASSERT( sizeof( ToolSoundItemContainer ) <= nBufferSize );

        m_pItems = reinterpret_cast<item_type*>( (unsigned char*)pContainer + sizeof(ToolSoundItemContainer) );
    }

    // 次のアイテムを取得します。
    item_type* Next()
    {
        if( m_nBufferSize <= GetSize() + sizeof( item_type ) ) { return NULL; }

        item_type* pItem = m_pItems + m_nCurrentPos;
        m_nCurrentPos++;

        return pItem;
    }

    // コンテナに含まれているアイテム数を取得します。
    int GetCount()
    {
        return m_nCurrentPos;
    }

    // コンテナ全体の出力サイズを取得します。
    int GetSize()
    {
        return (int)( sizeof( ToolSoundItemContainer ) + ( sizeof( item_type ) * GetCount() ) );
    }

    // バッファを取得します。
    ToolSoundItemContainer* GetBuffer()
    {
        return m_pHeader;
    }

    // 出力を開始します。
    void WriteBegin()
    {
        m_nCurrentPos = 0;
    }

    // 出力を終了します。
    void WriteEnd()
    {
        m_pHeader->offsetForData = sizeof( ToolSoundItemContainer );
        m_pHeader->itemCount     = (unsigned long)GetCount();
        m_pHeader->itemSize      = sizeof( item_type );
    }

private:
    // パラメータ
    ToolSoundItemContainer* m_pHeader;          // コンテナヘッダ
    item_type*              m_pItems;           // アイテムデータ
    u32                     m_nCurrentPos;      // ポインタ
    u32                     m_nBufferSize;      // バッファサイズ
};


} // namespace snd
} // namespace nw

#endif // NW_ITEMCONTAINER_H_
