﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/ut/ut_Delegate.h>
#include <nw/ut/os/ut_CriticalSection.h>
#include <nw/ut/os/ut_MessageQueue.h>
#include <nw/ut/os/ut_Thread.h>

namespace nw  {
namespace snd {

//---------------------------------------------------------------------------
//! @brief タスクを処理するスレッドです。
//---------------------------------------------------------------------------
class TaskThread
{
public:
    typedef ut::IDelegate0<void> Task;

private:
    static const u32 STACK_SIZE     = 4 * 1024;
    static const u32 MAX_TASK_COUNT = 128;

private:
    //---------------------------------------------------------------------------
    //! @brief タスクスレッドのメイン関数オブジェクトです。
    //---------------------------------------------------------------------------
    class MainHandler : public nw::ut::ThreadHandler
    {
    public:
        MainHandler(TaskThread& owner) :
          m_Owner(owner),
          m_IsRunning(false),
          m_IsExiting(true) { }

        bool IsRunning()
        {
            return m_IsRunning;
        }

        bool IsExiting()
        {
            return m_IsExiting;
        }

        void Reset()
        {
            m_IsRunning = false;
            m_IsExiting = false;
        }

        void Exit()
        {
            m_IsExiting = true;
        }

        /*override*/ void ThreadHandlerProc();

    private:
        TaskThread& m_Owner;
        bool m_IsRunning;
        bool m_IsExiting;
    };

    //---------------------------------------------------------------------------
    //! @brief タスクスレッドの停止用タスクです。
    //---------------------------------------------------------------------------
    class StopTask : public Task
    {
    public:
        StopTask(TaskThread& thread) : m_Thread(thread) { }

        /*override*/ void Invoke()
        {
            m_Thread.m_MainHandler.Exit();
        }

    private:
        TaskThread& m_Thread;
    };

public:
    //---------------------------------------------------------------------------
    //! @brief       コンストラクタです。
    //---------------------------------------------------------------------------
    /* ctor */ TaskThread();

    //---------------------------------------------------------------------------
    //! @brief       デストラクタです。
    //---------------------------------------------------------------------------
    /* dtor */ ~TaskThread()
    {
    }

public:
    //---------------------------------------------------------------------------
    //! @brief       Task のスレッドを開始します。
    //!
    //! @param[in]  priority スレッドのプライオリティ
    //---------------------------------------------------------------------------
    void Start(s32 priority);

    //---------------------------------------------------------------------------
    //! @brief       Task のスレッドを停止します。
    //!
    //! @details     残りタスクを処理してからスレッドを停止します。
    //!              この関数はスレッド終了の完了を待たずに制御を返します。
    //---------------------------------------------------------------------------
    void Stop();

    //---------------------------------------------------------------------------
    //! @brief       Task のスレッドをすぐに停止します。
    //!
    //! @details     残りタスクを処理せず、すぐにスレッドを停止します。
    //!              この関数はスレッド終了が完了するまで制御を返しません。
    //---------------------------------------------------------------------------
    void StopImmediately();

    //---------------------------------------------------------------------------
    //! @brief       Task::Stop() による残りタスク処理が完了するまで待機します。
    //---------------------------------------------------------------------------
    void WaitStop();

    //---------------------------------------------------------------------------
    //! @brief       Taskオブジェクトを追加
    //!
    //! @param[in]  task    追加するTaskオブジェクト
    //---------------------------------------------------------------------------
    void RegisterTask(Task& task);

    //---------------------------------------------------------------------------
    //! @brief       スレッドの実行状態を取得します。
    //!
    //! @return      スレッドが実行中の場合は true、それ以外の場合は false を返します。
    //---------------------------------------------------------------------------
    bool IsRunning()
    {
        return m_MainHandler.IsRunning();
    }

    //---------------------------------------------------------------------------
    //! @brief       スレッドが停止途中かどうかを取得します。
    //!
    //! @return      スレッドが停止途中の場合は true、それ以外の場合は false を返します。
    //---------------------------------------------------------------------------
    bool IsStopping()
    {
        return m_IsStopping || m_MainHandler.IsExiting();
    }

private:
    //---------------------------------------------------------------------------
    //! @brief       タスクリストから先頭のタスクを取り出します。
    //!
    //! @param[in]  isBlocking ブロッキングの有無
    //---------------------------------------------------------------------------
    Task* PopTask(bool isBlocking);

    //! @brief スレッドの終了処理を行います。
    void OnExitThread();

private:
    MainHandler    m_MainHandler;                    //!< タスクスレッドハンドラ
    nw::ut::Thread m_TaskThread;                     //!< バックグラウンドスレッドオブジェクト
    u8             m_TaskThreadStack[STACK_SIZE];    //!< スタックサイズは固定
    StopTask       m_StopTask;                       //!< 停止タスク
    bool           m_IsStopping;                     //!< スレッド停止中フラグ

    nw::ut::MessageQueue m_MessageQueue;
    nw::ut::MessageQueue::BufferType m_OSMessages[MAX_TASK_COUNT];

    nw::ut::CriticalSection m_stateLock;            //!< TaskThread の状態を保護するロックです。
};


} // namespace snd
} // namespace nw
