﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SmfParser.h"
#include "smfconv.h"


using namespace std;
using namespace sndlib;


//////////////////////////////////////////////////////////////////////
// 構築/消滅
//////////////////////////////////////////////////////////////////////

SmfParser::SmfParser()
{
}

SmfParser::~SmfParser()
{
}

//////////////////////////////////////////////////////////////////////
// ヘッダ解析
//////////////////////////////////////////////////////////////////////

void SmfParser::parseHeader(strm::fstream& in)
{
    strm::char4 chunkType;
    u32 chunkSize;

    in >> chunkType >> chunkSize;

    if (chunkType != 'MThd') {
        throw ParseException("Not SMF file");
    }
    if (chunkSize < 6) {
        ostringstream ostm;
        ostm << "Too small MThd chunkSize - " << chunkSize;
        throw ParseException(ostm.rdbuf()->str());
    }

    u16 division;
    in >> mFormat >> mTracks >> division;
    if (mFormat != 0 && mFormat != 1) {
        ostringstream ostm;
        ostm << "Unsupported Format - " << mFormat;
        throw ParseException(ostm.rdbuf()->str());
    }

    if (division & 0x8000)      // SMPTE
    {
        char bSMPTE = -((division >> 8) & 0xff);
        char bTicks =  ((division     ) & 0xff);
//      strTemp.Format("SMPTE: %d, Ticks: %d\r\n", bSMPTE, bTicks);

        // TODO: SMPTE
        throw ParseException("Not Support SMPTE division");
    }
    else                        // TickPerQuaterNote
    {
        mDivison = division;
    }

    // TODO: mDivison は OUTPUT_TPQN の倍数

    const byte_t data[4] = {4,2,24,8};
    mBeatStream.push_back( new BeatEvent(0, 4, data, mDivison) );

    in.seek( chunkSize - 6 , SEEK_CUR );
}


//////////////////////////////////////////////////////////////////////
// システムエクスクルーシブイベント
//////////////////////////////////////////////////////////////////////

void SmfParser::parseSysExEvent(strm::fstream& in, u8 status, TrackStatus* track)
{
    var_t len;

    in >> len;
    in.seek( len , SEEK_CUR ); // ignore
}

//////////////////////////////////////////////////////////////////////
// メタイベント
//////////////////////////////////////////////////////////////////////

void SmfParser::parseMetaEvent(
    strm::fstream& in,
    TrackStatus* track,
    const CommandLineArg& cmdInfo,
    const string& labelPrefix
)
{
    var_t len;
    u8 type;

    in >> type >> len;
    vector<unsigned char> data(len);
    if ( len > 0 ) {
        in.read( &data[0], len );
    }

    switch( type ) {
    case 0x00: // Sequence Number
        break;

    case 0x01: // Text Event
    case 0x02: // Copyright Notice
    case 0x03: // Sequence/Track Name
    case 0x04: // Instrument Name
    case 0x05: // Lyric
    case 0x06: // Marker
    case 0x07: // Cue Point
    case 0x08: case 0x09: case 0x0a: case 0x0b: // Other Text Event
    case 0x0c: case 0x0d: case 0x0e: case 0x0f: {
        string text;
        if ( len > 0 ) text.assign( reinterpret_cast<const char*>(&data[0]) , len );

        std::string::size_type pos = text.find('\0');
        if ( pos != std::string::npos ) {
            text.erase( pos );
        }

        if ( text == "[" || text == "loop_start" ) {
            for ( int ch = 0; ch < CHANNEL_MAX ; ch++ ) {
                mMidiEventStream[ ch ].push_back(
                    new TextAllEvent( track->tick, MakeTrackLabelPrefix( labelPrefix, ch ) + "_LoopStart:" ) );
            }
        }
        else if ( text == "]" || text == "loop_end" ) {
            for ( int ch = 0; ch < CHANNEL_MAX ; ch++ ) {
                mMidiEventStream[ ch ].push_back(
                    new TextAllEvent( track->tick, "    jump\t" + MakeTrackLabelPrefix( labelPrefix, ch ) + "_LoopStart" ) );
            }
        }
        else if ( text.find("text_") == 0 ) {
            string::size_type p = strlen("text_");

            if ( text.find( "all", p ) == p ) {
                p += strlen( "all" );

                while( p < len && isspace( text[p] ) ) {
                    p++;
                }

                if ( p < len && text[p] == ':' ) {
                    p++;

                    const string str = text.substr( p );
                    for( int chNo = 0; chNo < CHANNEL_MAX ; chNo++ ) {
                        string s = str;
                        string::size_type p;
                        while( ( p = s.find( "$$" ) ) != string::npos ) {
                            s.replace( p, 2, MakeLocalTrackLabelPrefix( chNo ) + "_" );
                        }
                        while( ( p = s.find( '$' ) ) != string::npos ) {
                            s.replace( p, 1, MakeTrackLabelPrefix( labelPrefix, chNo ) + "_" );
                        }
                        mMidiEventStream[ chNo ].push_back( new TextAllEvent( track->tick, s ) );
                    }
                }
                else {
                    if ( cmdInfo.verbose_flag ) {
                        cout << "smfconv: Ignored text event: " << text << endl;
                    }
                }
            }
            else if ( text.find_first_of( "0123456789", p ) == p ) {
                int chNo = 0;

                while ( p < len && isdigit( text[p] ) ) {
                    chNo *= 10;
                    chNo += text[p] - '0';
                    p++;
                }

                while( p < len && isspace( text[p] ) ) {
                    p++;
                }

                if ( p < len && text[p] == ':' ) {
                    p++;

                    const string str = text.substr( p );

                    string s = str;
                    string::size_type p;
                    while( ( p = s.find( "$$" ) ) != string::npos ) {
                        s.replace( p, 2, MakeLocalTrackLabelPrefix( chNo ) + "_" );
                    }
                    while( ( p = s.find( '$' ) ) != string::npos ) {
                        s.replace( p, 1, MakeTrackLabelPrefix( labelPrefix, chNo )+"_" );
                    }
                    mMidiEventStream[ chNo ].push_back( new TextEvent( track->tick, s ) );
                }
                else {
                    if ( cmdInfo.verbose_flag ) {
                        cout << "smfconv: Ignored text event: " << text << endl;
                    }
                }
            }
            else {
                if ( cmdInfo.verbose_flag ) {
                    cout << "smfconv: Ignored text event: " << text << endl;
                }
            }
        }
        else {
            if ( cmdInfo.verbose_flag ) {
                cout << "smfconv: Ignored text event: " << text << endl;
            }
        }
        break;
    }

    case 0x20: // MIDI Channel Prefix
        break;
    case 0x2f: // End of Track
        track->end_of_track = true;
        break;
    case 0x51: // Set Tempo
        if (len != 3) {
            throw MetaEventException("Set Tempo Event len != 3", track->tick);
        }
        mMidiEventStream[ 0 ].push_back( new TempoEvent( track->tick, len, &data[0] ) );
        break;

    case 0x54: // SMPTE Offset
        break;

    case 0x58: // Time Signature
        assert(! mBeatStream.empty() );
        if ( ! ((len == 4) || (len == 2)) ) {
            throw MetaEventException("Time Signature Event len != 4", track->tick);
        }
        mBeatStream.back()->setNext( track->tick );
        mBeatStream.push_back( new BeatEvent( track->tick, len, &data[0], mDivison ) );
        break;

    case 0x59: // Key Signature
        break;
    case 0x7f: // Sequencer Specific
        break;

    default: // Unknown
        break;
    }
}

//////////////////////////////////////////////////////////////////////
// MIDIイベント
//////////////////////////////////////////////////////////////////////

void SmfParser::parseMidiEvent(strm::fstream& in, u8 status, TrackStatus* track, const std::string& labelPrefix)
{
    u8 data1;
    u8 data2;
    // status に応じた引数の数
    static const int arg_count[8] = {
        2,2,2,2,1,1,2,0
    };

    if (status < 0x80) {
        data1 = status;
        status = track->running_status;
        if (arg_count[((status & 0xf0)>>4)-8] == 2) {
            in >> data2;
        }
    }
    else {
        track->running_status = status;
        const int argc = arg_count[((status & 0xf0)>>4)-8];
        if (argc == 1) {
            in >> data1;
        }
        else if (argc == 2) {
            in >> data1 >> data2;
        }
    }

    int channel = status & 0x0f;

    switch( status & 0xf0 ) {
    case 0x80: // Note Off Event
        eventNoteOff(channel, data1, data2, track);
        break;
    case 0x90:
        if ( data2 != 0 ) {
            eventNoteOn(channel, data1, data2, track);
        }
        else {
            eventNoteOff(channel, data1, 64, track);
        }
        break;
    case 0xa0: // Poly Key Pressure
        break;
    case 0xb0:
        if (data1 < 120) { // Control Change
            eventControlChange(channel, data1, data2, track, labelPrefix );
        }
        else { // Channel Mode Message
            switch( data1 ) {
            case 120: // All Sound Off
            case 121: // Reset All Controller
            case 122: // Local Control
            case 123: // All Note Off
            case 124: // Omni Off
            case 125: // Omni On
            case 126: // Mono Mode On
            case 127: // Poly Mode On
                break;
            }
        }
        break;
    case 0xc0: // Program Change
        eventProgramChange(channel, data1, track);
        break;

    case 0xd0: // Channel Pressure
        break;

    case 0xe0: // Pitch Bend
        short value = ( ( static_cast<short>(data2) << 7 ) | data1 ) - 0x2000;
        eventPitchBend(channel, value, track);
        break;
    }

    if ( track->tick > mMidiEventStream[channel].lastTick )
    {
        mMidiEventStream[channel].lastTick = track->tick;
    }
}

void SmfParser::eventNoteOn( int channel, int key, int velocity, TrackStatus* track )
{
    NoteEvent* e = new NoteEvent( track->tick, key, velocity );
    mMidiEventStream[ channel ].push_back( e );

    NoteEventMap::iterator p =
        track->note_event_map[ channel ].find( key );

    if ( p != track->note_event_map[ channel ].end() )
    {
        p->second.push_back( e );
    }
    else
    {
        track->note_event_map[ channel ][ key ].push_back( e ) ;
    }
}

void SmfParser::eventNoteOff(int channel, int key, int velocity, TrackStatus* track)
{
    NoteEventMap::iterator p =
        track->note_event_map[channel].find(key);
    if (p == track->note_event_map[channel].end()) {
        throw ChannelEventException("Unmatch Note Off Event", channel, track->tick);
    }
    p->second.front()->noteOff(track->tick);
    p->second.pop_front();
    if ( p->second.empty() ) {
        track->note_event_map[channel].erase(p);
    }
}

void SmfParser::eventControlChange(int channel, int control, int value, TrackStatus* track, const string& labelPrefix )
{
    switch( control )
    {
    case 6: // data entry
        mMidiEventStream[channel].push_back(
            new RpnNrpnEvent(
                track->tick,
                track->rpn_flag? RpnNrpnEvent::RPN: RpnNrpnEvent::NRPN,
                track->rpn_flag? track->rpnMSB: track->nrpnMSB,
                track->rpn_flag? track->rpnLSB: track->nrpnLSB,
                value
            )
        );
        break;
    case 6+32: // data entry MSB
        // 未対応（何もしない）
        break;
    case 98: // NRPN LSB
        track->nrpnLSB = value;
        track->rpn_flag = false;
        break;
    case 99: // NRPN MSB
        track->nrpnMSB = value;
        track->rpn_flag = false;
        break;
    case 100: // RPN LSB
        track->rpnLSB = value;
        track->rpn_flag = true;
        break;
    case 101: // RPN MSB
        track->rpnMSB = value;
        track->rpn_flag = true;
        break;
    default:
        mMidiEventStream[channel].push_back(
            new ControlChangeEvent(
                track->tick,
                control,
                value,
                MakeTrackLabelPrefix( labelPrefix, channel )
            )
        );
        break;
    }
}

void SmfParser::eventProgramChange(int channel, int program, TrackStatus* track)
{
    // TODO: なんでもかんでも登録すると、wait が分断されてしまう

    mMidiEventStream[channel].push_back( new ProgramChangeEvent(track->tick, program) );
}

void SmfParser::eventPitchBend(int channel, int value, TrackStatus* track)
{
    mMidiEventStream[channel].push_back( new PitchBendEvent(track->tick, value) );
}


//////////////////////////////////////////////////////////////////////
// メイン解析処理
//////////////////////////////////////////////////////////////////////

void SmfParser::parse( const char* filename, const CommandLineArg& cmdInfo, const string& labelPrefix )
{
    strm::fstream in;

    try {
        if (! in.open(filename, "rb") ) {
            throw ParseException("Cannot open file");
        }

        parseHeader(in);

        int nCurTrack = 0;
        while( in.peek() != EOF )
        {
            strm::char4 chunkType;
            u32 chunkSize;
            in >> chunkType >> chunkSize;

            //==========================
            // トラックチャンク
            //==========================
            if (chunkType == 'MTrk') {
                if (nCurTrack >= mTracks) {
                    throw ParseException("Too much MTrk chunk");
                }

                long end = in.tell() + chunkSize;

                var_t delta;
                u8 status;

                TrackStatus track;

                while( ! track.end_of_track )
                {
                    if ( in.tell() >= end ) {
                        ostringstream ostm;
                        ostm << "MTrk chunk parse overflow - " << nCurTrack;
                        throw ParseException(ostm.rdbuf()->str());
                    }

                    in >> delta >> status;
                    track.tick += delta;

                    if (status < 0xf0) {
                        parseMidiEvent(in, status, &track, labelPrefix );
                    }
                    else if (status == 0xf0 || status == 0xf7) {
                        parseSysExEvent(in, status, &track);
                    }
                    else if (status == 0xff) {
                        parseMetaEvent( in, &track, cmdInfo, labelPrefix );
                    }
                    else {
                        ostringstream ostm;
                        ostm << "Unexpected status - " << status;
                        throw MetaEventException(ostm.rdbuf()->str(), track.tick);
                    }
                }

                track.checkRestNoteOnEvent();

                ++nCurTrack;
            }
            else {
                in.seek( chunkSize , SEEK_CUR );
            }
        }

        if (nCurTrack < mTracks) {
            ostringstream ostm;
            ostm << "Too few MTrk chunk ( " << nCurTrack << " / " << mTracks << " )";
            throw ParseException(ostm.rdbuf()->str());
        }
    }
    catch( const strm::failure& ) {
        ostringstream ostm;
        ostm << "Unexpected EOF - " << filename;
        throw ParseException(ostm.rdbuf()->str());
    }

}


//////////////////////////////////////////////////////////////////////
// SmfParser::TrackStatus
//////////////////////////////////////////////////////////////////////

SmfParser::TrackStatus::TrackStatus()
: tick(0),
  running_status(0),
  end_of_track(false),
  rpnLSB(0x7f),
  rpnMSB(0x7f),
  nrpnLSB(0x7f),
  nrpnMSB(0x7f),
  rpn_flag(true)
{
}

bool SmfParser::TrackStatus::checkRestNoteOnEvent() const
{
    int i;
    for( i = 0; i < CHANNEL_MAX; ++i )
    {
        if (! note_event_map[i].empty())
        {
            NoteEventMap::const_iterator p = note_event_map[ i ].begin();
            NoteEvent* e = p->second.front();
            throw ChannelEventException("Unmatch Note On Event", i, e->tick);
        }
    }
    return false;
}


//////////////////////////////////////////////////////////////////////
// BeatStream
//////////////////////////////////////////////////////////////////////

void BeatStream::tickToBeat(tick_t tick, BeatInfo* info) const
{
    info->measure = 0;

    // サイズがゼロの BeatStream は p++ や *p できない
    if ( size() == 0 )
    {
        info->beat = 0;
        info->tick = tick;
        return;
    }

    list<BeatEvent*>::const_iterator p = begin();
    const BeatEvent* e = *p++;

    while( p != end() ) {
        const BeatEvent* next = *p;

        if ( tick < next->tick ) {
            break;
        }
        info->measure += e->measure;
        e = *p++;
    }

    tick -= e->tick;
    int beat = tick / e->division;
    tick -= beat * e->division;
    int measure = beat / e->numerator;
    beat -= measure * e->numerator;

    info->measure += measure;
    info->beat = beat;
    info->tick = tick;
}

tick_t BeatStream::measureToTick(unsigned int measure) const
{
    tick_t tick = 0;

    list<BeatEvent*>::const_iterator p = begin();

    while( p != end() ) {
        const BeatEvent* e = *p;

        if ( measure < e->measure ) {
            tick += measure * e->division * e->numerator;
            break;
        }

        tick += e->measure * e->division * e->numerator;
        measure -= e->measure;
        ++p;
    }

    return tick;
}

//////////////////////////////////////////////////////////////////////
// Measure
//////////////////////////////////////////////////////////////////////

bool Measure::isEqual( const Measure& rhs ) const
{
    if ( size() != rhs.size() ) return false;

    const_iterator p1, p2;
    for ( p1 = begin(), p2 = rhs.begin() ;
          p1 != end() && p2 != rhs.end() ; ++p1 , ++p2 )
    {
        if (! (*p1)->isEqual( *p2 )) return false;
    }

    return true;
}
