﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#if !defined(AFX_CMDLINE_H__1F7DEA24_379A_49A8_A97A_55C456684AE9__INCLUDED_)
#define AFX_CMDLINE_H__1F7DEA24_379A_49A8_A97A_55C456684AE9__INCLUDED_

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000

#include <exception>
#include <map>
#include <string>
#include <sstream>

namespace sndlib
{

class CommandLineArgParserException : public std::exception {
public:
    explicit CommandLineArgParserException(const std::string& _S )
        : exception(""), _Str(_S) {}
    virtual const char *what() const
        {return (_Str.c_str()); }
private:
    std::string _Str;
};

class CommandLineArgParser
{
public:
    CommandLineArgParser(){}

    void AddOption( std::string str )
    {
        if ( str.empty() ) return;

        if ( str.length() == 1 )
        {
            mCharOptMap[ str[0] ] = false;
        }
        else
        {
            mStringOptMap[ str ] = false;
        }
    }
    void AddOptionWithArg( std::string str )
    {
        if ( str.empty() ) return;

        if ( str.length() == 1 )
        {
            mCharOptMap[ str[0] ] = true;
        }
        else
        {
            mStringOptMap[ str ] = true;
        }
    }

    void Parse( int argc, char* argv[] )
    {
        std::string pre_option;
        bool next_flag = false;

        for (int i = 1 ; i < argc ; i++ )
        {
            const char* a = argv[i];

            if ( next_flag )
            {
                OptionHandler( pre_option, a );

                next_flag = false;
            }
            else
            {
                if ( a[0] == '-')
                {
                    if ( a[1] == '\0')
                    {
                        // only '-'
                        throw CommandLineArgParserException("unrecognized option \'-\'");
                    }
                    else
                    {
                        if ( a[1] == '-' )
                        {
                            // '--opt'
                            StringOptMap::const_iterator p = mStringOptMap.find( &a[2] );
                            if ( p == mStringOptMap.end() ) {
                                std::stringstream s;
                                s << "unrecognized option \'" << a <<"\'";
                                throw CommandLineArgParserException( s.str() );
                            }
                            if ( p->second ) {
                                pre_option = &a[2];
                                next_flag = true;
                            }
                            else {
                                OptionHandler( &a[2], std::string() );
                            }
                        }
                        else
                        {
                            // '-opt'
                            const char* cp = &a[1];
                            while( *cp != '\0' )
                            {
                                CharOptMap::const_iterator p = mCharOptMap.find( *cp );
                                if ( p == mCharOptMap.end() ) {
                                    std::stringstream s;
                                    s << "invalid option -- " << *cp;
                                    throw CommandLineArgParserException( s.str() );
                                }
                                if ( p->second ) {
                                    pre_option = *cp;
                                    next_flag = true;
                                }
                                else {
                                    OptionHandler( std::string(1, *cp), std::string() );
                                }

                                cp++;
                            }
                        }
                    }
                }
                else
                {
                    ArgHandler( a );
                }
            }
        }

        if ( next_flag )
        {
            std::stringstream s;
            if ( pre_option.length() == 1 ) {
                s << "option requires an argument -- " << pre_option;
            }
            else {
                s << "option '--" << pre_option << "' requires an argument";
            }
            throw CommandLineArgParserException( s.str() );
        }
    }

protected:
    virtual void OptionHandler( std::string opt, std::string next ) {}
    virtual void ArgHandler( std::string arg ) {}

private:
    typedef std::map< char, bool > CharOptMap;
    typedef std::map< std::string, bool > StringOptMap;

    CharOptMap mCharOptMap;
    StringOptMap mStringOptMap;
};

} // namespace sndlib

#endif // !defined(AFX_CMDLINE_H__1F7DEA24_379A_49A8_A97A_55C456684AE9__INCLUDED_)

