﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Xml;
using System.IO;
using System.Collections;
using NintendoWare.SoundFoundation.Legacies.FileFormat;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat
{

    public class Nw4rXmlAnimSound : Nw4rXmlFile
    {
        EventCollection events = new EventCollection();
        int frameSize = 0;

        public class Event
        {
            int startFrame;
            int endFrame;
            string soundName;
            string comment;
            double volume = 1.0;
            double pitch = 1.0;
            bool isTrigger = true;
            bool isEnabledEndFrame = false;
            UInt32 userParam = 0;

            public bool IsTriggerEvent
            {
                get { return isTrigger; }
                set { isTrigger = value; }
            }

            public bool IsEnabledEndFrame
            {
                get { return isEnabledEndFrame; }
                set { isEnabledEndFrame = value; }
            }

            public int StartFrame
            {
                get { return startFrame; }
                set { startFrame = value; }
            }
            public int EndFrame
            {
                get { return endFrame; }
                set { endFrame = value; }
            }
            public string SoundName
            {
                get { return soundName; }
                set { soundName = value; }
            }
            public string Comment
            {
                get { return comment; }
                set { comment = value; }
            }
            public int Volume
            {
                get { return (int)(volume * 127); }
                set { volume = value / 127.0f; }
            }
            public double Pitch
            {
                get { return pitch; }
                set { pitch = value; }
            }
            public UInt32 UserParam
            {
                set { userParam = value; }
                get { return userParam; }
            }

            internal static Event Parse(XmlDocument doc, XmlElement eventElem)
            {
                Event e = new Event();

                XmlElement frameElem = (XmlElement)eventElem.SelectSingleNode("frame");
                if (frameElem != null)
                {
                    if (frameElem.HasAttribute("type"))
                    {
                        if (frameElem.GetAttribute("type") == "trigger")
                        {
                            e.isTrigger = true;
                        }
                        else if (frameElem.GetAttribute("type") == "range")
                        {
                            e.isTrigger = false;
                        }
                    }
                    XmlElement startFrameElem = (XmlElement)frameElem.SelectSingleNode("start");
                    if (startFrameElem != null)
                    {
                        if (startFrameElem.HasAttribute("type"))
                        {
                            if (startFrameElem.GetAttribute("type") == "frame")
                            {
                                e.startFrame = Int32.Parse(startFrameElem.InnerText);
                            }
                            else if (startFrameElem.GetAttribute("type") == "percent")
                            {
                                // TODO:
                            }
                        }
                        e.volume = Double.Parse(startFrameElem.InnerText);
                    }

                    XmlElement endFrameElem = (XmlElement)frameElem.SelectSingleNode("end");
                    if (endFrameElem != null)
                    {
                        e.IsEnabledEndFrame = true;
                        if (endFrameElem.HasAttribute("type"))
                        {
                            if (endFrameElem.GetAttribute("type") == "frame")
                            {
                                e.endFrame = Int32.Parse(endFrameElem.InnerText);
                            }
                            else if (endFrameElem.GetAttribute("type") == "percent")
                            {
                                // TODO:
                            }
                        }
                        e.volume = Double.Parse(startFrameElem.InnerText);
                    }
                }

                XmlElement soundElem = (XmlElement)eventElem.SelectSingleNode("sound");
                if (soundElem != null)
                {
                    XmlElement soundIdElem = (XmlElement)soundElem.SelectSingleNode("id");
                    if (soundIdElem != null)
                    {
                        if (soundIdElem.HasAttribute("type"))
                        {
                            if (soundIdElem.GetAttribute("type") == "name")
                            {
                                e.soundName = soundIdElem.InnerText;
                            }
                        }
                    }
                }

                XmlElement commentElem = (XmlElement)eventElem.SelectSingleNode("comment");
                if (commentElem != null)
                {
                    e.comment = commentElem.InnerText;
                }
                XmlElement volumeElem = (XmlElement)eventElem.SelectSingleNode("volume");
                if (volumeElem != null)
                {
                    e.volume = Double.Parse(volumeElem.InnerText);
                }
                XmlElement pitchElem = (XmlElement)eventElem.SelectSingleNode("pitch");
                if (pitchElem != null)
                {
                    e.pitch = Double.Parse(pitchElem.InnerText);
                }
                XmlElement userParamElem = (XmlElement)eventElem.SelectSingleNode("user_param");
                if (userParamElem != null)
                {
                    e.userParam = UInt32.Parse(userParamElem.InnerText);
                }

                return e;
            }
            internal XmlElement ToXmlElement(XmlDocument doc)
            {
                XmlElement eventElem = doc.CreateElement("event");

                XmlElement frameElem = doc.CreateElement("frame");
                if (isTrigger) frameElem.SetAttribute("type", "trigger");
                else frameElem.SetAttribute("type", "range");

                XmlElement startFrameElem = doc.CreateElement("start");
                startFrameElem.SetAttribute("type", "frame");
                startFrameElem.InnerText = startFrame.ToString();
                frameElem.AppendChild(startFrameElem);

                if (IsEnabledEndFrame)
                {
                    XmlElement endFrameElem = doc.CreateElement("end");
                    endFrameElem.SetAttribute("type", "frame");
                    endFrameElem.InnerText = endFrame.ToString();
                    frameElem.AppendChild(endFrameElem);
                }

                eventElem.AppendChild(frameElem);

                if (soundName != null)
                {
                    XmlElement soundElem = doc.CreateElement("sound");
                    XmlElement soundIdElem = doc.CreateElement("id");
                    soundIdElem.SetAttribute("type", "name");
                    soundIdElem.InnerText = soundName.ToString();
                    soundElem.AppendChild(soundIdElem);
                    eventElem.AppendChild(soundElem);
                }

                if (comment != null)
                {
                    XmlElement commentElem = doc.CreateElement("comment");
                    commentElem.InnerText = comment.ToString();
                    eventElem.AppendChild(commentElem);
                }

                XmlElement volumeElem = doc.CreateElement("volume");
                volumeElem.InnerText = volume.ToString();
                eventElem.AppendChild(volumeElem);

                XmlElement pitchElem = doc.CreateElement("pitch");
                pitchElem.InnerText = pitch.ToString();
                eventElem.AppendChild(pitchElem);

                XmlElement userParamElem = doc.CreateElement("user_param");
                userParamElem.InnerText = userParam.ToString();
                eventElem.AppendChild(userParamElem);

                return eventElem;
            }
        }

        public class EventCollection : IEnumerable
        {
            ArrayList itemList = new ArrayList();

            /******************************************************************************
               interface
             ******************************************************************************/
            IEnumerator IEnumerable.GetEnumerator()
            {
                return itemList.GetEnumerator();
            }

            /******************************************************************************
               public
             ******************************************************************************/
            public int Count
            {
                get { return itemList.Count; }
            }

            public void Clear()
            {
                itemList.Clear();
            }

            public void Add(Event e)
            {
                itemList.Add(e);
            }
            public void Remove(Event e)
            {
                itemList.Remove(e);
            }
        }

        public EventCollection Events
        {
            get { return events; }
            set { events = value; }
        }
        public int FrameSize
        {
            get { return frameSize; }
            set { frameSize = value; }
        }

        protected override string FileTitle { get { return "NintendoWare Animation Sound"; } }

        protected override void LoadBody(XmlDocument doc, XmlElement bodyElem)
        {
            XmlElement animSoundElem = (XmlElement)bodyElem.SelectSingleNode("anim_sound");
            if (animSoundElem != null)
            {
                XmlElement frameSizeElem = (XmlElement)animSoundElem.SelectSingleNode("frame_size");
                if (frameSizeElem != null)
                {
                    frameSize = Int32.Parse(frameSizeElem.InnerText);
                }

                XmlElement eventArrayElem = (XmlElement)animSoundElem.SelectSingleNode("event_array");
                if (eventArrayElem != null)
                {
                    foreach (XmlElement eventElem in eventArrayElem.SelectNodes("event"))
                    {
                        Event e = Event.Parse(doc, eventElem);
                        if (e != null)
                        {
                            events.Add(e);
                        }
                    }
                }
            }
        }

        protected override void SaveBody(XmlDocument doc, XmlElement bodyElem)
        {
            XmlElement animSoundElem = doc.CreateElement("anim_sound");

            XmlElement frameSizeElem = doc.CreateElement("frame_size");
            frameSizeElem.InnerText = frameSize.ToString();
            animSoundElem.AppendChild(frameSizeElem);

            if (events.Count > 0)
            {
                XmlElement eventArrayElem = doc.CreateElement("event_array");
                eventArrayElem.SetAttribute("size", events.Count.ToString());
                foreach (Event e in events)
                {
                    XmlElement eventElem = e.ToXmlElement(doc);
                    eventArrayElem.AppendChild(eventElem);
                }

                animSoundElem.AppendChild(eventArrayElem);
            }
            bodyElem.AppendChild(animSoundElem);
        }
    }



}

