﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat
{
    #region ** シーケンスサウンドファイル

    internal class Nw4rSequenceSoundFile : Nw4rFileRoot
    {
        public Nw4rSequenceSoundFile() : base("SequenceSoundFileRoot") { }

        #region ** メソッド

        public static Nw4rSequenceSoundFile FromFile(string fileName)
        {
            if (null == fileName) { throw new ArgumentNullException("fileName"); }

            Nw4rSequenceSoundFile root = new Nw4rSequenceSoundFile();
            root.LoadFile(fileName);

            return root;
        }

        protected void LoadFile(string fileName)
        {
            if (null == fileName) { throw new ArgumentNullException("fileName"); }

            using (FileStream stream = File.OpenRead(fileName))
            {
                Read(BinaryReaderBigEndian.CreateInstance(stream));
            }
        }

        protected override Nw4rFileDataBlock ReadDataBlock(BinaryReader reader)
        {
            Nw4rFileDataBlockHeader header = new Nw4rFileDataBlockHeader();
            header.Read(reader);


            Nw4rFileDataBlock newBlock = null;

            switch (header.Kind)
            {
                case "DATA":
                    newBlock = new Nw4rSequenceDataBlock(header);
                    break;

                case "LABL":
                    newBlock = new Nw4rSequenceLabelBlock(header);
                    break;

                default:
                    newBlock = new Nw4rFileBinaryDataBlock("UnknownBlock", header);
                    break;
            }

            newBlock.ReadBody(reader);

            return newBlock;
        }

        #endregion
    }

    #region ** 各ブロック

    internal class Nw4rSequenceDataBlock : Nw4rFileBinaryDataBlock
    {
        public static readonly string NodeName = "DataBlock";

        public Nw4rSequenceDataBlock(Nw4rFileDataBlockHeader header) : base(NodeName, header, 32) { }
    }

    internal class Nw4rSequenceLabelBlock : Nw4rFileDataBlock
    {
        public static readonly string NodeName = "LabelBlock";

        public Nw4rSequenceLabelBlock(Nw4rFileDataBlockHeader header) : base(NodeName, header) { }

        #region ** イベントハンドラ

        protected override void OnReadBody(BinaryReader reader)
        {
            Nw4rSequenceLabelBlockBody body = new Nw4rSequenceLabelBlockBody();
            body.Read(reader);

            AddNode(body);
        }

        #endregion

        #region ** メソッド

        public Nw4rSequenceLabelData FindLabelData(string label)
        {
            return (ChildNodes[Nw4rSequenceLabelBlockBody.NodeName] as Nw4rSequenceLabelBlockBody).FindLabelData(label);
        }

        #endregion
    }

    internal class Nw4rSequenceLabelBlockBody : Nw4rFileDataBlockBody
    {
        private Nw4rFileInnerOffsetTable _table = new Nw4rFileInnerOffsetTable();

        public Nw4rSequenceLabelBlockBody() { }

        #region ** イベントハンドラ

        protected override void OnReadBinary(BinaryReader reader)
        {
            _table.Read(reader);

            foreach (Nw4rFileInnerOffsetTableItem item in _table.Items)
            {

                reader.BaseStream.Position = Offset + item.Offset;

                Nw4rSequenceLabelData newLabelData = new Nw4rSequenceLabelData();
                newLabelData.Read(reader);

                AddNode(newLabelData);

                item.Data = newLabelData;

            }

            new Nw4rByteAligner(reader.BaseStream).SeekForward(32);
        }

        #endregion

        #region ** メソッド

        public Nw4rSequenceLabelData FindLabelData(string label)
        {
            if (null == label) { throw new ArgumentNullException("label"); }

            foreach (Nw4rSequenceLabelData labelData in ChildNodes)
            {

                if (labelData.Label == label)
                {
                    return labelData;
                }

            }

            return null;
        }

        #endregion
    }

    internal class Nw4rSequenceLabelData : Nw4rFileNode
    {
        private long _dataOffset = 0;
        private string _label = string.Empty;

        public Nw4rSequenceLabelData() : base("LabelData") { }

        #region ** プロパティ

        public long DataOffset
        {
            get { return _dataOffset; }
        }

        public string Label
        {
            get { return _label; }
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnReadBinary(BinaryReader reader)
        {
            // シーケンスデータへのオフセット
            _dataOffset = reader.ReadUInt32();

            // ラベルの長さ
            int labelLength = (int)reader.ReadUInt32();

            if (0 == labelLength)
            {
                _label = string.Empty;
            }
            else
            {
                _label = new string(reader.ReadChars(labelLength));
            }

            new Nw4rByteAligner(reader.BaseStream).SeekForward(4);
        }

        #endregion
    }

    #endregion

    #endregion
}
