﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Text;
using System.Collections;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat
{

    /// <summary>
    /// ExprParser の概要の説明です。
    /// </summary>
    public class ExprParser
    {
        public delegate bool ParseSymbolCallback(string symbol, out long val);

        public ExprParser()
        {
        }

        public long ParseTerm(Lexer lexer, ParseSymbolCallback symbolParser)
        {
            LexerToken token = lexer.ReadToken();
            long x;

            switch (token)
            {
                case LexerToken.Number:
                    return lexer.NumberValue;

                case LexerToken.Minus:
                    return -ParseTerm(lexer, symbolParser);

                case LexerToken.Plus:
                    return +ParseTerm(lexer, symbolParser);

                case LexerToken.LeftParen:
                    x = ParseOr(lexer, symbolParser);
                    token = lexer.ReadToken();
                    if (token != LexerToken.RightParen)
                    {
                        throw new LexerException("Unmatch ( )", lexer);
                    }
                    return x;

                case LexerToken.Symbol:
                    string symbol = lexer.StringValue;
                    if (symbolParser == null || !symbolParser(symbol, out x))
                    {
                        throw new LexerException("Undefined symbol \"" + symbol + "\"", lexer);
                    }
                    return x;

                default:
                    throw new LexerException("Syntax error \"" + lexer.LastTokenString + "\"", lexer);
            }
        }

        public long ParseMul(Lexer lexer, ParseSymbolCallback symbolParser)
        {
            long left = ParseTerm(lexer, symbolParser);

            while (true)
            {
                LexerToken token = lexer.PeekToken();

                switch (token)
                {
                    case LexerToken.Mul:
                        lexer.ReadToken(); // eat
                        left *= ParseTerm(lexer, symbolParser);
                        break;
                    case LexerToken.Div:
                        lexer.ReadToken(); // eat
                        long d = ParseTerm(lexer, symbolParser);
                        if (d == 0)
                        {
                            throw new LexerException("Devide by zero", lexer);
                        }
                        left /= d;
                        break;

                    default:
                        return left;
                }
            }
        }

        public long ParsePlus(Lexer lexer, ParseSymbolCallback symbolParser)
        {
            long left = ParseMul(lexer, symbolParser);

            while (true)
            {
                LexerToken token = lexer.PeekToken();

                switch (token)
                {
                    case LexerToken.Plus:
                        lexer.ReadToken(); // eat
                        left += ParseMul(lexer, symbolParser);
                        break;
                    case LexerToken.Minus:
                        lexer.ReadToken(); // eat
                        left -= ParseMul(lexer, symbolParser);
                        break;

                    default:
                        return left;
                }
            }
        }

        public long ParseShift(Lexer lexer, ParseSymbolCallback symbolParser)
        {
            long left = ParsePlus(lexer, symbolParser);

            while (true)
            {
                LexerToken token = lexer.PeekToken();

                switch (token)
                {
                    case LexerToken.LeftShift:
                        lexer.ReadToken(); // eat
                        left <<= (int)ParsePlus(lexer, symbolParser);
                        break;
                    case LexerToken.RightShift:
                        lexer.ReadToken(); // eat
                        left >>= (int)ParsePlus(lexer, symbolParser);
                        break;

                    default:
                        return left;
                }
            }
        }

        public long ParseCompare(Lexer lexer, ParseSymbolCallback symbolParser)
        {
            long left = ParseShift(lexer, symbolParser);

            while (true)
            {
                LexerToken token = lexer.PeekToken();

                switch (token)
                {
                    case LexerToken.LessThan:
                        lexer.ReadToken(); // eat
                        left = Convert.ToInt64(left < ParseShift(lexer, symbolParser));
                        break;
                    case LexerToken.LessEqual:
                        lexer.ReadToken(); // eat
                        left = Convert.ToInt64(left <= ParseShift(lexer, symbolParser));
                        break;
                    case LexerToken.GreaterThan:
                        lexer.ReadToken(); // eat
                        left = Convert.ToInt64(left > ParseShift(lexer, symbolParser));
                        break;
                    case LexerToken.GreaterEqual:
                        lexer.ReadToken(); // eat
                        left = Convert.ToInt64(left >= ParseShift(lexer, symbolParser));
                        break;

                    default:
                        return left;
                }
            }
        }

        public long ParseEqual(Lexer lexer, ParseSymbolCallback symbolParser)
        {
            long left = ParseCompare(lexer, symbolParser);

            while (true)
            {
                LexerToken token = lexer.PeekToken();

                switch (token)
                {
                    case LexerToken.Equal:
                        lexer.ReadToken(); // eat
                        left = (left == ParseCompare(lexer, symbolParser)) ? 1 : 0;
                        break;

                    default:
                        return left;
                }
            }
        }

        public long ParseAnd(Lexer lexer, ParseSymbolCallback symbolParser)
        {
            long left = ParseEqual(lexer, symbolParser);

            while (true)
            {
                LexerToken token = lexer.PeekToken();

                switch (token)
                {
                    case LexerToken.And:
                        lexer.ReadToken(); // eat
                        left &= ParseEqual(lexer, symbolParser);
                        break;

                    default:
                        return left;
                }
            }
        }

        public long ParseOr(Lexer lexer, ParseSymbolCallback symbolParser)
        {
            long left = ParseAnd(lexer, symbolParser);

            while (true)
            {
                LexerToken token = lexer.PeekToken();

                switch (token)
                {
                    case LexerToken.Or:
                        lexer.ReadToken(); // eat
                        left |= ParseAnd(lexer, symbolParser);
                        break;

                    default:
                        return left;
                }
            }
        }

        public long Parse(Lexer lexer)
        {
            return ParseOr(lexer, null);
        }
        public long Parse(Lexer lexer, ParseSymbolCallback symbolParser)
        {
            return ParseOr(lexer, symbolParser);
        }
    }


}

