﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Runtime.InteropServices;

namespace Nintendo.Sound.Driver
{
/// <summary>
/// Nw4rSoundDriver の概要の説明です。
/// </summary>
public class Nw4rSoundDriver
{
    delegate int ReadInstDataCallback( IntPtr bank, int prgNo, int key, out SNDInstData inst );
    public delegate bool BankCallback( int prgNo, int key, out SNDInstData inst );

    public enum SNDInstType : byte
    {
        Invalid = 0,
            Pcm = 1,
            Psg = 2,
          Noise = 3,
      DirectPcm = 4,
           Null = 5,

        DrumSet = 0x10,
       KeySplit = 0x11
                  }

    [StructLayout(LayoutKind.Sequential)]
    struct SNDBinaryFileHeader {
        public UInt32 signature;
        public UInt16 byteOrder;
        public UInt16 version;
        public UInt32 fileSize;
        public UInt16 headerSize;
        public UInt16 dataBlocks;
    }
    [StructLayout(LayoutKind.Sequential)]
    struct SNDBinaryBlockHeader {
        public UInt32 kind;
        public UInt32 size;
    }
    [StructLayout(LayoutKind.Sequential)]
    struct SNDWaveArcLink {
        public UInt32 waveArc;
        public UInt32 next;
    }
    [StructLayout(LayoutKind.Sequential)]
    struct SNDBankData {
        public SNDBinaryFileHeader fileHeader;
        public SNDBinaryBlockHeader blockHeader;
        public SNDWaveArcLink waveArcLink1;
        public SNDWaveArcLink waveArcLink2;
        public SNDWaveArcLink waveArcLink3;
        public SNDWaveArcLink waveArcLink4;
        public UInt32 instCount;
        public UInt32 instOffsets;
    }
    [StructLayout(LayoutKind.Sequential)]
    struct SNDBankDataCallback {
        public SNDBinaryFileHeader fileHeader;
        public SNDBinaryBlockHeader blockHeader;
        public UInt32 arg;
        public ReadInstDataCallback readInstDataFunc;
    }
    [StructLayout(LayoutKind.Sequential)]
    public struct SNDInstData {
        public SNDInstType type;
        public byte padding_;
        public SNDInstParam param;
    }
    [StructLayout(LayoutKind.Sequential)]
    public struct SNDInstParam {
        public UInt16 wave1;
        public UInt16 wave2;
        public byte original_key;
        public byte attack;
        public byte decay;
        public byte sustain;
        public byte release;
        public byte pan;
    }

    [DllImport("SoundDriver.dll")]
    extern static void Nw4rSoundDriverLock();
    [DllImport("SoundDriver.dll")]
    extern static void Nw4rSoundDriverUnlock();
    [DllImport("SoundDriver.dll")]
    extern static void Nw4rSoundDriverReset();
    [DllImport("SoundDriver.dll")]
    extern static bool MidiInIsActive( UInt32 msec );
    [DllImport("SoundDriver.dll")]
    extern static void MidiPlayerSendMessage( byte status, byte data1, byte data2 );
    [DllImport("SoundDriver.dll")]
    extern static void MidiPlayerProgramChange( int channel, int prgNo );
    [DllImport("SoundDriver.dll")]
    extern static void MidiPlayerStartChannel( ref SNDInstData inst, int velocity );
    [DllImport("SoundDriver.dll")]
    extern static void MidiPlayerStopChannel();

    [DllImport("SoundDriver.dll")]
    extern static void NW4R_AIInit();
    [DllImport("SoundDriver.dll")]
    extern static void NW4R_AXInit();
    [DllImport("SoundDriver.dll")]
    extern static void NW4R_InitSoundSystem( Int32 soundThreadPrio, Int32 dvdThreadPrio );

    static Nw4rSoundDriver sInstance = new Nw4rSoundDriver();

    ReadInstDataCallback readInstDataCallback;
    SNDBankDataCallback bankCallback;
    BankCallback userCallback;
    IntPtr umBankCallback;

    public static Nw4rSoundDriver Instance
    {
        get { return sInstance; }
    }

    private Nw4rSoundDriver()
    {
        readInstDataCallback = new ReadInstDataCallback( ReadInstDataProc );
        userCallback = null;

        bankCallback = new SNDBankDataCallback();
        bankCallback.fileHeader.signature = ( ('S' << 0) | ('B' << 8) | ('C' << 16) | ('B' << 24) );
        bankCallback.fileHeader.byteOrder = 0xfeff;
        bankCallback.fileHeader.version = 0;
        bankCallback.fileHeader.headerSize = 0x10;
        bankCallback.fileHeader.dataBlocks = 1;
        bankCallback.blockHeader.kind = ( ('D' << 0) | ('A' << 8) | ('T' << 16) | ('A' << 24) );
        bankCallback.blockHeader.size = 0x10;

        umBankCallback = Marshal.AllocCoTaskMem( Marshal.SizeOf( bankCallback ) );
    }

    public void InitSoundSystem( Int32 soundThreadPrio, Int32 dvdThreadPrio )
    {
        NW4R_InitSoundSystem( soundThreadPrio, dvdThreadPrio );
    }
    public void AXInit()
    {
        NW4R_AXInit();
    }
    public void AIInit()
    {
        NW4R_AIInit();
    }

    public void Lock()
    {
        Nw4rSoundDriverLock();
    }
    public void Unlock()
    {
        Nw4rSoundDriverUnlock();
    }
    public void Reset()
    {
        Nw4rSoundDriverReset();
    }
    public int GetActiveChannelCount()
    {
        return 0; // TODO
    }
    public bool IsMidiInActive( UInt32 msec )
    {
        return MidiInIsActive( msec );
    }
    public void SendMidiMessage( byte status, byte data1, byte data2 )
    {
        MidiPlayerSendMessage( status, data1, data2 );
    }
    public void NoteOn( int prgNo, int key, int velocity )
    {
        MidiPlayerProgramChange( 0, prgNo );
        MidiPlayerSendMessage( 0x90, (byte)key, (byte)velocity );
    }
    public void NoteOff( int key )
    {
        MidiPlayerSendMessage( 0x80, (byte)key, 0 );
    }
    public void StartChannel( ref SNDInstData inst, int velocity )
    {
        MidiPlayerStartChannel( ref inst, velocity );
    }
    public void StopChannel()
    {
        MidiPlayerStopChannel();
    }
    public void SetBankCallback( BankCallback func )
    {
        // TODO
    }
    int ReadInstDataProc( IntPtr bank, int prgNo, int key, out SNDInstData inst )
    {
        if ( userCallback == null ) {
            inst = new SNDInstData();
            return 0;
        }
        Lock();
        int ret = userCallback( prgNo, key, out inst ) ? 1 : 0;
        Unlock();
        return ret;
    }
}
}
