﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Collections;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat
{

    public class CommandLineArgParserException : SoundException
    {
        public CommandLineArgParserException() { }
        public CommandLineArgParserException(string s) : base(s) { }
    }

    public class CommandLineArgParser
    {
        Hashtable optCharTable = new Hashtable();
        Hashtable optStringTable = new Hashtable();

        public delegate void OptionHandler(string opt);
        public delegate void OptionWithArgHandler(string opt, string arg);
        struct OptInfo
        {
            OptionHandler handler;
            OptionWithArgHandler withArgHandler;

            public bool WithArg
            {
                get { return withArgHandler != null; }
            }
            public OptInfo(OptionHandler handler)
            {
                this.handler = handler;
                this.withArgHandler = null;
            }
            public OptInfo(OptionWithArgHandler handler)
            {
                this.handler = null;
                this.withArgHandler = handler;
            }
            public void CallHandler(char opt)
            {
                CallHandler(opt.ToString(), null);
            }
            public void CallHandler(string opt)
            {
                CallHandler(opt, null);
            }
            public void CallHandler(char opt, string arg)
            {
                CallHandler(opt.ToString(), arg);
            }
            public void CallHandler(string opt, string arg)
            {
                if (handler != null) handler(opt);
                if (withArgHandler != null) withArgHandler(opt, arg);
            }
        }

        public CommandLineArgParser() { }

        void AddOption(string opt, OptInfo info)
        {
            if (opt.Length == 1)
            {
                optCharTable[opt[0]] = info;
            }
            else
            {
                optStringTable[opt] = info;
            }
        }

        public void AddOption(string opt, OptionHandler handler)
        {
            OptInfo info = new OptInfo(handler);
            AddOption(opt, info);
        }
        public void AddOption(string opt, string opt2, OptionHandler handler)
        {
            OptInfo info = new OptInfo(handler);
            AddOption(opt, info);
            AddOption(opt2, info);
        }
        public void AddOptionWithArg(string opt, OptionWithArgHandler handler)
        {
            OptInfo info = new OptInfo(handler);
            AddOption(opt, info);
        }
        public void AddOptionWithArg(string opt, string opt2, OptionWithArgHandler handler)
        {
            OptInfo info = new OptInfo(handler);
            AddOption(opt, info);
            AddOption(opt2, info);
        }

        void ParseCharOption(ref string arg)
        {
            char opt = arg[0];

            if (!optCharTable.Contains(opt))
            {
                throw new CommandLineArgParserException(String.Format(
                                                        "invalid option -- {0}", opt));
            }

            OptInfo info = (OptInfo)optCharTable[opt];
            if (info.WithArg)
            {
                // 引数要
                if (arg.Length < 2)
                {
                    throw new CommandLineArgParserException(String.Format(
                                                            "option requires an argument -- {0}", opt));
                }
                string optArg = arg.Substring(1, arg.Length);

                info.CallHandler(opt, optArg);

                arg = String.Empty;
            }
            else
            {
                info.CallHandler(opt);

                arg = arg.Substring(1);
            }
        }

        public string[] Parse(string[] args)
        {
            ArrayList argList = new ArrayList();

            for (int i = 0; i < args.Length; i++)
            {
                string arg = args[i];

                if (arg[0] == '-')
                {
                    if (arg.Length == 1)
                    {
                        // only '-'
                        // オプションでないものとして扱う
                        argList.Add(arg);
                    }
                    else
                    {
                        if (arg[1] == '-')
                        {
                            // '--opt'

                            int eqIndex = arg.IndexOf('=');
                            bool withArg = (eqIndex >= 0);
                            int optEnd = withArg ? eqIndex : arg.Length;
                            string opt = arg.Substring(2, optEnd - 2);
                            string optArg = withArg ? arg.Substring(optEnd + 1, arg.Length - (optEnd + 1)) : null;

                            if (!optStringTable.Contains(opt))
                            {
                                throw new CommandLineArgParserException(String.Format(
                                                                        "unrecognized option \'{0}\'", arg));
                            }

                            OptInfo info = (OptInfo)optStringTable[opt];
                            if (info.WithArg)
                            {
                                // 引数要
                                if (!withArg)
                                {
                                    i++;
                                    if (i > args.Length - 1)
                                    {
                                        throw new CommandLineArgParserException(String.Format(
                                                                                "option \'{0}\' requires an argument", arg));
                                    }
                                    withArg = true;
                                    optArg = args[i];
                                }
                            }
                            else
                            {
                                // 引数不要
                                if (withArg)
                                {
                                    throw new CommandLineArgParserException(String.Format(
                                                                            "option \'{0}\' doesn\'t allow an argument", arg));
                                }
                            }

                            info.CallHandler(opt, optArg);
                        }
                        else
                        {
                            // '-opt'

                            char opt = arg[1];

                            if (!optCharTable.Contains(opt))
                            {
                                throw new CommandLineArgParserException(String.Format(
                                                                        "invalid option -- {0}", opt));
                            }
                            OptInfo info = (OptInfo)optCharTable[opt];
                            if (info.WithArg)
                            {
                                // 引数要
                                string optArg;
                                if (arg.Length < 3)
                                {
                                    i++;
                                    if (i > args.Length - 1)
                                    {
                                        throw new CommandLineArgParserException(String.Format(
                                                                                     "option requires an argument -- {0}", opt));
                                    }
                                    optArg = args[i];
                                }
                                else
                                {
                                    optArg = arg.Substring(2, arg.Length - 2);
                                }

                                info.CallHandler(opt, optArg);
                            }
                            else
                            {
                                string optStr = arg.Substring(1);
                                while (optStr != String.Empty)
                                {
                                    ParseCharOption(ref optStr);
                                }
                            }
                        }
                    }
                }
                else
                {
                    argList.Add(arg);
                }
            }

            string[] argArray = new string[argList.Count];
            int index = 0;
            foreach (string arg in argList)
            {
                argArray[index] = arg;
                index++;
            }

            return argArray;
        }
    }

}

