﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.IO;
    using System.Xml;
    using System.Xml.Serialization;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareIntermediate;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// CTRサウンドセットドキュメントの読み込みをサポートします。
    /// </summary>
    public class SoundSetDocumentReaderCtr : SoundSetDocumentReader
    {
        private SoundSetXml2ModelTranslatorCtr translator;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public SoundSetDocumentReaderCtr(ObjectAggregateFactory<Type, Component> componentFactory)
            : base(componentFactory)
        {
            this.DocumentTypeName = Platforms.Ctr.SoundSetDocument;
            this.DocumentVersion = Platforms.Ctr.SoundSetDocumentVersion;
            this.Platform = Platforms.Ctr.PlatformName;
            this.translator = new SoundSetXml2ModelTranslatorCtr(componentFactory);
        }

        public override bool IsSupported(IStreamResource resource)
        {
            if (!(resource is FileResource))
            {
                return false;
            }

            using (FileStream stream =
                   File.Open(resource.Key, FileMode.Open, FileAccess.Read, FileShare.Read))
            {
                using (XmlReader reader = XmlReader.Create(stream))
                {
                    try
                    {
                        XmlSoundSetFileCtr.Validate(reader, this.Platform, this.DocumentVersion);
                    }
                    catch(NotSupportedException)
                    {
                        return false;
                    }
                    catch
                    {
                        throw;
                    }
                }
            }

            return true;
        }

        public override void Read(IStreamResource resource, Document document)
        {
            Ensure.Argument.NotNull(resource);
            Ensure.Argument.True(resource is FileResource);
            Ensure.Argument.True(document is SoundSetDocument);

            SoundSetDocument soundSetDocument = document as SoundSetDocument;

            XmlSoundSetFileCtr xmlSoundSetFileCtr = CreateXmlSoundSetFileCtr(resource);

            translator.PathResolver.BasePath =
                Path.GetDirectoryName(resource.Key.GetFullPath());

            SoundSet soundSet = translator.Run(xmlSoundSetFileCtr.Body.SoundSet);
            this.TranslateSoundSetCTRParameter2Parameter(soundSet);
            soundSetDocument.SoundSet = soundSet;
        }

        private XmlSoundSetFileCtr CreateXmlSoundSetFileCtr(IStreamResource resource)
        {
            Assertion.Argument.NotNull(resource);
            Assertion.Argument.True(resource is FileResource);

            using (FileStream fileStream = File.Open(resource.Key, FileMode.Open, FileAccess.Read, FileShare.Read))
            {
                return new XmlSerializer(typeof(XmlSoundSetFileCtr)).Deserialize(fileStream) as XmlSoundSetFileCtr;
            }
        }

        private void TranslateSoundSetCTRParameter2Parameter(SoundSet soundSet)
        {
            foreach (Component component in soundSet.Children)
            {
                if (component is StreamSoundPack ||
                    component is SequenceSoundPack ||
                    component is SequenceSoundSetPack ||
                    component is WaveSoundSetPack)
                {
                    this.TranslateCTRParameter2Parameter(component);
                }
            }
        }

        private void TranslateCTRParameter2Parameter(Component component)
        {
            if (component is Sound)
            {
                if (component.Parameters.ContainsKey(ProjectParameterNamesCtr.Sound.CTRParams))
                {
                    CTRParameterValue value = component.Parameters[ProjectParameterNamesCtr.Sound.CTRParams] as CTRParameterValue;
                    if (value == null) goto Remove;

                    if (component is StreamSoundBase &&
                        0 < component.Children.Count &&
                        component.Children[0] is StreamSoundTrackBase)
                    {
                        StreamSoundTrackBase track = component.Children[0] as StreamSoundTrackBase;
                        track.FrontBypass = value.FrontBypass;
                    }
                    else
                    {
                        component.Parameters[ProjectParameterNames.FrontBypass].Value = value.FrontBypass.Value;
                    }

                Remove:
                    component.Parameters.RemoveValue(ProjectParameterNamesCtr.Sound.CTRParams);
                }
            }
            else
            {
                foreach (Component comp in component.Children)
                {
                    this.TranslateCTRParameter2Parameter(comp);
                }
            }
        }
    }
}
