﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

/// SoundMakerのPreviewクラス群を騙す為にnamespace、クラス名が特殊になっています。
namespace NintendoWare.SoundMaker.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Reflection;
    using System.Windows.Forms;

    using Nintendo.Foundation.IO;
    using NintendoWare.SoundFoundation;
    using NintendoWare.SoundFoundation.Conversion;
    using NintendoWare.SoundFoundation.Conversion.NintendoWareBinary;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.Core.Reflection;
    using NintendoWare.SoundFoundation.Core.Threading;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.FileFormats.Wave;
    using NintendoWare.SoundFoundation.Logs;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.FileFormats;
    using NintendoWare.SoundMaker.Framework;
    using NintendoWare.SoundMaker.Framework.FileFormats;
    using NintendoWare.SoundMaker.Framework.Projects;
    using NintendoWare.SoundMaker.Preview;
    using NintendoWare.SoundMaker.Preview.Service;
    using NintendoWare.SoundMakerPlugin;
    using NintendoWare.ToolDevelopmentKit;
    using NintendoWare.WaveExporter;
    using SoundProjectWaveExporter.Resources;

    using We = NintendoWare.WaveExporter;

    /// <summary>
    ///
    /// </summary>
    public class FormsApplicationGeneric : ApplicationBase
    {
        private Settings settings = null;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public FormsApplicationGeneric() : this(CreateTraits())
        {
            ApplicationBase.Instance = this;

            ComponentConfigurationCommon.Instance.Initialize();

            string location = Assembly.GetExecutingAssembly().Location;
            string path = Path.GetDirectoryName(location);
            // SoundMaker と同じファイルを参照するため直接書く↓（しかたない）
            path = Path.Combine(path, "SoundMaker_enable_48kHz");
            Settings.Enabled48kHz = File.Exists(path);

            FileUtil.EnabledOpus = true; // Opus 対応を有効にします。
        }

        /// <summary>
        /// アプリケーションを実行します。
        /// </summary>
        public new bool Run(string[] args)
        {
            try
            {
                CommandLineParserSettings commandLineParserSettings = new CommandLineParserSettings()
                {
                    ApplicationDescription = MessageResourceWaveExporter.CommandLine_ApplicationDescription,
                };

                if (new Nintendo.Foundation.IO.CommandLineParser(commandLineParserSettings).ParseArgs(args, out this.settings) == false)
                {
                    return true;
                }
            }
            catch (ArgumentException)
            {
                return false;
            }

            try
            {
                InitializeSoundSystem();
                ShutdownWaveOutput();

                if (settings.FilePaths == null || settings.FilePaths.Length == 0)
                {
                    this.WriteErrorLine(MessageResourceWaveExporter.CommandLine_ErrorMessage_Error +
                                        MessageResourceWaveExporter.CommandLine_ErrorMessage_NotSpecifiedFile);
                    return false;
                }

                //
                foreach (string filePath in settings.FilePaths)
                {
                    string fullFilePath = Path.GetFullPath(filePath);
                    ProjectService.Open(fullFilePath);

                    WaveFileManager.Instance.PreprocessExePath = this.GetWavePreprocessExeFullPath();

                    // ストリームサウンド
                    if (settings.SoundTypes.HasFlag(Settings.SoundType.StreamSound))
                    {
                        foreach (StreamSoundBase sound in ProjectService.StreamSounds)
                        {
                            OutputWaveFile(sound);
                        }
                    }

                    // ウェーブサウンド
                    if (settings.SoundTypes.HasFlag(Settings.SoundType.WaveSound))
                    {
                        foreach (WaveSoundBase sound in ProjectService.WaveSounds)
                        {
                            if (sound.Parent != null && !sound.Parent.IsEnabled)
                            {
                                continue;
                            }

                            OutputWaveFile(sound);
                        }
                    }

                    // シーケンスサウンド
                    if (settings.SoundTypes.HasFlag(Settings.SoundType.SequenceSound))
                    {
                        foreach (SequenceSoundBase sound in ProjectService.SequenceSounds)
                        {
                            if (sound.Parent != null && !sound.Parent.IsEnabled)
                            {
                                continue;
                            }

                            OutputSequenceSoundWaveFile(sound);
                        }
                    }
                }
            }
            catch (FileNotFoundException e)
            {
                this.WriteErrorLine(MessageResourceWaveExporter.CommandLine_ErrorMessage_Error +
                                    MessageResourceWaveExporter.CommandLine_ErrorMessage_NotFoundFile +
                                    "\"" + e.FileName + "\"");
                this.WriteErrorLine(MessageResourceWaveExporter.CommandLine_ErrorMessage_ExporterFaied);
                return false;
            }
            catch (Exception e)
            {
                this.WriteErrorLine(MessageResourceWaveExporter.CommandLine_ErrorMessage_Error + e.Message);
                this.WriteErrorLine(MessageResourceWaveExporter.CommandLine_ErrorMessage_ExporterFaied);
                return false;
            }
            finally
            {
                this.CurrentPlugin.RuntimeSoundSystem_ShutdownSoundSystem();
                ShutdownWaveOutput();
                this.CurrentPlugin.RuntimeGlobal_AXQuit();
            }

            return true;
        }

        /// <summary>
        /// 現在選択されているプラットフォームのインスタンスを取得します。
        /// </summary>
        private ISoundMakerPlugin CurrentPlugin
        {
            get
            {
                return SoundMakerPluginManager.Instance.CurrentSoundMakerPlugin;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void InitializeSoundSystem()
        {
            this.CurrentPlugin.RuntimeGlobal_AXInit();
            StartupWaveOutput();
            try
            {
                this.CurrentPlugin.RuntimeSoundSystem_InitSoundSystem(3, 4);
            }
            catch
            {
                this.WriteLine(MessageResourceWaveExporter.Message_CanNotInitializeAudioDevice);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void StartupWaveOutput()
        {
            ShutdownWaveOutput();

            this.CurrentPlugin.InitializeAudioDevice(string.Empty);
        }

        /// <summary>
        ///
        /// </summary>
        public void ShutdownWaveOutput()
        {
            // NOTE: nn::audio をシャットダウンすることはないが、過去の SoundMaker のロジックを踏襲するために、関数は残しておく
        }

        /// <summary>
        ///
        /// </summary>
        private void OutputWaveFile(Sound sound)
        {
            if (sound.IsEnabled == false)
            {
                return;
            }

            sound.UpdateWaveFile();

            PreviewSound previewSound = PreviewPlayer.CreatePreviewSound(sound, this.settings.SamplingRate);
            string outputFilePath = Path.Combine(this.settings.OutputDirectory, sound.Name);
            outputFilePath += ".wav";

            this.WriteLine(outputFilePath);

            try
            {
                previewSound.OutputWaveFile(outputFilePath, this.settings.MaxDuration, this.settings.IsMonaural);
            }
            catch (WaveFileFormatException e)
            {
                this.WriteErrorLine(MessageResourceWaveExporter.CommandLine_ErrorMessage_Error + e.Message);

                if (File.Exists(outputFilePath))
                {
                    File.Delete(outputFilePath);
                }
            }

            WaveFileManager.Instance.ClearWaveFileCache();
        }

        private void OutputSequenceSoundWaveFile(SequenceSoundBase sound)
        {
            if (settings.SplitSequenceSoundTracks == true)
            {
                OutputSequenceSoundSplitTrackWaveFiles(sound);
            }
            else
            {
                OutputWaveFile(sound);
            }
        }

        private void OutputSequenceSoundSplitTrackWaveFiles(SequenceSoundBase sound)
        {
            if (sound.IsEnabled == false)
            {
                return;
            }

            sound.UpdateWaveFile();

            uint trackMask;
            using (var previewSound = (PreviewSequenceSound)PreviewPlayer.CreatePreviewSound(sound, this.settings.SamplingRate))
            {
                trackMask = previewSound.GetTrackMask();
            }

            for (int trackNo = 1; trackMask != 0 && trackNo <= 32; trackMask >>= 1, trackNo++)
            {
                if ((trackMask & 0x01) != 0)
                {
                    this.OutputSequenceSoundSingleTrackWaveFile(sound, trackNo);
                }
            }

            WaveFileManager.Instance.ClearWaveFileCache();
        }

        private void OutputSequenceSoundSingleTrackWaveFile(SequenceSoundBase sound, int trackNo)
        {
            using (var previewSound = (PreviewSequenceSound)PreviewPlayer.CreatePreviewSound(sound, this.settings.SamplingRate))
            {
                previewSound.SingleTrack = trackNo;

                string outputFilePath = Path.Combine(this.settings.OutputDirectory, string.Format("{0}_{1:d02}.wav", sound.Name, trackNo));
                this.WriteLine(outputFilePath);

                try
                {
                    previewSound.OutputWaveFile(outputFilePath, this.settings.MaxDuration, this.settings.IsMonaural);
                }
                catch (WaveFileFormatException e)
                {
                    this.WriteErrorLine(MessageResourceWaveExporter.CommandLine_ErrorMessage_Error + e.Message);

                    if (File.Exists(outputFilePath))
                    {
                        File.Delete(outputFilePath);
                    }

                    return;
                }

                // 無音だったらファイルを削除します。
                if (this.IsSilentWaveFile(outputFilePath))
                {
                    File.Delete(outputFilePath);
                    this.WriteLine(MessageResourceWaveExporter.Message_Info + string.Format(MessageResourceWaveExporter.Message_SilentTrackWasDeleted, outputFilePath));
                }
            }
        }

        private bool IsSilentWaveFile(string filePath)
        {
            if (!File.Exists(filePath))
            {
                return false;
            }

            using (var reader = WaveFileReader.CreateInstance(filePath))
            {
                var wavFile = reader.Open(filePath);

                var buffer = new byte[4 * 1024];
                using (var stream = reader.OpenDataStream())
                {
                    int readBytes;
                    while ((readBytes = stream.Read(buffer, 0, buffer.Length)) > 0)
                    {
                        if (Array.FindIndex(buffer, 0, readBytes, value => value != 0) >= 0)
                        {
                            return false;
                        }
                    }
                }

                return true;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void WriteLine(string text)
        {
            if (this.settings.IsSilent == true)
            {
                return;
            }

            Console.WriteLine(text);
        }

        /// <summary>
        ///
        /// </summary>
        private void WriteErrorLine(string text)
        {
            Console.Error.WriteLine(text);
        }

        /// <summary>
        ///
        /// </summary>
        private class Settings
        {
            [Flags]
            public enum SoundType
            {
                StreamSound = 1 << 0,
                WaveSound = 1 << 1,
                SequenceSound = 1 << 2,
                All = StreamSound | WaveSound | SequenceSound,
                None = 0,

                // CommandLineParser がフラグをうまく扱えないので、全フラグの組み合わせを定義しておきます。
                StreamOrWaveSound = StreamSound | WaveSound,
                WaveOrSequenceSound = WaveSound | SequenceSound,
                StreamOrSequenceSound = StreamSound | SequenceSound,
            }

            public static bool Enabled48kHz { get; set; }

            private List<string> filePaths = new List<string>();

            public Settings()
            {
                this.OutputDirectory = string.Empty;

                this.IsSilent = false;
                this.IsMonaural = false;
            }

            [CommandLineValues(ValueName = "input-filepath", Description = "CommandLine_UsageMessage_InputFilePaths", DescriptionConverterName = "LocalizeDescription")]
            public string[] FilePaths { get; set; }

            [CommandLineOption('o', "output", ValueName = "output-folder", Description = "CommandLine_UsageMessage_Option_o", DescriptionConverterName = "LocalizeDescription")]
            public string OutputDirectory { get; set; }

            [CommandLineOption('t', "time", ValueName = "time", DefaultValue = "30", Description = "CommandLine_UsageMessage_Option_t", DescriptionConverterName = "LocalizeDescription")]
            public int MaxDuration { get; set; }

            [CommandLineOption('r', "sampling-rate", ConverterName = "ConvertSamplingRate", ValueName = "rate", DefaultValue = OutputWaveFileRenderType.k32KHz, Description = "CommandLine_UsageMessage_Option_r", DescriptionConverterName = "LocalizeDescription")]
            public OutputWaveFileRenderType SamplingRate { get; set; }

            [CommandLineOption('m', "monaural", Description = "CommandLine_UsageMessage_Option_m", DescriptionConverterName = "LocalizeDescription")]
            public bool IsMonaural { get; set; }

            [CommandLineOption('s', "silent", Description = "CommandLine_UsageMessage_Option_s", DescriptionConverterName = "LocalizeDescription")]
            public bool IsSilent { get; set; }

            [CommandLineOption(
                "sound-type",
                ConverterName = nameof(ConvertSoundType),
                Description = nameof(MessageResourceWaveExporter.CommandLine_UsageMessage_Option_SoundTypes),
                DescriptionConverterName = nameof(LocalizeDescription))]
            public SoundType SoundTypes { get; set; } = Settings.SoundType.All;

            [CommandLineOption(
                "split-sequence-sound-tracks",
                Description = nameof(MessageResourceWaveExporter.CommandLine_UsageMessage_Option_SplitSequenceSoundTracks),
                DescriptionConverterName = nameof(LocalizeDescription))]
            public bool SplitSequenceSoundTracks { get; set; }

            public OutputWaveFileRenderType ConvertSamplingRate(string samplingRate, string optionName)
            {
                if (samplingRate == "32" || Enabled48kHz == false)
                {
                    return OutputWaveFileRenderType.k32KHz;
                }
                else if (samplingRate == "48")
                {
                    return OutputWaveFileRenderType.k48KHz;
                }
                else
                {
                    throw new Exception();
                }
            }

            public SoundType ConvertSoundType(string arg, string optionName)
            {
                var soundType = SoundType.None;

                foreach (var value in arg.Split(','))
                {
                    switch (value)
                    {
                        case "stream":
                            soundType |= SoundType.StreamSound;
                            break;

                        case "wave":
                            soundType |= SoundType.WaveSound;
                            break;

                        case "sequence":
                            soundType |= SoundType.SequenceSound;
                            break;

                        default:
                            throw new ArgumentException(string.Format(MessageResourceWaveExporter.CommandLine_ErrorMessage_UnexpectedValue, value));
                    }
                }

                return soundType;
            }

            /// <summary>
            /// ローカライズしたオプションの説明を取得します。
            /// </summary>
            /// <param name="description">Descriptionプロパティの値</param>
            /// <param name="valueName">引数名またはオプション名</param>
            /// <returns>ローカライズされたコマンドラインオプションの説明を返します。</returns>
            public static string LocalizeDescription(string description, string valueName)
            {
                string result = MessageResourceWaveExporter.ResourceManager.GetString(description, MessageResourceWaveExporter.Culture);
                System.Diagnostics.Debug.Assert(result != null, "コマンドオプションの Description が null");

                return result;
            }
        }


        private class AppConfigurationWaveExpoter : AppConfiguration
        {
            // バージョンはダミー（プラットフォームもバージョンも使用しない）
            public AppConfigurationWaveExpoter() : base(Platforms.Any.PlatformName, "0.0.0.0")
            {
                this.EnabledAACInternal = true; // 必要なのはこれだけ
            }

            protected override string RootElementName
            {
                get { return string.Empty; }
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected FormsApplicationGeneric(IApplicationTraits traits)
            : base(traits)
        {
        }

        /// <summary>
        /// ウェーブサウンド編集の有効、無効を取得します。
        /// </summary>
        private static bool IsWaveSoundEditEnabled()
        {
            string location = Assembly.GetExecutingAssembly().Location;
            string path = Path.GetDirectoryName(location);
            return File.Exists(Path.Combine(path, "SoundMaker_enable_wsdedit"));
        }

        ///
        private static IApplicationTraits CreateTraits()
        {
            ApplicationTraits traits = new ApplicationTraits()
            {
                IntermediateOutputTraits = CreateIntermediateOutputTraits(),
            };

            //
            string appPath = Path.GetDirectoryName(Assembly.GetEntryAssembly().Location);

            SequenceSoundTextConverter sequenceSoundTextConverter =
                new SequenceSoundTextConverter(ConfigurationFileUtility.SequenceSoundConverterFilePath);

            SmfConverter smfConverter = new SmfConverter(ConfigurationFileUtility.SmfConverterFilePath);

            traits.ConversionTraits = new SoundProjectConversionTraits(true)
            {
                IntermediateOutputTraits = traits.IntermediateOutputTraits,

                SequenceSoundTextConverterPath = sequenceSoundTextConverter.ConverterExePath,
                BuiltInWavePreprocessExePath = BuiltInWavePreprocessExePath,
                SmfConverterPath = smfConverter.ConverterExePath,
                SequenceSoundTextConverter = sequenceSoundTextConverter,
                SmfConverter = smfConverter,
                SequenceSoundBinaryReader = new SequenceSoundBinaryReader(),
                IsWaveSound2BinaryEnabled = IsWaveSoundEditEnabled(),
            };

            return traits;
        }

        /// <summary>
        ///
        /// </summary>
        protected override object CreateService(Type serviceType)
        {
            if (serviceType == typeof(DocumentService))
            {
                return new DocumentService(this.CreateDocumentServiceTraits2());
            }

            if (serviceType == typeof(SoundProjectService))
            {
                return new SoundProjectServiceCommon(DocumentService, CreateIntermediateOutputTraits());
            }

            if (serviceType == typeof(BankService))
            {
                return new BankServiceManager(DocumentService, CreateIntermediateOutputTraits());
            }

            return base.CreateService(serviceType);
        }

        /// <summary>
        ///
        /// </summary>
        protected override AppConfiguration CreateAppConfiguration()
        {
            return new AppConfigurationWaveExpoter();
        }

        /// <summary>
        ///
        /// </summary>
        protected override SoundProjectConfiguration CreateProjectConfiguration()
        {
            return null;
        }

        /// <summary>
        ///
        /// </summary>
        protected override PresetListColumnsConfiguration CreatePresetListColumnsConfiguration()
        {
            return null;
        }

        /// <summary>
        ///
        /// </summary>
        protected override BookmarkConfiguration CreateBookmarkConfiguration()
        {
            return null;
        }

        /// <summary>
        ///
        /// </summary>
        protected override IDocumentServiceTraits CreateDocumentServiceTraits()
        {
            return new DocumentServiceTraits();
        }

        /// <summary>
        ///
        /// </summary>
        protected override DocumentImportServiceTraits CreateDocumentImportServiceTraits()
        {
            return new DocumentImportServiceTraits();
        }

        /// <summary>
        ///
        /// </summary>
        protected override void RunApplication(string[] args)
        {
        }

        /// <summary>
        ///
        /// </summary>
        protected override bool ExitApplication()
        {
            return true;
        }

        /// <summary>
        ///
        /// </summary>
        public RealtimeEditService RealtimeTransferService
        {
            get
            {
                return null;
            }
        }

        /// <summary>
        /// サウンド中間出力特性を生成します。
        /// </summary>
        /// <returns>サウンド中間出力特性を返します。</returns>
        private static SoundIntermediateOutputTraits CreateIntermediateOutputTraits()
        {
            return new SoundIntermediateOutputTraits()
            {
                SoundProjectDocumentTypeName = Platforms.Any.SoundProjectDocument,
                SoundSetDocumentTypeName = Platforms.Any.SoundSetDocument,
                BankDocumentTypeName = Platforms.Any.BankDocument,
                SoundProjectFileExtension = SoundFileExtensions.SoundProjectFile,
                SoundSetFileExtension = SoundFileExtensions.SoundSetFile,
                BankFileExtension = SoundFileExtensions.BankFile,
                TextSequenceSoundFileExtension = SoundFileExtensions.TextSequenceSoundFile,
                BankIncludeFileExtension = SoundFileExtensions.BankIncludeFile,
                SoundIDCppHeaderFileExtension = SoundFileExtensions.SoundIDCppHeaderFile,
            };
        }

        /// <summary>
        /// ドキュメントサービス特性を生成します。
        /// </summary>
        /// <returns>ドキュメントサービス特性を返します。</returns>
        private IDocumentServiceTraits CreateDocumentServiceTraits2()
        {
            string productName = Assembly.GetEntryAssembly().GetTitle();
            string productVersion = Assembly.GetEntryAssembly().GetName().Version.ToString(3);

            DocumentServiceTraits traits = new DocumentServiceTraits();

            // DocumentFactory
            traits.DocumentFactorys.Add(new SoundProjectDocumentFactory());
            traits.DocumentFactorys.Add(new SoundSetDocumentFactory());
            traits.DocumentFactorys.Add(new BankDocumentFactory());

            // DocumentReader
            traits.DocumentReaders.Add(new SoundProjectDocumentReader());
            traits.DocumentReaders.Add(new SoundSetDocumentReaderCommon(new SoundSetComponentFactory()));
            traits.DocumentReaders.Add(new BankDocumentReader());

            // DocumentWriter
            traits.DocumentWriters.Add(
                new SoundProjectDocumentWriter()
                {
                    ProductName = productName,
                    ProductVersion = productVersion
                });
            traits.DocumentWriters.Add(
                new SoundSetDocumentWriterCommon()
                {
                    ProductName = productName,
                    ProductVersion = productVersion
                });
            traits.DocumentWriters.Add(
                new BankDocumentWriter()
                {
                    ProductName = productName,
                    ProductVersion = productVersion
                });

            return traits;
        }
    }
}
