﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Linq;
using NintendoWare.SoundFoundation.FileFormats.NintendoSdk;

namespace NintendoWare.SoundProjectConverter
{
    using System;
    using System.IO;
    using System.Reflection;
    using System.Management;
    using Nintendo.Foundation.IO;
    using NintendoWare.SoundFoundation;
    using NintendoWare.SoundFoundation.Conversion;
    using NintendoWare.SoundFoundation.Conversion.NintendoWareBinary;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.Core.Reflection;
    using NintendoWare.SoundFoundation.Core.Threading;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.Logs;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// コンソールアプリケーション機能を提供します。
    /// </summary>
    public class ConsoleApplication
    {
        private const string BuiltInWavePreprocessExeRelativePath = @"sox\\sox.exe";

        // サービス
        private DocumentService documentService;
        private SoundProjectService projectService;
        private BankServiceManager bankServices;
        private SoundProjectConvertService convertService;

        // 設定
        private Settings settings = null;

        // 状態
        private bool succeeded = true;

        //-----------------------------------------------------------------

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ConsoleApplication()
        {
            ComponentConfigurationCommon.Instance.Initialize();

            SoundIntermediateOutputTraits intermediateOutputTraits = this.CreateIntermediateOutputTraits();

            this.documentService = new DocumentService(this.CreateDocumentServiceTraits());

            this.projectService = new SoundProjectServiceCommon(
                this.documentService, intermediateOutputTraits);

            this.bankServices = new BankServiceManager(
                this.documentService, intermediateOutputTraits);

            this.convertService = new SoundProjectConvertService(
                    this.CreateConversionTraits(intermediateOutputTraits, null),
                    new DirectInvoker(),
                    this.bankServices);
            this.convertService.OutputLine += OnOutputLine;
        }

        private string BuiltInWavePreprocessExePath
        {
            get
            {
                return Path.Combine(
                    Path.GetDirectoryName(Assembly.GetAssembly(typeof(ConsoleApplication)).Location),
                    BuiltInWavePreprocessExeRelativePath);
            }
        }

        /// <summary>
        /// アプリケーションを実行します。
        /// </summary>
        /// <param name="args">コマンドライン引数。</param>
        public bool Run(string[] args)
        {
            try
            {
                CommandLineParserSettings commandLineParserSettings = new CommandLineParserSettings()
                {
                    ApplicationDescription = Resources.MessageResource.CommandLine_ApplicationDescription,
                };
                if (new Nintendo.Foundation.IO.CommandLineParser(commandLineParserSettings).ParseArgs(args, out this.settings) == false)
                {
                    return true;
                }
            }
            catch
            {
                return false;
            }

            if (string.IsNullOrEmpty(settings.FilePath) == true)
            {
                this.WriteErrorLine(Resources.MessageResource.Message_ProjectFileNotSet);
                return false;
            }

            this.WriteLine(
                string.Format(
                    Resources.MessageResource.Message_LoadingProject,
                    Path.GetFileName(this.settings.FilePath)
                    ),
                true);

            try
            {
                try
                {
                    this.projectService.Open(settings.FilePath);
                }
                catch (Exception exception)
                {
                    this.WriteErrorLine(Resources.MessageResource.Message_FailedToLoadProjectFile);
                    this.WriteErrorLine(exception.ToString());
                    return false;
                }

                if (!this.LoadItemFilterSettings(settings.FilePath))
                {
                    return false;
                }

                // プロジェクトファイルに書かれた出力先よりも、コマンドライン引数で指定された出力先を優先する
                // その際、カレントディレクトリからの相対パスをプロジェクトディレクトリからの相対パスに変換している
                if (!string.IsNullOrEmpty(this.settings.OutputDirectory))
                {
                    this.projectService.Project.OutputDirectoryPath = PathEx.MakeRelative(
                            PathUtility.GetFullPath(this.settings.OutputDirectory),
                            Path.GetDirectoryName(this.projectService.ProjectFilePath));
                }

                this.convertService.ParallelConversionCountMax = settings.ParallelConversionCountMax;
                this.convertService.IsPreConvertCommandsIgnored = settings.IsPreConvertCommandsIgnored;
                this.convertService.IsPostConvertCommandsIgnored = settings.IsPostConvertCommandsIgnored;

                this.convertService.EndConvert += (sender, e) => this.SetConversionResult(e.Succeeded);

                if (this.settings.IsReconvert)
                {
                    // TODO : 有効なプラットフォームを指定する
                    this.convertService.Convert(
                        this.projectService,
                        SoundProjectConvertSettings.CreateForReconvert(string.Empty));
                }
                else
                {
                    // TODO : 有効なプラットフォームを指定する
                    this.convertService.Convert(
                        this.projectService,
                        SoundProjectConvertSettings.CreateForConvert(string.Empty));
                }

                this.convertService.Wait();
            }
            finally
            {
                this.projectService.Close();
            }

            return succeeded;
        }

        /// <summary>
        /// ウェーブサウンド編集の有効、無効を取得します。
        /// </summary>
        private static bool IsWaveSoundEditEnabled()
        {
            string location = Assembly.GetExecutingAssembly().Location;
            string path = Path.GetDirectoryName(location);
            return File.Exists(Path.Combine(path, "SoundMaker_enable_wsdedit"));
        }

        private static void ConvertToFullPathForItemFilterTargets(ItemFilterSettings itemFillterSettings, string basePath)
        {
            var filters = itemFillterSettings?.Filters;

            if (filters == null)
            {
                return;
            }

            foreach (var itemFilter in filters.Where(filter => !string.IsNullOrEmpty(filter.Target)))
            {
                var path = Environment.ExpandEnvironmentVariables(itemFilter.Target);

                if (!Path.IsPathRooted(path))
                {
                    path = Path.Combine(basePath, path);
                }

                itemFilter.Target = PathUtility.GetFullPath(path);
            }
        }

        private bool LoadItemFilterSettings(string projectFilePath)
        {
            var projectDirectoryPath = PathUtility.GetFullPath(Path.GetDirectoryName(projectFilePath));

            try
            {
                if (!string.IsNullOrEmpty(settings.ExcludeListFilePath))
                {
                    using (var stream = File.OpenRead(settings.ExcludeListFilePath))
                    {
                        var itemFilterSettings = ItemFilterParser.Parse(stream);
                        ConvertToFullPathForItemFilterTargets(itemFilterSettings, projectDirectoryPath);
                        this.convertService.ExcludeItemFilterSettings = itemFilterSettings;
                    }
                }
            }
            catch (Exception exception)
            {
                this.WriteErrorLine(
                    Resources.MessageResource.Message_FailedToLoadItemFilterSettingsFile
                    + Environment.NewLine
                    + settings.ExcludeListFilePath
                    + Environment.NewLine
                    + exception.Message);
                return false;
            }

            try
            {
                if (!string.IsNullOrEmpty(settings.IncludeListFilePath))
                {
                    using (var stream = File.OpenRead(settings.IncludeListFilePath))
                    {
                        var itemFilterSettings = ItemFilterParser.Parse(stream);
                        ConvertToFullPathForItemFilterTargets(itemFilterSettings, projectDirectoryPath);
                        this.convertService.IncludeItemFilterSettings = itemFilterSettings;
                    }
                }
            }
            catch (Exception exception)
            {
                this.WriteErrorLine(
                    Resources.MessageResource.Message_FailedToLoadItemFilterSettingsFile
                    + Environment.NewLine
                    + settings.IncludeListFilePath
                    + Environment.NewLine
                    + exception.Message);
                return false;
            }

            return true;
        }

        /// <summary>
        /// コンバートの結果を設定します。
        /// </summary>
        /// <param name="succeeded">
        /// コンバートが成功した場合は true、失敗した場合は false を返します。
        /// </param>
        private void SetConversionResult(bool succeeded)
        {
            this.succeeded = succeeded;
        }

        /// <summary>
        /// サウンド中間出力特性を生成します。
        /// </summary>
        /// <returns>サウンド中間出力特性を返します。</returns>
        private SoundIntermediateOutputTraits CreateIntermediateOutputTraits()
        {
            return new SoundIntermediateOutputTraits()
            {
                SoundProjectDocumentTypeName = Platforms.Any.SoundProjectDocument,
                SoundSetDocumentTypeName = Platforms.Any.SoundSetDocument,
                BankDocumentTypeName = Platforms.Any.BankDocument,
                SoundProjectFileExtension = SoundFileExtensions.SoundProjectFile,
                SoundSetFileExtension = SoundFileExtensions.SoundSetFile,
                BankFileExtension = SoundFileExtensions.BankFile,
                TextSequenceSoundFileExtension = SoundFileExtensions.TextSequenceSoundFile,
                BankIncludeFileExtension = SoundFileExtensions.BankIncludeFile,
                SoundIDCppHeaderFileExtension = SoundFileExtensions.SoundIDCppHeaderFile,
            };
        }

        /// <summary>
        /// サウンドバイナリ出力特性を生成します。
        /// </summary>
        /// <returns>サウンドバイナリ出力特性を返します。</returns>
        private SoundBinaryOutputTraits CreateBinaryOutputTraits()
        {
            return new SoundBinaryOutputTraits()
            {
                BankBinaryFileExtension = SoundFileExtensions.BankBinaryFile,
                SequenceSoundBinaryFileExtension = SoundFileExtensions.SequenceSoundBinaryFile,
                SoundArchiveBinaryFileExtension = SoundFileExtensions.SoundArchiveBinaryFile,
                StreamSoundBinaryFileExtension = SoundFileExtensions.StreamSoundBinaryFile,
                StreamSoundPrefetchBinaryFileExtension = SoundFileExtensions.StreamSoundPrefetchBinaryFile,
                WaveArchiveBinaryFileExtension = SoundFileExtensions.WaveArchiveBinaryFile,
                WaveBinaryFileExtension = SoundFileExtensions.WaveBinaryFile,
                WaveSoundBinaryFileExtension = SoundFileExtensions.WaveSoundBinaryFile,
                WaveSoundBinary2FileExtension = SoundFileExtensions.WaveSoundBinary2File,
                GroupBinaryFileExtension = SoundFileExtensions.GroupBinaryFile,
            };
        }

        /// <summary>
        /// コンバート特性を生成します。
        /// </summary>
        /// <returns>コンバート特性を返します。</returns>
        private SoundProjectConversionTraits CreateConversionTraits(
            SoundIntermediateOutputTraits intermediateOutputraits,
            SoundBinaryOutputTraits binaryOutputTraits)
        {
            string appPath = Path.GetDirectoryName(Assembly.GetEntryAssembly().Location);

            SequenceSoundTextConverter sequenceSoundTextConverter =
                new SequenceSoundTextConverter(ConfigurationFileUtility.SequenceSoundConverterFilePath);

            SmfConverter smfConverter = new SmfConverter(ConfigurationFileUtility.SmfConverterFilePath);

            return new SoundProjectConversionTraits(true)
            {
                IntermediateOutputTraits = intermediateOutputraits != null ?
                    intermediateOutputraits : this.CreateIntermediateOutputTraits(),
                BinaryOutputTraits = binaryOutputTraits != null ?
                    binaryOutputTraits : this.CreateBinaryOutputTraits(),
                BinaryFileInfo = new BinaryFileInfo()
                {
                    BankSignature = "FBNK",
                    GroupSignature = "FGRP",
                    SequenceSoundSignature = "FSEQ",
                    SoundArchiveSignature = "FSAR",
                    StreamSoundSignature = "FSTM",
                    StreamSoundPrefetchSignature = "FSTP",
                    WaveArchiveSignature = "FWAR",
                    WaveSignature = "FWAV",
                    WaveSoundSignature = "FWSD",
                    WaveSound2Signature = "BAWSD   ",
                    BankVersion = new BinaryVersion(0, 1, 0, 0),
                    GroupVersion = new BinaryVersion(0, 1, 0, 0),
                    SequenceSoundVersion = new BinaryVersion(0, 1, 0, 0),
                    SoundArchiveVersion = new BinaryVersion(0, 2, 5, 0),
                    StreamSoundVersion = new BinaryVersion(0, 6, 3, 0),
                    StreamSoundPrefetchVersion = new BinaryVersion(0, 5, 2, 0),
                    WaveArchiveVersion = new BinaryVersion(0, 1, 0, 0),
                    WaveVersion = new BinaryVersion(0, 1, 2, 0),
                    WaveSoundVersion = new BinaryVersion(0, 1, 1, 0),
                    WaveSound2Version = new BinaryVersion(1, 0, 0, 0),
                },
                SequenceSoundTextConverterPath = sequenceSoundTextConverter.ConverterExePath,
                BuiltInWavePreprocessExePath = BuiltInWavePreprocessExePath,
                SmfConverterPath = smfConverter.ConverterExePath,
                SequenceSoundTextConverter = sequenceSoundTextConverter,
                SmfConverter = smfConverter,
                SequenceSoundBinaryReader = new SequenceSoundBinaryReader(),
                IsWaveSound2BinaryEnabled = IsWaveSoundEditEnabled(),
            };
        }

        /// <summary>
        /// ドキュメントサービス特性を生成します。
        /// </summary>
        /// <returns>ドキュメントサービス特性を返します。</returns>
        private IDocumentServiceTraits CreateDocumentServiceTraits()
        {
            string productName = Assembly.GetEntryAssembly().GetTitle();
            string productVersion = Assembly.GetEntryAssembly().GetName().Version.ToString(3);

            DocumentServiceTraits traits = new DocumentServiceTraits();

            // DocumentFactory
            traits.DocumentFactorys.Add(new SoundProjectDocumentFactory());
            traits.DocumentFactorys.Add(new SoundSetDocumentFactory());
            traits.DocumentFactorys.Add(new BankDocumentFactory());
            traits.DocumentFactorys.Add(new SoundProjectDocumentFactoryCafe());
            traits.DocumentFactorys.Add(new SoundSetDocumentFactoryCafe());
            traits.DocumentFactorys.Add(new BankDocumentFactoryCafe());

            // DocumentReader
            SoundSetComponentFactory soundSetComponentFactory = new SoundSetComponentFactory();
            traits.DocumentReaders.Add(new SoundProjectDocumentReader());
            traits.DocumentReaders.Add(new SoundSetDocumentReaderCommon(soundSetComponentFactory));
            traits.DocumentReaders.Add(new BankDocumentReader());
            traits.DocumentReaders.Add(new SoundProjectDocumentReaderCafe());
            traits.DocumentReaders.Add(new SoundSetDocumentReaderCafe(soundSetComponentFactory));
            traits.DocumentReaders.Add(new BankDocumentReaderCafe());

            // DocumentWriter
            traits.DocumentWriters.Add(
                new SoundProjectDocumentWriter()
                {
                    ProductName = productName,
                    ProductVersion = productVersion
                });
            traits.DocumentWriters.Add(
                new SoundSetDocumentWriterCommon()
                {
                    ProductName = productName,
                    ProductVersion = productVersion
                });
            traits.DocumentWriters.Add(
                new BankDocumentWriter()
                {
                    ProductName = productName,
                    ProductVersion = productVersion
                });

            return traits;
        }

        private void WriteLine(string text)
        {
            this.WriteLine(text, false);
        }

        private void WriteLine(string text, bool isVerboseText)
        {
            if (this.settings.IsSilent)
            {
                return;
            }

            if (isVerboseText && !this.settings.IsVerbose)
            {
                return;
            }

            Console.WriteLine(text);
        }

        private void WriteErrorLine(string text)
        {
            Console.Error.WriteLine(text);
        }

        private void OnOutputLine(object sender, OutputLineEventArgs e)
        {
            foreach (OutputLine line in e.Lines)
            {
                switch (line.Level)
                {
                    case OutputLevel.Information:
                    case OutputLevel.Warning:
                        this.WriteLine(line.Text);
                        break;

                    case OutputLevel.Error:
                        this.WriteErrorLine(line.Text);
                        break;

                    case OutputLevel.Detail:
                        this.WriteLine(line.Text, true);
                        break;
                }
            }
        }

        /// <summary>
        /// 設定を格納します。
        /// </summary>
        private class Settings
        {
            private string filePath = string.Empty;
            private uint parallelConversionCountMax = 0;

            public Settings()
            {
                this.IsReconvert = false;
                this.IsSilent = false;
                this.IsVerbose = false;
                this.IsPreConvertCommandsIgnored = false;
                this.IsPostConvertCommandsIgnored = false;
            }

            [CommandLineValue(0, IsRequired = true, ValueName = "input-filepath", Description = "CommandLine_InputDescription", DescriptionConverterName = "LocalizeDescription")]
            public string FilePath
            {
                get { return this.filePath; }
                set
                {
                    Ensure.Argument.NotNull(value);
                    this.filePath = value;
                }
            }

            [CommandLineOption('c', Description = "CommandLine_UsageMessage_Option_c", DescriptionConverterName = "LocalizeDescription")]
            public bool IsReconvert { get; set; }

            [CommandLineOption('j', "jobs", ValueName = "conversion-count-max", DefaultValue = 0, Description = "CommandLine_UsageMessage_Option_j", DescriptionConverterName = "LocalizeDescription")]
            public uint ParallelConversionCountMax
            {
                get { return this.parallelConversionCountMax; }
                set
                {
                    if (value < 0 || 99 < value)
                    {
                        throw new ArgumentOutOfRangeException("value");
                    }

                    this.parallelConversionCountMax = value;
                }
            }

            [CommandLineOption('s', "silent", Description = "CommandLine_UsageMessage_Option_s", DescriptionConverterName = "LocalizeDescription")]
            public bool IsSilent { get; set; }

            [CommandLineOption('v', "verbose", Description = "CommandLine_UsageMessage_Option_v", DescriptionConverterName = "LocalizeDescription")]
            public bool IsVerbose { get; set; }

            [CommandLineOption("ignore-pre-convert-commands", Description = "CommandLine_UsageMessage_Option_ignore_pre_convert_commands", DescriptionConverterName = "LocalizeDescription")]
            public bool IsPreConvertCommandsIgnored { get; set; }

            [CommandLineOption("ignore-post-convert-commands", Description = "CommandLine_UsageMessage_Option_ignore_post_convert_commands", DescriptionConverterName = "LocalizeDescription")]
            public bool IsPostConvertCommandsIgnored { get; set; }

            [CommandLineOption(
                'o',
                "output",
                ValueName = "output-dirpath",
                Description = "CommandLine_UsageMessage_Option_output",
                DescriptionConverterName = "LocalizeDescription")]
            public string OutputDirectory { get; set; }

            [CommandLineOption("exclude-list", Description = "CommandLine_UsageMessage_Option_exclude_list", DescriptionConverterName = "LocalizeDescription")]
            public string ExcludeListFilePath { get; set; }

            [CommandLineOption("include-list", Description = "CommandLine_UsageMessage_Option_include_list", DescriptionConverterName = "LocalizeDescription")]
            public string IncludeListFilePath { get; set; }

            /// <summary>
            /// ローカライズしたオプションの説明を取得します。
            /// </summary>
            /// <param name="description">Descriptionプロパティの値</param>
            /// <param name="valueName">引数名またはオプション名</param>
            /// <returns>ローカライズされたコマンドラインオプションの説明を返します。</returns>
            public static string LocalizeDescription(string description, string valueName)
            {
                string result = Resources.MessageResource.ResourceManager.GetString(description, Resources.MessageResource.Culture);
                System.Diagnostics.Debug.Assert(result != null, "コマンドオプションの Description が null");

                return result;
            }
        }
    }
}
