﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Text.RegularExpressions;
    using System.Windows.Forms;

    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.Resources;

    public partial class FindPanel : UserControl
    {
        /// <summary>
        ///
        /// </summary>
        private class ComboBoxRangeItem
        {
            private string text = null;
            private FindArgs.Ranges range = FindArgs.Ranges.None;

            public ComboBoxRangeItem(string text, FindArgs.Ranges range)
            {
                this.text = text;
                this.range = range;
            }

            public override string ToString()
            {
                return this.text;
            }

            public FindArgs.Ranges Range
            {
                get { return this.range; }
            }
        }

        ///
        private FindArgs.TargetDocumentKinds targetDocumentKind = FindArgs.TargetDocumentKinds.SoundSet;
        private string[] targetSoundSetNames = null;

        ///
        private bool updatingUI = false;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public FindPanel()
        {
            InitializeComponent();

            this.comboBox_FindText.Focus();
            UpdateUI();

            checkBox_FindName.Checked = true;
            checkBox_FindFilePath.Checked = true;
            checkBox_FindComment.Checked = true;
            comboBox_FindText.MaxDropDownItems = 10;

            //
            this.btn_SelectSoundSet.Enabled = false;

            if (ProjectService != null)
            {
                ProjectService.Opened += delegate (Object sender, EventArgs e)
                    {
                        this.targetSoundSetNames = ProjectService.SoundSetDocuments
                        .Select(d => d.SoundSet.Name)
                        .ToArray();
                        UpdateUI();
                    };
                ProjectService.Closed += delegate (Object sender, EventArgs e)
                    {
                        this.targetSoundSetNames = null;
                        UpdateUI();
                    };
            }
        }

        /// <summary>
        ///
        /// </summary>
        public FindArgs.TargetDocumentKinds TargetDocumentKind
        {
            get
            {
                return this.targetDocumentKind;
            }
            set
            {
                this.targetDocumentKind = value;
                UpdateUI();
            }
        }

        /// <summary>
        ///
        /// </summary>
        private FormsApplication Application
        {
            get { return FormsApplication.Instance; }
        }

        /// <summary>
        ///
        /// </summary>
        private MainWindow MainWindow
        {
            get
            {
                if (FormsApplication.Instance == null)
                {
                    return null;
                }
                return FormsApplication.Instance.UIService.MainWindow;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void UpdateInnerData()
        {
            UpdateRangeComboBox();

            ComboBoxRangeItem item = this.comboBox_FindRange.SelectedItem as ComboBoxRangeItem;
            if (item != null)
            {
                FindArgs.Ranges range = item.Range;
                SetSelectedIndexRangeComboBox(this.comboBox_FindRange, range);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void SelectFindText()
        {
            this.comboBox_FindText.Focus();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected ComponentListItem FindNextCandidate(FindArgs e, CommonListCtrl ctrl, ComponentListItem beginItem, int direction, bool next)
        {
            ComponentListItem item = null;
            int index = -1;
            int maxIndex = ctrl.ItemCount;

            if (beginItem == null)
            {
                index = direction > 0 ? 0 : maxIndex - 1;
            }
            else
            {

                index = ctrl.ToIndex(beginItem);
                if (next == true)
                {
                    index += direction;
                }
            }

            while (true)
            {
                if (index < 0 || index >= maxIndex)
                {
                    return null;
                }

                item = ctrl.ToItem(index) as ComponentListItem;
                if (Finder.IsMatchCondition(e, item.Target) == true)
                {
                    return item;
                }

                index += direction;
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected override void OnVisibleChanged(EventArgs e)
        {
            base.OnVisibleChanged(e);

            if (Visible)
            {
                this.comboBox_FindText.Select();
                this.comboBox_FindText.SelectAll();
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void UpdateUI()
        {
            if (this.updatingUI == true)
            {
                return;
            }
            this.updatingUI = true;

            try
            {
                switch (this.targetDocumentKind)
                {
                    case FindArgs.TargetDocumentKinds.SoundSet:
                        ComboBoxRangeItem item =
                            this.comboBox_FindRange.SelectedItem as ComboBoxRangeItem;
                        bool enabled = false;
                        if (item != null)
                        {
                            if (item.Range == FindArgs.Ranges.SoundSet)
                            {
                                enabled = true;
                            }
                        }
                        this.btn_SelectSoundSet.Enabled = enabled;
                        this.btn_SelectSoundSet.Visible = true;
                        ShowControl(this.chk_ItemReference);
                        ShowControl(this.chk_StartPosition);
                        break;

                    case FindArgs.TargetDocumentKinds.Bank:
                        this.btn_SelectSoundSet.Enabled = false;
                        this.btn_SelectSoundSet.Visible = false;
                        HideControl(this.chk_ItemReference);
                        HideControl(this.chk_StartPosition);
                        break;
                }

                UpdateRangeComboBox();
            }
            finally
            {
                this.updatingUI = false;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void UpdateRangeComboBox(ComboBox ctrl)
        {
            CommonTabPage page = null;
            int selectedIndex = ctrl.SelectedIndex;

            ctrl.Items.Clear();
            ctrl.Items.Add(new ComboBoxRangeItem
                            (MessageResource.FindWindowText_Project,
                              FindArgs.Ranges.Project));

            if (TargetDocumentKind == FindArgs.TargetDocumentKinds.SoundSet)
            {
                ctrl.Items.Add(new ComboBoxRangeItem
                                (MessageResource.FindWindowText_SoundSet,
                                  FindArgs.Ranges.SoundSet));
            }

            //
            if (MainWindow != null &&
               (page = MainWindow.ActivePage) != null)
            {
                if (page.Panel is BankPanel &&
                    TargetDocumentKind == FindArgs.TargetDocumentKinds.Bank)
                {
                    ctrl.Items.Add(new ComboBoxRangeItem
                                    (MessageResource.FindWindowText_CurrentBank,
                                      FindArgs.Ranges.CurrentBank));

                    ctrl.Items.Add(new ComboBoxRangeItem
                                    (MessageResource.FindWindowText_CurrentList,
                                      FindArgs.Ranges.CurrentList));
                }

                if (page.Panel is SoundSetPanel &&
                    TargetDocumentKind == FindArgs.TargetDocumentKinds.SoundSet)
                {
                    ctrl.Items.Add(new ComboBoxRangeItem
                                    (MessageResource.FindWindowText_CurrentList,
                                      FindArgs.Ranges.CurrentList));
                }
            }

            //
            if (selectedIndex >= 0 && selectedIndex < ctrl.Items.Count)
            {
                ctrl.SelectedIndex = selectedIndex;
            }
            else
            {
                ctrl.SelectedIndex = 0;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void AddTextToFindTextComboBox()
        {
            string text = this.comboBox_FindText.Text;
            int index = -1;

            if (text == string.Empty)
            {
                return;
            }

            if ((index = this.comboBox_FindText.Items.IndexOf(text)) >= 0)
            {
                this.comboBox_FindText.Items.RemoveAt(index);
            }

            this.comboBox_FindText.Items.Insert(0, text);

            if (this.comboBox_FindText.Items.Count > 10)
            {
                this.comboBox_FindText.Items.RemoveAt(this.comboBox_FindText.Items.Count - 1);
            }

            this.comboBox_FindText.Text = text;
            this.comboBox_FindText.SelectionStart = 0;
            this.comboBox_FindText.SelectionLength = text.Length;
        }

        /// <summary>
        ///
        /// </summary>
        private void UpdateRangeComboBox()
        {
            UpdateRangeComboBox(comboBox_FindRange);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void SetSelectedIndexRangeComboBox(ComboBox comboBox, FindArgs.Ranges range)
        {
            int index = 0;

            foreach (ComboBoxRangeItem item in comboBox.Items)
            {
                if (item.Range == range)
                {
                    comboBox.SelectedIndex = index;
                }
                index++;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private FindArgs GetFindArgs()
        {
            FindArgs args = new FindArgs();
            ComboBoxRangeItem comboBoxItem = null;

            //
            comboBoxItem = comboBox_FindRange.SelectedItem as ComboBoxRangeItem;
            args.Range = comboBoxItem.Range;
            args.EnableLabel = checkBox_FindName.Checked;
            args.EnableFilePath = checkBox_FindFilePath.Checked;
            args.EnableComment = checkBox_FindComment.Checked;
            args.EnableItemReference = this.chk_ItemReference.Checked;
            args.EnableStartPosition = this.chk_StartPosition.Checked;
            args.CaseSensitive = checkBox_FindCaseSensitive.Checked;
            args.Repeat = checkBox_FindLoop.Checked;
            args.Text = Finder.ReplaceMetaStrings(comboBox_FindText.Text);
            args.TargetSoundSetNames = this.targetSoundSetNames;

            //
            if (checkBox_FindExactMatch.Checked == true)
            {
                args.Text = args.Text.Replace("*", ".*");
                args.Text = "^" + args.Text + "$";
            }
            else
            {
                args.Text = args.Text.Replace("*", "\\*");
            }

            return args;

        }

        /// <summary>
        ///
        /// </summary>
        private FindWindow FindWindow
        {
            get
            {
                Control control = Parent;
                while (!(control is FindWindow))
                {
                    Debug.Assert(control != null, "Control is null.");
                    control = control.Parent;
                }
                return control as FindWindow;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void Close()
        {
            //TODO: 未実装
            // 親ウインドウを閉じます

            FindWindow.Close();
        }

        /// <summary>
        /// "前を検索"、"次を検索"を行います。
        /// </summary>
        private void Find(int direction)
        {
            if (ProjectDocument == null)
            {
                return;
            }

            FindArgs args = GetFindArgs();
            CommonTabPage page = null;
            CommonPanel panel = null;

            //
            if ((page = MainWindow.ActivePage) != null)
            {
                panel = page.Panel;
                if (panel is FindResultPanel2)
                {
                    panel = null;
                }
            }

            switch (args.Range)
            {
                case FindArgs.Ranges.Project:
                    FindForProject(args, direction, panel);
                    break;

                case FindArgs.Ranges.SoundSet:
                    FindForSoundSet(args, direction, panel);
                    break;

                //case FindArgs.Ranges.CurrentSoundSet:
                case FindArgs.Ranges.CurrentBank:
                    FindForCurrentPanel(args, direction, panel);
                    break;

                case FindArgs.Ranges.CurrentList:
                    FindForCurrentList(args, direction, panel);
                    break;
            }
        }

        /// <summary>
        /// プロジェクト全体から検索
        /// </summary>
        private void FindForProject(FindArgs e, int direction, CommonPanel panel)
        {
            ImaginaryDocument[] documents = GetFindTargetDocuments(e);
            if (documents.Length <= 0)
            {
                return;
            }

            FindForSoundSet(e, documents, direction, panel);
        }

        /// <summary>
        /// 指定サウンドセットから検索
        /// </summary>
        private void FindForSoundSet(FindArgs e, int direction, CommonPanel panel)
        {
            HashSet<string> targetNames = new HashSet<string>();
            if (this.targetSoundSetNames != null)
            {
                foreach (string name in this.targetSoundSetNames)
                {
                    targetNames.Add(name);
                }
            }

            ImaginaryDocument[] documents = GetFindTargetDocuments(e)
                .Where(i => i.Document is SoundSetDocument &&
                       targetNames.Contains(((SoundSetDocument)i.Document).SoundSet.Name))
                .ToArray();
            if (documents.Length <= 0)
            {
                return;
            }

            FindForSoundSet(e, documents, direction, panel);
        }

        /// <summary>
        /// 指定サウンドセットから検索
        /// </summary>
        private void FindForSoundSet(FindArgs e, ImaginaryDocument[] documents, int direction, CommonPanel panel)
        {
            ImaginaryDocument document = null;
            int index = 0;
            int beginIndex = 0;

            // 開かれているパネルのドキュメントを検索開始位置にします。
            if (panel != null)
            {
                if ((index = GetIndex(documents, panel.TargetDocument)) < 0)
                {
                    index = 0;
                }
            }
            else
            {
                if ((index = GetIndexByMatchCondition(e, documents, direction)) < 0)
                {
                    return;
                }

                document = documents[index];
                panel = document.GetPanel();
            }

            beginIndex = index;

            //
            ComponentListItem[] items = null;
            Component currentComponent = null;
            bool firstCtrl = true;
            bool retry = false;

            while (true)
            {
                //
                document = documents[index];
                panel = document.GetPanel();

                //
                currentComponent = null;
                if (panel != null && firstCtrl == true)
                {
                    items = panel.CurrentListCtrl.GetSelectedItems();
                    if (items != null && items.Length > 0)
                    {
                        currentComponent = items[0].Target;
                    }
                    firstCtrl = false;
                }

                if ((currentComponent = document.NextComponent
                     (e, currentComponent, direction)) != null)
                {
                    if (panel != null)
                    {
                        MainWindow.ActivatePage(panel.Parent as CommonTabPage);
                    }
                    return;
                }

                //
                index += direction;
                if (index < 0)
                {
                    if (e.Repeat == false) { return; }
                    index = documents.Length - 1;
                }

                if (index >= documents.Length)
                {
                    if (e.Repeat == false) { return; }
                    index = 0;
                }

                //
                if (index == beginIndex)
                {
                    if (retry == true)
                    {
                        return;
                    }
                    retry = true;
                }
            }
        }

        /// <summary>
        /// 現在のパネルから検索します。
        /// </summary>
        private bool FindForCurrentPanel(FindArgs e, int direction, CommonPanel panel)
        {
            ImaginaryDocument imaginaryDocument = null;
            ComponentListItem[] items = null;
            Component currentComponent = null;
            bool firstCtrl = true;
            string filePath = null;
            int loopCount = 0;

            if (panel is SoundSetPanel)
            {
                filePath = panel.Document.Resource.Key;
                imaginaryDocument = new ImaginarySoundSetDocument
                    (panel.Document as SoundSetDocument, filePath);
            }

            if (panel is BankPanel)
            {
                filePath = panel.Document.Resource.Key;
                imaginaryDocument = new ImaginaryBankDocument
                    (panel.Document as BankDocument, filePath);
            }

            //
            while (true)
            {
                //
                currentComponent = null;
                if (panel != null && firstCtrl == true)
                {
                    items = panel.CurrentListCtrl.GetSelectedItems();
                    if (items != null && items.Length > 0)
                    {
                        currentComponent = items[0].Target;
                    }
                    firstCtrl = false;
                }

                if ((currentComponent = imaginaryDocument.NextComponent
                     (e, currentComponent, direction)) != null)
                {
                    if (panel != null)
                    {
                        MainWindow.ActivatePage(panel.Parent as CommonTabPage);
                    }
                    return true;
                }

                //
                if (e.Repeat == false)
                {
                    return true;
                }

                //
                if (++loopCount > 1)
                {
                    break;
                }
            }

            return false;
        }

        /// <summary>
        /// 現在のリストから検索
        /// </summary>
        private bool FindForCurrentList(FindArgs e, int direction, CommonPanel panel)
        {
            CommonListCtrl currentCtrl = null;
            CommonListCtrl beginCtrl = null;
            ComponentListItem[] items = null;
            ComponentListItem currentItem = null;
            bool next = false;
            int loopCount = 0;

            currentCtrl = panel.CurrentListCtrl;
            beginCtrl = currentCtrl;

            if ((items = currentCtrl.GetSelectedItems()) != null && items.Length > 0)
            {
                currentItem = items[0];
                next = true;
            }

            while ((currentItem = FindNextCandidate
                    (e, currentCtrl, currentItem, direction, next)) == null)
            {
                if (e.Repeat == false)
                {
                    return false;
                }

                //
                if (++loopCount > 1)
                {
                    break;
                }

                //
                currentItem = null;
                next = false;
            }

            currentCtrl.ShowByItem(currentItem);
            return true;
        }

        /// <summary>
        /// 検索対象となるドキュメントを取得します。
        /// </summary>
        private ImaginaryDocument[] GetFindTargetDocuments(FindArgs ea)
        {
            ImaginaryDocumentCollection collection = null;
            ImaginaryDocument imaginaryDocument = null;
            BankService bankService = null;
            BankDocument bankDocument = null;
            string filePath = null;

            collection = new ImaginaryDocumentCollection();

            switch (TargetDocumentKind)
            {
                case FindArgs.TargetDocumentKinds.SoundSet:
                    foreach (SoundSetDocument soundSetDocument in ProjectService.SoundSetDocuments)
                    {
                        filePath = soundSetDocument.Resource.Key;
                        if (collection.Contains(filePath) == false)
                        {
                            imaginaryDocument = new ImaginarySoundSetDocument
                                (soundSetDocument, filePath);
                            collection.Add(imaginaryDocument);
                        }
                    }
                    break;

                case FindArgs.TargetDocumentKinds.Bank:
                    foreach (SoundSetBankBase bank in ProjectService.SoundSetBanks)
                    {
                        if (Application.BankServices.Contains(bank.FilePath) == true)
                        {
                            bankService = Application.BankServices[bank.FilePath];
                            bankDocument = bankService.BankDocument;
                        }
                        else
                        {
                            bankDocument = null;
                        }

                        filePath = bank.FilePath;
                        if (collection.Contains(filePath) == false)
                        {
                            imaginaryDocument = new ImaginaryBankDocument(bankDocument, filePath);
                            collection.Add(imaginaryDocument);
                        }
                    }
                    break;
            }

            return collection.ToArray();
        }

        ///--------------------------------
        ///
        private class ImaginaryDocumentCollection : KeyedCollection<string, ImaginaryDocument>
        {
            protected override string GetKeyForItem(ImaginaryDocument imaginaryDocument)
            {
                return imaginaryDocument.FilePath;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private int GetIndex(ImaginaryDocument[] imaginaryDocuments, SoundDocument document)
        {
            Debug.Assert(document != null, "Argument document is null");

            int index = 0;

            foreach (ImaginaryDocument imaginaryDocument in imaginaryDocuments)
            {
                if (imaginaryDocument.Document == document)
                {
                    return index;
                }
                index++;
            }
            return -1;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private int GetIndexByMatchCondition(FindArgs e, ImaginaryDocument[] imaginaryDocuments, int direction)
        {
            ImaginaryDocument imaginaryDocument = null;
            int index = -1;

            if (direction > 0)
            {
                index = 0;
                while (index < imaginaryDocuments.Length)
                {
                    imaginaryDocument = imaginaryDocuments[index];
                    if (imaginaryDocument.ContainsMatchCondition(e) == true)
                    {
                        return index;
                    }
                    index++;
                }
            }
            else
            {

                index = imaginaryDocuments.Length - 1;
                while (index >= 0)
                {
                    imaginaryDocument = imaginaryDocuments[index];
                    if (imaginaryDocument.ContainsMatchCondition(e) == true)
                    {
                        return index;
                    }
                    index--;
                }
            }

            return -1;
        }

        /// <summary>
        ///
        /// </summary>
        private SoundProjectService ProjectService
        {
            get
            {
                if (FormsApplication.Instance == null)
                {
                    return null;
                }

                return FormsApplication.Instance.ProjectService;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private SoundProjectDocument ProjectDocument
        {
            get { return FormsApplication.Instance.ProjectService.ProjectDocument; }
        }

        ///--------------------------------
        /// <summary>
        /// 前に検索
        /// </summary>
        private void OnFindPrevious(object sender, EventArgs e)
        {
            this.FindPrevious();
        }

        private void FindPrevious()
        {
            Find(-1);
            AddTextToFindTextComboBox();
        }

        ///--------------------------------
        /// <summary>
        /// 次に検索
        /// </summary>
        private void OnFindNext(object sender, EventArgs e)
        {
            this.FindNext();
        }

        private void FindNext()
        {
            Find(1);
            AddTextToFindTextComboBox();
        }

        ///--------------------------------
        /// <summary>
        /// 全て検索
        /// </summary>
        private void OnFindAll(object sender, EventArgs e)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void ShowControl(Control control)
        {
            control.Enabled = true;
            control.Visible = true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void HideControl(Control control)
        {
            control.Enabled = false;
            control.Visible = false;
        }

        ///--------------------------------
        /// <summary>
        /// 仮想アイテムを実際のアイテムへの関連付けを行う
        /// </summary>
        public void Linkage()
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override bool ProcessDialogKey(Keys keyData)
        {
            if (keyData == Keys.Escape)
            {
                Close();
                return true;
            }
            else if (keyData == Keys.F3)
            {
                this.FindNext();
                return true;
            }
            else if (keyData == (Keys.F3 | Keys.Shift) ||
                     keyData == (Keys.Enter | Keys.Shift))
            {
                this.FindPrevious();
                return true;
            }

            return base.ProcessDialogKey(keyData);
        }

        /// <summary>
        ///
        /// </summary>
        private void OnClick_SelectSoundSet(object sender, EventArgs e)
        {
            SoundSetSelectDialog dialog = new SoundSetSelectDialog
               (ProjectService, this.targetSoundSetNames);

            if (dialog.ShowDialog() != DialogResult.OK)
            {
                return;
            }

            this.targetSoundSetNames = dialog.SelectedSoundSets
                .Select(s => s.Name)
                .ToArray();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnSelectedIndexChanged_Range(object sender, EventArgs e)
        {
            UpdateUI();
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class FindArgs
    {
        /// <summary>
        ///
        /// </summary>
        public enum TargetDocumentKinds
        {
            SoundSet,
            Bank,
        }

        /// <summary>
        ///
        /// </summary>
        public enum Ranges
        {
            None,
            Project,
            SoundSet,
            CurrentSoundSet,
            CurrentBank,
            CurrentList,
        }

        private string text = null;
        private Regex regex = null;
        private bool caseSensitive = false;
        private string[] targetSoundSetNames = null;
        private bool onlyOneThing = false;

        /// <summary>
        ///
        /// </summary>
        public string Text
        {
            get
            {
                return this.text;
            }
            set
            {
                this.text = value;
                this.regex = null;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public bool CaseSensitive
        {
            get
            {
                return this.caseSensitive;
            }
            set
            {
                this.caseSensitive = value;
                this.regex = null;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public string[] TargetSoundSetNames
        {
            get
            {
                return this.targetSoundSetNames;
            }
            set
            {
                this.targetSoundSetNames = value;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public bool OnlyOneThing
        {
            get { return this.onlyOneThing; }
            set { this.onlyOneThing = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public bool IsMatch(IEnumerable<string> texts)
        {
            foreach (string text in texts)
            {
                if (IsMatch(text) == true)
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        ///
        /// </summary>
        public bool IsMatch(string text)
        {
            // 比較用 Regexを作成します。
            if (this.regex == null)
            {
                string conditionText = this.text;
                RegexOptions options = RegexOptions.Compiled;

                if (CaseSensitive == false)
                {
                    options |= RegexOptions.IgnoreCase;
                    //conditionText = conditionText.ToLower();
                }

                this.regex = new Regex(conditionText, options);
            }

            // 比較テキストの調整を行います。
            if (text == null)
            {
                text = String.Empty;
            }

            //
            return this.regex.IsMatch(text);
        }

        /// <summary>
        ///
        /// </summary>
        public bool EnableLabel { get; set; }
        public bool EnableFilePath { get; set; }
        public bool EnableComment { get; set; }
        public bool EnableItemReference { get; set; }
        public bool EnableStartPosition { get; set; }

        public bool Repeat { get; set; }

        public Ranges Range { get; set; }

        public SoundSetDocument SoundSetDocument { get; set; }
        public BankDocument BankDocument { get; set; }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public abstract class ImaginaryDocument
    {
        private SoundDocument _Document = null;
        private string _FilePath = null;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ImaginaryDocument(SoundDocument document, string filePath)
        {
            _Document = document;
            _FilePath = filePath;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public SoundDocument Document
        {
            get { return _Document; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string FilePath
        {
            get { return _FilePath; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual Component Component
        {
            get { return null; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual bool ContainsMatchCondition(FindArgs e)
        {
            return false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual CommonPanel GetPanel()
        {
            return null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual Component NextComponent(FindArgs e, Component currentComponent, int direction)
        {
            return null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual Component[] GetMatchConditionComponents(FindArgs e)
        {
            return null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected FormsApplication Application
        {
            get { return FormsApplication.Instance; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected MainWindow MainWindow
        {
            get { return FormsApplication.Instance.UIService.MainWindow; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected ComponentListItem GetListItem(CommonListCtrl listCtrl, Component component)
        {
            foreach (ComponentListItem item in listCtrl.ItemsSource.Items)
            {
                if (item.Target == component)
                {
                    return item;
                }
            }
            return null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected int ToIndex(CommonListCtrl listCtrl, Component component)
        {
            int index = 0;

            foreach (ComponentListItem item in listCtrl.ItemsSource.Items)
            {
                if (item.Target == component)
                {
                    return index;
                }
                index++;
            }
            return -1;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected ComponentListItem ToItem(CommonListCtrl listCtrl, int index)
        {
            if (index < 0 || index >= listCtrl.ItemsSource.Items.Count)
            {
                return null;
            }
            return listCtrl.ItemsSource.Items[index] as ComponentListItem;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class ImaginarySoundSetDocument : ImaginaryDocument
    {
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ImaginarySoundSetDocument(SoundDocument document, string filePath) : base(document, filePath)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private SoundSet SoundSet
        {
            get { return Document != null ? ((SoundSetDocument)Document).SoundSet : null; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override bool ContainsMatchCondition(FindArgs e)
        {
            SoundSetDocument soundSetDocument = Document as SoundSetDocument;
            return GetMatchConditionComponent(e, soundSetDocument, null, 1) != null ? true : false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override CommonPanel GetPanel()
        {
            CommonTabPage page = null;

            if ((page = MainWindow.FindPage(SoundSet)) == null)
            {
                return null;
            }

            return page.Panel;
        }

        /// <summary>
        ///
        /// </summary>
        public override Component NextComponent(FindArgs e, Component beginComponent, int direction)
        {
            SoundSetDocument soundSetDocument = Document as SoundSetDocument;
            Component result = null;
            CommonTabPage page = null;
            SoundSetPanel panel = null;
            IListItem listItem = null;

            if ((page = MainWindow.FindPage(soundSetDocument.SoundSet)) == null)
            {
                result = GetMatchConditionComponent
                    (e, soundSetDocument, beginComponent, direction);

                if (result != null)
                {
                    page = MainWindow.AddPage(soundSetDocument.SoundSet);
                    panel = page.Panel as SoundSetPanel;
                }
            }
            else
            {
                page = MainWindow.FindPage(soundSetDocument.SoundSet);
                panel = page.Panel as SoundSetPanel;

                result = GetMatchConditionComponent
                    (e, soundSetDocument, beginComponent, direction);
            }

            //
            if (result != null)
            {
                if (result.Parent is StreamSoundBase ||
                    result.Parent is WaveSoundSetBase ||
                    result.Parent is SequenceSoundSetBase)
                {
                    panel.Show(result.Parent.Parent as SoundSetItem);

                    listItem = GetListItem(panel.ListCtrlMain, result.Parent);
                    panel.ListCtrlMain.ShowByItem(listItem);

                    listItem = GetListItem(panel.ListCtrlSub, result);
                    panel.ListCtrlSub.ShowByItem(listItem);

                    panel.SetCurrentListCtrl(panel.ListCtrlSub);
                }
                else
                {
                    panel.SetCurrentListCtrl(panel.ListCtrlMain);

                    panel.Show(result.Parent as SoundSetItem);
                    listItem = GetListItem(panel.CurrentListCtrl, result);
                    panel.CurrentListCtrl.ShowByItem(listItem);
                }
            }

            return result;
        }

        /// <summary>
        ///
        /// </summary>
        public override Component[] GetMatchConditionComponents(FindArgs e)
        {
            List<Component> list = new List<Component>();
            SoundSetDocument soundSetDocument = Document as SoundSetDocument;
            Component component = null;

            while ((component = GetMatchConditionComponent
                    (e, soundSetDocument, component, 1)) != null)
            {
                list.Add(component);
            }

            return list.ToArray();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private Component GetMatchConditionComponent(FindArgs e, SoundSetDocument soundSetDocument, Component beginComponent, int direction)
        {
            List<Component> list = new List<Component>();
            SoundSet soundSet = soundSetDocument.SoundSet;
            Component component = null;
            int index = 0;
            bool next = true;

            foreach (Component item in soundSet.Children)
            {
                list.AddRange(GetComponents(item));
            }

            if (list.Count <= 0)
            {
                return null;
            }

            //
            if (beginComponent != null)
            {
                index = list.IndexOf(beginComponent);
            }
            else
            {
                index = direction < 0 ? list.Count - 1 : 0;
                next = false;
            }

            while (true)
            {
                if (next == true)
                {
                    index += direction;
                    if (index < 0 || index >= list.Count)
                    {
                        return null;
                    }
                }
                next = true;

                component = list[index];
                if (Finder.IsMatchCondition(e, component) == true)
                {
                    return component;
                }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private Component[] GetComponents(Component component)
        {
            List<Component> list = new List<Component>();
            Component[] items = null;

            items = component.Children.Cast<Component>().ToArray();
            list.AddRange(items);

            if (component is WaveSoundSetPack ||
                component is SequenceSoundSetPack ||
                component is StreamSoundPack)
            {
                foreach (Component item in items)
                {
                    list.AddRange(item.Children.Cast<Component>().ToArray());
                }
            }

            return list.ToArray();
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class ImaginaryBankDocument : ImaginaryDocument
    {
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ImaginaryBankDocument(SoundDocument document, string filePath) : base(document, filePath)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override Component Component
        {
            get { return Document != null ? ((BankDocument)Document).Bank : null; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override bool ContainsMatchCondition(FindArgs e)
        {
            BankDocument bankDocument = null;
            DocumentReference docRef = null;
            FileResource resource = null;
            bool result = false;

            if (Document == null)
            {
                resource = new FileResource(FilePath);
                using (docRef = Application.DocumentService.OpenDocument(resource))
                {
                    bankDocument = docRef.Document as BankDocument;

                    if (bankDocument != null)
                    {
                        result = GetMatchConditionComponent
                            (e, bankDocument, null, 1) != null ? true : false;
                    }
                }
            }
            else
            {

                result = GetMatchConditionComponent
                    (e, Document as BankDocument, null, 1) != null ? true : false;
            }

            return result;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override CommonPanel GetPanel()
        {
            CommonTabPage page = null;

            if (Document == null)
            {
                //page = AddBankDocument( FilePath);
            }
            else
            {

                if ((page = MainWindow.FindPage(Component)) != null)
                {
                    return page.Panel;
                }
                //MainWindow.ActivatePage( page);
            }

            return null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override Component NextComponent(FindArgs e, Component beginComponent, int direction)
        {
            CommonTabPage page = null;
            BankPanel panel = null;
            DocumentReference docRef = null;
            FileResource resource = null;
            BankDocument bankDocument = null;
            Component result = null;

            if (Document == null)
            {

                if (File.Exists(FilePath) == false)
                {
                    return null;
                }

                resource = new FileResource(FilePath);
                using (docRef = Application.DocumentService.OpenDocument(resource))
                {
                    bankDocument = docRef.Document as BankDocument;

                    if (bankDocument != null)
                    {
                        result = GetMatchConditionComponent
                            (e, bankDocument, beginComponent, direction);

                        if (result != null)
                        {
                            page = MainWindow.AddPage(bankDocument.Bank, null, true);
                            panel = page.Panel as BankPanel;

                        }
                    }
                }
            }
            else
            {

                CommonListCtrl listCtrl = null;
                bankDocument = Document as BankDocument;
                ComponentListItem item = null;

                if ((page = MainWindow.FindPage(bankDocument.Bank)) == null)
                {
                    page = MainWindow.AddPage(bankDocument.Bank, null, true);
                }

                panel = page.Panel as BankPanel;
                listCtrl = panel.CurrentListCtrl;

                int index = ToIndex(listCtrl, beginComponent);

                if (index < 0)
                {
                    index = direction < 0 ? listCtrl.ItemsSource.Items.Count - 1 : 0;
                }
                else
                {

                    index += direction;
                }

                while (true)
                {
                    if ((item = ToItem(panel.CurrentListCtrl, index)) == null)
                    {
                        return null;
                    }

                    if (Finder.IsMatchCondition(e, item.Target) == true)
                    {
                        result = item.Target;
                        break;
                    }

                    index += direction;
                }
            }

            //
            if (result != null)
            {
                IListItem listItem = GetListItem(panel.CurrentListCtrl, result);
                panel.CurrentListCtrl.ShowByItem(listItem);
            }

            return result;
        }

        /// <summary>
        ///
        /// </summary>
        public override Component[] GetMatchConditionComponents(FindArgs e)
        {
            List<Component> list = new List<Component>();
            Component component = null;
            DocumentReference docRef = null;
            BankDocument bankDocument = Document as BankDocument;
            ImaginaryInstrument instrument = null;
            FileResource resource = null;

            try
            {
                if (bankDocument == null)
                {
                    if (File.Exists(FilePath) == false)
                    {
                        return new Component[0];
                    }

                    resource = new FileResource(FilePath);

                    if ((docRef = Application.DocumentService.OpenDocument(resource)) != null)
                    {
                        bankDocument = docRef.Document as BankDocument;
                    }
                }

                if (bankDocument != null)
                {
                    while ((component = GetMatchConditionComponent
                            (e, bankDocument, component, 1)) != null)
                    {
                        instrument = new ImaginaryInstrument(component, bankDocument.Resource.Key);
                        list.Add(instrument);

                        if (e.OnlyOneThing == true)
                        {
                            break;
                        }
                    }
                }
            }
            finally
            {
                if (docRef != null)
                {
                    docRef.Close();
                }
            }

            return list.ToArray();
        }

        /// <summary>
        ///
        /// </summary>
        private Component GetMatchConditionComponent(FindArgs e, BankDocument bankDocument, Component beginComponent, int direction)
        {
            Instrument instrument = null;
            int beginIndex = 0;
            int endIndex = 0;

            if (bankDocument.Bank.Children.Count <= 0)
            {
                return null;
            }

            //
            endIndex = direction < 0 ? 0 : bankDocument.Bank.Children.Count - 1;

            if (beginComponent != null)
            {
                beginIndex = bankDocument.Bank.Children.IndexOf(beginComponent) + direction;
                if (beginIndex < 0 || beginIndex >= endIndex + 1)
                {
                    return null;
                }
            }
            else
            {

                beginIndex = direction < 0 ? bankDocument.Bank.Children.Count - 1 : 0;
            }

            while (true)
            {
                instrument = bankDocument.Bank.Children[beginIndex] as Instrument;
                if (Finder.IsMatchCondition(e, instrument) == true)
                {
                    return instrument;
                }

                //
                beginIndex += direction;
                if (direction > 0)
                {
                    if (beginIndex > endIndex)
                    {
                        break;
                    }
                }
                else
                {
                    if (beginIndex < endIndex)
                    {
                        break;
                    }
                }
            }

            return null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private BankDocument[] BankDocuments
        {
            get
            {
                return Application.DocumentService.Documents
                    .Where(d => d is BankDocument).Cast<BankDocument>()
                    .ToArray();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool AlreadyOpenedBankDocument(string filePath)
        {
            foreach (BankDocument bankDocument in BankDocuments)
            {
                if (bankDocument.Resource.Key == filePath)
                {
                    return true;
                }
            }
            return false;
        }
    }
}
