﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Drawing;
    using System.Linq;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Drawing;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Configurations;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.Preset;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Framework.Utilities;
    using NintendoWare.SoundMaker.Framework.Windows.Forms.CommandHandlers;
    using NintendoWare.ToolDevelopmentKit.Collections;

    using SFresrc = NintendoWare.SoundFoundation.Resources;

    /// <summary>
    /// コメントのテキストを変更するクラスです。
    /// </summary>
    public static class CommentColumnTextApplier
    {
        /// <summary>
        ///
        /// </summary>
        private class ColumnText
        {
            public string Text { get; set; }
            public string OriginalText { get; set; }

            public ColumnText(string text, string originalText)
            {
                Text = text;
                OriginalText = originalText;
            }
        }

        /// <summary>
        /// プロジェクト設定のコメント用テキストを IHeaderSourceに反映させます。
        /// </summary>
        public static void Apply(IHeaderSource[] headerAdapters)
        {
            foreach (IHeaderSource headerAdapter in headerAdapters)
            {
                Apply(headerAdapter);
            }
        }

        /// <summary>
        /// プロジェクト設定のコメント用テキストを IHeaderSourceに反映させます。
        /// </summary>
        public static void Apply(IHeaderSource headerAdapter)
        {
            if (headerAdapter == null)
            {
                return;
            }

            /// 変更するテキストはプロジェクト設定から取得されます。
            SoundProject project = FormsApplication.Instance.ProjectService.Project;
            if (project == null)
            {
                return;
            }

            //
            Dictionary<string, ColumnText> dictionary = new Dictionary<string, ColumnText>();
            dictionary.Add(ProjectParameterNames.Comment,
                            new ColumnText(project.CommentColumnText,
                                            SFresrc.MessageResource.ColumnText_Comment));
            dictionary.Add(ProjectParameterNames.Comment1,
                            new ColumnText(project.Comment1ColumnText,
                                            SFresrc.MessageResource.ColumnText_Comment1));
            dictionary.Add(ProjectParameterNames.Comment2,
                            new ColumnText(project.Comment2ColumnText,
                                            SFresrc.MessageResource.ColumnText_Comment2));
            dictionary.Add(ProjectParameterNames.Comment3,
                            new ColumnText(project.Comment3ColumnText,
                                            SFresrc.MessageResource.ColumnText_Comment3));
            dictionary.Add(ProjectParameterNames.Comment4,
                            new ColumnText(project.Comment4ColumnText,
                                            SFresrc.MessageResource.ColumnText_Comment4));
            dictionary.Add(ProjectParameterNames.Comment5,
                            new ColumnText(project.Comment5ColumnText,
                                            SFresrc.MessageResource.ColumnText_Comment5));
            dictionary.Add(ProjectParameterNames.Comment6,
                            new ColumnText(project.Comment6ColumnText,
                                            SFresrc.MessageResource.ColumnText_Comment6));
            dictionary.Add(ProjectParameterNames.Comment7,
                            new ColumnText(project.Comment7ColumnText,
                                            SFresrc.MessageResource.ColumnText_Comment7));
            dictionary.Add(ProjectParameterNames.Comment8,
                            new ColumnText(project.Comment8ColumnText,
                                            SFresrc.MessageResource.ColumnText_Comment8));
            dictionary.Add(ProjectParameterNames.Comment9,
                            new ColumnText(project.Comment9ColumnText,
                                            SFresrc.MessageResource.ColumnText_Comment9));

            //
            string formatTemplate = "{0}\r\n({1})";

            foreach (IHeaderItem headerItem in headerAdapter.Items)
            {
                if (dictionary.ContainsKey(headerItem.Name) != false)
                {
                    ColumnText columnText = dictionary[headerItem.Name];
                    if (columnText.Text != columnText.OriginalText)
                    {
                        headerItem.Text = String.Format
                            (formatTemplate,
                              columnText.Text,
                              columnText.OriginalText);
                    }
                    else
                    {
                        // 元の文字列に戻す処理は不要になりました。(2013.1.23 aoyagi)
                        // headerItem.Text = columnText.OriginalText;
                    }
                }
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class TransactionPack : IDisposable
    {
        private UserTransaction2[] transactions = null;

        public TransactionPack(UserTransaction2[] transactions)
        {
            this.transactions = transactions;
        }

        /// <summary>
        ///
        /// </summary>
        void IDisposable.Dispose()
        {
        }


        /// <summary>
        ///
        /// </summary>
        public bool Execute()
        {
            for (int index = 0; index < this.transactions.Length; index++)
            {
                if (this.transactions[index].Document.OperationHistory.Redo() == false)
                {
                    // Execute()が失敗したので、それまでに行った Execute()をキャンセルします
                    for (int rindex = index - 1; rindex >= 0; rindex--)
                    {
                        this.transactions[index].Document.OperationHistory.Undo();
                    }
                    return false;
                }
            }
            return true;
        }

        /// <summary>
        ///
        /// </summary>
        public bool Rollback()
        {
            for (int index = 0; index < this.transactions.Length; index++)
            {
                if (this.transactions[index].Document.OperationHistory.Undo() == false)
                {
                    // Rollback()が失敗したので、それまでに行った Rollback()をキャンセルします
                    for (int rindex = index - 1; rindex >= 0; rindex--)
                    {
                        this.transactions[index].Document.OperationHistory.Redo();
                    }
                    return false;
                }
            }
            return true;
        }

        /// <summary>
        ///
        /// </summary>
        public bool CanExecute()
        {
            foreach (UserTransaction2 transaction in this.transactions)
            {
                if (transaction.CanExecute() == false)
                {
                    return false;
                }
            }
            return true;
        }

        /// <summary>
        ///
        /// </summary>
        public bool CanRollback()
        {
            foreach (UserTransaction2 transaction in this.transactions)
            {
                if (transaction.CanRollback() == false)
                {
                    return false;
                }
            }
            return true;
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class TransactionPackStack
    {
        /// Undoスタック
        private LinkedList<TransactionPack> _undoStack = new LinkedList<TransactionPack>();

        /// Redoスタック
        private LinkedList<TransactionPack> _redoStack = new LinkedList<TransactionPack>();

        /// 保存位置のトランザクション
        //private TransactionPack _savedItem = null;

#if false
        /// <summary>
        /// トランザクションスタックの IEnumerator を返します。
        /// </summary>
        public IEnumerable<TransactionPack> AllItems
        {
            get
            {
                foreach ( TransactionPack transaction in _undoStack ) {
                    yield return transaction;
                }

                foreach ( TransactionPack transaction in _redoStack ) {
                    yield return transaction;
                }
            }
        }
#endif

        /// <summary>
        /// Undo 対象トランザクションの IEnumerator を返します。
        /// </summary>
        public IEnumerable<TransactionPack> UndoItems
        {
            get { return this._undoStack; }
        }

        /// <summary>
        /// Redo 対象トランザクションの IEnumerator を返します。
        /// </summary>
        public IEnumerable<TransactionPack> RedoItems
        {
            get { return this._redoStack; }
        }

#if false
        /// <summary>
        /// ドキュメントが編集されているかどうかを調べます。
        /// </summary>
        public bool IsDirty
        {
            get
            {
                if ( 0 == this._undoStack.Count ) { return ( null != this._savedItem ); }
                return ( this._undoStack.First.Value != this._savedItem );
            }
        }
#endif

        /// <summary>
        /// 現在位置のトランザクションを取得します。
        /// </summary>
        public TransactionPack CurrentItem
        {
            get { return UndoItem; }
        }

        /// <summary>
        /// Undo対象のトランザクションを取得します。
        /// </summary>
        public TransactionPack UndoItem
        {
            get { return (0 == this._undoStack.Count) ? null : this._undoStack.First.Value; }
        }

        /// <summary>
        /// Redo対象のトランザクションを取得します。
        /// </summary>
        public TransactionPack RedoItem
        {
            get { return (0 == this._redoStack.Count) ? null : this._redoStack.Last.Value; }
        }

#if false
        /// <summary>
        /// 保存位置のトランザクションを取得します。
        /// </summary>
        public TransactionPack SavedItem
        {
            get { return this._savedItem; }
        }
#endif

        /// <summary>
        /// トランザクションスタックが変更されたときに発生します。
        /// </summary>
        public event EventHandler ItemsChanged;

        /// <summary>
        /// Dirty 状態が変更されたときに発生します。
        /// </summary>
        public event EventHandler DirtyChanged;

        /// <summary>
        /// 現在位置のトランザクションが変更されたときに発生します。
        /// </summary>
        public event EventHandler CurrentItemChanged;

        /// <summary>
        /// 保存位置のトランザクションが変更されたときに発生します。
        /// </summary>
        public event EventHandler SavedItemChanged;

        /// <summary>
        /// スタックにトランザクションを追加します。
        /// </summary>
        /// <param name="transaction">追加するトランザクション。</param>
        public void Push(TransactionPack transaction)
        {
            if (null == transaction) { throw new ArgumentNullException("transaction"); }

            using (new DirtyChecker(this))
            {

                this._redoStack.Clear();
                this._undoStack.AddFirst(transaction);

                try
                {

                    //this.transaction.Stacks.Add( this );

                }
                catch (Exception exception)
                {

                    Debug.Assert(_undoStack.First.Value == transaction, "Internal Error : Failed to push transaction?");
                    _undoStack.RemoveFirst();

                    throw exception;

                }

                OnItemsChanged(EventArgs.Empty);

            }
        }

        /// <summary>
        /// スタックをクリアします。
        /// </summary>
        public void Clear()
        {
            using (new DirtyChecker(this))
            {

                if (0 < this._undoStack.Count || 0 < this._redoStack.Count)
                {

                    ClearTransactionStack(this._undoStack);
                    ClearTransactionStack(this._redoStack);

                    OnItemsChanged(EventArgs.Empty);
                    OnCurrentItemChanged(EventArgs.Empty);

                }

#if false
                if ( null != this._savedItem ) {
                    this._savedItem = null;
                    OnSavedItemChanged( EventArgs.Empty );
                }
#endif

                OnItemsChanged(EventArgs.Empty);

            }
        }

#if false
        /// <summary>
        /// 保存位置を現在位置にリセットします。
        /// </summary>
        public void ResetSavedItem()
        {
            if ( this._savedItem == UndoItem ) { return; }

            using ( new DirtyChecker( this ) ) {

                _savedItem = UndoItem;
                OnSavedItemChanged( EventArgs.Empty );

            }
        }
#endif

        /// <summary>
        ///
        /// </summary>
        public bool Undo()
        {
            if (null == UndoItem) { return false; }

            if (!UndoItem.Rollback()) { return false; }

            using (new DirtyChecker(this))
            {

                this._redoStack.AddLast(_undoStack.First.Value);
                this._undoStack.RemoveFirst();

                OnCurrentItemChanged(EventArgs.Empty);

            }

            return true;
        }

        /// <summary>
        ///
        /// </summary>
        public bool Redo()
        {
            if (null == RedoItem) { return false; }

            if (!RedoItem.Execute()) { return false; }

            using (new DirtyChecker(this))
            {

                this._undoStack.AddFirst(_redoStack.Last.Value);
                this._redoStack.RemoveLast();

                OnCurrentItemChanged(EventArgs.Empty);

            }

            return true;
        }

        /// <summary>
        ///
        /// </summary>
        public bool CanUndo()
        {
            if (null == UndoItem) { return false; }

            return UndoItem.CanRollback();
        }

        /// <summary>
        ///
        /// </summary>
        public bool CanRedo()
        {
            if (null == RedoItem) { return false; }

            return RedoItem.CanExecute();
        }

        /// <summary>
        /// トランザクションスタックが変更されたときに実行されます。
        /// </summary>
        /// <param name="e">イベントパラメータ。</param>
        protected virtual void OnItemsChanged(EventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (null != ItemsChanged)
            {
                ItemsChanged(this, e);
            }
        }

        /// <summary>
        /// Dirty 状態が変更されたときに実行されます。
        /// </summary>
        /// <param name="e">イベントパラメータ。</param>
        protected virtual void OnDirtyChanged(EventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (null != DirtyChanged)
            {
                DirtyChanged(this, e);
            }
        }

        /// <summary>
        /// 現在位置のトランザクションが変更されたときに実行されます。
        /// </summary>
        /// <param name="e">イベントパラメータ。</param>
        protected virtual void OnCurrentItemChanged(EventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (null != CurrentItemChanged)
            {
                CurrentItemChanged(this, e);
            }
        }

        /// <summary>
        /// 保存位置のトランザクションが変更されたときに実行されます。
        /// </summary>
        /// <param name="e">イベントパラメータ。</param>
        protected virtual void OnSavedItemChanged(EventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (null != SavedItemChanged)
            {
                SavedItemChanged(this, e);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void ClearTransactionStack(LinkedList<TransactionPack> stack)
        {
            Debug.Assert(null != stack, "invalid argument.");

            try
            {
                foreach (TransactionPack transaction in stack)
                {
                    IDisposable disposable = transaction as IDisposable;
                    if (disposable != null)
                    {
                        disposable.Dispose();
                    }
                }
            }
            finally
            {
                stack.Clear();
            }
        }

        /// <summary>
        ///
        /// </summary>
        private class DirtyChecker : IDisposable
        {
            private TransactionPackStack _owner;
            //private bool             _isDirtyOld = false;
            private bool _disposed = false;

            public DirtyChecker(TransactionPackStack owner)
            {
                Debug.Assert(null != owner, "unexpected error");

                _owner = owner;
                //_isDirtyOld = owner.IsDirty;
            }

            ~DirtyChecker()
            {
                Dispose();
            }

            void IDisposable.Dispose()
            {
                Dispose();
            }

            private void Dispose()
            {
                if (_disposed) { return; }

#if false
                if ( _isDirtyOld != _owner.IsDirty ) {
                    _owner.OnDirtyChanged( EventArgs.Empty );
                }
#endif

                _disposed = true;
            }
        }
    }


    /// <summary>
    ///
    /// </summary>
    public class UserTransaction2 : UserTransaction
    {
        /// <summary>
        ///
        /// </summary>
        public UserTransaction2(string description) : base(description)
        {
        }

        /// <summary>
        ///
        /// </summary>
        public Document Document
        {
            get;
            set;
        }
    }
}
