﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Drawing;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Conversion.NintendoWareReport;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Drawing;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Core.Reflection;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.FileFormats.Audio;
    using NintendoWare.SoundFoundation.FileFormats.Wave;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.CommandBars;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Configurations;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.FileFormats;
    using NintendoWare.SoundMaker.Framework.Preset;
    using NintendoWare.SoundMaker.Framework.Projects;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Framework.Utilities;
    using NintendoWare.SoundMaker.Framework.Windows.Forms.CommandHandlers;
    using NintendoWare.SoundMaker.Preview.Service;
    using CoreResources = SoundFoundation.Resources;
    using SFresrc = NintendoWare.SoundFoundation.Resources;

    /// <summary>
    ///
    /// </summary>
    public partial class SoundSetPanel : CommonListPanel, IQueryCommandParameter, IParameterPanelOperationExecutedListener, IEditableControlContainer
    {
        private CommonListCtrl _ListCtrlMain;
        private CommonListCtrl _ListCtrlSub;

        private CommonListAdapter _ListAdapterMain = null;
        private CommonListAdapter _ListAdapterSub = null;

        private Dictionary<Component, ListCtrlState> _ListCtrlState = new Dictionary<Component, ListCtrlState>();
        private bool enabledSaveListState = true;

        private DocumentReference _SoundSetDocumentRef = null;
        private SoundSet _SoundSet = null;

        private Component _CurrentItem = null;
        private CommonListCtrl _CurrentListCtrl = null;
        private ComponentListItem _CurrentListItem = null;

        private ListHeaderAdapterDictionary _headerAdapters = new ListHeaderAdapterDictionary();
        private IDictionary<string, XmlList> _xmlLists;

        private ToolStripAdapter _soundListMenuAdapter;
        private ToolStripMenuItem _HeaderContextMenuItem_AddPresetListColumns = null;
        private ToolStripMenuItem _HeaderContextMenuItem_ApplyPresetListColumns = null;

        private double _StreamSoundViewSplitterRatio = 0.5;
        private double _WaveSoundSetViewSplitterRatio = 0.5;
        private double _SequenceSoundSetViewSplitterRatio = 0.5;
        private double _GroupViewSplitterRatio = 0.5;

        private List<Tuple<int, IHeaderItem>> _aacUseHeaderItems = new List<Tuple<int, IHeaderItem>>();
        private List<Tuple<int, IHeaderItem>> _wavUseHeaderItems = new List<Tuple<int, IHeaderItem>>();

        private ListPartTextDrawer textDrawer = null;
        private ListPartCommonRowHeaderDrawer rowHeaderDrawer = null;

        public event EventHandler PreviewPlay;
        public event EventHandler PreviewMute;
        public event EventHandler PreviewSoloPlay;
        public event EventHandler SelectChangedListItem;

        public event EventHandler ShowedPanel;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public SoundSetPanel(DocumentReference soundSetDocumentRef)
        {
            this.textDrawer = new ListPartTextDrawer();
            this.rowHeaderDrawer = new ListPartCommonRowHeaderDrawer();
            this.ListDecorationEvaluator.AddDoShowErrorIconDelegate(null, this.MultiChannelDoShowErrorIcon);

            this._ListCtrlMain = CreateListCtrl();
            this._ListCtrlSub = CreateListCtrl();

            InitializeComponent();

            NPanel mainListPanel =
                (NPanel)this.splitContainer.Panel1.Controls.Find("mainListPanel", false)[0];

            mainListPanel.Controls.Add(this._ListCtrlMain);
            this._ListCtrlMain.Parent = mainListPanel;
            this._ListCtrlMain.AllowDragItem = true;
            this._ListCtrlMain.AttachedContextMenuStrip = this._soundListMenu;
            this._ListCtrlMain.ClickedOutsideUnselectItem = false;
            this._ListCtrlMain.DecideFileDropPosition = false;
            this._ListCtrlMain.DropOnDragItem = false;
            this._ListCtrlMain.Dock = DockStyle.Fill;
            this._ListCtrlMain.FollowSelectedItemToCaret = true;
            this._ListCtrlMain.InterlockSelectedItemAndSubItem = true;
            this._ListCtrlMain.MoveNewLineCaretLeftAndRight = true;
            this._ListCtrlMain.Name = "_ListCtrlMain";
            this._ListCtrlMain.ReadOnly = false;
            this._ListCtrlMain.StopCaretLeftAndRight = false;
            this._ListCtrlMain.SuspendTransaction = false;

            NPanel subListPanel =
                (NPanel)this.splitContainer.Panel2.Controls.Find("subListPanel", false)[0];

            subListPanel.Controls.Add(this._ListCtrlSub);
            this._ListCtrlSub.Parent = subListPanel;
            this._ListCtrlSub.AllowDragItem = true;
            this._ListCtrlSub.AttachedContextMenuStrip = this._soundListMenu;
            this._ListCtrlSub.ClickedOutsideUnselectItem = false;
            this._ListCtrlSub.DecideFileDropPosition = false;
            this._ListCtrlSub.Dock = DockStyle.Fill;
            this._ListCtrlSub.DropOnDragItem = false;
            this._ListCtrlSub.FollowSelectedItemToCaret = true;
            this._ListCtrlSub.InterlockSelectedItemAndSubItem = true;
            this._ListCtrlSub.MoveNewLineCaretLeftAndRight = true;
            this._ListCtrlSub.Name = "_ListCtrlSub";
            this._ListCtrlSub.ReadOnly = false;
            this._ListCtrlSub.StopCaretLeftAndRight = false;
            this._ListCtrlSub.SuspendTransaction = false;

            //
            _SoundSetDocumentRef = soundSetDocumentRef.Clone();
            SoundSetDocument.DirtyChanged += OnDirtyChanged;
            SoundSetDocument.SoundSetChanged += OnSoundSetChanged;
            SoundSetDocument.OperationHistory.ItemsChanged += OnOperationHistoryItemsChanged;
            SoundSetDocument.OperationHistory.CurrentItemChanged += OnOperationHistoryCurrentItemChanged;

            SetSoundSet(SoundSetDocument.SoundSet);

            //
            _ListAdapterMain = CreateListAdapter();
            _ListAdapterMain.OperationHistory = SoundSetDocument.OperationHistory;
            _ListAdapterMain.PreviewPlay += OnPreviewPlay;
            _ListAdapterMain.PreviewMute += OnPreviewMute;
            _ListAdapterMain.PreviewSoloPlay += OnPreviewSoloPlay;

            _ListAdapterSub = CreateListAdapter();
            _ListAdapterSub.OperationHistory = SoundSetDocument.OperationHistory;
            _ListAdapterSub.PreviewPlay += OnPreviewPlay;
            _ListAdapterSub.PreviewMute += OnPreviewMute;
            _ListAdapterSub.PreviewSoloPlay += OnPreviewSoloPlay;
            _ListAdapterSub.Updated += OnUpdatedSub;

            //
            _ListCtrlMain.HeaderHeight = 28;
            _ListCtrlMain.ItemsSource = _ListAdapterMain;
            _ListCtrlMain.SelectChanged += OnSelectChangedListCtrlMain;
            _ListCtrlMain.SelectChanged += OnSelectChangedListItem;
            _ListCtrlMain.ItemDoubleClicked += OnItemDoubleClicked;
            _ListCtrlMain.Enter += OnEnterd;
            _ListCtrlMain.Leave += OnLeave;
            _ListCtrlMain.QueryFileDropped += OnQueryFileDropped;
            _ListCtrlMain.FileDropped += OnFileDropped;
            _ListCtrlMain.QueryDropped += OnQueryDropped;
            _ListCtrlMain.Dropped += OnDropped;
            _ListCtrlMain.EditBegan += OnEditBegan;
            _ListCtrlMain.EditEnded += OnEditEnded;
            _ListCtrlMain.PartDrawers.Replace(this.textDrawer);
            _ListCtrlMain.PartDrawers.Replace(this.rowHeaderDrawer);
            _ListCtrlMain.VerticalScrollValueChanged += OnListCtrlMainVerticalScrollValueChanged;

            _HeaderContextMenuItem_AddPresetListColumns = new ToolStripMenuItem
                (Resources.MessageResource.HeaderMenuItem_AddPresetListColumns);
            _HeaderContextMenuItem_AddPresetListColumns.Click +=
                OnHeaderMenuItemAddPresetListColumnsClick;
            _HeaderContextMenuItem_ApplyPresetListColumns = new ToolStripMenuItem
                (Resources.MessageResource.HeaderMenuItem_ApplyPresetListColumns);

            _ListCtrlMain.HeaderCtrl.InsertContextMenuItems =
                new ToolStripMenuItem[] { _HeaderContextMenuItem_AddPresetListColumns,
                                          _HeaderContextMenuItem_ApplyPresetListColumns };
            _ListCtrlMain.HeaderCtrl.ContextMenuOpening += OnHeaderCtrlContextMenuOpening;

            //
            _ListCtrlSub.HeaderHeight = 28;
            _ListCtrlSub.ItemsSource = _ListAdapterSub;
            _ListCtrlSub.SelectChanged += OnSelectChangedListCtrlSub;
            _ListCtrlSub.SelectChanged += OnSelectChangedListItem;
            _ListCtrlSub.ItemDoubleClicked += OnItemDoubleClicked;
            _ListCtrlSub.Enter += OnEnterd;
            _ListCtrlSub.Leave += OnLeave;
            _ListCtrlSub.QueryFileDropped += OnQueryFileDropped;
            _ListCtrlSub.FileDropped += OnFileDropped;
            _ListCtrlSub.QueryDropped += OnQueryDropped;
            _ListCtrlSub.Dropped += OnDropped;
            _ListCtrlSub.EditBegan += OnEditBegan;
            _ListCtrlSub.EditEnded += OnEditEnded;
            _ListCtrlSub.PartDrawers.Replace(this.textDrawer);
            _ListCtrlSub.PartDrawers.Replace(this.rowHeaderDrawer);
            _ListCtrlSub.VerticalScrollValueChanged += OnListCtrlSubVerticalScrollValueChanged;

            _HeaderContextMenuItem_AddPresetListColumns = new ToolStripMenuItem
                (Resources.MessageResource.HeaderMenuItem_AddPresetListColumns);
            _HeaderContextMenuItem_AddPresetListColumns.Click +=
                OnHeaderMenuItemAddPresetListColumnsClick;

            _ListCtrlSub.HeaderCtrl.InsertContextMenuItems =
                new ToolStripMenuItem[] { _HeaderContextMenuItem_AddPresetListColumns,
                                          _HeaderContextMenuItem_ApplyPresetListColumns };
            _ListCtrlSub.HeaderCtrl.ContextMenuOpening += OnHeaderCtrlContextMenuOpening;

            //
            _CurrentListCtrl = _ListCtrlMain;

            InitializeCommandBindings();

            ApplySoundSetConfiguration();
            UpdatedOptions();
            UpdateReadOnly();
        }

        /// <summary>
        ///
        /// </summary>
        void IEditableControlContainer.CancelEdit()
        {
            this._ListCtrlMain.CancelEdit();
            this._ListCtrlSub.CancelEdit();
        }

        /// <summary>
        /// パネル内のリストコントロールを取得します。
        /// </summary>
        public override CommonListCtrl[] ListControls
        {
            get
            {
                return new CommonListCtrl[] {
                    this._ListCtrlMain,
                    this._ListCtrlSub,
                };
            }
        }

        /// <summary>
        /// アクティブになった時に呼ばれます。
        /// </summary>
        public override void Activated()
        {
            SetSelectedItemsToParameterPanel(LastActivedListControl);
            ParameterPanel.SetSoundSetPanel(this);
        }

        /// <summary>
        /// ディアクティブになった時に呼ばれます。
        /// </summary>
        public override void Deactivated()
        {
        }

        /// <summary>
        /// クリップボードが更新された時に呼ばれる
        /// </summary>
        public override void UpdatedClipboard()
        {
            _ListCtrlMain.ClearDashLine();
            _ListCtrlSub.ClearDashLine();
        }

        /// <summary>
        /// プロジェクト設定によりユーザーパラメータが変更された時に呼ばれます。
        /// </summary>
        public override void ApplyUserParameterColumns()
        {
            this._ListCtrlMain.UpdateUserParameterColumnStyles();
            this._ListCtrlSub.UpdateUserParameterColumnStyles();

            ListConfigurationApplier applier = FormsApplication.Instance.UIService.CreateListConfigurationApplier(this._headerAdapters);
            foreach (XmlList xmlList in _xmlLists.Values)
            {
                applier.Extract(xmlList);
            }

            foreach (XmlList xmlList in _xmlLists.Values)
            {
                applier.Apply(xmlList);
            }

            this._ListCtrlMain.UpdateInfos();
            this._ListCtrlSub.UpdateInfos();
            this._ListCtrlMain.Invalidate();
            this._ListCtrlSub.Invalidate();
        }

        /// <summary>
        /// プロジェクト設定によりカラム名が変更された時に呼ばれます。
        /// </summary>
        public override void ApplyColumnsText()
        {
            CommentColumnTextApplier.Apply(this._headerAdapters.Values.ToArray());

            this._ListCtrlMain.InvalidateHeader();
            this._ListCtrlSub.InvalidateHeader();
        }

        /// <summary>
        /// オプション設定が変更された時に呼ばれます。
        /// </summary>
        public override void ApplyOptionSettings()
        {
            this._ListCtrlMain.ApplyOptionConfigurations();
            this._ListCtrlSub.ApplyOptionConfigurations();
        }

        /// <summary>
        ///
        /// </summary>
        public override void UpdatedOptions()
        {
            _ListCtrlMain.UpdatePartDrawerOptions();
            _ListCtrlSub.UpdatePartDrawerOptions();
        }

        /// <summary>
        ///
        /// </summary>
        public override void ApplySettings(SettingKinds kind)
        {
            switch (kind)
            {
                case SettingKinds.ColorComment:
                    CommandTextReplacer.ReplaceColorCommentCommandTexts
                        (ProjectService, this._soundListMenuAdapter);
                    break;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public SoundSet SoundSet
        {
            get { return _SoundSet; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public SoundSetDocument SoundSetDocument
        {
            get { return _SoundSetDocumentRef.Document as SoundSetDocument; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override Document Document
        {
            get { return SoundSetDocument; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public Component CurrentItem
        {
            get { return _CurrentItem; }
        }

        public bool IsSingleList
        {
            get { return !_ListCtrlSub.Visible; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public Type TargetItemType
        {
            get
            {
                if (null == _CurrentItem) { return null; }
                return _CurrentItem.GetType();
            }
        }

        /// <summary>
        ///
        /// </summary>
        public string TargetItemPath
        {
            get
            {
                if (null == _CurrentItem) { return string.Empty; }

                Component targetItem = _CurrentItem;

                if (0 < _ListCtrlMain.SelectedItems.Length)
                {
                    targetItem = (_ListCtrlMain.SelectedItems[0] as ComponentListItem).Target;
                }

                if (_CurrentItem is WaveSoundSetPack ||
                     _CurrentItem is SequenceSoundSetPack ||
                     _CurrentItem is GroupPack)
                {

                    if (0 < _ListCtrlSub.SelectedItems.Length)
                    {
                        targetItem = (_ListCtrlSub.SelectedItems[0] as ComponentListItem).Target;
                    }
                    else if (0 < _ListAdapterSub.Items.Count)
                    {
                        targetItem = (_ListAdapterSub.Items[0] as ComponentListItem).Target;
                    }

                }

                if (null == targetItem)
                {
                    Debug.Fail("targetItem must not be null");
                    return string.Empty;
                }

                return new TreeTracer('/').GetPath(targetItem, SoundSet);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public CommonListCtrl ListCtrlMain
        {
            get { return _ListCtrlMain; }
        }

        /// <summary>
        ///
        /// </summary>
        public CommonListCtrl ListCtrlSub
        {
            get { return _ListCtrlSub; }
        }

        /// <summary>
        /// カレントコントロールを取得します。
        /// 検索で使用されます。
        /// </summary>
        public override CommonListCtrl CurrentListCtrl
        {
            get { return _CurrentListCtrl; }
        }

        /// <summary>
        /// セルの装飾に関する情報を評価します。
        /// </summary>
        protected CommonListDecorationEvaluator ListDecorationEvaluator { get; } =
            new CommonListDecorationEvaluator();

        /// <summary>
        /// カレントコントロールの設定を行います。
        /// 検索で使用されます。
        /// </summary>
        public void SetCurrentListCtrl(CommonListCtrl listCtrl)
        {
            _CurrentListCtrl = listCtrl;
        }

        /// <summary>
        /// コマンドが実行された後に呼ばれます。
        /// </summary>
        public override void CommandExecuted(Command command)
        {
            //
            if (command == EditCommands.CopyCell)
            {
                this._CurrentListCtrl.SetDashLineBySelectedSubItem();
            }

            //
            if (command == ToolCommands.ShowOptions ||
                command == ProjectCommands.ShowProjectSettings)
            {
                MainWindow.UpdatedOptions();
            }

            //
            if (command == EditCommands.EnableItem ||
                command == EditCommands.DisableItem)
            {
                MainWindow.ProjectPanel.UpdateProjectCtrl();
            }

            //
            if (ActiveListCtrl == _ListCtrlMain)
            {
                UpdateListCtrlSub();
            }

            RedrawControls();
            MainWindow.RedrawPanels();
            BuildListCommandUI();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override SoundDocument TargetDocument
        {
            get { return SoundSetDocument; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override Component[] TargetComponents
        {
            get
            {
                Component[] components = null;

                if (SoundSetDocument.IsReadOnly == false)
                {
                    if (null != ActiveListCtrl)
                    {
                        components = ActiveListCtrl.GetComponentsBySelectedItem();
                        components = ActiveListCtrl.SortToDisplayedOrder(components);
                    }
                }

                if (components == null)
                {
                    components = new Component[0];
                }

                return components;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private Component InsertParentComponent
        {
            get
            {
                if (null == ActiveListCtrl) { return null; }

                if (ActiveListCtrl == _ListCtrlSub)
                {

                    Component[] mainListItems = _ListCtrlMain.GetComponentsBySelectedItem();

                    if (1 == mainListItems.Length)
                    {
                        return mainListItems[0];
                    }

                }
                else if (ActiveListCtrl == _ListCtrlMain)
                {

                    Component[] components = _ListCtrlMain.GetComponentsBySelectedItem();
                    if (0 < components.Length) { return components[0]; }

                    return _CurrentItem;

                }

                return null;
            }
        }

        private double CurrentViewSplitterRatio
        {
            get
            {
                if (_CurrentItem is StreamSoundPack)
                {
                    return _StreamSoundViewSplitterRatio;
                }
                if (_CurrentItem is WaveSoundSetPack)
                {
                    return _WaveSoundSetViewSplitterRatio;
                }
                if (_CurrentItem is SequenceSoundSetPack)
                {
                    return _SequenceSoundSetViewSplitterRatio;
                }
                if (_CurrentItem is GroupPack)
                {
                    return _GroupViewSplitterRatio;
                }

                return 0.0;
            }
            set
            {
                if (!(_CurrentItem is StreamSoundPack ||
                        _CurrentItem is WaveSoundSetPack ||
                        _CurrentItem is SequenceSoundSetPack ||
                        _CurrentItem is GroupPack))
                {
                    return;
                }

                if (value <= 0.0 || 1.0 <= value)
                {
                    throw new ArgumentOutOfRangeException("value");
                }

                if (_CurrentItem is StreamSoundPack)
                {
                    _StreamSoundViewSplitterRatio = value;
                }
                else if (_CurrentItem is WaveSoundSetPack)
                {
                    _WaveSoundSetViewSplitterRatio = value;
                }
                else if (_CurrentItem is SequenceSoundSetPack)
                {
                    _SequenceSoundSetViewSplitterRatio = value;
                }
                else if (_CurrentItem is GroupPack)
                {
                    _GroupViewSplitterRatio = value;
                }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool Show(SoundSetItem item, bool force = false)
        {
            if (item is SequenceSoundPack ||
                item is SoundSetBankPack ||
                item is WaveArchivePack ||
                item is PlayerPack ||
                item is WaveSoundSetBase ||
                item is SequenceSoundSetBase)
            {
                Show(item, null, 0.0, force);
                return true;
            }

            if (item is StreamSoundPack)
            {
                Show(item, null, _StreamSoundViewSplitterRatio, force);
                return true;
            }

            if (item is WaveSoundSetPack)
            {
                Show(item, null, _WaveSoundSetViewSplitterRatio, force);
                return true;
            }

            if (item is SequenceSoundSetPack)
            {
                Show(item, null, _SequenceSoundSetViewSplitterRatio, force);
                return true;
            }

            if (item is GroupPack)
            {
                Show(item, null, _GroupViewSplitterRatio, force);
                return true;
            }

            return false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override void RefreshPanel()
        {
            if (null == SoundSet || 0 == SoundSet.Children.Count) { return; }

            if (null != _CurrentItem)
            {
                Show(FindItem(_CurrentItem.Name, SoundSet), true);
            }
            else
            {
                Show(SoundSet.Children[0] as SoundSetItem, true);
            }
        }

        public void UpdateTabImage()
        {
            this.UpdateImage(FindItem(_CurrentItem.Name, SoundSet));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected SoundProjectConfiguration ProjectConfiguration
        {
            get { return FormsApplication.Instance.ProjectConfiguration; }
        }

        protected XmlSoundSetDocumentView SoundSetViewConfiguration
        {
            get
            {
                if (null == ProjectConfiguration) { return null; }
                if (!ProjectConfiguration.DocumentViews.
                      ContainsKey(SoundSetDocument.Resource.Key)) { return null; }

                return ProjectConfiguration.DocumentViews[SoundSetDocument.Resource.Key]
                            as XmlSoundSetDocumentView;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected SoundProjectService ProjectService
        {
            get { return FormsApplication.Instance.ProjectService; }
        }

        ///--------------------------------
        /// <summary>
        /// コンポーネント作成サービスを取得します。
        /// </summary>
        protected CreateComponentService CreateComponentService
        {
            get { return FormsApplication.Instance.CreateComponentService; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected MainWindow MainWindow
        {
            get { return FormsApplication.Instance.UIService.MainWindow; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected ParameterPanel ParameterPanel
        {
            get
            {
                return FormsApplication.Instance.UIService.
                    MainWindow.ToolPages[ParameterPanel.PageName] as ParameterPanel;
            }
        }

        ///--------------------------------
        /// <summary>
        /// アクティブなリストコントロールを取得します。
        /// </summary>
        protected override CommonListCtrl ActiveListCtrl
        {
            get
            {
                if (_ListCtrlMain.ContainsFocus) { return _ListCtrlMain; }
                if (_ListCtrlSub.ContainsFocus) { return _ListCtrlSub; }

                return null;
            }
        }

        ///--------------------------------
        /// <summary>
        /// アクティブなリストアダプタを取得します。
        /// </summary>
        protected override ComponentListAdapter ActiveListAdapter
        {
            get
            {
                if (_ListCtrlMain.ContainsFocus) { return _ListAdapterMain; }
                if (_ListCtrlSub.ContainsFocus) { return _ListAdapterSub; }

                return null;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override ComponentService ComponentService
        {
            get { return ProjectService; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private ComponentService TargetComponentService
        {
            get { return FormsApplication.Instance.ProjectService; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override void UpdateTitle()
        {
            if (this.InvokeRequired == true)
            {
                this.Invoke(new MethodInvoker(delegate () { UpdateTitle(_CurrentItem as SoundSetItem); }));
            }
            else
            {
                UpdateTitle(_CurrentItem as SoundSetItem);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override void RedrawPanel()
        {
            RedrawControls();
        }

        /// <summary>
        /// アクティブになった時に呼ばれます。
        /// </summary>
        public override void ActivatePanel(bool resetSelection, bool mainList)
        {
            ActivateList((mainList) ? _ListCtrlMain : _ListCtrlSub, resetSelection);
        }

        /// <summary>
        ///
        /// </summary>
        public override void OnClosed()
        {
            ApplicationBase.Instance.ProjectConfiguration.Saving -= OnProjectConfigurationSaving;

            ExtractSoundSetConfiguration();

            SoundSetDocument.DirtyChanged -= OnDirtyChanged;
            SoundSetDocument.SoundSetChanged -= OnSoundSetChanged;

            _SoundSetDocumentRef.Close();
            SetSoundSet(null);

            base.OnClosed();
        }

        public bool AddBank(string filePath)
        {
            SoundSetBankBase bank = ApplicationBase.Instance.Traits.ComponentTraits.SoundSetComponentFactory.Create(typeof(SoundSetBankBase)) as SoundSetBankBase;
            bank.FilePath = filePath;
            bank.Name = ItemNamer.CreateUniqueNameFromFileName
                (ProjectService.ComponentDictionary,
                 new ItemNamingSettings(ProjectService.Project),
                 ProjectService.Project.BankNamePrefix, String.Empty, filePath);

            return AddSoundSetItemHandler.Execute(SoundSetDocument,
                                                  (SoundSetItem)_CurrentItem, null, bank);
        }

        public override void ApplyProjectConfigurations()
        {
            ApplySoundSetConfiguration();
        }

        /// <summary>
        /// プリセット名からをプリセットを適用します。
        /// </summary>
        public override void ApplyPresetListColumns(string presetName)
        {
            CommonListCtrl listCtrl = this.GetListCtrl();
            string listName = GetListName(_CurrentItem, _ListCtrlMain == listCtrl);
            ListConfigurationApplier applier = FormsApplication.Instance.UIService.CreateListConfigurationApplier(_headerAdapters);

            ListHeaderHelper.ApplyPresetListColumns(applier, listName, presetName);

            listCtrl.UpdateInfos();

            MainWindow.PresetListColumnsPresetCurrentNameOnSplitButton = presetName;
            PresetListColumnsService preset = ApplicationBase.Instance.PresetListColumnsService;
            preset.SetCurrentPresetName(listName, presetName);
        }

        /// <summary>
        /// プリセットを更新します。
        /// </summary>
        public override void UpdatePresetListColumns()
        {
            if (_CurrentListCtrl != null)
            {
                this.UpdatePresetListColumnsSplitButton(_CurrentListCtrl);
            }
        }

        /// <summary>
        ///
        /// </summary>
        void IParameterPanelOperationExecutedListener.OperationExecuted(OperationExecutedEventArgs e)
        {
            RedrawPanel();
        }

        /// <summary>
        ///
        /// </summary>
        bool IQueryCommandParameter.ContainsParameter(string parameterName)
        {
            switch (parameterName)
            {
                case CommandParameterNames.TargetComponentService:
                case CommandParameterNames.TargetDocuments:
                case CommandParameterNames.TargetComponents:
                case CommandParameterNames.InsertParentComponent:
                case CommandParameterNames.InsertTargetComponent:
                case "ColumnNames":
                case CaretItemSetter.CommandParameterOfListCtrl:
                case "Instance":
                    return true;
            }

            return false;
        }

        object IQueryCommandParameter.GetParameter(string parameterName)
        {
            switch (parameterName)
            {
                case CommandParameterNames.TargetComponentService:
                    return TargetComponentService;

                case CommandParameterNames.TargetDocuments:
                    if (null == TargetDocument) { return new SoundDocument[0]; }
                    return new SoundDocument[] { TargetDocument };

                case CommandParameterNames.TargetComponents:
                    return TargetComponents;

                case CommandParameterNames.InsertParentComponent:
                    return InsertParentComponent;

                case CommandParameterNames.InsertTargetComponent:
                    Component[] components = TargetComponents;
                    return (0 == components.Length) ? null : TargetComponents[0];

                case "ColumnNames":
                    return ActiveListCtrl.ItemsName
                        .Where(s => s != "RowHeader").ToArray();

                case CaretItemSetter.CommandParameterOfListCtrl:
                    return ActiveListCtrl;

                case "Instance":
                    return this;
            }

            throw new KeyNotFoundException();
        }

        /// <summary>
        /// メイン、サブのリストコントロールから選択されているサウンドを取得します。
        /// </summary>
        protected Sound[] GetSelectedSoundsFromListControl()
        {
            List<Sound> list = new List<Sound>();
            Component[] components = null;

            components = _ListCtrlMain.GetComponentsBySelectedItem();
            if (components.Length > 0)
            {
                list.AddRange(components
                               .Where(c => c is Sound)
                               .Cast<Sound>()
                               .ToArray());
            }

            components = _ListCtrlSub.GetComponentsBySelectedItem();
            if (components.Length > 0)
            {
                list.AddRange(components
                               .Where(c => c is Sound)
                               .Cast<Sound>()
                               .ToArray());
            }

            return list.ToArray();
        }

        /// <summary>
        /// ストリームトラックが追加、削除、編集された時に呼ばれます。
        /// </summary>
        protected virtual void StreamSoundTracksModified(StreamSoundBase streamSound)
        {
        }

        /// <summary>
        /// リストコントロールを作成します。
        /// </summary>
        protected virtual CommonListCtrl CreateListCtrl()
        {
            return new CommonListCtrl();
        }

        /// <summary>
        /// リストコントロール Main の縦スクロールボックスが移動した時に呼ばれます。
        /// </summary>
        protected virtual void OnListCtrlMainVerticalScrollValueChanged(object sender, EventArgs e)
        {
        }

        /// <summary>
        /// リストコントロール Sub の縦スクロールボックスが移動した時に呼ばれます。
        /// </summary>
        protected virtual void OnListCtrlSubVerticalScrollValueChanged(object sender, EventArgs e)
        {
        }

        ///--------------------------------
        /// <summary>
        /// コントロールがロードされると発生します。
        /// </summary>
        /// <param name="e">空のイベントデータ。</param>
        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);

            InitializeMenu();
            ApplicationBase.Instance.ProjectConfiguration.Saving += OnProjectConfigurationSaving;

            // Load直後のリストにフォーカスが渡るタイミングを検知する術がないので、
            // BeginInvoke でタイミングを遅らせる
            BeginInvoke(new MethodInvoker(BuildCommandUIs));
        }

        ///--------------------------------
        /// <summary>
        /// コントロールが入力されると発生します。
        /// </summary>
        /// <param name="e">空のイベントデータ。</param>
        protected override void OnEnter(EventArgs e)
        {
            base.OnEnter(e);

            BuildListCommandUI();
        }

        ///--------------------------------
        /// <summary>
        /// アクティブリストに対応したアイテムの追加コマンドを取得します。
        /// </summary>
        private Command AddItemCommand
        {
            get
            {
                ComponentListAdapter adapter = null;

                if ((adapter = ActiveListAdapter) == null)
                {
                    return null;
                }

                if (adapter.ItemsOwner is StreamSoundPack)
                {
                    return SoundListCommands.AddStreamSound;
                }

                if (adapter.ItemsOwner is StreamSoundBase)
                {
                    return SoundListCommands.AddStreamSoundTrack;
                }

                if (adapter.ItemsOwner is WaveSoundSetPack)
                {
                    return SoundListCommands.AddWaveSoundSet;
                }
                if (adapter.ItemsOwner is WaveSoundSetBase)
                {
                    return SoundListCommands.AddWaveSound;
                }
                if (adapter.ItemsOwner is SequenceSoundSetPack)
                {
                    return SoundListCommands.AddSequenceSoundSet;
                }
                if (adapter.ItemsOwner is SequenceSoundSetBase)
                {
                    return SoundListCommands.AddSequenceSound;
                }
                if (adapter.ItemsOwner is SequenceSoundPack)
                {
                    return SoundListCommands.AddSequenceSound;
                }
                if (adapter.ItemsOwner is SoundSetBankPack)
                {
                    return SoundListCommands.AddBank;
                }
                if (adapter.ItemsOwner is WaveArchivePack)
                {
                    return SoundListCommands.AddWaveArchive;
                }
                if (adapter.ItemsOwner is PlayerPack)
                {
                    return SoundListCommands.AddPlayer;
                }
                if (adapter.ItemsOwner is GroupPack)
                {
                    return SoundListCommands.AddGroup;
                }
                if (adapter.ItemsOwner is GroupBase)
                {
                    return SoundListCommands.AddGroupItem;
                }

                return null;
            }
        }

        ///--------------------------------
        /// <summary>
        /// アクティブリストに対応したアイテムの追加コマンドを取得します。
        /// </summary>
        protected virtual Command InsertItemCommand
        {
            get
            {
                ComponentListAdapter adapter = null;

                if ((adapter = ActiveListAdapter) == null)
                {
                    return null;
                }

                if (adapter.ItemsOwner is StreamSoundPack)
                {
                    return SoundListCommands.InsertStreamSound;
                }
                if (adapter.ItemsOwner is StreamSoundBase)
                {
                    return SoundListCommands.InsertStreamSoundTrack;
                }
                if (adapter.ItemsOwner is WaveSoundSetPack)
                {
                    return SoundListCommands.InsertWaveSoundSet;
                }
                if (adapter.ItemsOwner is WaveSoundSetBase)
                {
                    return SoundListCommands.InsertWaveSound;
                }
                if (adapter.ItemsOwner is SequenceSoundSetPack)
                {
                    return SoundListCommands.InsertSequenceSoundSet;
                }
                if (adapter.ItemsOwner is SequenceSoundSetBase)
                {
                    return SoundListCommands.InsertSequenceSound;
                }
                if (adapter.ItemsOwner is SequenceSoundPack)
                {
                    return SoundListCommands.InsertSequenceSound;
                }
                if (adapter.ItemsOwner is SoundSetBankPack)
                {
                    return SoundListCommands.InsertBank;
                }
                if (adapter.ItemsOwner is WaveArchivePack)
                {
                    return SoundListCommands.InsertWaveArchive;
                }
                if (adapter.ItemsOwner is PlayerPack)
                {
                    return SoundListCommands.InsertPlayer;
                }
                if (adapter.ItemsOwner is GroupPack)
                {
                    return SoundListCommands.InsertGroup;
                }
                if (adapter.ItemsOwner is GroupBase)
                {
                    return SoundListCommands.InsertGroupItem;
                }

                return null;
            }
        }

        /// <summary>
        /// リストの行ヘッダがダブルクリックされたら呼ばれます。
        /// </summary>
        protected virtual void ExecuteItem(ComponentListItem item)
        {
            if (null == item) { throw new ArgumentNullException("item"); }

            if (item.Target is SoundSetBankBase)
            {
                SoundSetBankBase bank = item.Target as SoundSetBankBase;

                string validFilePath = new ReferenceFileFinder()
                {
                    Text = bank.FilePath,
                }.Validate(bank.FilePath);

                if (null == validFilePath) { return; }

                if (validFilePath != bank.FilePath)
                {
                    Operation operation = new SetParameterOperation
                        (item.Target.Parameters,
                          ProjectParameterNames.FilePath, validFilePath);
                    operation.Execute();
                    SoundSetDocument.OperationHistory.AddOperation(operation);
                }

                MainWindow.OpenFile(bank);
            }

            // グループアイテムでは、対象アイテムにジャンプします。(2012.5.31 aoyagi)
            if (item.Target is GroupItemBase)
            {
                GroupItemBase groupItem = item.Target as GroupItemBase;
                MainWindow.ShowPageByComponent(groupItem.Target);
            }
        }

        ///--------------------------------
        /// <summary>
        /// コマンドの関連付けを初期化します。
        /// </summary>
        protected virtual void InitializeCommandBindings()
        {
            // ファイル関連コマンドハンドラ
            BindCommand(new CommandBinding(this, new QueryAndImportBankHandler(this)));
            BindCommand(new CommandBinding(this, FileCommands.ImportDLS.ID,
                                             QueryStatusImportDLS, ExecuteImportDLS));
            BindCommand(new CommandBinding(this, FileCommands.ImportStxt.ID,
                                             QueryStatusImportStxt, ExecuteImportStxt));

            // 編集関連コマンドハンドラ
            BindCommand(new CommandBinding(this, EditCommands.SelectAll.ID,
                                             QueryStatusSelectAll, ExecuteSelectAll));
            BindCommand(new CommandBinding(this, EditCommands.Undo.ID, QueryStatusUndo, ExecuteUndo));
            BindCommand(new CommandBinding(this, EditCommands.Redo.ID, QueryStatusRedo, ExecuteRedo));
            //BindCommand( new CommandBinding( this, new CutHandler( this)));
            BindCommand(new CommandBinding(this, new SoundSetPanelCutHandler(this)));
            BindCommand(new CommandBinding(this, new CopyHandler(this)));
            BindCommand(new CommandBinding(this, EditCommands.Paste.ID, QueryStatusPaste, ExecutePaste));
            //BindCommand( new CommandBinding( this, new DeleteHandler( this)));
            BindCommand(new CommandBinding(this, new SoundSetPanelDeleteHandler(this)));
            BindCommand(new CommandBinding(this, EditCommands.CopyCell.ID,
                                             QueryStatusCopyCell, ExecuteCopyCell));
            BindCommand(new CommandBinding(this, EditCommands.PasteSpecial.ID,
                                             QueryStatusPasteSpecial, ExecutePasteSpecial));
            BindCommand(new CommandBinding(this, EditCommands.BatchEdit.ID,
                                             QueryStatusBatchEdit, ExecuteBatchEdit));
            BindCommand(new CommandBinding(this, new EnableItemHandler(this, true)));
            BindCommand(new CommandBinding(this, new EnableItemHandler(this, false)));
            BindCommand(new CommandBinding(this, new SetItemColorHandler(this)));
            BindCommand(new CommandBinding(this, EditCommands.FindReferences.ID,
                                             QueryStatusFindReferences, ExecuteFindReferences));
            BindCommand(new CommandBinding(this, EditCommands.EnableSndEdit.ID,
                                             QueryStatusSndEdit, ExecuteEnableSndEdit));
            BindCommand(new CommandBinding(this, EditCommands.DisableSndEdit.ID,
                                             QueryStatusSndEdit, ExecuteDisableSndEdit));
            BindCommand(new CommandBinding(this, EditCommands.CombineAACTrack.ID,
                                           QueryCombineAACTrack, ExecuteCombineAACTrack));
            BindCommand(new CommandBinding(this, EditCommands.DivideAACTrack.ID,
                                           QueryDivideAACTrack, ExecuteDivideAACTrack));

            /// デフォルト非表示です。表示するには派生先で BindCommand で上書きしてください。
            BindCommand(EditCommands.UseWaveSoundWaveFile.ID, (c) => CommandStatus.Supported, (c) => true);
            BindCommand(EditCommands.UseWaveSoundResource.ID, (c) => CommandStatus.Supported, (c) => true);


            // プロジェクト関連コマンドハンドラ
            BindCommand(new CommandBinding(this, ProjectCommands.AddItem.ID,
                                             QueryStatusAddItem, ExecuteAddItem));
            BindCommand(new CommandBinding(this, ProjectCommands.InsertItem.ID,
                                             QueryStatusInsertItem, ExecuteInsertItem));
            BindCommand(new CommandBinding(this,
                        new QueryAndAddWaveSoundSetHandler(true, this)));
            BindCommand(new CommandBinding(this,
                        new QueryAndAddWaveSoundHandler(true, this, ApplicationBase.DefaultPlayerName)));
            BindCommand(new CommandBinding(this,
                        new QueryAndAddSequenceSoundHandler(true, this, ApplicationBase.DefaultPlayerName)));
            BindCommand(new CommandBinding(this,
                        new QueryAndAddSequenceSoundSetHandler(true, this)));
            BindCommand(new CommandBinding(this,
                        new QueryAndAddPlayerHandler(true, this)));
            BindCommand(new CommandBinding(this,
                        new QueryAndAddBankHandler(true, this)));

            BindCommand(new CommandBinding(this,
                        new QueryAndAddWaveArchiveHandler(true, this)));

            BindCommand(new CommandBinding(this,
                        new QueryAndAddGroupHandler(true, this)));
            BindCommand(new CommandBinding(this,
                        new QueryAndAddGroupItemHandler(true, this)));

            BindCommand(new CommandBinding(this,
                        new QueryAndAddWaveSoundSetHandler(false, this)));
            BindCommand(new CommandBinding(this,
                        new QueryAndAddWaveSoundHandler(false, this, ApplicationBase.DefaultPlayerName)));
            BindCommand(new CommandBinding(this,
                        new QueryAndAddSequenceSoundHandler(false, this, ApplicationBase.DefaultPlayerName)));
            BindCommand(new CommandBinding(this,
                        new QueryAndAddSequenceSoundSetHandler(false, this)));
            BindCommand(new CommandBinding(this,
                        new QueryAndAddPlayerHandler(false, this)));
            BindCommand(new CommandBinding(this,
                        new QueryAndAddBankHandler(false, this)));

            BindCommand(new CommandBinding(this,
                        new QueryAndAddWaveArchiveHandler(false, this)));

            BindCommand(new CommandBinding(this,
                        new QueryAndAddGroupHandler(false, this)));
            BindCommand(new CommandBinding(this,
                        new QueryAndAddGroupItemHandler(false, this)));
            BindCommand(new CommandBinding(this, new OpenItemFolderHandler(this)));
            BindCommand(new CommandBinding(this, new ExecuteItemFolderHandler(this)));
            BindCommand(new CommandBinding(this, new OpenBankItemHandler(this)));

            BindCommand(new CommandBinding(this, new QueryAndOutputTemplateTextSequence(this)));

            // サウンドリスト関連コマンドハンドラ
            BindCommand(new CommandBinding(this,
                            new SoundListQueryAndAddWaveSoundSetHandler(true, this)));
            BindCommand(new CommandBinding(this,
                            new SoundListQueryAndAddWaveSoundHandler(true, this, ApplicationBase.DefaultPlayerName)));
            BindCommand(new CommandBinding(this,
                            new SoundListQueryAndAddSequenceSoundHandler(true, this, ApplicationBase.DefaultPlayerName)));
            BindCommand(new CommandBinding(this,
                            new SoundListQueryAndAddSequenceSoundSetHandler(true, this)));
            BindCommand(new CommandBinding(this,
                            new SoundListQueryAndAddBankHandler(true, this)));
            BindCommand(new CommandBinding(this,
                            new SoundListQueryAndAddWaveArchiveHandler(true, this)));
            BindCommand(new CommandBinding(this,
                            new SoundListQueryAndAddPlayerHandler(true, this)));
            BindCommand(new CommandBinding(this,
                            new SoundListQueryAndAddGroupHandler(true, this)));
            BindCommand(new CommandBinding(this,
                            new SoundListQueryAndAddGroupItemHandler(true, this)));

            BindCommand(new CommandBinding(this,
                            new SoundListQueryAndAddWaveSoundSetHandler(false, this)));
            BindCommand(new CommandBinding(this,
                            new SoundListQueryAndAddWaveSoundHandler(false, this, ApplicationBase.DefaultPlayerName)));
            BindCommand(new CommandBinding(this,
                            new SoundListQueryAndAddSequenceSoundHandler(false, this, ApplicationBase.DefaultPlayerName)));
            BindCommand(new CommandBinding(this,
                            new SoundListQueryAndAddSequenceSoundSetHandler(false, this)));
            BindCommand(new CommandBinding(this,
                            new SoundListQueryAndAddBankHandler(false, this)));
            BindCommand(new CommandBinding(this,
                            new SoundListQueryAndAddWaveArchiveHandler(false, this)));
            BindCommand(new CommandBinding(this,
                            new SoundListQueryAndAddPlayerHandler(false, this)));
            BindCommand(new CommandBinding(this,
                            new SoundListQueryAndAddGroupHandler(false, this)));
            BindCommand(new CommandBinding(this,
                            new SoundListQueryAndAddGroupItemHandler(false, this)));
            BindCommand(new CommandBinding(this, SoundListCommands.AddToGroup.ID,
                                             QueryStatusAddToGroup, ExecuteAddToGroup));
            BindCommand(new CommandBinding(this, SoundListCommands.OutputHtmlInstrumentList.ID,
                                             QueryStatusOutputHtmlInstrumentList, ExecuteOutputHtmlInstrumentList));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override string ConvertText(string name, string text)
        {
            if (name == ProjectParameterNames.WaveArchive.LoadType)
            {
                if (text.ToLower() == "true")
                {
                    IParameterValue individual =
                        new WaveArchiveLoadTypeParameterValue(WaveArchiveLoadType.Individual);
                    text = individual.ToString();
                }
                else if (text.ToLower() == "false")
                {
                    IParameterValue whole =
                        new WaveArchiveLoadTypeParameterValue(WaveArchiveLoadType.Whole);
                    text = whole.ToString();
                }
            }

            return text;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override IParameterValue ConvertValue(
                                                        string name,
                                                        string text,
                                                        IParameterValue value
                                                        )
        {
            if (name == ProjectParameterNames.WaveArchive.LoadType)
            {
                IParameterValue individual =
                    new WaveArchiveLoadTypeParameterValue(WaveArchiveLoadType.Individual);
                if (text == individual.ToString())
                {
                    return individual;
                }

                IParameterValue whole =
                    new WaveArchiveLoadTypeParameterValue(WaveArchiveLoadType.Whole);
                if (text == whole.ToString())
                {
                    return whole;
                }
            }

            return value;
        }

        /// <summary>
        /// <see cref="CommonListAdapter"/>  を生成します。
        /// <para>
        /// 本メソッドをオーバーライドし生成方法を変更する場合は <see cref="ListDecorationEvaluator"/> を与えるようにしてください。
        /// </para>
        /// </summary>
        protected virtual CommonListAdapter CreateListAdapter()
        {
            return new CommonListAdapter(ListDecorationEvaluator);
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual bool IsSupportWaveFile(string filePath)
        {
            if (AACUtil.IsAACFile(filePath) == true)
            {
                return true;
            }

            try
            {
                WaveFileReader.CreateInstance(filePath);
            }
            catch
            {
                return false;
            }
            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual bool ValidateSoundSetItemForDragDrop(Component parent, Component component, out string message)
        {
            message = string.Empty;
            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual Component CreateSoundSetItemForDragDrop(Component component, string filePath, int channelCount)
        {
            if (component is StreamSoundPack)
            {
                StreamSoundBase streamSound = null;

                if (IsSupportWaveFile(filePath) == false)
                {
                    return null;
                }

                if (this.IsMultiChannelAACFile(filePath) == true)
                {
                    using (Stream stream = File.OpenRead(filePath))
                    {
                        AdtsHeader adtsHeader = AdtsHeader.Parse(stream);
                        if (adtsHeader != null && adtsHeader.IsValid() == true)
                        {
                            streamSound = this.CreateComponentService.Create<StreamSoundBase>();
                            streamSound.Name = ItemNamer.CreateUniqueNameFromFileName
                                (ProjectService.ComponentDictionary,
                                 new ItemNamingSettings(ProjectService.Project),
                                 ProjectService.Project.StreamSoundNamePrefix, String.Empty, filePath);
                            streamSound.PlayerReference = GetInitialPlayer();

                            int trackCount = (adtsHeader.Channel + (adtsHeader.Channel % channelCount)) / channelCount;
                            for (int i = 0; i < trackCount; i++)
                            {
                                StreamSoundTrackBase track = this.CreateComponentService.Create<StreamSoundTrackBase>();
                                track.FilePath = filePath;
                                if (i * channelCount <= adtsHeader.Channel)
                                {
                                    track.ChannelCount = channelCount;
                                }
                                else
                                {
                                    track.ChannelCount = 1;
                                }

                                streamSound.Children.Add(track);
                                streamSound.Encoding = WaveEncoding.NoConvert;
                            }
                        }
                        else
                        {
                            return null;
                        }
                    }
                }
                else
                {
                    StreamSoundTrackBase streamSoundTrack = this.CreateComponentService.Create<StreamSoundTrackBase>();
                    streamSoundTrack.FilePath = filePath;
                    streamSoundTrack.ChannelCount = channelCount;

                    streamSound = this.CreateComponentService.Create<StreamSoundBase>();
                    streamSound.Name = ItemNamer.CreateUniqueNameFromFileName
                        (ProjectService.ComponentDictionary,
                         new ItemNamingSettings(ProjectService.Project),
                         ProjectService.Project.StreamSoundNamePrefix, String.Empty, filePath);
                    if (string.IsNullOrEmpty(streamSound.PlayerReference) == true)
                    {
                        streamSound.PlayerReference = GetInitialPlayer();
                    }
                    if (AACUtil.IsAACFile(filePath) == true)
                    {
                        streamSound.Encoding = WaveEncoding.NoConvert;
                    }
                    streamSound.Children.Add(streamSoundTrack);
                }

                return streamSound;
            }

            //
            if (component is StreamSoundBase)
            {
                if (IsSupportWaveFile(filePath) == false)
                {
                    return null;
                }

                StreamSoundTrackBase streamSoundTrack = this.CreateComponentService.Create<StreamSoundTrackBase>();
                streamSoundTrack.FilePath = filePath;
                return streamSoundTrack;
            }

            //
            if (component is WaveSoundSetBase)
            {
                if (IsSupportWaveFile(filePath) == false)
                {
                    return null;
                }

                WaveSoundBase waveSound = this.CreateComponentService.Create<WaveSoundBase>();
                waveSound.FilePath = filePath;
                waveSound.Name = ItemNamer.CreateUniqueNameFromFileName
                    (ProjectService.ComponentDictionary,
                      new ItemNamingSettings(ProjectService.Project),
                      ProjectService.Project.WaveSoundNamePrefix, String.Empty, filePath);
                if (string.IsNullOrEmpty(waveSound.PlayerReference) == true)
                {
                    waveSound.PlayerReference = GetInitialPlayer();
                }
                if (AACUtil.IsAACFile(filePath) == true)
                {
                    waveSound.Encoding = WaveEncoding.NoConvert;
                }

                return waveSound;
            }

            //
            if (component is SequenceSoundPack ||
                component is SequenceSoundSetBase)
            {
                SequenceSoundBase sequenceSound = this.CreateComponentService.Create<SequenceSoundBase>();
                sequenceSound.FilePath = filePath;
                sequenceSound.Name = ItemNamer.CreateUniqueNameFromFileName
                    (ProjectService.ComponentDictionary,
                      new ItemNamingSettings(ProjectService.Project),
                      ProjectService.Project.SequenceSoundNamePrefix, String.Empty, filePath);
                if (string.IsNullOrEmpty(sequenceSound.PlayerReference) == true)
                {
                    sequenceSound.PlayerReference = GetInitialPlayer();
                }
                if (sequenceSound.SoundSetBankReferences.Count <= 0)
                {
                    sequenceSound.InitializeBankReferences(GetInitialBank());
                }

                return sequenceSound;
            }

            //
            if (component is SoundSetBankPack)
            {
                try
                {
                    if (ApplicationBase.Instance.DocumentService.GetDocumentTypeName(new FileResource(filePath))
                        != ApplicationBase.Instance.Traits.IntermediateOutputTraits.BankDocumentTypeName)
                    {
                        return null;
                    }
                }
                catch
                {
                    return null;
                }

                SoundSetBankBase bank = this.CreateComponentService.Create<SoundSetBankBase>();
                bank.FilePath = filePath;
                bank.Name = ItemNamer.CreateUniqueNameFromFileName
                    (ProjectService.ComponentDictionary,
                      new ItemNamingSettings(ProjectService.Project),
                      ProjectService.Project.BankNamePrefix, String.Empty, filePath);
                return bank;
            }

            return null;
        }

        /// <summary>
        /// パネルが表示された時に呼ばれます。
        /// </summary>
        protected virtual void OnShowedPanel(object sender, EventArgs e)
        {
            this.ShowedPanel?.Invoke(sender, e);
        }

        /// <summary>
        ///
        /// </summary>
        private ListCtrl LastActivedListControl
        {
            get
            {
                //if( CurrentListCtrl == this._ListCtrlMain ) { Console.WriteLine( "Main"); }
                //if( CurrentListCtrl == this._ListCtrlSub  ) { Console.WriteLine( "Sub"); }
                return CurrentListCtrl != null ? CurrentListCtrl : this._ListCtrlMain;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void SetSoundSet(SoundSet soundSet)
        {
            if (null != _SoundSet)
            {
                _SoundSet.NameChanged -= OnSoundSetNameChanged;
            }

            _SoundSet = soundSet;

            if (null != _SoundSet)
            {
                _SoundSet.NameChanged += OnSoundSetNameChanged;
            }
        }

        /// <summary>
        /// コマンドが実行されると発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">コマンドイベントデータ。</param>
        private void OnSoundListCommandExecuted(object sender, CommandEventArgs e)
        {
            CommandExecuted(e.Command);
            MainWindow.BuildCommandUI();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnEnterd(object sender, EventArgs e)
        {
            _CurrentListCtrl = sender as CommonListCtrl;
            SetSelectedItemsToParameterPanel(_CurrentListCtrl);

            BuildCommandUIs();

            this.UpdatePresetListColumnsSplitButton(_CurrentListCtrl);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnLeave(object sender, EventArgs e)
        {
            MainWindow.ClearPresetListColumnsOnSplitButton();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnDirtyChanged(object sender, EventArgs e)
        {
            MainWindow.ProjectPanel.UpdateProjectCtrl();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnSoundSetChanged(object sender, EventArgs e)
        {
            SetSoundSet(SoundSetDocument.SoundSet);
        }

        /// <summary>
        ///
        /// </summary>
        private void OnSoundSetNameChanged(object sender, NameChangedEventArgs e)
        {
            UpdateTitle();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnOperationHistoryItemsChanged(object sender, EventArgs e)
        {
            if (e is TransactionStackEventArgs)
            {
                TransactionStackEventArgs args = e as TransactionStackEventArgs;
                if ((args.Transaction is UserTransaction2) == false)
                {
                    //FindResultPanel.ClearTransactions();

                    foreach (FindResultPanel2 panel in MainWindow.FindResultPanels)
                    {
                        panel.ClearTransactions();
                    }
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnOperationHistoryCurrentItemChanged(object sender, EventArgs e)
        {
            OnOperationHistoryItemsChanged(sender, e);
        }

        ///--------------------------------
        /// <summary>
        /// ファイルがドロップできるかどうか判断する時に呼ばれる
        /// </summary>
        private void OnQueryFileDropped(object sender, FileDroppedEventArgs e)
        {
            //
            ListCtrl listCtrl = sender as ListCtrl;
            CommonListAdapter adapter = listCtrl.ItemsSource as CommonListAdapter;
            Component component = adapter.ItemsOwner;

            if (SoundSetDocument.IsReadOnly == true)
            {
                e.Cancel = true;
                return;
            }

            if (null == component)
            {
                e.Cancel = true;
                return;
            }

            foreach (string filePath in e.FilePaths)
            {
                if ((AACUtil.IsAACFile(filePath) == true || FileUtil.IsOpusFile(filePath) == true) &&
                    (listCtrl == _ListCtrlSub ||
                     (listCtrl == _ListCtrlMain && component is StreamSoundPack == false)))
                {
                    e.Cancel = true;
                    return;
                }
            }

            if (component is StreamSoundPack ||
                component is StreamSoundBase ||
                component is WaveSoundSetBase ||
                component is SequenceSoundPack ||
                component is SequenceSoundSetBase ||
                component is SoundSetBankPack) { return; }

            e.Cancel = true;
            return;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private Component NextSiblingComponent(ListCtrl listCtrl, ComponentListItem item, DropPosition dropPosition)
        {
            CommonListAdapter adapter = listCtrl.ItemsSource as CommonListAdapter;
            int index = -1;

            index = adapter.Items.IndexOf(item);

            switch (dropPosition)
            {
                case DropPosition.Upper:
                    return ((ComponentListItem)adapter.Items[index]).Target;

                case DropPosition.Lower:
                    if (index + 1 >= adapter.Items.Count) { break; }
                    return ((ComponentListItem)adapter.Items[index + 1]).Target;

                case DropPosition.None:
                case DropPosition.Center:
                    break;
            }
            return null;
        }

        ///--------------------------------
        /// <summary>
        /// ファイルがドロップされた時に呼ばれる
        /// </summary>
        private void OnFileDropped(object sender, FileDroppedEventArgs e)
        {
            BeginInvoke(new ListCtrl.FileDroppedEventHandler(OnFileDroppedInternal),
                        new object[] { sender, e });
        }

        ///--------------------------------
        /// <summary>
        /// ファイルがドロップされた時に呼ばれる
        /// </summary>
        private void OnFileDroppedInternal(object sender, FileDroppedEventArgs e)
        {
            MainWindow.Activate();

            //
            ListCtrl listCtrl = sender as ListCtrl;
            CommonListAdapter adapter = listCtrl.ItemsSource as CommonListAdapter;
            Component ownerComponent = adapter.ItemsOwner;
            Component parentSoundSetItem = null;
            Component nextSiblingSoundSetItem = null;
            int addCount = 0;

            if (ownerComponent == null)
            {
                return;
            }

            //
            if (ownerComponent is SequenceSoundPack ||
                ownerComponent is SequenceSoundSetBase)
            {
                if (ProjectService.SoundSetBanks.Length <= 0)
                {
                    FormsApplication.Instance.UIService.ShowMessageBox
                        (MessageResource.Message_CanNotCreateSequenceSoundNothingBank,
                          AppMessageBoxButton.OK,
                          AppMessageBoxImage.Warning);
                    return;
                }
            }

            StringBuilder unsupportedFileNames = new StringBuilder();

            //
            try
            {
                parentSoundSetItem = ownerComponent;
                nextSiblingSoundSetItem = NextSiblingComponent
                    ((ListCtrl)sender, (ComponentListItem)e.DropTarget, e.DropPosition);

                //
                SoundSetDocument.OperationHistory.BeginTransaction();

                bool entryOutsideFilePath = false;
                bool entryChannelCount = false;
                int channelCount = 0;

                foreach (string filePath in e.FilePaths)
                {
                    string message = string.Empty;
                    // プロジェクトファイル外のファイルを登録するのか問い合わせます。
                    // D&Dだと同じディレクトリからしかありえないので、一度しか問い合わせません。
                    if (entryOutsideFilePath == false)
                    {
                        if (ProjectFilePathUtility.ConfirmFilePath(filePath) == false)
                        {
                            return;
                        }
                        entryOutsideFilePath = true;
                    }

                    if (entryChannelCount == false &&
                        this.IsMultiChannelAACFile(filePath) == true)
                    {
                        CreateStreamSoundTrackDropDialog dialog = new CreateStreamSoundTrackDropDialog();
                        if (dialog.ShowDialog() != DialogResult.OK)
                        {
                            return;
                        }
                        else
                        {
                            channelCount = dialog.ChannelCount;
                            entryChannelCount = true;
                        }
                    }

                    //
                    Component component = CreateSoundSetItemForDragDrop(ownerComponent, filePath, channelCount);
                    if (component != null)
                    {
                        if (ValidateSoundSetItemForDragDrop(parentSoundSetItem, component, out message) == false)
                        {
                            if (string.IsNullOrEmpty(message) == true)
                            {
                                message = filePath;
                            }
                            else
                            {
                                message = message + " " + filePath;
                            }
                            unsupportedFileNames.AppendLine(message);
                            continue;
                        }

                        if (component is SoundSetItem)
                        {
                            AddSoundSetItemHandler.Execute(SoundSetDocument,
                                                           parentSoundSetItem,
                                                           nextSiblingSoundSetItem,
                                                           component);
                            addCount++;
                        }
                        else if (component is StreamSoundTrackBase)
                        {
                            int maxCount = ComponentConfiguration.Instance.StreamSoundTrackNumberMaximum;
                            if (ownerComponent.Children.Count >= maxCount)
                            {
                                break;
                            }

                            QueryAndAddStreamSoundTrackHandler.Execute(SoundSetDocument,
                                                                       ownerComponent,
                                                                       nextSiblingSoundSetItem,
                                                                       component);
                            addCount++;
                        }
                    }
                    else
                    {
                        unsupportedFileNames.AppendLine(filePath);
                    }
                }
            }
            catch
            {
                addCount = 0;
            }
            finally
            {
                if (addCount > 0)
                {
                    SoundSetDocument.OperationHistory.EndTransaction();
                    ProjectService.SendUpdateSoundProjectItemBindingEvent();
                }
                else
                {
                    SoundSetDocument.OperationHistory.CancelTransaction();
                }
            }

            if (0 < unsupportedFileNames.Length)
            {
                string message = Resources.MessageResource.Message_CanNotUseUnsupportedFileWithAAC;

                if (ownerComponent is SoundSetBankPack == true)
                {
                    message = String.Format(Resources.MessageResource.Message_CanNotUseUnsupportedBankFile, ApplicationBase.Instance.Traits.IntermediateOutputTraits.BankFileExtension);
                }

                TextDisplayMessageBox dialog = new TextDisplayMessageBox
                    (message, unsupportedFileNames.ToString(),
                      TextDisplayMessageBoxStyle.OKButton);
                dialog.ShowDialog();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnQueryDropped(object sender, DroppedEventArgs e)
        {
            CommonListCtrl listCtrl = sender as CommonListCtrl;
            CommonListAdapter listAdapter = listCtrl.ItemsSource as CommonListAdapter;
            Component ownerComponent = listAdapter.ItemsOwner;

            if (!(ownerComponent is GroupBase) ||
                listCtrl == _ListCtrlMain ||
                CanPasteToGroupItemByDragDrop() == false)
            {
                e.Cancel = true;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnDropped(object sender, DroppedEventArgs e)
        {
            CommonListCtrl listCtrl = null;

            if ((listCtrl = sender as CommonListCtrl) == null)
            {
                return;
            }

            //
            CommonListAdapter adapter = listCtrl.ItemsSource as CommonListAdapter;
            Component parentComponent = null;
            Component nextSiblingComponent = null;

            parentComponent = adapter.ItemsOwner;
            nextSiblingComponent = NextSiblingComponent
                (listCtrl, (ComponentListItem)e.DropTarget, e.DropPosition);

            PasteToGroupItemByDragDrop(parentComponent, nextSiblingComponent);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnEditBegan(object sender, EventArgs e)
        {
            ApplicationBase.Instance.DisableCommandKeyProcess();
            MainWindow.EnableAllCommands(false);
            BuildListCommandUI();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnEditEnded(object sender, EventArgs e)
        {
            ApplicationBase.Instance.EnableCommandKeyProcess();
            UpdateStatusBar(ActiveListCtrl);
            if (ActiveListCtrl == _ListCtrlSub)
            {
                UpdateListCtrlSub();
            }
            MainWindow.EnableAllCommands(true);
            MainWindow.RedrawPanels();
            BuildListCommandUI();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private string GetInitialPlayer()
        {
            PlayerBase[] players = ProjectService.Players;
            return players.Length > 0 ? players[0].Name : ApplicationBase.DefaultPlayerName;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private string GetInitialBank()
        {
            SoundSetBankBase[] banks = ProjectService.SoundSetBanks;
            return banks.Length > 0 ? banks[0].Name : String.Empty;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void UpdateListCtrlSub()
        {

            ComponentListItem[] items = _ListCtrlMain.GetSelectedItems();
            SoundSetItem item = null;

            if (_CurrentListItem != null)
            {
                if (items.Length == 1 && _CurrentListItem == items[0] &&
                    _CurrentListItem.Target is StreamSoundBase == false)
                {
                    return;
                }
            }

            this.statusStrip.Visible = false;

            if (items.Length == 1)
            {

                item = items[0].Target as SoundSetItem;

                //
                if (this._CurrentListItem != null)
                {
                    SaveListState(this._ListCtrlSub, this._ListCtrlState,
                                   this._CurrentListItem.Target);
                }

                //
                if (item is StreamSoundBase)
                {
                    ListHeaderAdapterBase headerAdapter = _headerAdapters[ApplicationBase.Instance.Traits.ListTraits.ListNames[typeof(StreamSoundBase)]];

                    if (AppConfiguration.EnabledMultiChannelAAC == true)
                    {
                        if ((item as StreamSoundBase).IsMultiChannelAAC() == true)
                        {
                            if (_aacUseHeaderItems.Count() != 0)
                            {
                                foreach (Tuple<int, IHeaderItem> tuple in _aacUseHeaderItems)
                                {
                                    headerAdapter.Items.Insert(tuple.Item1, tuple.Item2);
                                }
                                _aacUseHeaderItems.Clear();
                            }

                            if (_wavUseHeaderItems.Count() == 0)
                            {
                                int index = 0;
                                foreach (IHeaderItem hItem in headerAdapter.Items)
                                {
                                    switch (hItem.Name)
                                    {
                                        case ListTraits.ColumnName_PreviewMute:
                                        case ListTraits.ColumnName_PreviewSoloPlay:
                                            Tuple<int, IHeaderItem> tuple = new Tuple<int, IHeaderItem>(index, hItem);
                                            _wavUseHeaderItems.Add(tuple);
                                            break;
                                    }
                                    index++;
                                }
                            }

                            foreach (Tuple<int, IHeaderItem> tuple in _wavUseHeaderItems)
                            {
                                headerAdapter.Items.Remove(tuple.Item2);
                            }

                            CheckMultiChannelAAC(item as StreamSoundBase);
                        }
                        else
                        {
                            if (_wavUseHeaderItems.Count() != 0)
                            {
                                foreach (Tuple<int, IHeaderItem> tuple in _wavUseHeaderItems)
                                {
                                    headerAdapter.Items.Insert(tuple.Item1, tuple.Item2);
                                }
                                _wavUseHeaderItems.Clear();
                            }

                            if (_aacUseHeaderItems.Count() == 0)
                            {
                                int index = 0;
                                foreach (IHeaderItem hItem in headerAdapter.Items)
                                {
                                    switch (hItem.Name)
                                    {
                                        case ListTraits.ColumnName_ChannelNo:
                                            Tuple<int, IHeaderItem> tuple = new Tuple<int, IHeaderItem>(index, hItem);
                                            _aacUseHeaderItems.Add(tuple);
                                            break;
                                    }
                                    index++;
                                }
                            }

                            foreach (Tuple<int, IHeaderItem> tuple in _aacUseHeaderItems)
                            {
                                headerAdapter.Items.Remove(tuple.Item2);
                            }

                            CheckMultiChannelWaveFile(item as StreamSoundBase);
                        }
                    }
                    else // AppConfiguration.EnabledMultiChannelAAC == false
                    {
                        // 「チャンネル番号」カラムを非表示にします。（Multi Channel aac 用カラム）
                        var removeItem = headerAdapter.Items.Where(i => i.Name == ListTraits.ColumnName_ChannelNo).FirstOrDefault();
                        if (removeItem != null)
                        {
                            headerAdapter.Items.Remove(removeItem);
                        }
                    }

                    _ListCtrlSub.HeaderSource = headerAdapter;
                    _ListCtrlSub.ItemsSource = null; // 高速化のため一旦外して
                    _ListAdapterSub.AddItems(item.Children);
                    _ListCtrlSub.ItemsSource = _ListAdapterSub; // 再度、付け直す
                    _ListCtrlSub.Sort();
                }
                else if (item is WaveSoundSetBase ||
                    item is SequenceSoundSetBase ||
                    item is GroupBase)
                {
                    _ListCtrlSub.HeaderSource =
                        //_headerAdapters[ListTraits.ListNames[item.GetType()]];
                        _headerAdapters[ApplicationBase.Instance.Traits.ListTraits.ListNames[item.GetType()]];
                    _ListCtrlSub.ItemsSource = null; // 高速化のため一旦外して
                    _ListAdapterSub.AddItems(item.Children);
                    _ListCtrlSub.ItemsSource = _ListAdapterSub; // 再度、付け直す
                    _ListCtrlSub.Sort();
                }

                _CurrentListItem = items[0];
            }
            else
            {

                _ListAdapterSub.RemoveAllItems();
                //_ListAdapterSub.Items.Clear();

                _CurrentListItem = null;
            }
        }

        private bool IsMultiChannelAACFile(string filePath)
        {
            return (AppConfiguration.EnabledMultiChannelAAC == true &&
                    AACUtil.IsAACFile(filePath) == true);
        }

        private void CheckMultiChannelAAC(StreamSoundBase streamSound)
        {
            int count = 0;
            foreach (StreamSoundTrackBase track in streamSound.Children)
            {
                count += track.ChannelCount;
            }

            StreamSoundTrackBase firstTrack = streamSound.Children[0] as StreamSoundTrackBase;
            firstTrack.UpdateWaveFile();
            if (firstTrack.WaveFile != null &&
                firstTrack.WaveFile.ChannelCount != count)
            {
                this.statusStrip.Visible = true;
                this.toolStripStatusLabel.Text = Resources.MessageResource.Message_ErrorMultiChannelAACChannelCount;
                this.toolStripStatusLabel.Image = ImageResource.BitmapIconItemError.MakeNewTransparent(UIServiceBase.TransparentColor);
            }
            else
            {
                this.statusStrip.Visible = false;
                this.toolStripStatusLabel.Text = string.Empty;
                this.toolStripStatusLabel.Image = null;
            }
        }

        private void CheckMultiChannelWaveFile(StreamSoundBase streamSound)
        {
            this.statusStrip.Visible = false;
            this.toolStripStatusLabel.Text = string.Empty;
            this.toolStripStatusLabel.Image = null;

            foreach (StreamSoundTrackBase track in streamSound.Children)
            {
                track.UpdateWaveFile();

                if (track.WaveFile != null &&
                    AppConfiguration.WaveFileMultiChannelCountMax < track.WaveFile.ChannelCount)
                {
                    this.statusStrip.Visible = true;
                    this.toolStripStatusLabel.Text = Resources.MessageResource.Message_ErrorMultiChannelWaveFile;
                    this.toolStripStatusLabel.Image = ImageResource.BitmapIconItemError.MakeNewTransparent(UIServiceBase.TransparentColor);
                    break;
                }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void UpdateStatusBar(CommonListCtrl listCtrl)
        {
            CommonListItem[] items = null;
            string message = String.Empty;

            if (listCtrl != null)
            {
                items = listCtrl.GetSelectedItems();
                if (items.Length == 1)
                {
                    message = items[0].FilePath;
                }
            }

            MainWindow.UpdateStatusText(message);
        }

        /// <summary>
        /// パラメータパネルに選択されているアイテムを設定します。
        /// </summary>
        private void SetSelectedItemsToParameterPanel(ListCtrl listCtrl)
        {
            List<Component> list = new List<Component>();

            foreach (ComponentListItem item in listCtrl.SelectedItems)
            {
                if (item.Target is Sound || item.Target is StreamSoundTrackBase)
                {
                    list.Add(item.Target);
                }
            }

            ParameterPanel.Items = list.Count > 0 ? list.ToArray() : null;
        }

        /// <summary>
        /// メイン側(上)のリストの選択状態が変更された時に呼ばれます
        /// </summary>
        private void OnSelectChangedListCtrlMain(object sender, EventArgs e)
        {
            if (_ListCtrlMain.ContainsFocus == true)
            {
                SetSelectedItemsToParameterPanel(_ListCtrlMain);
            }

            UpdateListCtrlSub();

            ///
            if (this._CurrentListItem != null)
            {
                LoadListState(_ListCtrlSub, _ListAdapterSub, this._ListCtrlState,
                               this._CurrentListItem.Target);
            }

            UpdateStatusBar(_ListCtrlMain);
            BuildCommandUIs();
        }

        /// <summary>
        /// サブ側(下)のリストの選択状態が変更された時に呼ばれます
        /// </summary>
        private void OnSelectChangedListCtrlSub(object sender, EventArgs e)
        {
            if (_ListCtrlSub.ContainsFocus == true)
            {
                SetSelectedItemsToParameterPanel(_ListCtrlSub);
            }

            UpdateStatusBar(_ListCtrlSub);
            BuildCommandUIs();
        }

        /// <summary>
        /// リストの選択状態が変更された時に呼ばれます
        /// </summary>
        private void OnSelectChangedListItem(object sender, EventArgs e)
        {
            if (SelectChangedListItem != null)
            {
                SelectChangedListItem(sender, e);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnItemDoubleClicked(object sender, ListItemDoubleClickedEventArgs e)
        {
            ExecuteItem(e.Item as ComponentListItem);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private IHeaderSource GetHeaderAdapter(Type itemType)
        {
            if (itemType == null) { return null; }
            //return _headerAdapters[ListTraits.ListNames[itemType]];
            return _headerAdapters[ApplicationBase.Instance.Traits.ListTraits.ListNames[itemType]];
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void Show(SoundSetItem mainItem, SoundSetItem subItem, double splitterRatio, bool force)
        {
            Type subItemType = null;

            if (_CurrentItem == mainItem && force == false)
            {
                return;
            }

            if (_CurrentItem != null)
            {
                SaveListsState();
            }

            //
            this.UpdateReadOnly();
            this.UpdateTitle(mainItem);
            this.UpdateImage(mainItem);

            _CurrentItem = mainItem;

            if (mainItem != null)
            {
                _ListCtrlMain.HeaderSource = GetHeaderAdapter(mainItem.GetType());
                _ListCtrlMain.ItemsSource = null; // 高速化のため一旦外して
                _ListAdapterMain.AddItems(mainItem.Children);
                _ListCtrlMain.ItemsSource = _ListAdapterMain; // 再度、付け直す。
                _ListCtrlMain.Sort();

                splitContainer.Panel1Collapsed = false;

                if (mainItem is StreamSoundPack)
                {
                    subItemType = typeof(StreamSoundBase);
                }
                else if (mainItem is WaveSoundSetPack)
                {
                    subItemType = typeof(WaveSoundSetBase);
                }
                else if (mainItem is SequenceSoundSetPack)
                {
                    subItemType = typeof(SequenceSoundSetBase);
                }
                else if (mainItem is GroupPack)
                {
                    subItemType = typeof(GroupBase);
                }

            }
            else
            {
                _ListCtrlMain.HeaderSource = null;
                _ListAdapterMain.Items.Clear();
                splitContainer.Panel1Collapsed = true;
            }

            //
            if (subItem != null)
            {
                _ListCtrlSub.HeaderSource = GetHeaderAdapter(subItem.GetType());
                _ListCtrlSub.ItemsSource = null; // 高速化のため一旦外して
                _ListAdapterSub.AddItems(subItem.Children);
                _ListCtrlSub.ItemsSource = _ListAdapterSub; // 再度、付け直す
                _ListCtrlSub.Sort();

                splitContainer.Panel2Collapsed = false;
            }
            else
            {
                _ListCtrlSub.HeaderSource = GetHeaderAdapter(subItemType);

                _ListAdapterSub.RemoveAllItems();
                //_ListAdapterSub.Items.Clear();
                splitContainer.Panel2Collapsed = (0.0 >= splitterRatio);

                try
                {
                    splitContainer.SplitterDistance = (int)(splitContainer.Height * splitterRatio);
                }
                catch
                {
                }
            }

            //ResetListSeletion();
            if (_CurrentItem != null)
            {
                LoadListsState();
            }

            SetSelectedItemsToParameterPanel(this._ListCtrlMain);

            if (_CurrentListCtrl != null)
            {
                this.UpdatePresetListColumnsSplitButton(_CurrentListCtrl);
            }
            else
            {
                MainWindow.ClearPresetListColumnsOnSplitButton();
            }

            this.OnShowedPanel(this, EventArgs.Empty);
        }

        private class ListCtrlState
        {
            private Dictionary<Component, ListItemSelectedState> _ListItemSelectedStateDictionary;

            public ListCtrlState(ListItemSelectedDictionary dictionary)
            {
                _ListItemSelectedStateDictionary = new Dictionary<Component, ListItemSelectedState>();

                foreach (IListItem item in dictionary.Keys)
                {
                    if (item is CommonListItem == true)
                    {
                        CommonListItem common = item as CommonListItem;
                        Component component = common.Target;
                        _ListItemSelectedStateDictionary[component] = dictionary[item];
                    }
                }
            }

            public Component CaretComponent { get; set; }
            public string CaretColumnName { get; set; }
            public int VerticalScrollBarValue { get; set; }
            public int HorizontalScrollBarValue { get; set; }
            public Dictionary<Component, ListItemSelectedState> ListItemSelectedStateDictionary
            {
                get
                {
                    return _ListItemSelectedStateDictionary;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void SaveListsState()
        {
            SaveListState(_ListCtrlMain, _ListCtrlState, _CurrentItem);

            if (IsSingleList == false)
            {
                if (this._CurrentListItem != null &&
                    this._CurrentListItem.Target != this._CurrentItem)
                {
                    SaveListState(this._ListCtrlSub, this._ListCtrlState,
                                   this._CurrentListItem.Target);
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void SaveListState(CommonListCtrl ctrl, Dictionary<Component, ListCtrlState> stateDictionary, Component currentItem)
        {
            if (currentItem == null ||
                this.enabledSaveListState == false)
            {
                return;
            }

            ListItemSelectedDictionary dictionary;
            dictionary = ctrl.GetItemSelecteds().Clone() as ListItemSelectedDictionary;
            stateDictionary[currentItem] = new ListCtrlState(dictionary);

            IListItem item = null;
            string columnName = string.Empty;

            ctrl.GetCaret(out item, out columnName);
            if (item != null && item is CommonListItem)
            {
                stateDictionary[currentItem].CaretComponent = (item as CommonListItem).Target;
                stateDictionary[currentItem].CaretColumnName = columnName;
            }
            stateDictionary[currentItem].VerticalScrollBarValue = ctrl.VerticalScrollBar.Value;
            stateDictionary[currentItem].HorizontalScrollBarValue = ctrl.HorizontalScrollBar.Value;
        }

        /// <summary>
        ///
        /// </summary>
        private void LoadListsState()
        {
            try
            {
                this.enabledSaveListState = false;

                LoadListState(_ListCtrlMain, _ListAdapterMain,
                               this._ListCtrlState, _CurrentItem);
            }
            finally
            {
                this.enabledSaveListState = true;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void LoadListState(CommonListCtrl ctrl, CommonListAdapter adapter, Dictionary<Component, ListCtrlState> listCtrlStateDictionary, Component currentItem)
        {
            if (currentItem == null)
            {
                return;
            }

            bool restored = false;

            if (listCtrlStateDictionary.ContainsKey(currentItem) == true)
            {
                Dictionary<Component, ListItemSelectedState> stateDictionary =
                    listCtrlStateDictionary[currentItem].ListItemSelectedStateDictionary;

                if (stateDictionary.Count > 0)
                {
                    SetSelectedList(ctrl.GetItemSelecteds(), stateDictionary, adapter.Items);

                    SetCaret(ctrl, adapter.Items,
                              listCtrlStateDictionary[currentItem].CaretComponent,
                              listCtrlStateDictionary[currentItem].CaretColumnName);

                    ctrl.VerticalScrollBar.Value =
                        listCtrlStateDictionary[currentItem].VerticalScrollBarValue;
                    ctrl.HorizontalScrollBar.Value =
                        listCtrlStateDictionary[currentItem].HorizontalScrollBarValue;

                    restored = true;
                }
            }

            if (restored == false)
            {
                ctrl.GetItemSelecteds().Clear();
                ctrl.SetCaret();
            }

            ctrl.OnSelectChanged();
        }

        /// <summary>
        ///
        /// </summary>
        private void SetSelectedList(ListItemSelectedDictionary dictionary,
                                     Dictionary<Component, ListItemSelectedState> selectDictionary,
                                     ComponentListItemCollection items)
        {
            foreach (CommonListItem item in items)
            {
                if (selectDictionary.ContainsKey(item.Target) == true)
                {
                    if (selectDictionary[item.Target].Selected == true)
                    {
                        dictionary.Select(item, true);
                    }
                    foreach (string name in selectDictionary[item.Target].SubSelected)
                    {
                        dictionary.SubSelect(item, name, true);
                    }
                }
            }
        }

        private void SetCaret(CommonListCtrl ctrl, ComponentListItemCollection items,
                              Component component, string columnName)
        {
            if (string.IsNullOrEmpty(columnName) == true)
            {
                return;
            }

            foreach (CommonListItem item in items)
            {
                if (item.Target == component)
                {
                    ctrl.SetCaretWithoutFollow(item, columnName);
                    break;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void ResetListSeletion()
        {
            _ListCtrlMain.GetItemSelecteds().Clear();
            _ListCtrlSub.GetItemSelecteds().Clear();
            _ListCtrlMain.SetCaret();
            _ListCtrlSub.SetCaret();
        }

        /// <summary>
        ///
        /// </summary>
        private void ActivateList(ListCtrl listCtrl, bool resetSelection)
        {
            Debug.Assert(null != listCtrl, "invalid argument");

            if (!listCtrl.Visible)
            {
                return;
            }

            if (resetSelection && 0 < listCtrl.ItemCount)
            {
                ResetListSeletion();
            }

            listCtrl.Select();
            BuildListCommandUI();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private SoundSetItem FindItem(string targetName, Component parentComponent)
        {
            if (null == targetName) { throw new ArgumentNullException("targetName"); }
            if (null == parentComponent) { throw new ArgumentNullException("parentComponent"); }

            foreach (Component child in parentComponent.Children)
            {

                if (child.Name == targetName)
                {
                    return child as SoundSetItem;
                }

                SoundSetItem result = FindItem(targetName, child);
                if (null != result) { return result; }

            }

            return null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void UpdateReadOnly()
        {
            bool readOnly = SoundSetDocument.IsReadOnly;

            _ListCtrlMain.ReadOnly = readOnly;
            _ListCtrlSub.ReadOnly = readOnly;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void UpdateImage(SoundSetItem item)
        {
            ImageIndex = TabCtrl.ToImageIndex(item);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void UpdateTitle(SoundSetItem item)
        {
            string title = CreateTitle(item);

            if (SoundSetDocument.IsReadOnly == true)
            {
                title += String.Format(" [{0}]", MessageResource.Message_ReadOnly);
            }

            Title = title;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private string CreateTitle(SoundSetItem item)
        {
            if (null == item) { return string.Empty; }

            if (item is StreamSoundPack)
            {
                return "STRM - " + SoundSet.Name;
            }
            if (item is WaveSoundSetPack)
            {
                return "WSDSET - " + SoundSet.Name;
            }
            if (item is WaveSoundSetBase)
            {
                return item.Name + " - " + SoundSet.Name;
            }
            if (item is SequenceSoundSetPack)
            {
                return "SEQSET - " + SoundSet.Name;
            }
            if (item is SequenceSoundSetBase)
            {
                return item.Name + " - " + SoundSet.Name;
            }
            if (item is SequenceSoundPack)
            {
                return "SEQ - " + SoundSet.Name;
            }
            if (item is SoundSetBankPack)
            {
                return "BANK - " + SoundSet.Name;
            }
            if (item is WaveArchivePack)
            {
                return "WARC - " + SoundSet.Name;
            }
            if (item is PlayerPack)
            {
                return "PLAYER - " + SoundSet.Name;
            }
            if (item is GroupPack)
            {
                return "GROUP - " + SoundSet.Name;
            }

            return string.Empty;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void RedrawControls()
        {
            _ListCtrlMain.Invalidate();
            _ListCtrlSub.Invalidate();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private CommandStatus QueryStatusUndo(Command command)
        {
            return SoundSetDocument.OperationHistory.CanUndo() ?
                CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private CommandStatus QueryStatusRedo(Command command)
        {
            return SoundSetDocument.OperationHistory.CanRedo() ?
                CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool ExecuteUndo(Command command)
        {
            if (SoundSetDocument.OperationHistory.Undo() == false)
            {
                return false;
            }
            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool ExecuteRedo(Command command)
        {
            if (SoundSetDocument.OperationHistory.Redo() == false)
            {
                return false;
            }
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// "すべて選択"が実行可能なのか調べる
        /// </summary>
        private CommandStatus QueryStatusSelectAll(Command command)
        {
            return CanSelectAll() ? CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        ///--------------------------------
        /// <summary>
        /// "すべて選択"の実行
        /// </summary>
        private bool ExecuteSelectAll(Command command)
        {
            return SelectAll();
        }

        ///--------------------------------
        /// <summary>
        /// "貼り付け"が実行可能なのか調べる
        /// </summary>
        private CommandStatus QueryStatusPaste(Command command)
        {
            return CanPaste() ? CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        ///--------------------------------
        /// <summary>
        /// "貼り付け"の実行
        /// </summary>
        private bool ExecutePaste(Command command)
        {
            if (Paste() == false)
            {
                return false;
            }

            ProjectService.SendUpdateSoundProjectItemBindingEvent();
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// "セルのコピー"が実行可能なのか調べる
        /// </summary>
        private CommandStatus QueryStatusCopyCell(Command command)
        {
            return CanCopyCell() ? CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        ///--------------------------------
        /// <summary>
        /// "セルのコピー"の実行
        /// </summary>
        private bool ExecuteCopyCell(Command command)
        {
            return CopyCell();
        }

        ///--------------------------------
        /// <summary>
        /// "項目を選択して貼り付け"が実行可能なのか調べる
        /// </summary>
        private CommandStatus QueryStatusPasteSpecial(Command command)
        {
            return CanPasteSpecial() ? CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        ///--------------------------------
        /// <summary>
        /// "項目を選択して貼り付け"の実行
        /// </summary>
        private bool ExecutePasteSpecial(Command command)
        {
            if (PasteSpecial() == false)
            {
                return false;
            }

            return true;
        }

        ///--------------------------------
        /// <summary>
        /// "一括編集"が実行可能なのか調べる
        /// </summary>
        private CommandStatus QueryStatusBatchEdit(Command command)
        {
            if (SoundSetDocument.IsReadOnly == true)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return CanBatchEdit() ? CommandStatus.SupportedAndEnabledAndVisible :
                CommandStatus.SupportedAndVisible;
        }

        ///--------------------------------
        /// <summary>
        /// "一括編集"の実行
        /// </summary>
        private bool ExecuteBatchEdit(Command command)
        {
            if (BatchEdit() == false)
            {
                return false;
            }

            return true;
        }

        ///--------------------------------
        /// <summary>
        /// "DLSのインポート"が実行可能なのか調べる
        /// </summary>
        private CommandStatus QueryStatusImportDLS(Command command)
        {
            if (_CurrentItem is SoundSetBankPack)
            {
                return CommandStatus.SupportedAndEnabledAndVisible;
            }

            return CommandStatus.Supported;
        }

        ///--------------------------------
        /// <summary>
        /// "DLSのインポート"の実行
        /// </summary>
        private bool ExecuteImportDLS(Command command)
        {
            return ImportDLS();
        }

        ///--------------------------------
        /// <summary>
        /// DLSファイルのインポート
        /// </summary>
        private bool ImportDLS()
        {
            OpenFileDialog dialog = new OpenFileDialog();

            dialog.Filter = ("DLS Files (*.dls)|*.dls" +
                              "|All Files (*.*)|*.*");
            dialog.RestoreDirectory = true;
            if (dialog.ShowDialog() != DialogResult.OK)
            {
                return false;
            }

            //ここで DLSから rbnkにコンバートする
            string filePath = dialog.FileName;
            string outputFilePath = null;

            outputFilePath = Path.Combine
                (Path.GetDirectoryName(filePath),
                  Path.GetFileNameWithoutExtension(filePath)) + ".rbnk";

            if (ConvertDLSFile(filePath, outputFilePath) == false)
            {
                return false;
            }

            //
            return this.AddBank(outputFilePath);
        }

        ///--------------------------------
        /// <summary>
        /// DLSファイルのコンバート
        /// </summary>
        private bool ConvertDLSFile(string filePath, string outputFilePath)
        {
            //
            if (File.Exists(outputFilePath) == true)
            {
                string message = null;
                message = MessageResource.Inquire_OverwriteFileDLS;
                message = String.Format(message, outputFilePath);

                if (MessageBox.Show(message, Application.ProductName,
                                     MessageBoxButtons.OKCancel,
                                     MessageBoxIcon.Warning) != DialogResult.OK)
                {
                    return false;
                }
            }

            //
            ImportDLSDialog dialog = new ImportDLSDialog();

            dialog.FilePath = filePath;
            dialog.OutputDirectory = Path.GetDirectoryName(filePath);

            if (dialog.ShowDialog() != DialogResult.OK)
            {
                return false;
            }

            return true;
        }

        ///--------------------------------
        /// <summary>
        /// "Stxtのインポート"が実行可能なのか調べる
        /// </summary>
        private CommandStatus QueryStatusImportStxt(Command command)
        {
            if (_CurrentItem is WaveSoundSetPack)
            {
                return CommandStatus.SupportedAndEnabledAndVisible;
            }

            return CommandStatus.Supported;
        }

        ///--------------------------------
        /// <summary>
        /// "Stxtのインポート"の実行
        /// </summary>
        private bool ExecuteImportStxt(Command command)
        {
            return ImportStxt();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool ImportStxt()
        {
            OpenFileDialog dialog = new OpenFileDialog();
            StxtData[] datas = null;
            string filePath = null;

            dialog.Title = MessageResource.DialogTitle_File_ImportStxt_Text;
            dialog.Filter = ("Stxt Files (*.stxt)|*.stxt" +
                              "|All Files (*.*)|*.*");
            dialog.RestoreDirectory = true;
            if (dialog.ShowDialog() != DialogResult.OK)
            {
                return false;
            }

            filePath = dialog.FileName;

            try
            {
                datas = StxtReader.Read(filePath);
            }

            catch (Exception e)
            {
                ApplicationBase.Instance.UIService.ShowMessageBox(e.Message);
                return false;
            }

            WaveSoundSetBase waveSoundSet = null;
            WaveSoundBase waveSound = null;

            waveSoundSet = new WaveSoundSetBase();
            waveSoundSet.Name = ItemNamer.CreateUniqueNameFromFileName
                (ProjectService.ComponentDictionary,
                  new ItemNamingSettings(ProjectService.Project),
                  ProjectService.Project.WaveSoundSetNamePrefix, String.Empty, filePath);

            foreach (StxtData data in datas)
            {
                waveSound = ApplicationBase.Instance.Traits.ComponentTraits.SoundSetComponentFactory.Create(typeof(WaveSoundBase)) as WaveSoundBase;
                waveSound.Name = data.Label;
                waveSound.PlayerReference = GetInitialPlayer();
                waveSound.Encoding = WaveEncoding.Adpcm;
                waveSound.FilePath = data.FilePath;
                waveSound.ChannelPriority = data.ChannelPriority;
                waveSound.Volume = data.Volume;
                waveSound.Pan = data.Pan;
                waveSound.Pitch = Convert.ToSingle(data.Pitch);
                waveSound.SurroundPan = data.Span;
                waveSound.Sends.AuxASend = data.SendA;
                waveSound.Sends.AuxBSend = data.SendB;
                waveSound.Comment = data.Comment;

                waveSoundSet.Children.Add(waveSound);
            }

            try
            {
                SoundSetDocument.OperationHistory.BeginTransaction();

                if (ProjectService.Players.Count() <= 0)
                {
                    AddSoundHandler.AddDefaultPlayer
                        (ProjectService, SoundSetDocument, ApplicationBase.DefaultPlayerName);
                }

                if (AddSoundSetItemHandler.Execute(SoundSetDocument,
                                                    (SoundSetItem)_CurrentItem,
                                                    null, waveSoundSet) == false)
                {
                    SoundSetDocument.OperationHistory.CancelTransaction();
                    return false;
                }

                //
                EventRouter router = null;

                router = ((IEventRoutable)waveSoundSet).Router;
                router.BubbleEvent(new RoutingEventArgs
                                    (new RequestUpdateSoundProjectItemBindingEvent(),
                                      waveSoundSet));

                SoundSetDocument.OperationHistory.EndTransaction();
                return true;
            }
            catch
            {
                SoundSetDocument.OperationHistory.CancelTransaction();
            }

            return false;
        }

        /// <summary>
        /// "すべての参照の検索"が実行可能なのか調べる
        /// </summary>
        private CommandStatus QueryStatusFindReferences(Command command)
        {
            Component[] components = TargetComponents;
            return components.Length == 1 ? CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// "全ての参照の検索"の実行
        /// </summary>
        private bool ExecuteFindReferences(Command command)
        {
            Component[] components = TargetComponents;

            if (components.Length != 1)
            {
                return false;
            }

            FindReferences(components[0]);
            return true;
        }

        /// <summary>
        ///
        /// </summary>
        private void FindReferences(Component targetComponent)
        {
            List<Component> list = new List<Component>();
            Component[] components = GetSoundSetItems();

            foreach (Component component in components)
            {
                if (DoReference(component, targetComponent) == true)
                {
                    list.Add(component);
                }
            }

            MainWindow.AddFindTabPage(FindArgs.TargetDocumentKinds.SoundSet, list.ToArray());
        }

        /// <summary>
        ///
        /// </summary>
        private bool DoReference(Component component, Component targetComponent)
        {
            Sound sound = null;
            GroupBase group = null;

            if (component is GroupBase)
            {
                group = component as GroupBase;

                foreach (GroupItemBase groupItem in group.Children)
                {
                    if (groupItem.Target == targetComponent)
                    {
                        return true;
                    }
                }
            }

            if (component is Sound)
            {
                sound = component as Sound;
                return sound.IsReference(targetComponent);
            }

            if (component is SoundSetItem)
            {
                SoundSetItem soundSetItem = component as SoundSetItem;
                return soundSetItem.IsReference(targetComponent);
            }

            return false;
        }


        /// <summary>
        /// "1トラックに統合する"が実行可能か調べます
        /// </summary>
        private CommandStatus QueryCombineAACTrack(Command command)
        {
            if (AppConfiguration.EnabledMultiChannelAAC == false)
            {
                return CommandStatus.Unsupported;
            }

            if (ActiveListCtrl == _ListCtrlSub &&
                _ListCtrlSub.SelectedItems.Length == 2 &&
                (_ListCtrlSub.SelectedItems[0] as CommonListItem).Target is StreamSoundTrackBase == true)
            {
                StreamSoundTrackBase track = (_ListCtrlSub.SelectedItems[0] as CommonListItem).Target as StreamSoundTrackBase;
                if (track.IsMultiChannelAACTrack() == true)
                {
                    StreamSoundTrackBase track1 = (_ListCtrlSub.SelectedItems[0] as CommonListItem).Target as StreamSoundTrackBase;
                    StreamSoundTrackBase track2 = (_ListCtrlSub.SelectedItems[1] as CommonListItem).Target as StreamSoundTrackBase;
                    // 隣同士判定
                    if (this.IsNextToEachOther(track1, track2) == true &&
                        track1.ChannelCount == 1 && track2.ChannelCount == 1)
                    {
                        return CommandStatus.SupportedAndEnabledAndVisible;
                    }
                }
            }

            return CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// "1トラックに統合する"の実行
        /// </summary>
        private bool ExecuteCombineAACTrack(Command command)
        {
            if (AppConfiguration.EnabledMultiChannelAAC == true)
            {
                if (ActiveListCtrl == _ListCtrlSub &&
                    _ListCtrlSub.SelectedItems.Length == 2 &&
                    (_ListCtrlSub.SelectedItems[0] as CommonListItem).Target is StreamSoundTrackBase == true)
                {
                    StreamSoundTrackBase track = (_ListCtrlSub.SelectedItems[0] as CommonListItem).Target as StreamSoundTrackBase;
                    if (track.IsMultiChannelAACTrack() == true)
                    {
                        StreamSoundTrackBase track1 = (_ListCtrlSub.SelectedItems[0] as CommonListItem).Target as StreamSoundTrackBase;
                        StreamSoundTrackBase track2 = (_ListCtrlSub.SelectedItems[1] as CommonListItem).Target as StreamSoundTrackBase;


                        if (this.IsNextToEachOther(track1, track2) == true && // 隣同士判定
                            track1.ChannelCount == 1 && track2.ChannelCount == 1)
                        {
                            try
                            {
                                SoundSetDocument.OperationHistory.BeginTransaction();
                                Operation operation = new SetParameterOperation(track1.Parameters, ProjectParameterNames.StreamSoundTrack.ChannelCount, 2);
                                if (operation.Execute() == false)
                                {
                                    SoundSetDocument.OperationHistory.CancelTransaction();
                                    return false;
                                }
                                SoundSetDocument.OperationHistory.AddOperation(operation);

                                operation = new RemoveComponentOperation(track2);
                                if (operation.Execute() == false)
                                {
                                    SoundSetDocument.OperationHistory.CancelTransaction();
                                    return false;
                                }
                                SoundSetDocument.OperationHistory.AddOperation(operation);
                                SoundSetDocument.OperationHistory.EndTransaction();
                            }
                            catch
                            {
                                SoundSetDocument.OperationHistory.CancelTransaction();
                            }
                        }
                    }
                }
            }

            return true;
        }

        private bool IsNextToEachOther(StreamSoundTrackBase track1, StreamSoundTrackBase track2)
        {
            Debug.Assert(track1.Parent == track2.Parent);

            StreamSoundBase streamSound = track1.Parent as StreamSoundBase;

            int index1 = streamSound.Children.IndexOf(track1);
            int index2 = streamSound.Children.IndexOf(track2);
            int sub = Math.Abs(index1 - index2);

            return sub == 1; // 隣同士判定
        }

        /// <summary>
        /// "2トラックに分割する"が実行可能か調べます
        /// </summary>
        private CommandStatus QueryDivideAACTrack(Command command)
        {
            if (AppConfiguration.EnabledMultiChannelAAC == true)
            {
                if (ActiveListCtrl == _ListCtrlSub &&
                    _ListCtrlSub.SelectedItems.Length == 1 &&
                    (_ListCtrlSub.SelectedItems[0] as CommonListItem).Target is StreamSoundTrackBase == true)
                {
                    StreamSoundTrackBase track = (_ListCtrlSub.SelectedItems[0] as CommonListItem).Target as StreamSoundTrackBase;
                    if (track.IsMultiChannelAACTrack() == true)
                    {
                        // 仕様は単数選択だが、念のため複数選択に対応しておく。
                        foreach (CommonListItem item in _ListCtrlSub.SelectedItems)
                        {
                            track = item.Target as StreamSoundTrackBase;
                            if (track.ChannelCount < 2)
                            {
                                return CommandStatus.SupportedAndVisible;
                            }
                        }

                        return CommandStatus.SupportedAndEnabledAndVisible;
                    }
                }

                return CommandStatus.SupportedAndVisible;
            }
            else
            {
                return CommandStatus.Unsupported;
            }
        }

        /// <summary>
        /// "2トラックに分割する"の実行
        /// </summary>
        private bool ExecuteDivideAACTrack(Command command)
        {
            if (AppConfiguration.EnabledMultiChannelAAC == true)
            {
                if (ActiveListCtrl == _ListCtrlSub &&
                    _ListCtrlSub.SelectedItems.Length == 1 &&
                    (_ListCtrlSub.SelectedItems[0] as CommonListItem).Target is StreamSoundTrackBase == true)
                {
                    StreamSoundTrackBase track = (_ListCtrlSub.SelectedItems[0] as CommonListItem).Target as StreamSoundTrackBase;
                    if (track.IsMultiChannelAACTrack() == true)
                    {
                        // 仕様は単数選択だが、念のため複数選択に対応しておく。
                        foreach (CommonListItem item in _ListCtrlSub.SelectedItems)
                        {
                            track = item.Target as StreamSoundTrackBase;
                            if (track.ChannelCount == 2)
                            {
                                try
                                {
                                    SoundSetDocument.OperationHistory.BeginTransaction();
                                    Operation operation = new SetParameterOperation(track.Parameters, ProjectParameterNames.StreamSoundTrack.ChannelCount, 1);
                                    if (operation.Execute() == false)
                                    {
                                        SoundSetDocument.OperationHistory.CancelTransaction();
                                        return false;
                                    }
                                    SoundSetDocument.OperationHistory.AddOperation(operation);

                                    StreamSoundTrackBase newTrack = track.Clone() as StreamSoundTrackBase;
                                    operation = new InsertComponentOperation(track.Parent, track, newTrack);
                                    if (operation.Execute() == false)
                                    {
                                        SoundSetDocument.OperationHistory.CancelTransaction();
                                        return false;
                                    }
                                    SoundSetDocument.OperationHistory.AddOperation(operation);
                                    SoundSetDocument.OperationHistory.EndTransaction();
                                }
                                catch
                                {
                                    SoundSetDocument.OperationHistory.CancelTransaction();
                                }
                            }
                        }
                    }
                }
            }

            return true;
        }

        /// <summary>
        /// "インゲーム編集対象にする"、"インゲーム編集対象から外す"、
        /// </summary>
        private CommandStatus QueryStatusSndEdit(Command command)
        {
            if (SoundSetDocument.IsReadOnly == true)
            {
                return CommandStatus.SupportedAndVisible;
            }

            bool enabled = false;

            if (ActiveListCtrl != null)
            {
                Component[] components = ActiveListCtrl.GetSelectedItems()
                    .Select(i => i.Target)
                    .Cast<Component>()
                    .Where(c => c.HasSndEdit())
                    .ToArray();
                if (components.Length > 0)
                {
                    enabled = true;
                }
            }

            return (enabled != false ?
                     CommandStatus.SupportedAndEnabledAndVisible :
                     CommandStatus.SupportedAndVisible);
        }

        /// <summary>
        /// "インゲーム編集対象にする"の実行です。
        /// </summary>
        private bool ExecuteEnableSndEdit(Command command)
        {
            Component[] components = ActiveListCtrl.GetSelectedItems()
                .Select(i => i.Target)
                .Cast<Component>()
                .ToArray();
            EnableSndEdit(components, true);
            return true;
        }

        /// <summary>
        /// "インゲーム編集対象にする"の実行です。
        /// </summary>
        private bool ExecuteDisableSndEdit(Command command)
        {
            Component[] components = ActiveListCtrl.GetSelectedItems()
                .Select(i => i.Target)
                .Cast<Component>()
                .ToArray();
            EnableSndEdit(components, false);
            return true;
        }

        /// <summary>
        ///
        /// </summary>
        private void EnableSndEdit(Component[] components, bool b)
        {
            foreach (Component component in components)
            {
                component.Parameters[ProjectParameterNames.SndEdit].Value = b;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private Component[] GetSoundSetItems()
        {
            List<Component> list = new List<Component>();

            foreach (SoundSetDocument soundSetDocument in ProjectService.SoundSetDocuments)
            {
                list.AddRange(GetSoundSetItems(soundSetDocument));
            }
            return list.ToArray();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private Component[] GetSoundSetItems(SoundSetDocument soundSetDocument)
        {
            List<Component> list = new List<Component>();
            SoundSet soundSet = soundSetDocument.SoundSet;

            foreach (Component component in soundSet.Children)
            {
                list.AddRange(component.Children);

                if (component is StreamSoundBase ||
                    component is WaveSoundSetPack ||
                    component is SequenceSoundSetPack)
                {
                    foreach (Component childComponent in component.Children)
                    {
                        list.AddRange(childComponent.Children);
                    }
                }
            }

            return list.ToArray();
        }

        ///--------------------------------
        /// <summary>
        /// "グループに登録"が実行可能なのか調べる
        /// </summary>
        private CommandStatus QueryStatusAddToGroup(Command command)
        {
            Component[] components = TargetComponents;

            if (components.Length <= 0)
            {
                return CommandStatus.SupportedAndVisible;
            }

            foreach (Component component in components)
            {
                if (CanAddGroup(component) == false)
                {
                    return CommandStatus.SupportedAndVisible;
                }
            }

            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        ///--------------------------------
        /// <summary>
        /// "グループに登録"の実行
        /// </summary>
        private bool ExecuteAddToGroup(Command command)
        {
            Component[] components = TargetComponents;
            GroupBase[] groups = null;

            if (components.Length <= 0)
            {
                return false;
            }

            GroupRegisteredState[] states = null;
            GroupRegisteredState.States state;
            int[] registeredCount = null;
            int groupCount = 0;

            groups = ExcludeGroup(ProjectService.Groups, components);
            groupCount = groups.Length;

            //
            registeredCount = new int[groupCount];
            for (int index = 0; index < groupCount; index++)
            {
                registeredCount[index] = 0;
            }

            for (int index = 0; index < groupCount; index++)
            {
                if (IncludedGroup(groups[index], components) == true)
                {
                    registeredCount[index]++;
                }
            }

            //
            states = new GroupRegisteredState[groupCount];

            for (int index = 0; index < groupCount; index++)
            {

                state = GroupRegisteredState.States.Intermingled;
                if (registeredCount[index] == 0)
                {
                    state = GroupRegisteredState.States.Unregistered;
                }

                if (registeredCount[index] == components.Length)
                {
                    state = GroupRegisteredState.States.Registered;
                }

                states[index] = new GroupRegisteredState();
                states[index].State = state;
                states[index].Group = groups[index];
            }

            //
            GroupItemEntryDialog dialog = null;

            dialog = new GroupItemEntryDialog(states, components);
            if (dialog.ShowDialog() != DialogResult.OK)
            {
                return true;
            }

            return AddToGroup(dialog.GetGroupRegisteredStates(), components);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool AddToGroup(GroupRegisteredState[] states, Component[] components)
        {
            SoundSetDocument soundSetDocument = null;
            GroupItemBase[] groupItems = null;
            GroupItemBase groupItem = null;
            OperationHistory operationHistory = null;
            Operation operation = null;
            GroupBelong[] groupBelongs = null;
            bool included = false;

            //グループの所属するサウンドセットごとに分類する
            if ((groupBelongs = ClassificationGroup(states)) == null)
            {
                return false;
            }

            //サウンドセットごとに登録する
            foreach (GroupBelong groupBelong in groupBelongs)
            {

                soundSetDocument = groupBelong.SoundSetDocument;
                operationHistory = soundSetDocument.OperationHistory;
                operationHistory.BeginTransaction();

                //
                foreach (GroupRegisteredState state in groupBelong.States)
                {
                    HashSet<GroupItemBase> hashSet = new HashSet<GroupItemBase>();

                    //
                    foreach (Component component in components)
                    {

                        included = IncludedGroupItemInGroup(state.Group, component);
                        operation = null;

                        switch (state.State)
                        {
                            case GroupRegisteredState.States.Unregistered:
                                if (included == true)
                                {
                                    groupItems = state.Group.Children
                                        .Cast<GroupItemBase>().Where(c => c.Target == component)
                                        .ToArray();

                                    if (groupItems.Length > 0)
                                    {
                                        operation = new RemoveComponentOperation(groupItems);
                                    }
                                }
                                break;

                            case GroupRegisteredState.States.Registered:
                                if (included == false)
                                {
                                    groupItem = this.CreateComponentService.Create<GroupItemBase>();
                                    groupItem.TargetItemReference = component.Name;
                                    groupItem.TargetSoundSet = soundSetDocument.SoundSet.Name;
                                    groupItem.Target = component as SoundSetItem;

                                    operation = new InsertComponentOperation
                                        (state.Group, null, new[] { groupItem });

#if true
                                    // シーケンスサウンドセット内に含まれているシーケンスサウンド
                                    // をグループから削除します。(ユーザーに問い合わせる)
                                    GroupItemBase[] removeGroupItems = GroupUtility.GetRemoveGroupItem
                                        (state.Group, component);
                                    foreach (GroupItemBase removeGroupItem in removeGroupItems)
                                    {
                                        hashSet.Add(removeGroupItem);
                                    }
#endif
                                }
                                break;
                        }

                        //
                        if (operation != null)
                        {
                            operation.Execute();
                            operationHistory.AddOperation(operation);
                        }
                    }

                    //
                    if (hashSet.Count > 0)
                    {
                        operation = new RemoveComponentOperation(hashSet.ToArray());
                        operation.Execute();
                        operationHistory.AddOperation(operation);
                    }
                }

                //
                operationHistory.EndTransaction();
            }
            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool CanAddGroup(Component component)
        {
            if ((component is StreamSoundBase && AppConfiguration.EnabledPrefetch == true) ||
                component is SequenceSoundBase ||
                component is WaveSoundSetBase ||
                component is SequenceSoundSetBase ||
                component is SoundSetBankBase ||
                component is WaveArchiveBase ||
                component is GroupBase)
            {
                return true;
            }
            return false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool IncludedGroup(GroupBase group, Component[] components)
        {
            foreach (GroupItemBase groupItem in group.Children)
            {
                if (components.Contains(groupItem.Target) == true)
                {
                    return true;
                }
            }
            return false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool IncludedGroupInSoundSet(SoundSet soundSet, GroupBase targetGroup)
        {
            GroupPack[] groupPacks = null;

            groupPacks = soundSet.Children.Cast<Component>()
                .Where(c => c is GroupPack).Cast<GroupPack>().ToArray();

            foreach (GroupBase group in groupPacks[0].Children)
            {
                if (group == targetGroup)
                {
                    return true;
                }
            }
            return false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool IncludedGroupItemInGroup(GroupBase group, Component component)
        {
            foreach (GroupItemBase groupItem in group.Children)
            {
                if (groupItem.Target == component)
                {
                    return true;
                }
            }
            return false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private GroupBase[] ExcludeGroup(GroupBase[] groups, Component[] components)
        {
            List<GroupBase> list = new List<GroupBase>();

            foreach (GroupBase group in groups)
            {
                if (components.Contains(group) == false)
                {
                    list.Add(group);
                }
            }
            return list.ToArray();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private class GroupBelong
        {
            private SoundSetDocument _SoundSetDocument = null;
            private List<GroupRegisteredState> _States = new List<GroupRegisteredState>();

            ///
            public GroupBelong(SoundSetDocument soundSetDocument)
            {
                _SoundSetDocument = soundSetDocument;
            }

            ///
            public SoundSetDocument SoundSetDocument
            {
                get { return _SoundSetDocument; }
            }

            ///
            public List<GroupRegisteredState> States
            {
                get { return _States; }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private GroupBelong[] ClassificationGroup(GroupRegisteredState[] states)
        {
            SoundSetDocument[] soundSetDocuments = null;
            GroupBelong[] groupBelongs = null;

            soundSetDocuments = ProjectService.SoundSetDocuments;
            groupBelongs = new GroupBelong[soundSetDocuments.Length];

            for (int index = 0; index < soundSetDocuments.Length; index++)
            {
                groupBelongs[index] = new GroupBelong(soundSetDocuments[index]);
            }

            //
            foreach (GroupRegisteredState state in states)
            {

                foreach (GroupBelong groupBelong in groupBelongs)
                {
                    if (IncludedGroupInSoundSet(groupBelong.SoundSetDocument.SoundSet,
                                                 state.Group) == true)
                    {
                        groupBelong.States.Add(state);
                        break;
                    }
                }
            }

            return groupBelongs;
        }

        ///--------------------------------
        /// <summary>
        /// "HTMLインストルメントリストの出力" が実行可能かどうか調べます。
        /// </summary>
        private CommandStatus QueryStatusOutputHtmlInstrumentList(Command command)
        {
            if (!(_CurrentItem is SoundSetBankPack)) { return CommandStatus.Supported; }

            Component[] components = GetOutputHtmlInstrumentListTargets();

            if (0 == components.Length)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        ///--------------------------------
        /// <summary>
        /// "HTMLインストルメントリストの出力" を実行します。
        /// </summary>
        private bool ExecuteOutputHtmlInstrumentList(Command command)
        {
            Component[] components = GetOutputHtmlInstrumentListTargets();
            if (0 == components.Length) { return true; }

            HtmlInstrumentListSettingsDialog dialog = new HtmlInstrumentListSettingsDialog();
            dialog.LoadSettings(ProjectConfiguration.Output.HtmlInstrumentList);

            if (dialog.ShowDialog(this) != DialogResult.OK)
            {
                return false;
            }

            dialog.SaveSettings(ProjectConfiguration.Output.HtmlInstrumentList);

            // HTML ファイル出力
            foreach (SoundSetBankBase bank in components)
            {
                try
                {
                    BankReportExporter exporter = new BankReportExporter()
                    {
                        Generator = string.Format("{0} {1}",
                            ApplicationBase.Instance.Traits.ApplicationAssembly.GetTitle(),
                            ApplicationBase.Instance.Traits.ApplicationAssembly.GetName().Version.ToString(3)),
                    };

                    using (BankServiceReference bankServiceReference =
                        ApplicationBase.Instance.BankServices.OpenItem(bank.FilePath))
                    {
                        using (TextWriter writer = new StreamWriter(
                            Path.ChangeExtension(bank.FilePath, ".html"), false, Encoding.UTF8))
                        {
                            exporter.Export(writer, bankServiceReference.Target, dialog.ListOutput);
                        }
                    }
                }
                catch (FileNotFoundException exception)
                {
                    ApplicationBase.Instance.UIService.ShowMessageBox(
                        string.Format(MessageResource.Message_TargetFileNotFound, exception.FileName));
                }
                catch (System.Xml.XmlException)
                {
                    ApplicationBase.Instance.UIService.ShowMessageBox(
                        string.Format(CoreResources.MessageResource.Message_UnsupportedFileFormat, bank.FilePath));
                }
                catch
                {
                    ApplicationBase.Instance.UIService.ShowMessageBox(MessageResource.Message_UnknownError);
                    break;
                }
            }

            return true;
        }

        private Component[] GetOutputHtmlInstrumentListTargets()
        {
            Component[] components = TargetComponents;

            if (components.Length <= 0)
            {
                return new Component[0];
            }

            foreach (Component component in components)
            {
                if (!(component is SoundSetBankBase))
                {
                    return new Component[0];
                }
            }

            return components;
        }


        ///--------------------------------
        /// <summary>
        /// アイテムの追加コマンドが実行できるかどうかを取得します。
        /// </summary>
        /// <param name="command">アイテムの追加コマンド</param>
        /// <returns>実行できる場合は true、できない場合は false。</returns>
        private CommandStatus QueryStatusAddItem(Command command)
        {
            if (SoundSetDocument == null) { return CommandStatus.SupportedAndVisible; }
            if (SoundSetDocument.IsReadOnly == true) { return CommandStatus.SupportedAndVisible; }

            Command targetCommand = AddItemCommand;
            if (null == targetCommand) { return CommandStatus.SupportedAndVisible; }

            return ApplicationBase.Instance.QueryCommandStatus(targetCommand);
        }

        /// <summary>
        /// アイテムの追加コマンドを実行します。
        /// </summary>
        /// <param name="command"></param>
        /// <returns></returns>
        private bool ExecuteAddItem(Command command)
        {
            Command targetCommand = AddItemCommand;
            if (null == targetCommand) { return false; }

            return ApplicationBase.Instance.ExecuteCommand(targetCommand);
        }

        ///--------------------------------
        /// <summary>
        /// アイテムの挿入コマンドが実行できるかどうかを取得します。
        /// </summary>
        /// <param name="command">アイテムの挿入コマンド</param>
        /// <returns>実行できる場合は true、できない場合は false。</returns>
        private CommandStatus QueryStatusInsertItem(Command command)
        {
            if (SoundSetDocument == null) { return CommandStatus.SupportedAndVisible; }
            if (SoundSetDocument.IsReadOnly == true) { return CommandStatus.SupportedAndVisible; }

            Command targetCommand = InsertItemCommand;
            if (null == targetCommand) { return CommandStatus.SupportedAndVisible; }

            return ApplicationBase.Instance.QueryCommandStatus(targetCommand);
        }

        /// <summary>
        /// アイテムの挿入コマンドを実行します。
        /// </summary>
        /// <param name="command"></param>
        /// <returns></returns>
        private bool ExecuteInsertItem(Command command)
        {
            Command targetCommand = InsertItemCommand;
            if (null == targetCommand) { return false; }

            return ApplicationBase.Instance.ExecuteCommand(targetCommand);
        }

        /// <summary>
        ///
        /// </summary>
        private void ApplySoundSetConfiguration()
        {
            if (null == SoundSetViewConfiguration) { return; }

            XmlSoundSetDocumentView configuration = SoundSetViewConfiguration;
            _xmlLists = configuration.Lists.ExportDictionary();

            _StreamSoundViewSplitterRatio = configuration.StreamSoundTrackListView.SplitterPositionRatio;
            _WaveSoundSetViewSplitterRatio = configuration.WaveSoundSetListView.SplitterPositionRatio;
            _SequenceSoundSetViewSplitterRatio = configuration.SequenceSoundSetListView.SplitterPositionRatio;
            _GroupViewSplitterRatio = configuration.GroupListView.SplitterPositionRatio;

            //ListConfigurationApplier applier = new ListConfigurationApplier(_headerAdapters);
            ListConfigurationApplier applier = FormsApplication.Instance.UIService.CreateListConfigurationApplier(_headerAdapters);

            foreach (XmlList xmlList in _xmlLists.Values)
            {
                applier.Apply(xmlList);
            }
        }

        private void ExtractSoundSetConfiguration()
        {
            if (null == SoundSetViewConfiguration) { return; }
            if (null == _xmlLists) { return; }

            XmlSoundSetDocumentView configuration = SoundSetViewConfiguration;

            configuration.StreamSoundTrackListView.SplitterPositionRatio = _StreamSoundViewSplitterRatio;
            configuration.WaveSoundSetListView.SplitterPositionRatio = _WaveSoundSetViewSplitterRatio;
            configuration.SequenceSoundSetListView.SplitterPositionRatio = _SequenceSoundSetViewSplitterRatio;
            configuration.GroupListView.SplitterPositionRatio = _GroupViewSplitterRatio;

            //ListConfigurationApplier applier = new ListConfigurationApplier(_headerAdapters);
            ListConfigurationApplier applier = FormsApplication.Instance.UIService.CreateListConfigurationApplier(_headerAdapters);

            foreach (XmlList xmlList in _xmlLists.Values)
            {
                applier.Extract(xmlList);
            }

            configuration.Lists.List = _xmlLists.Values.ToArray();
        }

        /// <summary>
        /// メニューを初期化します。
        /// </summary>
        private void InitializeMenu()
        {
            if (null != _soundListMenuAdapter) { return; }

            _soundListMenuAdapter = ToolStripAdapter.FromToolStrip(FormsApplication.Instance.CommandService,
                                                                    _soundListMenu);
            _soundListMenuAdapter.CommandTarget = FormsApplication.Instance;
            _soundListMenuAdapter.CommandExecuted += OnSoundListCommandExecuted;

            ApplySettings(SettingKinds.ColorComment);
            BuildListCommandUI();
        }

        /// <summary>
        /// コマンドバーを再構築します。
        /// </summary>
        private void BuildCommandUIs()
        {
            BuildListCommandUI();
            MainWindow.BuildCommandUI();
        }

        /// <summary>
        /// コマンドバーを再構築します。
        /// </summary>
        private void BuildListCommandUI()
        {
            if (null != _soundListMenuAdapter)
            {
                _soundListMenuAdapter.BuildUI();
            }
        }

        private void UpdateSplitterRatio()
        {
            CurrentViewSplitterRatio = (double)splitContainer.SplitterDistance / splitContainer.Height;
        }

        private void OnSplitterMoved(object sender, SplitterEventArgs e)
        {
            UpdateSplitterRatio();
        }

        private void OnSplitContainerResize(object sender, EventArgs e)
        {
            UpdateSplitterRatio();
        }

        ///--------------------------------
        /// <summary>
        /// リスト項目設定関連
        /// </summary>
        private void OnHeaderMenuItemAddPresetListColumnsClick(object sender, EventArgs e)
        {
            CommonListCtrl listCtrl = this.GetListCtrl();
            string listName = GetListName(_CurrentItem, _ListCtrlMain == listCtrl);
            ListConfigurationApplier applier = FormsApplication.Instance.UIService.CreateListConfigurationApplier(_headerAdapters);

            ListHeaderHelper.AddPresetListColumns(listName, applier);

            PresetListColumnsService preset = ApplicationBase.Instance.PresetListColumnsService;
            MainWindow.PresetListColumnsPresetNamesOnSplitButton = preset.GetPresetNames(listName);
            MainWindow.PresetListColumnsPresetCurrentNameOnSplitButton = preset.GetCurrentPresetName(listName);
        }

        private void OnHeaderMenuItemApplyPresetListColumnsClick(object sender, EventArgs e)
        {
            ToolStripMenuItem menuItem = sender as ToolStripMenuItem;
            CommonListCtrl listCtrl = this.GetListCtrl();
            string listName = GetListName(_CurrentItem, _ListCtrlMain == listCtrl);
            ListConfigurationApplier applier = FormsApplication.Instance.UIService.CreateListConfigurationApplier(_headerAdapters);

            ListHeaderHelper.ApplyPresetListColumns(applier, listName, menuItem.Text);

            listCtrl.UpdateInfos();
        }

        private void OnHeaderCtrlContextMenuOpening(object sender, System.ComponentModel.CancelEventArgs e)
        {
            CommonListCtrl listCtrl = this.GetListCtrl();
            string listName = GetListName(_CurrentItem, _ListCtrlMain == listCtrl);

            ListHeaderHelper.CreateMeunApplyPresetListColumns(_HeaderContextMenuItem_ApplyPresetListColumns, listName, OnHeaderMenuItemApplyPresetListColumnsClick);
        }

        private void UpdatePresetListColumnsSplitButton(CommonListCtrl listCtrl)
        {
            string listName = GetListName(_CurrentItem, _ListCtrlMain == listCtrl);
            PresetListColumnsService preset = ApplicationBase.Instance.PresetListColumnsService;
            MainWindow.PresetListColumnsPresetNamesOnSplitButton = preset.GetPresetNames(listName);
            MainWindow.PresetListColumnsPresetCurrentNameOnSplitButton = preset.GetCurrentPresetName(listName);
        }

        private CommonListCtrl GetListCtrl()
        {
            CommonListCtrl listCtrl = ActiveListCtrl;
            if (listCtrl == null)
            {
                listCtrl = _ListCtrlMain;
            }

            return listCtrl;
        }

        /// <summary>
        /// プロジェクト設定が保存される前に発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">空のイベントデータ。</param>
        private void OnProjectConfigurationSaving(object sender, EventArgs e)
        {
            ExtractSoundSetConfiguration();
        }

        private string GetListName(Component item, bool main)
        {
            string name = string.Empty;

            if (item is StreamSoundPack)
            {
                if (main == true)
                {
                    name = ListTraits.ListName_StreamSound;
                }
                else
                {
                    name = ListTraits.ListName_StreamSoundTrack;
                }
            }
            else if (item is WaveSoundSetPack)
            {
                if (main == true)
                {
                    name = ListTraits.ListName_WaveSoundSet;
                }
                else
                {
                    name = ListTraits.ListName_WaveSoundSetItem;
                }
            }
            else if (item is WaveSoundSetBase)
            {
                name = ListTraits.ListName_WaveSoundSetItem;
            }
            else if (item is SequenceSoundSetPack)
            {
                if (main == true)
                {
                    name = ListTraits.ListName_SequenceSoundSet;
                }
                else
                {
                    name = ListTraits.ListName_SequenceSoundSetItem;
                }
            }
            else if (item is SequenceSoundSetBase)
            {
                name = ListTraits.ListName_SequenceSoundSetItem;
            }
            else if (item is SequenceSoundPack)
            {
                name = ListTraits.ListName_SequenceSound;
            }
            else if (item is SoundSetBankPack)
            {
                name = ListTraits.ListName_Bank;
            }
            else if (item is WaveArchivePack)
            {
                name = ListTraits.ListName_WaveArchive;
            }
            else if (item is PlayerPack)
            {
                name = ListTraits.ListName_Player;
            }
            else if (item is GroupPack)
            {
                if (main == true)
                {
                    name = ListTraits.ListName_Group;
                }
                else
                {
                    name = ListTraits.ListName_GroupItem;
                }
            }

            return name;
        }

        /// <summary>
        ///
        /// </summary>
        private void OnPreviewPlay(object sender, EventArgs e)
        {
            if (PreviewPlay != null)
            {
                Component component = sender as Component;
                component.UpdateWaveFile();
                PreviewPlay(sender, e);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnPreviewMute(object sender, EventArgs e)
        {
            if (PreviewMute != null)
            {
                PreviewMute(sender, e);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnPreviewSoloPlay(object sender, EventArgs e)
        {
            if (PreviewSoloPlay != null)
            {
                PreviewSoloPlay(sender, e);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnOpening_soundListMenu(object sender, System.ComponentModel.CancelEventArgs e)
        {
            BuildListCommandUI();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnUpdatedSub(object sender, EventArgs e)
        {
            _ListCtrlMain.Invalidate();
        }

        private string MultiChannelDoShowErrorIcon(Component component)
        {
            if (AppConfiguration.EnabledMultiChannelAAC == true)
            {
                if (component is StreamSoundTrackBase == true)
                {
                    StreamSoundTrackBase streamSoundTrack = component as StreamSoundTrackBase;
                    StreamSoundBase streamSound = streamSoundTrack.Parent as StreamSoundBase;

                    streamSoundTrack.UpdateWaveFile();

                    if (streamSoundTrack.WaveFile != null &&
                        streamSound.IsMultiChannelAAC() == true)
                    {
                        int count = 0;
                        foreach (StreamSoundTrackBase track in streamSound.Children)
                        {
                            count += track.ChannelCount;
                            if (track == streamSoundTrack)
                            {
                                if (streamSoundTrack.WaveFile.ChannelCount < count)
                                {
                                    // Siglo は AAC に対応しないのでメッセージ対応は省略します。
                                    return string.Empty;
                                }
                            }
                        }
                    }
                    else if (streamSoundTrack.WaveFile != null &&
                             AppConfiguration.WaveFileMultiChannelCountMax < streamSoundTrack.WaveFile.ChannelCount)
                    {
                        // Siglo は AAC に対応しないのでメッセージ対応は省略します。
                        return string.Empty;
                    }
                }
                else if (component is StreamSoundBase == true)
                {
                    StreamSoundBase streamSound = component as StreamSoundBase;

                    if (streamSound.IsMultiChannelAAC() == true)
                    {
                        StreamSoundTrackBase streamSoundTrack = streamSound.Children[0] as StreamSoundTrackBase;
                        streamSoundTrack.UpdateWaveFile();

                        int count = 0;

                        foreach (StreamSoundTrackBase track in streamSound.Children)
                        {
                            count += track.ChannelCount;
                        }

                        if (streamSoundTrack.WaveFile != null &&
                            streamSoundTrack.WaveFile.ChannelCount != count)
                        {
                            // Siglo は AAC に対応しないのでメッセージ対応は省略します。
                            return string.Empty;
                        }
                    }
                }
            }

            return null;
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class ListSubItemSndEditWithCheck : ListPartParameter, IListSubItem
    {
        /// <summary>
        ///
        /// </summary>
        string IListSubItem.Name
        {
            get { return "SndEdit"; }
        }

        /// <summary>
        ///
        /// </summary>
        bool IListSubItem.EditByKey
        {
            get { return true; }
        }

        /// <summary>
        ///
        /// </summary>
        IListPart[] IListSubItem.GetParts(ListDrawDescriptor desc)
        {
            ListPart part0 = null;
            ListPart part1 = null;
            Rectangle bounds = new Rectangle();

            int width = desc.Bounds.Width < 16 ? desc.Bounds.Width : 16;
            int height = desc.Bounds.Height < 16 ? desc.Bounds.Height : 16;
            int x = (desc.Bounds.Width) / 2 - width - 2;
            int y = (desc.Bounds.Height - height) / 2;

            if (x < 0)
            {
                x = 0;
            }

            bounds.X = x;
            bounds.Y = y;
            bounds.Width = width;
            bounds.Height = height;

            part0 = new ListPart("CheckBox", "CheckBox", null, bounds,
                                 (ListPartActions.EditByClick |
                                 ListPartActions.EditByClickOnCaret));

            //
            bounds.X = x + width + 4;
            part1 = new ListPart("SndEdit", "SndEdit", null, bounds,
                                 (ListPartActions.EditByClick |
                                 ListPartActions.EditByClickOnCaret));

            //
            return new[] { part0, part1 };
        }

        /// <summary>
        ///
        /// </summary>
        string IListSubItem.RepresentativeEditorName(bool immediateEdit)
        {
            return "CheckBox";
        }
    }
}
