﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections;
    using System.Collections.Generic;
    using System.Data;
    using System.Diagnostics;
    using System.Drawing;
    using System.Linq;
    using System.IO;
    using System.Windows.Forms;

    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows;

    public partial class InstrumentParameterPanel : System.Windows.Forms.UserControl
    {
        private bool readOnly = false;
        private Instrument _Instrument = null;
        private VelocityRegion _VelocityRegion = null;
        public event OperationExecutedEventHandler OperationExecuted;

        ///--------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public InstrumentParameterPanel()
        {
            InitializeComponent();
        }

        /// <summary>
        ///
        /// </summary>
        private MainWindow MainWindow
        {
            get { return FormsApplication.Instance.UIService.MainWindow; }
        }

        /// <summary>
        ///
        /// </summary>
        public Instrument Instrument
        {
            get { return _Instrument; }
            set
            {
                if (_Instrument != null)
                {
                    _Instrument.Parameters.ParameterValueChanged -= OnParameterValueChanged;
                }

                _Instrument = value;

                if (_Instrument != null)
                {
                    _Instrument.Parameters.ParameterValueChanged += OnParameterValueChanged;
                }

                UpdatePanel();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public VelocityRegion VelocityRegion
        {
            get { return _VelocityRegion; }
            set
            {
                if (_VelocityRegion != null)
                {
                    _VelocityRegion.Parameters.ParameterValueChanged -= OnParameterValueChanged;
                }

                _VelocityRegion = value;

                if (_VelocityRegion != null)
                {
                    _VelocityRegion.Parameters.ParameterValueChanged += OnParameterValueChanged;
                }

                UpdatePanel();
            }
        }

        public OperationHistory OperationHistory { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool ReadOnly
        {
            get
            {
                if (this.readOnly == false)
                {
                    if (_Instrument != null)
                    {
                        BankDocument[] BankDocuments =
                            FormsApplication.Instance.DocumentService.Documents
                            .Where(d => d is BankDocument).Cast<BankDocument>()
                            .ToArray();
                        foreach (BankDocument bankDocument in BankDocuments)
                        {
                            if (bankDocument.Bank == _Instrument.Bank)
                            {
                                return bankDocument.IsReadOnly;
                            }
                        }
                    }
                }

                return true;
            }
            set
            {
                this.readOnly = value;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void UpdateUI()
        {
            UpdatePanel();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool _SuspendEvent = false;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void OnOperationExecuted(Operation operation)
        {
            OnOperationExecuted(new OperationExecutedEventArgs(operation));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual void OnOperationExecuted(OperationExecutedEventArgs e)
        {
            if (OperationHistory != null)
            {
                OperationHistory.AddOperation(e.Operation);
            }

            if (OperationExecuted != null)
            {
                OperationExecuted(this, e);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void SetValue(Component component, string name, object value)
        {
            if (component.Parameters.ContainsKey(name) == false)
            {
                return;
            }

            SetParameterOperation operation =
                new SetParameterOperation(component.Parameters, name, value);
            operation.Execute();
            OnOperationExecuted(operation);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void BeginTransaction()
        {
            if (OperationHistory != null)
            {
                OperationHistory.BeginTransaction();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void EndTransaction()
        {
            if (OperationHistory != null)
            {
                OperationHistory.EndTransaction();
            }

            UpdatePanel();
        }

        ///
        private void SuspendEvent()
        {
            _SuspendEvent = true;
        }

        ///
        private void ResumeEvent()
        {
            _SuspendEvent = false;
        }

        ///
        private bool DoSuspendEvent
        {
            get { return _SuspendEvent; }
        }

        ///--------------------------------
        /// <summary>
        /// 使用されているリソースに後処理を実行します。
        /// </summary>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                if (components != null)
                {
                    components.Dispose();
                }
            }
            base.Dispose(disposing);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        delegate void UpdatePanelDelegate();
        private void UpdatePanel()
        {
            if (InvokeRequired == true)
            {
                Invoke(new UpdatePanelDelegate(UpdatePanel));
                return;
            }
            UpdateControlsStatus();
            UpdateControlsValue();
            UpdateReadOnlyControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void UpdateReadOnlyControls()
        {
            bool readOnly = ReadOnly;
            bool enabled = !ReadOnly;

            textBox_I_Volume.ReadOnly = readOnly;
            textBox_I_PitchSemitones.ReadOnly = readOnly;
            textBox_I_PitchCents.ReadOnly = readOnly;

            trackBar_I_Volume.Enabled = enabled;
            trackBar_I_PitchSemitones.Enabled = enabled;
            trackBar_I_PitchCents.Enabled = enabled;

            //
            textBox_A.ReadOnly = readOnly;
            textBox_H.ReadOnly = readOnly;
            textBox_D.ReadOnly = readOnly;
            textBox_S.ReadOnly = readOnly;
            textBox_R.ReadOnly = readOnly;

            trackBar_A.Enabled = enabled;
            trackBar_H.Enabled = enabled;
            trackBar_D.Enabled = enabled;
            trackBar_S.Enabled = enabled;
            trackBar_R.Enabled = enabled;

            //
            textBox_R_Volume.ReadOnly = readOnly;
            textBox_R_Pan.ReadOnly = readOnly;
            textBox_R_PitchSemitones.ReadOnly = readOnly;
            textBox_R_PitchCents.ReadOnly = readOnly;

            trackBar_R_Volume.Enabled = enabled;
            trackBar_R_Pan.Enabled = enabled;
            trackBar_R_PitchCents.Enabled = enabled;
            trackBar_R_PitchSemitones.Enabled = enabled;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void UpdateControlsStatus()
        {
            Enabled = (_Instrument != null);

            string instName = Enabled ? _Instrument.Name : string.Empty;
            groupBox_Inst.Text = instName;
            groupBox_Inst.Enabled = Enabled;

            {
                groupBox_Envelope.Text = instName;

                if (Enabled == true)
                {
                    if (_Instrument.IsInstListEnvelope == true)
                    {
                        groupBox_Envelope.Enabled = true;
                    }
                    else
                    {
                        if (_VelocityRegion != null)
                        {
                            groupBox_Envelope.Text = Path.GetFileName(_VelocityRegion.FilePath);
                            groupBox_Envelope.Enabled = true;
                        }
                        else
                        {
                            groupBox_Envelope.Enabled = false;
                        }
                    }
                }
                else
                {
                    groupBox_Envelope.Enabled = false;
                }
            }

            {
                string velocityRegionFileName = string.Empty;

                groupBox_Region.Enabled = (_VelocityRegion != null);
                if (groupBox_Region.Enabled)
                {
                    velocityRegionFileName = Path.GetFileName(_VelocityRegion.FilePath);
                }

                groupBox_Region.Text = velocityRegionFileName;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void UpdateControlsValue()
        {
            string instVolumeText = string.Empty;
            string instCoarseTuneText = string.Empty;
            string instFineTuneText = string.Empty;
            string attackText = string.Empty;
            string holdText = string.Empty;
            string decayText = string.Empty;
            string sustainText = string.Empty;
            string releaseText = string.Empty;
            string velPanText = string.Empty;
            string velVolumeText = string.Empty;
            string velCoarseTuneText = string.Empty;
            string velFineTuneText = string.Empty;
            int instVolume = 0;
            int instCoarseTune = 0;
            int instFineTune = 0;
            int attack = 0;
            int hold = 0;
            int decay = 0;
            int sustain = 0;
            int release = 0;
            int velPan = 0;
            int velVolume = 0;
            int velCoarseTune = 0;
            int velFineTune = 0;

            if (_Instrument != null)
            {
                instVolume = _Instrument.Volume;
                instCoarseTune = _Instrument.PitchSemitones;
                instFineTune = _Instrument.PitchCents;

                instVolumeText = instVolume.ToString();
                instCoarseTuneText = instCoarseTune.ToString();
                instFineTuneText = instFineTune.ToString();

                if (_Instrument.IsInstListEnvelope == true)
                {
                    attack = _Instrument.Envelope.Attack;
                    hold = _Instrument.Envelope.Hold;
                    decay = _Instrument.Envelope.Decay;
                    sustain = _Instrument.Envelope.Sustain;
                    release = _Instrument.Envelope.Release;

                    attackText = attack.ToString();
                    holdText = hold.ToString();
                    decayText = decay.ToString();
                    sustainText = sustain.ToString();
                    releaseText = release.ToString();
                }
                else if (_VelocityRegion != null)
                {
                    attack = _VelocityRegion.Envelope.Attack;
                    hold = _VelocityRegion.Envelope.Hold;
                    decay = _VelocityRegion.Envelope.Decay;
                    sustain = _VelocityRegion.Envelope.Sustain;
                    release = _VelocityRegion.Envelope.Release;

                    attackText = attack.ToString();
                    holdText = hold.ToString();
                    decayText = decay.ToString();
                    sustainText = sustain.ToString();
                    releaseText = release.ToString();
                }

                if (_VelocityRegion != null)
                {
                    velPan = _VelocityRegion.Pan;
                    velVolume = _VelocityRegion.Volume;
                    velCoarseTune = _VelocityRegion.PitchSemitones;
                    velFineTune = _VelocityRegion.PitchCents;

                    velPanText = velPan.ToString();
                    velVolumeText = velVolume.ToString();
                    velCoarseTuneText = velCoarseTune.ToString();
                    velFineTuneText = velFineTune.ToString();

                    textBox_PanLabel.Text = PanToText(velPan);
                }
                else
                {
                    textBox_PanLabel.Text = string.Empty;
                }
            }
            else
            {

                textBox_PanLabel.Text = string.Empty;
                attack = 0;
                hold = 0;
                decay = 0;
                sustain = 0;
                release = 0;
            }

            {
                SuspendEvent();

                textBox_I_Volume.SetText(instVolumeText);
                textBox_I_PitchSemitones.SetText(instCoarseTuneText);
                textBox_I_PitchCents.SetText(instFineTuneText);

                textBox_A.SetText(attackText);
                textBox_H.SetText(holdText);
                textBox_D.SetText(decayText);
                textBox_S.SetText(sustainText);
                textBox_R.SetText(releaseText);

                textBox_R_Pan.SetText(velPanText);
                textBox_R_Volume.SetText(velVolumeText);
                textBox_R_PitchSemitones.SetText(velCoarseTuneText);
                textBox_R_PitchCents.SetText(velFineTuneText);

                trackBar_I_Volume.Value = instVolume;
                trackBar_I_PitchSemitones.Value = instCoarseTune;
                trackBar_I_PitchCents.Value = instFineTune;

                trackBar_H.Value = hold;
                trackBar_A.Value = attack;
                trackBar_D.Value = decay;
                trackBar_S.Value = sustain;
                trackBar_R.Value = release;

                trackBar_R_Pan.Value = velPan;
                trackBar_R_Volume.Value = velVolume;
                trackBar_R_PitchSemitones.Value = velCoarseTune;
                trackBar_R_PitchCents.Value = velFineTune;

                if (adsrControl.Attack != attack ||
                    adsrControl.Hold != hold ||
                    adsrControl.Decay != decay ||
                    adsrControl.Sustain != sustain ||
                    adsrControl.Release != release)
                {
                    adsrControl.LockWindowUpdate(true);
                    adsrControl.Attack = attack;
                    adsrControl.Hold = hold;
                    adsrControl.Decay = decay;
                    adsrControl.Sustain = sustain;
                    adsrControl.Release = release;
                    adsrControl.LockWindowUpdate(false);
                }

                UpdateEnvelopeTime();

                ResumeEvent();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void UpdateTrackValue()
        {
            try
            {
                SuspendEvent();

                int instVolume = Convert.ToInt32(trackBar_I_Volume.Value);
                int instCoarseTune = Convert.ToInt32(trackBar_I_PitchSemitones.Value);
                int instFineTune = Convert.ToInt32(trackBar_I_PitchCents.Value);

                int attack = Convert.ToInt32(trackBar_A.Value);
                int hold = Convert.ToInt32(trackBar_H.Value);
                int decay = Convert.ToInt32(trackBar_D.Value);
                int sustain = Convert.ToInt32(trackBar_S.Value);
                int release = Convert.ToInt32(trackBar_R.Value);

                int velPan = Convert.ToInt32(trackBar_R_Pan.Value);
                int velVolume = Convert.ToInt32(trackBar_R_Volume.Value);
                int velCoarseTune = Convert.ToInt32(trackBar_R_PitchSemitones.Value);
                int velFineTune = Convert.ToInt32(trackBar_R_PitchCents.Value);

                adsrControl.LockWindowUpdate(true);
                adsrControl.Attack = attack;
                adsrControl.Hold = hold;
                adsrControl.Decay = decay;
                adsrControl.Sustain = sustain;
                adsrControl.Release = release;
                adsrControl.LockWindowUpdate(false);

                textBox_I_Volume.SetText(instVolume.ToString());
                textBox_I_PitchSemitones.SetText(instCoarseTune.ToString());
                textBox_I_PitchCents.SetText(instFineTune.ToString());

                textBox_A.SetText(attack.ToString());
                textBox_H.SetText(hold.ToString());
                textBox_D.SetText(decay.ToString());
                textBox_S.SetText(sustain.ToString());
                textBox_R.SetText(release.ToString());

                if (_VelocityRegion != null)
                {
                    textBox_R_Pan.SetText(velPan.ToString());
                    textBox_R_Volume.SetText(velVolume.ToString());
                    textBox_R_PitchSemitones.SetText(velCoarseTune.ToString());
                    textBox_R_PitchCents.SetText(velFineTune.ToString());
                    textBox_PanLabel.Text = PanToText(velPan);
                }
                else
                {
                    textBox_PanLabel.Text = String.Empty;
                }
            }
            finally
            {
                ResumeEvent();
            }

            UpdateEnvelopeTime();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void UpdateEnvelopeTime()
        {
            string attack = string.Empty;
            string hold = string.Empty;
            string decay = string.Empty;
            string release = string.Empty;

            if (_Instrument != null &&
                 (_Instrument.IsInstListEnvelope == true ||
                   _VelocityRegion != null))
            {
                attack = ((int)adsrControl.AttackTime).ToString();
                hold = ((int)adsrControl.HoldTime).ToString();
                decay = ((int)adsrControl.DecayTime).ToString();
                release = ((int)adsrControl.ReleaseTime).ToString();
            }

            textBox_AttackTime.Text = attack;
            textBox_HoldTime.Text = hold;
            textBox_DecayTime.Text = decay;
            textBox_ReleaseTime.Text = release;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private string PanToText(int pan)
        {
            if (pan == 64) { return "0"; }
            if (pan < 64) { return "L" + (64 - pan).ToString(); }
            return "R" + (pan - 64).ToString();
        }

        /// <summary>
        ///
        /// </summary>
        private void SetValueInstVolume(int value)
        {
            if (_Instrument != null && _Instrument.Volume != value)
            {
                SetValue(_Instrument, ProjectParameterNames.Volume, value);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void SetValueInstVolume(string text)
        {
            if (String.IsNullOrEmpty(text) != false)
            {
                return;
            }

            int value = int.Parse(text);
            SetValueInstVolume(value);
        }

        /// <summary>
        ///
        /// </summary>
        private void SetValueInstPitchSemitones(int value)
        {
            if (_Instrument != null && _Instrument.PitchSemitones != value)
            {
                SetValue(_Instrument, ProjectParameterNames.VelocityRegion.PitchSemitones, value);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void SetValueInstPitchSemitones(string text)
        {
            if (String.IsNullOrEmpty(text) != false)
            {
                return;
            }

            int value = int.Parse(text);
            SetValueInstPitchSemitones(value);
        }

        /// <summary>
        ///
        /// </summary>
        private void SetValueInstPitchCents(int value)
        {
            if (_Instrument != null && _Instrument.PitchCents != value)
            {
                SetValue(_Instrument, ProjectParameterNames.VelocityRegion.PitchCents, value);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void SetValueInstPitchCents(string text)
        {
            if (String.IsNullOrEmpty(text) != false)
            {
                return;
            }

            int value = int.Parse(text);
            SetValueInstPitchCents(value);
        }

        /// <summary>
        ///
        /// </summary>
        private void SetValueVelPan(int value)
        {
            if (_VelocityRegion != null && _VelocityRegion.Pan != value)
            {
                SetValue(_VelocityRegion, ProjectParameterNames.Pan, value);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void SetValueVelPan(string text)
        {
            if (String.IsNullOrEmpty(text) != false)
            {
                return;
            }

            int value = int.Parse(text);
            SetValueVelPan(value);
        }

        /// <summary>
        ///
        /// </summary>
        private void SetValueVelVolume(int value)
        {
            if (_VelocityRegion != null && _VelocityRegion.Volume != value)
            {
                SetValue(_VelocityRegion, ProjectParameterNames.Volume, value);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void SetValueVelVolume(string text)
        {
            if (String.IsNullOrEmpty(text) != false)
            {
                return;
            }

            int value = int.Parse(text);
            SetValueVelVolume(value);
        }

        /// <summary>
        ///
        /// </summary>
        private void SetValueVelPitchSemitones(int value)
        {
            if (_VelocityRegion != null && _VelocityRegion.PitchSemitones != value)
            {
                SetValue(
                         _VelocityRegion, ProjectParameterNames.VelocityRegion.PitchSemitones,
                         value
                         );
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void SetValueVelPitchSemitones(string text)
        {
            if (String.IsNullOrEmpty(text) != false)
            {
                return;
            }

            int value = int.Parse(text);
            SetValueVelPitchSemitones(value);
        }

        /// <summary>
        ///
        /// </summary>
        private void SetValueVelPitchCents(int value)
        {
            if (_VelocityRegion != null && _VelocityRegion.PitchCents != value)
            {
                SetValue(_VelocityRegion, ProjectParameterNames.VelocityRegion.PitchCents, value);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void SetValueVelPitchCents(string text)
        {
            if (String.IsNullOrEmpty(text) != false)
            {
                return;
            }

            int value = int.Parse(text);
            SetValueVelPitchCents(value);
        }

        /// <summary>
        ///
        /// </summary>
        private void OnInstValueChanged(object sender, ValueChangedEventArgs e)
        {
            if (_SuspendEvent == false)
            {
                NTrackBar trackBar = sender as NTrackBar;
                Debug.Assert(trackBar != null, "Sender is not NTrackBar.");
                if (trackBar.IsDragging == false)
                {
                    switch (trackBar.Name)
                    {
                        case "trackBar_I_Volume":
                            SetValueInstVolume(Convert.ToInt32(trackBar.Value));
                            break;

                        case "trackBar_I_PitchSemitones":
                            SetValueInstPitchSemitones(Convert.ToInt32(trackBar.Value));
                            break;

                        case "trackBar_I_PitchCents":
                            SetValueInstPitchCents(Convert.ToInt32(trackBar.Value));
                            break;

                        default:
                            Debug.Assert(false, "Unknown control name.");
                            break;
                    }
                }
                else
                {
                    UpdateTrackValue();
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnVelValueChanged(object sender, ValueChangedEventArgs e)
        {
            if (_SuspendEvent == false)
            {
                NTrackBar trackBar = sender as NTrackBar;
                Debug.Assert(trackBar != null, "Sender is not NTrackBar.");
                if (trackBar.IsDragging == false)
                {
                    switch (trackBar.Name)
                    {
                        case "trackBar_R_Pan":
                            SetValueVelPan(Convert.ToInt32(trackBar.Value));
                            break;

                        case "trackBar_R_Volume":
                            SetValueVelVolume(Convert.ToInt32(trackBar.Value));
                            break;

                        case "trackBar_R_PitchSemitones":
                            SetValueVelPitchSemitones(Convert.ToInt32(trackBar.Value));
                            break;

                        case "trackBar_R_PitchCents":
                            SetValueVelPitchCents(Convert.ToInt32(trackBar.Value));
                            break;

                        default:
                            Debug.Assert(false, "Unknown control name.");
                            break;
                    }
                }
                else
                {
                    UpdateTrackValue();
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnEnvelopeValueChanged(object sender, ValueChangedEventArgs e)
        {
            if (_SuspendEvent != false || _Instrument == null)
            {
                return;
            }

            NTrackBar trackBar = sender as NTrackBar;
            if (trackBar.IsDragging != false)
            {
                UpdateTrackValue();
                return;
            }

            int value = Convert.ToInt32(trackBar.Value);
            bool diff = false;
            string name = string.Empty;
            Envelope envelope;
            Component component = null;

            if (_Instrument.IsInstListEnvelope == true)
            {
                envelope = _Instrument.Envelope;
                component = _Instrument;
            }
            else if (_VelocityRegion == null)
            {
                return;
            }
            else
            {
                envelope = _VelocityRegion.Envelope;
                component = _VelocityRegion;
            }

            switch (trackBar.Name)
            {
                case "trackBar_A":
                    diff = envelope.Attack != value;
                    name = ProjectParameterNames.Envelope.Attack;
                    break;

                case "trackBar_H":
                    diff = envelope.Hold != value;
                    name = ProjectParameterNames.Envelope.Hold;
                    break;

                case "trackBar_D":
                    diff = envelope.Decay != value;
                    name = ProjectParameterNames.Envelope.Decay;
                    break;

                case "trackBar_S":
                    diff = envelope.Sustain != value;
                    name = ProjectParameterNames.Envelope.Sustain;
                    break;

                case "trackBar_R":
                    diff = envelope.Release != value;
                    name = ProjectParameterNames.Envelope.Release;
                    break;

                default:
                    Debug.Assert(false, "Unknown control name.");
                    break;
            }

            if (diff == true)
            {
                SetValue(component, name, value);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void SetEnvelopeTextValue(NNumericUpDown textBox)
        {
            if (_SuspendEvent != false || _Instrument == null)
            {
                return;
            }

            //NNumericUpDown textBox = sender as NNumericUpDown;
            int value = int.Parse(textBox.Text);
            bool diff = false;
            string name = string.Empty;
            Envelope envelope;
            Component component = null;

            if (_Instrument.IsInstListEnvelope == true)
            {
                envelope = _Instrument.Envelope;
                component = _Instrument;
            }
            else if (_VelocityRegion == null)
            {
                return;
            }
            else
            {
                envelope = _VelocityRegion.Envelope;
                component = _VelocityRegion;
            }

            switch (textBox.Name)
            {
                case "textBox_A":
                    diff = envelope.Attack != value;
                    name = ProjectParameterNames.Envelope.Attack;
                    break;

                case "textBox_H":
                    diff = envelope.Hold != value;
                    name = ProjectParameterNames.Envelope.Hold;
                    break;

                case "textBox_D":
                    diff = envelope.Decay != value;
                    name = ProjectParameterNames.Envelope.Decay;
                    break;

                case "textBox_S":
                    diff = envelope.Sustain != value;
                    name = ProjectParameterNames.Envelope.Sustain;
                    break;

                case "textBox_R":
                    diff = envelope.Release != value;
                    name = ProjectParameterNames.Envelope.Release;
                    break;

                default:
                    Debug.Assert(false, "Unknown control name.");
                    break;
            }

            if (diff == true)
            {
                SetValue(component, name, value);
            }
        }

        /// <summary>
        /// フォーカスが失われた時にエンベロープの数値入力を確定する
        /// </summary>
        private void OnLostFocusEnvelope(object sender, System.EventArgs e)
        {
            SetEnvelopeTextValue(sender as NNumericUpDown);
        }

        /// <summary>
        ///
        /// </summary>
        private void OnEnvelopeTextValueChanged(object sender, ValueChangedEventArgs e)
        {
            SetEnvelopeTextValue(sender as NNumericUpDown);
        }

        /// <summary>
        ///
        /// </summary>
        private void SetInstTextValue(NNumericUpDown textBox)
        {
            if (_SuspendEvent == false)
            {
                string text = textBox.Text;
                switch (textBox.Name)
                {
                    case "textBox_I_Volume":
                        SetValueInstVolume(text);
                        break;

                    case "textBox_I_PitchSemitones":
                        SetValueInstPitchSemitones(text);
                        break;

                    case "textBox_I_PitchCents":
                        SetValueInstPitchCents(text);
                        break;

                    default:
                        Debug.Assert(false, "Unknown control name.");
                        break;
                }
            }
        }

        /// <summary>
        /// フォーカスが失われた時にインストのボリュームの数値入力を確定する
        /// </summary>
        private void OnLostFocusInst(object sender, System.EventArgs e)
        {
            SetInstTextValue(sender as NNumericUpDown);
        }

        /// <summary>
        ///
        /// </summary>
        private void OnInstTextValueChanged(object sender, ValueChangedEventArgs e)
        {
            SetInstTextValue(sender as NNumericUpDown);
        }

        /// <summary>
        ///
        /// </summary>
        private void SetVelTextValue(NNumericUpDown textBox)
        {
            if (_SuspendEvent == false)
            {
                string text = textBox.Text;
                switch (textBox.Name)
                {
                    case "textBox_R_Pan":
                        SetValueVelPan(text);
                        break;

                    case "textBox_R_Volume":
                        SetValueVelVolume(text);
                        break;

                    case "textBox_R_PitchSemitones":
                        SetValueVelPitchSemitones(text);
                        break;

                    case "textBox_R_PitchCents":
                        SetValueVelPitchCents(text);
                        break;

                    default:
                        Debug.Assert(false, "Unknown control name.");
                        break;
                }
            }
        }

        /// <summary>
        /// フォーカスが失われた時に数値入力を確定する
        /// </summary>
        private void OnLostFocusVel(object sender, System.EventArgs e)
        {
            SetVelTextValue(sender as NNumericUpDown);
        }

        /// <summary>
        ///
        /// </summary>
        private void OnVelTextValueChanged(object sender, ValueChangedEventArgs e)
        {
            SetVelTextValue(sender as NNumericUpDown);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnParameterValueChanged(object sender, EventArgs e)
        {
            if (InvokeRequired == true)
            {
                Invoke((MethodInvoker)delegate ()
                {
                    UpdatePanel();
                    MainWindow.BuildCommandUI();
                });
            }
            else
            {
                UpdatePanel();
                MainWindow.BuildCommandUI();
            }
        }
    }
}
