﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using NintendoWare.SoundFoundation.FileFormats.Wave;

    public partial class WavePreviewControl : UserControl
    {
        public delegate bool ValidatePreviewFilePathDelegate(string filePath);

        private const string AliasName = "MediaFilePath";

        ///
        private static bool AutoPlay = true;

        private string filePath = null;
        private int soundLength = 0;

        private PlayStates playState = PlayStates.NotReady;
        private bool readySound = false;

        private Timer timer = null;
        private bool suspendValueChanged = false;

        private int pausedPosition = 0;

        private ValidatePreviewFilePathDelegate validatePreviewFilePathDelegate = null;

        /// <summary>
        ///
        /// </summary>
        public WavePreviewControl(ValidatePreviewFilePathDelegate validatePreviewFilePathDelegate)
        {
            InitializeComponent();

            this.timer = new Timer();
            this.timer.Interval = 100;
            this.timer.Tick += OnTick;

            this.chk_AutoPlay.Checked = AutoPlay;
            this.validatePreviewFilePathDelegate = validatePreviewFilePathDelegate;

            SetPlayState(PlayStates.NotReady);
            SetTimeUI(0);
        }

        /// <summary>
        ///
        /// </summary>
        public void DisposeAllSounds()
        {
            CloseSound();
        }

        /// <summary>
        ///
        /// </summary>
        public void SetFilePath(string filePath)
        {
            if (this.filePath != null &&
                (this.validatePreviewFilePathDelegate == null ||
                 this.validatePreviewFilePathDelegate(this.filePath) == true))
            {
                CloseSound();
            }

            this.filePath = null;
            if (Directory.Exists(filePath) != false)
            {
                return;
            }

            this.filePath = filePath;

            if (this.validatePreviewFilePathDelegate == null ||
                this.validatePreviewFilePathDelegate(filePath) == true)
            {
                OpenSound();
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                case MM.MCINOTIFY:
                    UInt16 wParam = (UInt16)m.WParam;
                    if (wParam == MCI_NOTIFY.SUCCESSFUL)
                    {
                        if (this.playState == PlayStates.Playing)
                        {
                            UpdateTime();
                            StopSound(false);
                        }
                    }
                    break;
            }

            base.WndProc(ref m);
        }

        ///
        [System.Runtime.InteropServices.DllImport("winmm.dll")]
        private static extern int mciSendString(String command, StringBuilder buffer, int bufferSize, IntPtr hwndCallback);
        [System.Runtime.InteropServices.DllImport("winmm.dll")]
        private static extern int mciSendCommand(int wDeviceID, int uMessage, int dwParam1, IntPtr dwParam2);

        /// <summary>
        ///
        /// </summary>
        private void OnTick(object sender, EventArgs e)
        {
            UpdateTime();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnClick_Play(object sender, EventArgs e)
        {
            PlaySound();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnClick_Pause(object sender, EventArgs e)
        {
            PauseSound();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnClick_Stop(object sender, EventArgs e)
        {
            StopSound();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnValueChanged(object sender, EventArgs e)
        {
            if (this.suspendValueChanged != false)
            {
                return;
            }

            PauseSound();

            int now = tkb_Time.Value;
            int max = tkb_Time.Maximum;
            int value = (int)(((float)now / (float)max) * this.soundLength);
            this.pausedPosition = value;

            string command = String.Format("seek {0} to {1}", AliasName, value);
            mciSendString(command, null, 0, IntPtr.Zero);

            UpdateTime();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnAutoPlayCheckedChanged(object sender, EventArgs e)
        {
            AutoPlay = this.chk_AutoPlay.Checked;
        }

        /// <summary>
        ///
        /// </summary>
        private void SetUI(int channelCount, int sampleRate, int sampleBit)
        {
            //float sampleRateText = (float)((float)sampleRate / 1000.0F);

            lbl_Channel.Text = String.Format("{0}", channelCount);
            lbl_SamplingRate.Text = String.Format("{0} Hz", sampleRate);
            lbl_BitPerSample.Text = String.Format("{0}", sampleBit);
            lbl_Time.Text = String.Empty;
        }

        /// <summary>
        ///
        /// </summary>
        private void SetTimeUI(int milliseconds)
        {
            int sec = milliseconds / 1000;
            int msec = milliseconds % 1000;
            int min = sec / 60;
            sec = sec % 60;
            lbl_Time.Text = String.Format("{0:00}:{1:00}.{2:000}", min, sec, msec);
        }

        /// <summary>
        ///
        /// </summary>
        private void UpdateTime()
        {
            StringBuilder sb = new StringBuilder(32);
            string command = String.Format("status {0} position", AliasName);
            mciSendString(command, sb, sb.Capacity, IntPtr.Zero);
            int sec = int.Parse(sb.ToString());

            int max = tkb_Time.Maximum;
            int value = (int)(((float)sec / (float)this.soundLength) * max);
            if (value > max)
            {
                value = max;
            }

            {
                this.suspendValueChanged = true;
                tkb_Time.Value = value;
                this.suspendValueChanged = false;
            }

            SetTimeUI(sec);
        }

        /// <summary>
        ///
        /// </summary>
        private void OpenSound()
        {
            if (this.filePath == null ||
                File.Exists(this.filePath) == false)
            {
                return;
            }

            WaveFile waveFile = null;
            try
            {
                WaveFileReader reader = WaveFileReader.CreateInstance(this.filePath);
                waveFile = reader.Open(this.filePath);
                reader.Close();
            }
            catch (Exception)
            {
                return;
            }

            string command = "open \"" + this.filePath + "\" type mpegvideo alias " + AliasName;
            if (mciSendString(command, null, 0, IntPtr.Zero) != 0)
            {
                return;
            }

            StringBuilder sb = new StringBuilder(32);
            command = String.Format("status {0} length", AliasName);
            if (mciSendString(command, sb, sb.Capacity, IntPtr.Zero) != 0)
            {
                return;
            }

            ///
            //WaveFileReader reader = WaveFileReader.CreateInstance( this.filePath);
            //WaveFile waveFile = reader.Open( this.filePath);
            //reader.Close();

            SetUI(waveFile.ChannelCount, waveFile.SampleRate, waveFile.SampleBit);

            this.soundLength = int.Parse(sb.ToString());

            this.readySound = true;
            SetPlayState(PlayStates.Stop);

            if (this.chk_AutoPlay.Checked != false)
            {
                PlaySound();
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void CloseSound()
        {
            StopSound();

            string command = String.Format("close {0}", AliasName);
            mciSendString(command, null, 0, IntPtr.Zero);

            this.readySound = false;
        }

        /// <summary>
        /// 再生
        /// </summary>
        private void PlaySound()
        {
            if (this.readySound == false)
            {
                return;
            }

            switch (this.playState)
            {
                case PlayStates.Stop:
                    {
                        string command = String.Format("seek {0} to start", AliasName);
                        mciSendString(command, null, 0, IntPtr.Zero);

                        command = String.Format("play {0} notify", AliasName);
                        mciSendString(command, null, 0, this.Handle);
                    }
                    break;

                case PlayStates.Paused:
                    {
                        string command = String.Format("seek {0} to {1}", AliasName, this.pausedPosition);
                        mciSendString(command, null, 0, IntPtr.Zero);

                        command = String.Format("play {0} notify", AliasName);
                        mciSendString(command, null, 0, this.Handle);
                    }
                    break;
            }

            SetPlayState(PlayStates.Playing);
            this.timer.Start();
        }

        /// <summary>
        /// 停止
        /// </summary>
        private void StopSound()
        {
            StopSound(true);
        }

        private void StopSound(bool seekToTop)
        {
            this.timer.Stop();

            string command = String.Format("stop {0}", AliasName);
            if (mciSendString(command, null, 0, IntPtr.Zero) != 0)
            {
                return;
            }

            if (seekToTop != false)
            {
                command = String.Format("seek {0} to start", AliasName);
                mciSendString(command, null, 0, IntPtr.Zero);
            }

            UpdateTime();
            SetPlayState(PlayStates.Stop);
        }

        /// <summary>
        ///
        /// </summary>
        private void PauseSound()
        {
            StringBuilder sb = new StringBuilder(32);
            string command = String.Format("status {0} position", AliasName);
            if (mciSendString(command, sb, sb.Capacity, IntPtr.Zero) != 0)
            {
                return;
            }

            command = String.Format("stop {0}", AliasName);
            if (mciSendString(command, null, 0, IntPtr.Zero) != 0)
            {
                return;
            }

            this.pausedPosition = int.Parse(sb.ToString());
            SetPlayState(PlayStates.Paused);
        }

        /// <summary>
        ///
        /// </summary>
        private void SetPlayState(PlayStates state)
        {
            this.playState = state;

            bool visiblePlay = true;
            bool enabledPlay = true;
            bool enabledStop = true;
            bool visiblePause = false;
            bool enabledPause = false;

            switch (this.playState)
            {
                case PlayStates.NotReady:
                    enabledPlay = false;
                    enabledPause = false;
                    enabledStop = false;
                    break;

                case PlayStates.Stop:
                case PlayStates.Paused:
                    visiblePlay = true;
                    enabledPlay = true;
                    visiblePause = false;
                    enabledPause = false;
                    enabledStop = true;
                    break;

                case PlayStates.Playing:
                    visiblePlay = false;
                    enabledPlay = false;
                    visiblePause = true;
                    enabledPause = true;
                    enabledStop = true;
                    break;
            }

            bool containsFocusedPlay = btn_Play.Focused;
            bool containsFocusedPause = btn_Pause.Focused;

            btn_Play.Visible = visiblePlay;
            btn_Play.Enabled = enabledPlay;
            btn_Pause.Visible = visiblePause;
            btn_Pause.Enabled = enabledPause;

            btn_Stop.Enabled = enabledStop;
            tkb_Time.Enabled = enabledStop;

            if (containsFocusedPlay != false)
            {
                if (enabledPlay == false)
                {
                    btn_Pause.Focus();
                }
            }

            if (containsFocusedPause != false)
            {
                if (enabledPause == false)
                {
                    btn_Play.Focus();
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private enum PlayStates
        {
            NotReady = 0,
            Stop = 1,
            Playing = 2,
            Paused = 3,
        }
    }

    /// <summary>
    ///
    /// </summary>
    internal class MM
    {
        public const UInt16 MCINOTIFY = 0x03B9;
    }

    /// <summary>
    ///
    /// </summary>
    internal class MCI_NOTIFY
    {
        public const UInt16 SUCCESSFUL = 1;
        public const UInt16 SUSPERSEDED = 2;
        public const UInt16 ABORTED = 4;
        public const UInt16 FAILURE = 8;
    }
}
