﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics.CodeAnalysis;
using System.Drawing;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Windows.Forms;
using NintendoWare.SoundFoundation.Core.Win32;

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    internal class ExtendableOpenFileDialog
    {
        /// コントロールのID番号(GetDlgItem)
        private const short ID_BTN_OPEN = 0x0001;
        private const short ID_BTN_CANCEL = 0x0002;
        private const short ID_CHK_READONLYOPEN = 0x0410;
        private const short ID_LBL_FILEFILTER = 0x0441;
        private const short ID_LBL_FILENAME = 0x0442;
        private const short ID_CMB_FILEFILTER = 0x0470;
        private const short ID_CMB_FILENAME = 0x047C;

        ///
        private const int MAX_PATH = 260;
        private const int MAX_PATH_UNI = (260 * 2);
        private const int MARGIN = 3;

        public delegate void SelectChangedHandler(string path);
        public event SelectChangedHandler SelectChanged;

        private IntPtr filePathPtr;
        private IntPtr fileTitlePtr;
        private IntPtr initialDirectoryPtr;

        private IntPtr titlePtr;
        private IntPtr extensionPtr;
        private IntPtr filterPtr;

        private OpenFileName ofn;
        private Control userControl;

        private IntPtr templatePtr;

        /// コントロールの位置情報保存用
        private bool sampled = false;
        private int contentMarginTop = 0;
        private int contentMarginBottom = 0;

        private int btnOpenMarginTop = 0;
        private int btnCancelMarginTop = 0;
        private int chkReadOnlyOpenMarginTop = 0;
        private int lblFileNameMarginTop = 0;
        private int lblFileFilterMarginTop = 0;
        private int cmbFileNameMarginTop = 0;
        private int cmbFileFilterMarginTop = 0;

        /// <summary>
        ///
        /// </summary>
        public ExtendableOpenFileDialog(string title, string extension, string fileName, string filter, string initialDirectory, Control userControl)
        {
            filePathPtr = Marshal.AllocCoTaskMem(MAX_PATH_UNI);
            fileTitlePtr = Marshal.AllocCoTaskMem(MAX_PATH_UNI);
            WIN32.ZeroMemory(this.filePathPtr, MAX_PATH_UNI);
            WIN32.ZeroMemory(this.fileTitlePtr, MAX_PATH_UNI);

            DlgTemplate template = new DlgTemplate();
            IntPtr ptr = Marshal.AllocCoTaskMem(Marshal.SizeOf(template));
            Marshal.StructureToPtr(template, ptr, true);
            this.templatePtr = ptr;

            this.initialDirectoryPtr = Marshal.StringToCoTaskMemUni(initialDirectory);

            this.titlePtr = Marshal.StringToCoTaskMemUni(title);
            this.extensionPtr = Marshal.StringToCoTaskMemUni(extension);
            this.filterPtr = Marshal.StringToCoTaskMemUni(ConvertFilter(filter));

            //
            ofn.lStructSize = Marshal.SizeOf(ofn);
            ofn.lpstrTitle = this.titlePtr;
            ofn.lpstrFile = this.filePathPtr;
            ofn.nMaxFile = MAX_PATH;
            ofn.lpstrDefExt = this.extensionPtr;
            ofn.lpstrFileTitle = this.fileTitlePtr;
            ofn.nMaxFileTitle = MAX_PATH;
            ofn.lpstrFilter = this.filterPtr;
            ofn.Flags = (OFN.ENABLEHOOK |
                                    OFN.ENABLETEMPLATEHANDLE |
                                    OFN.ENABLESIZING |
                                    OFN.EXPLORER);
            ofn.hInstance = this.templatePtr;
            ofn.lpfnHook = new WindowHookProc(WindowProc);
            ofn.lpstrInitialDir = this.initialDirectoryPtr;

            //
            byte[] buff = (new UnicodeEncoding()).GetBytes(fileName);
            Marshal.Copy(buff, 0, this.filePathPtr, buff.Length);

            //
            this.userControl = userControl;
        }

        /// <summary>
        ///
        /// </summary>
        ~ExtendableOpenFileDialog()
        {
            Dispose(false);
        }

        public Control Owner
        {
            get;
            set;
        }

        /// <summary>
        ///
        /// </summary>
        public bool Show()
        {
            //Debug.Assert( Onwer != null, "Owner is null");
            if (Owner != null)
            {
                ofn.hwndOwner = Owner.Handle;
            }

            return WIN32.GetOpenFileName(ref this.ofn);
        }

        /// <summary>
        /// ウインドウプロシージャ
        /// </summary>
        public IntPtr WindowProc(IntPtr hWnd, UInt16 msg, Int32 wParam, Int32 lParam)
        {
            if (hWnd == IntPtr.Zero)
            {
                return IntPtr.Zero;
            }

            //
            switch (msg)
            {
                default:
                    {
                        return IntPtr.Zero;
                    }

                case WM.WM_INITDIALOG:
                    WIN32.SetParent(this.userControl.Handle, WIN32.GetParent(hWnd));
                    return IntPtr.Zero;

                case WM.WM_SIZE:
                    ResizePanels(hWnd);
                    return IntPtr.Zero;

                case 0x03B9:
                    return IntPtr.Zero;

                case WM.WM_NOTIFY:
                    {
                        IntPtr ptr = new IntPtr(lParam);
                        NMHDR2 nmhdr = (NMHDR2)Marshal.PtrToStructure(ptr, typeof(NMHDR2));
                        UInt16 code = nmhdr.hdr.code;

                        switch (code)
                        {
                            case 0x03B9:
                                break;

                            case CommonDlgNotification.InitDone:
                                break;

                            case CommonDlgNotification.FolderChange:
                                // ダイアログが表示されるまでの間に、GetDlgItem(0x0461)がnullで
                                // なくなるタイミングがここしかないので、ここでResizePanels()を
                                // 一度だけ行います。
                                if (this.sampled == false)
                                {
                                    ResizePanels(hWnd);
                                }
                                break;

                            case CommonDlgNotification.SelChange:
                                ResizePanels(hWnd);

                                //
                                StringBuilder sb = new StringBuilder(MAX_PATH);
                                UInt32 ret = WIN32.SendMessage(WIN32.GetParent(hWnd),
                                                                DLGMSG.GETFILEPATH,
                                                                MAX_PATH, sb);
                                string filePath = sb.ToString();

                                //
                                byte[] buff = (new UnicodeEncoding()).GetBytes(filePath);
                                Marshal.Copy(buff, 0, this.filePathPtr, buff.Length);

                                //
                                if (SelectChanged != null)
                                {
                                    SelectChanged(filePath);
                                }
                                break;
                        }
                        return IntPtr.Zero;
                    }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public string FilePath
        {
            get
            {
                return Marshal.PtrToStringUni(filePathPtr);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void Dispose()
        {
            Dispose(true);
        }

        /// <summary>
        ///
        /// </summary>
        public void Dispose(bool disposing)
        {
            if (disposing)
            {
                GC.SuppressFinalize(this);
            }

            Marshal.FreeCoTaskMem(this.filePathPtr);
            Marshal.FreeCoTaskMem(this.fileTitlePtr);
            Marshal.FreeCoTaskMem(this.templatePtr);

            Marshal.FreeCoTaskMem(this.initialDirectoryPtr);

            Marshal.FreeCoTaskMem(this.titlePtr);
            Marshal.FreeCoTaskMem(this.extensionPtr);
            Marshal.FreeCoTaskMem(this.filterPtr);
        }

        /// <summary>
        ///
        /// </summary>
        private int GetMargin(IntPtr hWndParent, short id, int baseY)
        {
            RECT rect = new RECT();
            IntPtr dlgItem = WIN32.GetDlgItem(hWndParent, id);
            WIN32.GetWindowRect(dlgItem, ref rect);
            return rect.top - baseY;
        }

        /// <summary>
        ///
        /// </summary>
        private void MoveControl(IntPtr hWnd, short id, int y)
        {
            IntPtr hWndParent = WIN32.GetParent(hWnd);

            IntPtr hWndDlgItem = WIN32.GetDlgItem(hWndParent, id);
            RECT rect = new RECT();
            WIN32.GetWindowRect(hWndDlgItem, ref rect);

            POINT tl;
            tl.X = rect.left;
            tl.Y = rect.top;
            WIN32.ScreenToClient(hWnd, ref tl);

            POINT br;
            br.X = rect.right;
            br.Y = rect.bottom;
            WIN32.ScreenToClient(hWnd, ref br);

            WIN32.MoveWindow(hWndDlgItem,
                              tl.X,
                              //tl.Y + offsetY,
                              y,
                              br.X - tl.X,
                              br.Y - tl.Y,
                              true);
        }

        /// <summary>
        ///
        /// </summary>
        private void ResizePanels(IntPtr hWnd)
        {
            IntPtr hWndParent = WIN32.GetParent(hWnd);
            IntPtr hWndContent = WIN32.GetDlgItem(hWndParent, 0x0461);

            /// コントロールの位置情報を保存します。
            if (hWndContent != IntPtr.Zero &&
                this.sampled == false)
            {
                this.sampled = true;

                //
                RECT wndBounds = new RECT();
                WIN32.GetWindowRect(hWnd, ref wndBounds);

                RECT contentRect = new RECT();
                WIN32.GetWindowRect(hWndContent, ref contentRect);
                this.contentMarginTop = contentRect.top - wndBounds.top;
                this.contentMarginBottom = wndBounds.bottom - contentRect.bottom;

                //
                int baseY = contentRect.bottom;
                this.cmbFileFilterMarginTop = GetMargin(hWndParent, ID_CMB_FILEFILTER, baseY);
                this.cmbFileNameMarginTop = GetMargin(hWndParent, ID_CMB_FILENAME, baseY);
                this.btnOpenMarginTop = GetMargin(hWndParent, ID_BTN_OPEN, baseY);
                this.btnCancelMarginTop = GetMargin(hWndParent, ID_BTN_CANCEL, baseY);
                this.chkReadOnlyOpenMarginTop = GetMargin(hWndParent, ID_CHK_READONLYOPEN, baseY);
                this.lblFileNameMarginTop = GetMargin(hWndParent, ID_LBL_FILENAME, baseY);
                this.lblFileFilterMarginTop = GetMargin(hWndParent, ID_LBL_FILEFILTER, baseY);
            }

            ///
            Rectangle clientBounds = new Rectangle();
            Rectangle contentBounds = new Rectangle();

            //
            {
                RECT bounds = new RECT();
                WIN32.GetClientRect(hWndParent, ref bounds);
                clientBounds.X = bounds.left;
                clientBounds.Y = bounds.top;
                clientBounds.Width = bounds.right - bounds.left;
                clientBounds.Height = bounds.bottom - bounds.top;
            }

            //
            if (hWndContent != IntPtr.Zero)
            {
                // ファイルリストのコントロールの縦幅を決定します。
                RECT rect = new RECT();
                WIN32.GetWindowRect(hWnd, ref rect);
                int contentHeight = ((rect.bottom - rect.top) -
                                     (this.contentMarginTop + this.contentMarginBottom) -
                                     this.userControl.Height);

                //
                RECT bounds = new RECT();
                WIN32.GetWindowRect(hWndContent, ref bounds);

                POINT tl;
                tl.X = bounds.left;
                tl.Y = bounds.top;
                WIN32.ScreenToClient(hWndParent, ref tl);

                POINT br;
                br.X = bounds.right;
                br.Y = bounds.bottom;
                WIN32.ScreenToClient(hWndParent, ref br);

                contentBounds.X = tl.X;
                contentBounds.Y = tl.Y;
                contentBounds.Width = br.X - tl.X;
                //contentBounds.Height = br.Y - tl.Y;
                //contentBounds.Width = ( clientBounds.Width / 2 ) + MARGIN;
                contentBounds.Height = contentHeight;

                WIN32.MoveWindow(hWndContent,
                                  contentBounds.Left,
                                  contentBounds.Top,
                                  contentBounds.Width,
                                  contentBounds.Height,
                                  true);

                // コントロールの位置を変更します。
                int baseY = contentBounds.Top + contentBounds.Height;

                MoveControl(hWnd, ID_CMB_FILEFILTER, baseY + this.cmbFileFilterMarginTop);
                MoveControl(hWnd, ID_CMB_FILENAME, baseY + this.cmbFileNameMarginTop);

                MoveControl(hWnd, ID_LBL_FILEFILTER, baseY + this.lblFileFilterMarginTop);
                MoveControl(hWnd, ID_LBL_FILENAME, baseY + this.lblFileNameMarginTop);

                MoveControl(hWnd, ID_CHK_READONLYOPEN, baseY + this.chkReadOnlyOpenMarginTop);

                MoveControl(hWnd, ID_BTN_OPEN, baseY + this.btnOpenMarginTop);
                MoveControl(hWnd, ID_BTN_CANCEL, baseY + this.btnCancelMarginTop);
            }

            //
            {
#if false
                // 右側に付ける
                Rectangle bounds = new Rectangle
                    ( contentBounds.Right + (2 * MARGIN),
                      contentBounds.Top,
                      clientBounds.Right   - contentBounds.Right - (3 * MARGIN),
                      contentBounds.Bottom - contentBounds.Top );
#else
                // 下側に付ける
                Rectangle bounds = new Rectangle
                    (contentBounds.Left,
                      contentBounds.Bottom + this.contentMarginBottom,
                      this.userControl.Width,
                      this.userControl.Height);
#endif

                WIN32.MoveWindow(this.userControl.Handle,
                                  bounds.X,
                                  bounds.Y,
                                  bounds.Width,
                                  bounds.Height,
                                  true);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private string ConvertFilter(string filter)
        {
            StringBuilder sb = new StringBuilder();
            string[] values = filter.Split('|');
            foreach (string value in values)
            {
                sb.Append(String.Format("{0}\0", value));
            }
            sb.Append("\0\0");
            return sb.ToString();
        }
    }

    /// <summary>
    ///
    /// </summary>
    internal delegate IntPtr WindowHookProc(IntPtr hWnd, UInt16 msg, Int32 wParam, Int32 lParam);

    /// <summary>
    ///
    /// </summary>
    internal class OFN
    {
        public const Int32 ENABLEHOOK = 0x00000020;
        public const Int32 ENABLETEMPLATEHANDLE = 0x00000080;
        public const Int32 ENABLESIZING = 0x00800000;
        public const Int32 EXPLORER = 0x00080000;
    }

    /// <summary>
    ///
    /// </summary>
    internal class WS
    {
        public const Int32 CHILD = 0x40000000;
        public const Int32 CLIPSIBLINGS = 0x04000000;
        public const Int32 VISIBLE = 0x10000000;
        public const Int32 GROUP = 0x00020000;
    }

    /// <summary>
    ///
    /// </summary>
    internal class DS
    {
        public const Int32 SETFONT = 0x00000040;
        public const Int32 LOOK3D = 0x00000004;
        public const Int32 CONTROL = 0x00000400;
    }

    /// <summary>
    ///
    /// </summary>
    internal class SS
    {
        public const Int32 NOTIFY = 0x00000100;
    }

    /// <summary>
    ///
    /// </summary>
    internal class WSEX
    {
        public const Int32 NOPARENTNOTIFY = 0x00000004;
        public const Int32 CONTROLPARENT = 0x00010000;
    }

    /// <summary>
    ///
    /// </summary>
    internal class CommonDlgNotification
    {
        //private const UInt16 First = 0xFDA7;

        public const UInt16 InitDone = 0xFDA7;
        public const UInt16 SelChange = 0xFDA6;
        public const UInt16 FolderChange = 0xFDA5;
        public const UInt16 ShareViolation = 0xFDA4;
        public const UInt16 Help = 0xFDA3;
        public const UInt16 FileOk = 0xFDA2;
        public const UInt16 TypeChange = 0xFDA1;
        public const UInt16 IncludeItem = 0xFDA0;
    }

    /// <summary>
    ///
    /// </summary>
    internal class DLGMSG
    {
        public const UInt16 GETFILEPATH = 0x0465;
    }

    ///
    [SuppressMessage("Microsoft.StyleCop.CSharp.NamingRules",
                     "SA1307:AccessibleFieldsMustBeginWithUpperCaseLetter")]
    internal struct OpenFileName
    {
        public Int32 lStructSize;
        public IntPtr hwndOwner;
        public IntPtr hInstance;
        public IntPtr lpstrFilter;
        public IntPtr lpstrCustomFilter;
        public Int32 nMaxCustFilter;
        public Int32 nFilterIndex;
        public IntPtr lpstrFile;
        public Int32 nMaxFile;
        public IntPtr lpstrFileTitle;
        public Int32 nMaxFileTitle;
        public IntPtr lpstrInitialDir;
        public IntPtr lpstrTitle;
        public Int32 Flags;
        public Int16 nFileOffset;
        public Int16 nFileExtension;
        public IntPtr lpstrDefExt;
        public Int32 lCustData;
        public WindowHookProc lpfnHook;
        public IntPtr lpTemplateName;
        public IntPtr pvReserved;
        public Int32 dwReserved;
        public Int32 FlagsEx;
    }

    /// <summary>
    ///
    /// </summary>
    [SuppressMessage("Microsoft.StyleCop.CSharp.NamingRules",
                     "SA1307:AccessibleFieldsMustBeginWithUpperCaseLetter")]
    [StructLayout(LayoutKind.Sequential)]
    internal struct RECT
    {
        public int left;
        public int top;
        public int right;
        public int bottom;
    }

    /// <summary>
    ///
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    internal struct POINT
    {
        public int X;
        public int Y;
    }

    /// <summary>
    ///
    /// </summary>
    [SuppressMessage("Microsoft.StyleCop.CSharp.MaintainabilityRules",
                     "SA1401:FieldsMustBePrivate")]
    [SuppressMessage("Microsoft.StyleCop.CSharp.NamingRules",
                     "SA1307:AccessibleFieldsMustBeginWithUpperCaseLetter")]
    [StructLayout(LayoutKind.Sequential)]
    internal class DlgTemplate
    {
        public Int32 style = (DS.LOOK3D |
                               DS.CONTROL |
                               WS.CHILD |
                               WS.CLIPSIBLINGS |
                               SS.NOTIFY);
        public Int32 extendedStyle = WSEX.CONTROLPARENT;
        public Int16 numItems = 1;
        public Int16 x = 0;
        public Int16 y = 0;
        public Int16 cx = 0;
        public Int16 cy = 0;
        public Int16 reservedMenu = 0;
        public Int16 reservedClass = 0;
        public Int16 reservedTitle = 0;
        public Int32 itemStyle = WS.CHILD;
        public Int32 itemExtendedStyle = WSEX.NOPARENTNOTIFY;
        public Int16 itemX = 0;
        public Int16 itemY = 0;
        public Int16 itemCx = 0;
        public Int16 itemCy = 0;
        public Int16 itemId = 0;
        public UInt16 itemClassHdr = 0xffff;
        public Int16 itemClass = 0x0082;
        public Int16 itemText = 0x0000;
        public Int16 itemData = 0x0000;
    }

    /// <summary>
    ///
    /// </summary>
    [SuppressMessage("Microsoft.StyleCop.CSharp.NamingRules",
                     "SA1307:AccessibleFieldsMustBeginWithUpperCaseLetter")]
    [StructLayout(LayoutKind.Explicit)]
    internal struct NMHDR
    {
        [FieldOffset(0)]
        public IntPtr hWndFrom;
        [FieldOffset(4)]
        public UInt16 idFrom;
        [FieldOffset(8)]
        public UInt16 code;
    }

    /// <summary>
    ///
    /// </summary>
    [SuppressMessage("Microsoft.StyleCop.CSharp.NamingRules",
                     "SA1307:AccessibleFieldsMustBeginWithUpperCaseLetter")]
    [StructLayout(LayoutKind.Explicit)]
    internal struct NMHDR2
    {
        [FieldOffset(0)]
        public NMHDR hdr;
        [FieldOffset(12)]
        public IntPtr ipOfn;
        [FieldOffset(16)]
        public IntPtr ipFile;
    }

    /// <summary>
    ///
    /// </summary>
    internal class WIN32
    {
        [DllImport("User32.dll", CharSet = CharSet.Unicode)]
        internal static extern IntPtr GetDlgItem(IntPtr hWndDlg, Int16 Id);

        [DllImport("User32.dll", CharSet = CharSet.Unicode)]
        internal static extern IntPtr GetParent(IntPtr hWnd);

        [DllImport("User32.dll", CharSet = CharSet.Unicode)]
        internal static extern IntPtr SetParent(IntPtr hWndChild, IntPtr hWndNewParent);

        [DllImport("User32.dll", CharSet = CharSet.Unicode)]
        internal static extern UInt32 SendMessage(IntPtr hWnd, UInt32 msg, UInt32 wParam, StringBuilder buffer);

        [DllImport("user32.dll", CharSet = CharSet.Unicode)]
        internal static extern int GetWindowRect(IntPtr hWnd, ref RECT rc);

        [DllImport("user32.dll", CharSet = CharSet.Unicode)]
        internal static extern int GetClientRect(IntPtr hWnd, ref RECT rc);

        [DllImport("user32.dll", CharSet = CharSet.Unicode)]
        internal static extern bool ScreenToClient(IntPtr hWnd, ref POINT pt);

        [DllImport("user32.dll", CharSet = CharSet.Unicode)]
        internal static extern bool MoveWindow(IntPtr hWnd, int X, int Y, int Width, int Height, bool repaint);

        [DllImport("ComDlg32.dll", CharSet = CharSet.Unicode)]
        internal static extern bool GetOpenFileName(ref OpenFileName ofn);

        [DllImport("ComDlg32.dll", CharSet = CharSet.Unicode)]
        internal static extern Int32 CommDlgExtendedError();

        [DllImport("KERNEL32.DLL", EntryPoint = "RtlZeroMemory", SetLastError = true)]
        internal static extern bool ZeroMemory(IntPtr destinationObject, uint length);

        [System.Runtime.InteropServices.DllImport("winmm.dll")]
        internal static extern int mciSendString(String command, StringBuilder buffer, int bufferSize, IntPtr hwndCallback);
        [System.Runtime.InteropServices.DllImport("winmm.dll")]
        internal static extern int mciSendCommand(int wDeviceID, int uMessage, int dwParam1, IntPtr dwParam2);
    }
}
