﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.Data;
    using System.Diagnostics;
    using System.Drawing;
    using System.Linq;
    using System.Text;
    using System.Windows.Forms;

    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.Resources;

    public partial class UserParameterSettingPanel : UserControl
    {
        private UserParameterStructureSetting setting = null;
        private bool overlappedStructure = false;

        public event EventHandler ValueChanged;

        /// <summary>
        ///
        /// </summary>
        public static string StructureTypeToString(StructureTypes type)
        {
            switch (type)
            {
                case StructureTypes.Integer:
                    return MessageResource.Label_UserParameterTypeInteger;
                case StructureTypes.UInteger:
                    return MessageResource.Label_UserParameterTypeUInteger;
                case StructureTypes.Decimal:
                    return MessageResource.Label_UserParameterTypeDecimal;
                case StructureTypes.Boolean:
                    return MessageResource.Label_UserParameterTypeBoolean;
            }
            Debug.Assert(false, "Bad structure type");
            return String.Empty;
        }

        /// <summary>
        ///
        /// </summary>
        public UserParameterSettingPanel()
        {
            InitializeComponent();
        }

        /// <summary>
        ///
        /// </summary>
        public UserParameterStructureSetting Setting
        {
            get
            {
                return this.setting;
            }
            set
            {
                this.setting = value;

                UpdateListView();
                UpdateBitOccupancyControl();
                UpdateOverlappedStructure();
                UpdateControls();
            }
        }

        /// <summary>
        ///
        /// </summary>
        public bool OverlappedStructure
        {
            get
            {
                return this.overlappedStructure;
            }
        }

#if false
        /// <summary>
        ///
        /// </summary>
        public void HideEnabledCheckBox()
        {
            this.checkBox_Enabled.Visible = false;
            this.checkBox_Enabled.Checked = true;
        }
#endif

        /// <summary>
        ///
        /// </summary>
        private void OnValueChanged()
        {
            if (ValueChanged != null)
            {
                ValueChanged(this, EventArgs.Empty);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void UpdateListView()
        {
            // 選択状態を保持するために、現在の状態を覚えておきます。
            List<UserParameterStructure> selectedStructures = new List<UserParameterStructure>();

            foreach (ListViewItem item in this.listView.SelectedItems)
            {
                UserParameterStructure structure = item.Tag as UserParameterStructure;
                Debug.Assert(structure != null, "UserParameterStructure must be set.");

                selectedStructures.Add(structure);
            }

            listView.Items.Clear();
            if (this.setting != null)
            {

                foreach (UserParameterStructure structure in this.setting.Structures)
                {
                    ListViewItem item = new ListViewItem(structure.Label);
                    item.Tag = structure;

                    item.SubItems.Add(StructureTypeToString(structure.StructureType));

                    if (structure.Size == 1)
                    {
                        item.SubItems.Add(String.Format("{0}", structure.Location));
                    }
                    else
                    {
                        item.SubItems.Add(String.Format("{0} - {1}",
                                                          structure.Location + structure.Size - 1,
                                                          structure.Location));

                    }
                    item.SubItems.Add(String.Format("{0}",
                                                      structure.Size));

                    string candidate = String.Empty;
                    if (structure.ValueCandidate != null)
                    {
                        candidate = structure.ValueCandidate.Replace("\r\n", " ");
                    }
                    item.SubItems.Add(candidate);

                    listView.Items.Add(item);
                }
            }

            // 選択状態を戻します。
            // アイテム数の上限が32なので、線形検索でも遅くないはずです。
            foreach (UserParameterStructure structure in selectedStructures)
            {
                var targetItem = this.listView.Items.
                    OfType<ListViewItem>().
                    FirstOrDefault(item => item.Tag == structure);

                if (targetItem != null)
                {
                    targetItem.Selected = true;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void SortStructures()
        {
            UserParameterStructure[] structures = this.setting.Structures
                .OrderByDescending(s => s.Location)
                .ToArray();

            this.setting.Structures.Clear();
            foreach (UserParameterStructure structure in structures)
            {
                this.setting.Structures.Add(structure);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void UpdateControls()
        {
            ListView.SelectedListViewItemCollection items = this.listView.SelectedItems;

            if (items.Count > 0)
            {
                button_Edit.Enabled = items.Count == 1 ? true : false;
                button_Delete.Enabled = true;
            }
            else
            {
                button_Edit.Enabled = false;
                button_Delete.Enabled = false;
            }

            button_Append.Enabled = this.GetFirstUnusedBitLocation() >= 0;

            if (this.setting != null)
            {
                checkBox_Enabled.Checked = this.setting.Enabled;
            }
        }

        private void UpdateBitOccupancyControl()
        {
            this.bitOccupancyControl.UsedBits.Clear();
            this.bitOccupancyControl.ErrorBits.Clear();

            HashSet<int> usedBits = new HashSet<int>();

            foreach (ListViewItem item in this.listView.Items)
            {
                UserParameterStructure structure = item.Tag as UserParameterStructure;
                Debug.Assert(structure != null, "UserParameterStructure must be set.");

                this.bitOccupancyControl.UsedBits.Add(
                    new BitOccupancyControl.BitsInfo()
                    {
                        Location = structure.Location,
                        Size = structure.Size
                    });

                // ビットの重複チェック
                for (int bitLocation = structure.Location;
                    bitLocation < structure.Location + structure.Size;
                    ++bitLocation)
                {
                    if (usedBits.Contains(bitLocation))
                    {
                        this.bitOccupancyControl.ErrorBits.Add(bitLocation);
                        continue;
                    }

                    usedBits.Add(bitLocation);
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void UpdateHighlightedBits()
        {
            this.bitOccupancyControl.HighlightedBits.Clear();

            foreach (ListViewItem item in this.listView.SelectedItems)
            {
                UserParameterStructure structure = item.Tag as UserParameterStructure;
                Debug.Assert(structure != null, "UserParameterStructure must be set.");

                for (int bitLocation = structure.Location;
                    bitLocation < structure.Location + structure.Size;
                    ++bitLocation)
                {
                    this.bitOccupancyControl.HighlightedBits.Add(bitLocation);
                }
            }

            this.bitOccupancyControl.Invalidate();
        }

        /// <summary>
        ///
        /// </summary>
        private void UpdateOverlappedStructure()
        {
            this.overlappedStructure = ValidateOverlap();
        }

        /// <summary>
        ///
        /// </summary>
        private bool ValidateOverlap()
        {
            if (this.setting == null)
            {
                return false;
            }

            foreach (UserParameterStructure structure in this.setting.Structures)
            {
                if (OverlapStructure(structure) != false)
                {
                    return true;
                }
            }
            return false;
        }

        ///
        private bool OverlapStructure(UserParameterStructure targetStructure)
        {
            foreach (UserParameterStructure structure in this.setting.Structures)
            {
                if (structure == targetStructure)
                {
                    continue;
                }

                if ((structure.Location >= targetStructure.Location &&
                     structure.Location < targetStructure.Location + targetStructure.Size) ||
                   (targetStructure.Location >= structure.Location &&
                     targetStructure.Location < structure.Location + structure.Size))
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        ///
        /// </summary>
        private void EditItem()
        {
            if (this.listView.SelectedItems.Count != 1)
            {
                return;
            }

            ListViewItem item = this.listView.SelectedItems[0];
            UserParameterStructure structure = item.Tag as UserParameterStructure;

            UserParameterStructure duplicateStructure = structure.Clone() as UserParameterStructure;
            UserParameterEditingDialog dialog = new UserParameterEditingDialog();

            dialog.Structure = duplicateStructure;

            if (dialog.ShowDialog() != DialogResult.OK)
            {
                return;
            }

            structure.Label = dialog.Label;
            structure.StructureType = dialog.Type;
            structure.Location = dialog.BitLocation;
            structure.Size = dialog.BitSize;
            structure.ValueCandidate = dialog.Candidate;

            SortStructures();

            UpdateListView();
            UpdateBitOccupancyControl();
            UpdateHighlightedBits();
            UpdateOverlappedStructure();
            UpdateControls();
            OnValueChanged();
        }

        /// <summary>
        ///
        /// </summary>
        private void DeleteItem()
        {
            if (this.listView.SelectedItems.Count == 0)
            {
                return;
            }

            List<ListViewItem> deleteList = new List<ListViewItem>();

            foreach (ListViewItem item in this.listView.SelectedItems)
            {
                deleteList.Add(item);
            }

            int newSelectedIndex = this.listView.SelectedIndices[0];

            foreach (ListViewItem item in deleteList)
            {
                UserParameterStructure structure = item.Tag as UserParameterStructure;
                this.setting.Structures.Remove(structure);

                listView.Items.Remove(item);
            }

            SortStructures();

            UpdateListView();
            UpdateBitOccupancyControl();
            UpdateOverlappedStructure();
            UpdateControls();
            OnValueChanged();

            if (0 < newSelectedIndex && newSelectedIndex < this.listView.Items.Count)
            {
                this.listView.SelectedIndices.Add(newSelectedIndex);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private int GetFirstUnusedBitLocation()
        {
            if (this.setting == null)
            {
                return -1;
            }

            int bitLocation = 31;

            foreach (UserParameterStructure structure in this.setting.Structures)
            {
                int bitStart = structure.Location + structure.Size - 1;

                if (bitLocation > bitStart)
                {
                    return bitLocation;
                }

                bitLocation = structure.Location - 1;
            }

            return bitLocation < 0 ? -1 : bitLocation;
        }

        /// <summary>
        ///
        /// </summary>
        private void OnAppend(object sender, EventArgs e)
        {
            UserParameterStructure structure = new UserParameterStructure();
            UserParameterEditingDialog dialog = new UserParameterEditingDialog();

            int unusedBitLocation = this.GetFirstUnusedBitLocation();

            if (unusedBitLocation < 0)
            {
                Debug.Assert(false, "must not be append.");
                return;
            }

            structure.Location = unusedBitLocation;

            dialog.Structure = structure;

            if (dialog.ShowDialog() != DialogResult.OK)
            {
                return;
            }

            structure.Label = dialog.Label;
            structure.StructureType = dialog.Type;
            structure.Location = dialog.BitLocation;
            structure.Size = dialog.BitSize;
            structure.ValueCandidate = dialog.Candidate;

            this.setting.Structures.Add(structure);
            SortStructures();

            UpdateListView();
            UpdateBitOccupancyControl();
            UpdateOverlappedStructure();
            UpdateControls();
            OnValueChanged();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnEdit(object sender, EventArgs e)
        {
            this.EditItem();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnDelete(object sender, EventArgs e)
        {
            this.DeleteItem();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnItemSelectionChanged(object sender, ListViewItemSelectionChangedEventArgs e)
        {
            UpdateControls();
            UpdateHighlightedBits();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnEnabledChachedChanged(object sender, EventArgs e)
        {
            this.setting.Enabled = this.checkBox_Enabled.Checked;
        }

        /// <summary>
        ///
        /// </summary>
        private void OnMouseDoubleClick(object sender, MouseEventArgs e)
        {
            ListViewHitTestInfo result = this.listView.HitTest(e.Location);

            if (result.Item != null)
            {
                this.EditItem();
            }
        }

        private void OnKeyDown(object sender, KeyEventArgs e)
        {
            switch (e.KeyCode)
            {
                case Keys.Enter:
                    this.EditItem();
                    break;

                case Keys.Delete:
                    this.DeleteItem();
                    break;
            }
        }
    }
}
