﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.Data;
    using System.Diagnostics;
    using System.Drawing;
    using System.Linq;
    using System.Text;
    using System.Windows.Forms;

    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.Resources;

    public partial class UserParameterEditingDialog : Form
    {
        private const int MaxBitSize = 32;

        private UserParameterStructure structure = null;
        private bool isUpdateControlsLocked = false;

        /// <summary>
        ///
        /// </summary>
        public UserParameterEditingDialog()
        {
            InitializeComponent();
            this.errorProvider.SetIconAlignment(this.textBox_Candidate, ErrorIconAlignment.TopRight);
        }

        /// <summary>
        ///
        /// </summary>
        public UserParameterStructure Structure
        {
            get
            {
                return this.structure;
            }
            set
            {
                if (this.structure == value)
                {
                    return;
                }

                this.structure = value;

                this.isUpdateControlsLocked = true;

                try
                {
                    if (this.structure != null)
                    {
                        textBox_Label.Text = this.structure.Label;
                        comboBox_Type.SelectedIndex = StructureTypeToIndex(this.structure.StructureType);
                        upDown_BitStart.Value = this.structure.Location + this.structure.Size - 1;
                        upDown_BitEnd.Value = this.structure.Location;
                        upDown_BitSize.Value = this.structure.Size;
                        textBox_Candidate.Text = this.structure.ValueCandidate;
                    }
                }
                finally
                {
                    this.isUpdateControlsLocked = false;
                    UpdateControls();
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public string Label
        {
            get
            {
                return this.textBox_Label.Text;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public StructureTypes Type
        {
            get
            {
                return IndexToStructureType(this.comboBox_Type.SelectedIndex);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public int BitLocation
        {
            get
            {
                return Convert.ToInt32(this.upDown_BitEnd.Value);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public int BitSize
        {
            get
            {
                return Convert.ToInt32(this.upDown_BitSize.Value);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public string Candidate
        {
            get
            {
                return this.textBox_Candidate.Text;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void UpdateControls()
        {
            if (this.isUpdateControlsLocked)
            {
                return;
            }

            if (this.comboBox_Type.SelectedIndex < 0)
            {
                return;
            }

            bool oldIsUpdateControlsLocked = this.isUpdateControlsLocked;
            this.isUpdateControlsLocked = true;

            try
            {
                switch (this.Type)
                {
                    case StructureTypes.Integer:
                    case StructureTypes.UInteger:
                        upDown_BitStart.Enabled = true;
                        upDown_BitStart.Minimum = upDown_BitSize.Value - 1;
                        upDown_BitStart.Maximum = MaxBitSize - 1;
                        upDown_BitEnd.Enabled = true;
                        upDown_BitEnd.Minimum = 0;
                        upDown_BitEnd.Maximum = upDown_BitStart.Value;
                        upDown_BitSize.Enabled = true;
                        upDown_BitSize.Minimum = 1;
                        upDown_BitSize.Maximum = upDown_BitStart.Value + 1;
                        textBox_Candidate.Enabled = true;
                        break;

                    case StructureTypes.Decimal:
                        upDown_BitStart.Enabled = false;
                        upDown_BitStart.Minimum = 31;
                        upDown_BitStart.Maximum = 31;
                        upDown_BitStart.Value = 31;
                        upDown_BitEnd.Enabled = false;
                        upDown_BitEnd.Minimum = 0;
                        upDown_BitEnd.Maximum = 0;
                        upDown_BitEnd.Value = 0;
                        upDown_BitSize.Enabled = false;
                        upDown_BitSize.Minimum = 32;
                        upDown_BitSize.Maximum = 32;
                        upDown_BitSize.Value = 32;
                        textBox_Candidate.Enabled = true;
                        break;

                    case StructureTypes.Boolean:
                        upDown_BitStart.Enabled = true;
                        upDown_BitStart.Minimum = 0;
                        upDown_BitStart.Maximum = MaxBitSize - 1;
                        upDown_BitEnd.Enabled = false;
                        upDown_BitEnd.Minimum = 0;
                        upDown_BitEnd.Maximum = MaxBitSize - 1;
                        upDown_BitEnd.Value = upDown_BitStart.Value;
                        upDown_BitSize.Enabled = false;
                        upDown_BitSize.Minimum = 1;
                        upDown_BitSize.Maximum = 1;
                        upDown_BitSize.Value = 1;
                        textBox_Candidate.Enabled = false;
                        break;
                }

                if (ValidateValue() == true)
                {
                    button_OK.Enabled = true;
                }
                else
                {
                    button_OK.Enabled = false;
                }
            }
            finally
            {
                this.isUpdateControlsLocked = oldIsUpdateControlsLocked;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private bool ValidateValue()
        {
            int location = Convert.ToInt32(upDown_BitStart.Value);
            int size = Convert.ToInt32(upDown_BitSize.Value);
            bool error = false;

            {
                int min = 0;
                int max = 31;
                if (location < min || location > max)
                {
                    SetError(upDown_BitStart,
                              String.Format(MessageResource.Message_ErrorProvider_OutsideValueRange,
                                             min, max));
                    error = true;
                }
                else
                {
                    ResetError(upDown_BitStart);
                }
            }

            {
                int min = 0;
                int max = 31;
                if (location < min || location > max)
                {
                    SetError(upDown_BitEnd,
                              String.Format(MessageResource.Message_ErrorProvider_OutsideValueRange,
                                             min, max));
                    error = true;
                }
                else
                {
                    ResetError(upDown_BitStart);
                }
            }

            {
                int min = 1;
                int max = 32;
                if (size < min || size > max)
                {
                    SetError(upDown_BitSize,
                              String.Format(MessageResource.Message_ErrorProvider_OutsideValueRange,
                                             min, max));
                    error = true;
                }
                else
                {
                    ResetError(upDown_BitSize);
                }
            }

            //
            if (textBox_Label.Text == String.Empty)
            {
                SetError(textBox_Label, MessageResource.Message_ErrorProvider_EmptyName);
                error = true;
            }
            else
            {
                ResetError(textBox_Label);
            }

            string errorMessage;
            if (this.ValidateCandidateValue(out errorMessage) == false)
            {
                SetError(this.textBox_Candidate, errorMessage);
                error = true;
            }
            else
            {
                ResetError(this.textBox_Candidate);
            }

            return error == true ? false : true;
        }

        /// <summary>
        ///
        /// </summary>
        private bool ValidateCandidateValue(out string errorMessage)
        {
            errorMessage = string.Empty;

            if (string.IsNullOrEmpty(this.Candidate) == true || this.Type == StructureTypes.Boolean)
            {
                return true;
            }

            List<object> values = new List<object>();
            string[] lines = this.Candidate.Split(new Char[] { '\n' });
            foreach (string line in lines)
            {
                object value;

                // 空白行はスキップします。
                if (string.IsNullOrEmpty(line) == true)
                {
                    continue;
                }

                // '=' 付きの名前はエラーにします。
                if (line.Count(c => c == '=') != 1)
                {
                    errorMessage = MessageResource.Message_ErrorProvider_BadFormat;
                    return false;
                }

                // 単純な解析します。例："poi = 123" -> {"poi", "123"}
                string[] tokens = line.Split(new Char[] { '=', ' ' }, StringSplitOptions.RemoveEmptyEntries);
                if (tokens.Length != 2)
                {
                    errorMessage = MessageResource.Message_ErrorProvider_BadFormat;
                    return false;
                }

                // 文字数字を解析します。
                if (UserParameterUtility.TryParseCandidate(this.Type, tokens[1], out value) == false)
                {
                    // 数値化できない場合はエラーを返す。
                    errorMessage = MessageResource.Message_ErrorProvider_BadFormat;
                    return false;
                }

                if (values.Contains(value) == true)
                {
                    // 数値化できた値が他の定義と同じ値の場合はエラー返す。
                    errorMessage = MessageResource.Message_ErrorProvider_SameValue;
                    return false;
                }
                else
                {
                    values.Add(value);
                }
            }

            return true;
        }

        /// <summary>
        ///
        /// </summary>
        private int StructureTypeToIndex(StructureTypes type)
        {
            switch (type)
            {
                case StructureTypes.Integer:
                    return 0;
                case StructureTypes.UInteger:
                    return 1;
                case StructureTypes.Decimal:
                    return 2;
                case StructureTypes.Boolean:
                    return 3;
            }
            Debug.Assert(false, "Bad structure type");
            return -1;
        }

        /// <summary>
        ///
        /// </summary>
        private void SetError(Control control, string message)
        {
            if (null == control) { throw new ArgumentNullException("control"); }
            if (null == message) { throw new ArgumentNullException("message"); }

            this.errorProvider.SetError(control, message);
        }

        /// <summary>
        ///
        /// </summary>
        private void ResetError(Control control)
        {
            if (null == control) { throw new ArgumentNullException("control"); }

            this.errorProvider.SetError(control, null);
        }

        /// <summary>
        ///
        /// </summary>
        private StructureTypes IndexToStructureType(int index)
        {
            switch (index)
            {
                case 0:
                    return StructureTypes.Integer;
                case 1:
                    return StructureTypes.UInteger;
                case 2:
                    return StructureTypes.Decimal;
                case 3:
                    return StructureTypes.Boolean;
            }
            Debug.Assert(false, "Bad structure index");
            return StructureTypes.Integer;
        }

        /// <summary>
        ///
        /// </summary>
        private void OnSelectedIndexChanged(object sender, EventArgs e)
        {
            UpdateControls();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnLabelTextChanged(object sender, EventArgs e)
        {
            UpdateControls();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnBitStartValueChanged(object sender, EventArgs e)
        {
            if (this.isUpdateControlsLocked)
            {
                return;
            }

            bool oldIsUpdateControlsLocked = this.isUpdateControlsLocked;
            this.isUpdateControlsLocked = true;

            try
            {
                // Start - End と Size を同期
                decimal newEnd = this.upDown_BitStart.Value - this.upDown_BitSize.Value + 1;

                this.upDown_BitEnd.Maximum = newEnd;
                this.upDown_BitEnd.Value = newEnd;
            }
            finally
            {
                this.isUpdateControlsLocked = oldIsUpdateControlsLocked;
                UpdateControls();
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnBitEndValueChanged(object sender, EventArgs e)
        {
            if (this.isUpdateControlsLocked)
            {
                return;
            }

            bool oldIsUpdateControlsLocked = this.isUpdateControlsLocked;
            this.isUpdateControlsLocked = true;

            try
            {
                // Start - End と Size を同期
                this.upDown_BitSize.Maximum = upDown_BitStart.Value + 1;
                this.upDown_BitSize.Value = this.upDown_BitStart.Value - this.upDown_BitEnd.Value + 1;
            }
            finally
            {
                this.isUpdateControlsLocked = oldIsUpdateControlsLocked;
                UpdateControls();
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnBitSizeValueChanged(object sender, EventArgs e)
        {
            if (this.isUpdateControlsLocked)
            {
                return;
            }

            bool oldIsUpdateControlsLocked = this.isUpdateControlsLocked;
            this.isUpdateControlsLocked = true;

            try
            {
                // Start - End と Size を同期
                this.upDown_BitEnd.Value = this.upDown_BitStart.Value - this.upDown_BitSize.Value + 1;
            }
            finally
            {
                this.isUpdateControlsLocked = oldIsUpdateControlsLocked;
                UpdateControls();
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnCandidateTextChanged(object sender, EventArgs e)
        {
            UpdateControls();
        }
    }
}
