﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------


namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.Diagnostics;
    using System.Drawing;
    using System.IO;
    using System.Linq;
    using System.Reflection;
    using System.Text;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.Configurations;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Framework.Windows.Forms.CommandHandlers;
    using ToolDevelopmentKit;
    using NwSoundResources = NintendoWare.SoundFoundation.Resources;

    /// <summary>
    /// ProjectSettingsDialog の概要の説明です。
    /// </summary>
    public partial class ProjectSettingsDialog : Form
    {
        public delegate void ProjectSettingsDialogInitialize(ProjectSettingsDialog dialog);
        public static ProjectSettingsDialogInitialize InitializeInternal { get; set; }

        private OutputListInfo mCurrentSoundListOutput;

        private Dictionary<string, string> listTexts = new Dictionary<string, string>();
        private Dictionary<string, string> listColumnTexts = new Dictionary<string, string>();

        ///
        private ErrorProvider ErrorProvider;

        private List<SoundListOutput> soundListOutputs;
        private SoundListOutput currentSoundListOutput = null;

        const int SOUND_LIST_COUNT = 3;

        public string ProjectFilePath;
        private bool _SuspendUpdateSampleItemName = true;

        private int saveSelectedHTMLSoundListListBoxIndex = -1;
        private int saveHTMLSoundListListBoxItemsCount = 0;
        private bool isCancel = false;

        private UserParameterStructureSettings settings = null;
        private UserParameterStructureSetting[] duplicatedSettings = null;

        //
        private const string ColumnSoundSetName = "SoundSetName";
        private const string ColumnOutputType = "OutputType";

        private readonly Dictionary<SoundSet, SoundArchiveOutputTypes> soundArchiveOutputType = new Dictionary<SoundSet, SoundArchiveOutputTypes>();
        private ListViewHitTestInfo listViewHitTestInfo = null;
        private SoundSet editingSoundSet = null;
        private bool _readOnly = false;

        //
        class OutputListInfo
        {
            public string _text;
            public ListOutput ItemInfo;
            public List<OutputItemInfo> ListInfoArray;

            public OutputListInfo(string text)
            {
                if (null == text) { throw new ArgumentNullException("text"); }
                _text = text.Replace("\n", "");
            }
            public override string ToString()
            {
                return _text;
            }
        }

        class OutputItemInfo
        {
            public string _text;
            public ListColumnOutput ItemInfo;

            public OutputItemInfo(string text)
            {
                if (null == text) { throw new ArgumentNullException("text"); }
                _text = text.Replace("\n", "");
            }
            public override string ToString()
            {
                return _text;
            }
        }

        /// <summary>
        /// 非公開機能を表示します。
        /// </summary>
        public static bool ShowHiddenFunction
        {
            get; set;
        }

        /// <summary>
        ///
        /// </summary>
        static ProjectSettingsDialog()
        {
            //ShowPCBinariesAAC = false;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ProjectSettingsDialog()
        {
            InitializeComponent();
            InitializeStringDictionaries();

            if (ShowHiddenFunction == false)
            {
                this.HiddenFunctionTabPage.Visible = false;
                this.HiddenFunctionTabPage.Enabled = false;
                this.ProjectSettingTab.Controls.Remove(this.HiddenFunctionTabPage);
            }

            //
            this.StreamPathSummary.Text = string.Format(
                this.StreamPathSummary.Text,
                FormsApplication.Instance.Traits.BinaryOutputTraits.StreamSoundBinaryFileExtension,
                FormsApplication.Instance.Traits.BinaryOutputTraits.SoundArchiveBinaryFileExtension
                );

            ErrorProvider = new ErrorProvider(this);

            this.soundListOutputs = new List<SoundListOutput>();
            UpdateSoundListOutputSettingList();
            UpdateListMoveButton();
            UpdateItemListMoveButton();

            //
            //pnl_Color0.BackColor = CommonDrawParameter.RowColors[0];
            pnl_Color1.BackColor = CommonDrawParameter.RowColors[1];
            pnl_Color2.BackColor = CommonDrawParameter.RowColors[2];
            pnl_Color3.BackColor = CommonDrawParameter.RowColors[3];
            pnl_Color4.BackColor = CommonDrawParameter.RowColors[4];
            pnl_Color5.BackColor = CommonDrawParameter.RowColors[5];
            pnl_Color6.BackColor = CommonDrawParameter.RowColors[6];
            pnl_Color7.BackColor = CommonDrawParameter.RowColors[7];
            pnl_Color8.BackColor = CommonDrawParameter.RowColors[8];

            this.Initialize();

            this.listView.MouseDown += OnMouseDown_ListView;
            this.listView.MouseUp += OnMouseUp_ListView;
            this.listView.SmallImageList = new ImageList() { ImageSize = new Size(1, 22) };

            this.UpdateConnectionTimeOutEnabled();
        }

        public void HideAAC()
        {
            this.chk_WarnNotFoundPCBinariesAAC.Enabled = false;
            this.chk_WarnNotFoundPCBinariesAAC.Visible = false;
        }

        public void HideSndEditTab()
        {
            this.SndEditTabPage.Visible = false;
            this.SndEditTabPage.Enabled = false;
            this.ProjectSettingTab.Controls.Remove(this.SndEditTabPage);
        }

        public void HideSndEditChannelTab()
        {
            this.SndEditChannelTabPage.Visible = false;
            this.SndEditChannelTabPage.Enabled = false;
            this.ProjectSettingTab.Controls.Remove(this.SndEditChannelTabPage);
        }

        public void HideStreamBufferTimes()
        {
            this.comboBox_SoundArchiveStreamBufferTimes.Visible = false;
            this.comboBox_SoundArchiveStreamBufferTimes.Enabled = false;

            this.label_SoundArchiveStreamBufferTimes.Visible = false;
            this.label_SoundArchiveStreamBufferSize.Visible = false;
            this.label_SoundArchiveStreamBufferSizeValue.Visible = false;

            this.label_SoundArchiveStreamBufferConversionADPCM.Visible = false;
            this.label_SoundArchiveStreamBufferConversionPCM16.Visible = false;
            this.label_SoundArchiveStreamBufferConversionADPCMValue.Visible = false;
            this.label_SoundArchiveStreamBufferConversionPCM16Value.Visible = false;
            this.label_SoundArchiveStreamBufferConversionADPCMSec.Visible = false;
            this.label_SoundArchiveStreamBufferConversionPCM16Sec.Visible = false;
        }

        /// <summary>
        ///
        /// </summary>
        private void InitializeStringDictionaries()
        {
            this.listTexts.Add("Sounds", MessageResource.Label_HtmlSoundListDescription_SoundList);
            this.listTexts.Add("Groups", MessageResource.Label_HtmlSoundListDescription_GroupList);
            this.listTexts.Add("Players", MessageResource.Label_HtmlSoundListDescription_PlayerList);
            this.listTexts.Add("SoundSetBanks", MessageResource.Label_HtmlSoundListDescription_BankList);

            this.listColumnTexts.Add(ListTraits.ColumnName_ItemIndex, MessageResource.HeaderText_ItemIndex);
            this.listColumnTexts.Add(ListTraits.ColumnName_Name, MessageResource.HeaderText_Name);
            this.listColumnTexts.Add(ListTraits.ColumnName_ItemType, MessageResource.HeaderText_ItemType);
            this.listColumnTexts.Add(ListTraits.ColumnName_Volume, MessageResource.HeaderText_Volume);
            this.listColumnTexts.Add(ListTraits.ColumnName_PlayerReference, MessageResource.HeaderText_Player);
            this.listColumnTexts.Add(ListTraits.ColumnName_ActorPlayer, MessageResource.HeaderText_ActorPlayer);
            this.listColumnTexts.Add(ListTraits.ColumnName_PlayerPriority, MessageResource.HeaderText_PlayerPriority.Replace("\r\n", string.Empty));
            this.listColumnTexts.Add(ListTraits.ColumnName_PlayerSoundLimit, MessageResource.HeaderText_PlayerSoundLimit);
            this.listColumnTexts.Add(ListTraits.ColumnName_Comment, MessageResource.HeaderText_Comment);
            this.listColumnTexts.Add(ListTraits.ColumnName_Comment1, MessageResource.HeaderText_Comment1);
            this.listColumnTexts.Add(ListTraits.ColumnName_Comment2, MessageResource.HeaderText_Comment2);
            this.listColumnTexts.Add(ListTraits.ColumnName_Comment3, MessageResource.HeaderText_Comment3);
            this.listColumnTexts.Add(ListTraits.ColumnName_Comment4, MessageResource.HeaderText_Comment4);
            this.listColumnTexts.Add(ListTraits.ColumnName_Comment5, MessageResource.HeaderText_Comment5);
            this.listColumnTexts.Add(ListTraits.ColumnName_Comment6, MessageResource.HeaderText_Comment6);
            this.listColumnTexts.Add(ListTraits.ColumnName_Comment7, MessageResource.HeaderText_Comment7);
            this.listColumnTexts.Add(ListTraits.ColumnName_Comment8, MessageResource.HeaderText_Comment8);
            this.listColumnTexts.Add(ListTraits.ColumnName_Comment9, MessageResource.HeaderText_Comment9);
            this.listColumnTexts.Add(ListTraits.ColumnName_UserParameter0, MessageResource.HeaderText_UserParameter);
            this.listColumnTexts.Add(ListTraits.ColumnName_UserParameter1, MessageResource.HeaderText_UserParameter1);
            this.listColumnTexts.Add(ListTraits.ColumnName_UserParameter2, MessageResource.HeaderText_UserParameter2);
            this.listColumnTexts.Add(ListTraits.ColumnName_UserParameter3, MessageResource.HeaderText_UserParameter3);
        }

        /// <summary>
        ///
        /// </summary>
        private void Initialize()
        {
            if (InitializeInternal != null)
            {
                InitializeInternal(this);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public List<SoundListOutput> SoundListOutputSettings
        {
            get { return this.soundListOutputs; }
            set
            {
                this.soundListOutputs = new List<SoundListOutput>();
                foreach (SoundListOutput soundListOutput in value)
                {
                    this.soundListOutputs.Add(soundListOutput.Clone());
                }
                UpdateSoundListOutputSettingList();
            }
        }

        ///--------------------------------
        /// <summary>
        /// ダイアログを読み取り専用にします。
        /// </summary>
        public bool ReadOnly
        {
            get
            {
                return _readOnly;
            }
            set
            {
                _readOnly = value;

                this.Text = _readOnly == true ? MessageResource.ProjectSettingsDialogTitleReadOnly : MessageResource.ProjectSettingsDialogTitle;

                bool enabled = _readOnly == true ? false : true;

                this.OK.Enabled = enabled && CanDecide;
                this.ProjectSettingTab.TabPages.Cast<TabPage>().ForEach(p => p.Enabled = enabled);
                this.mSoundListOutputSettingTabControl.TabPages.Cast<TabPage>().ForEach(p => p.Enabled = enabled);
                this.mSoundListOutputSettingCreateButton.Enabled = enabled;
                this.mOutputSoundListDeleteButton.Enabled = enabled;
                this.tabControl.TabPages.Cast<TabPage>().ForEach(p => p.Enabled = enabled);
                this.flowLayoutPanel1.Controls.Cast<Control>().ForEach(c => c.Enabled = enabled);

                // 「HTML サウンドリスト」タブページは有効にします。
                // （「基本設定」「出力項目」タブの切り替えのため）
                this.HTMLSoundListTabPage.Enabled = true;

                // 「ユーザーパラメータ」タブページは有効にします。
                // （「ユーザーパラメータ０〜３」タブの切り替えのため）
                this.UserParameterTabPage.Enabled = true;

                // 「ユーザーコマンド」タブページは有効にします。
                // （タブページ内のスクロールバーを有効にするため）
                this.UserCommandTabPage.Enabled = true;

                this.UpdateControls();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string StreamName
        {
            get { return textBox_Stream.Text; }
            set { textBox_Stream.Text = value; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string WaveSoundSetName
        {
            get { return textBox_WaveSoundSet.Text; }
            set { textBox_WaveSoundSet.Text = value; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string WaveSoundCoName
        {
            get { return textBox_WaveSoundCo.Text; }
            set { textBox_WaveSoundCo.Text = value; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string SeqSetName
        {
            get { return textBox_SeqSet.Text; }
            set { textBox_SeqSet.Text = value; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string SeqName
        {
            get { return textBox_Seq.Text; }
            set { textBox_Seq.Text = value; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string BankName
        {
            get { return textBox_Bank.Text; }
            set { textBox_Bank.Text = value; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string PlayerName
        {
            get { return textBox_Player.Text; }
            set { textBox_Player.Text = value; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string GroupName
        {
            get { return textBox_Group.Text; }
            set { textBox_Group.Text = value; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string InstName
        {
            get { return textBox_Inst.Text; }
            set { textBox_Inst.Text = value; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string WarcName
        {
            get { return textBox_Warc.Text; }
            set { textBox_Warc.Text = value; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool EnablePrefix
        {
            get { return checkBox_Prefix.Checked; }
            set { checkBox_Prefix.Checked = value; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public CaseChangeType UpperLowerLetterProcess
        {
            get
            {
                switch (comboBox_UpperLowerProcess.SelectedIndex)
                {
                    case 0:
                        return CaseChangeType.ToUpper;
                    case 1:
                        return CaseChangeType.ToLower;
                    case 2:
                        return CaseChangeType.None;
                }

                Debug.Assert(false);
                return CaseChangeType.None;
            }

            set
            {
                int index = -1;

                switch (value)
                {
                    case CaseChangeType.ToUpper:
                        index = 0;
                        break;
                    case CaseChangeType.ToLower:
                        index = 1;
                        break;
                    case CaseChangeType.None:
                        index = 2;
                        break;
                    default:
                        Debug.Assert(false);
                        break;
                }
                comboBox_UpperLowerProcess.SelectedIndex = index;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public InvalidCharChangeType IllegalLetterProcess
        {
            get
            {
                switch (comboBox_IllegalLetterProcess.SelectedIndex)
                {
                    case 0:
                        return InvalidCharChangeType.ReplaceToUnderscore;
                    case 1:
                        return InvalidCharChangeType.RemoveChar;
                }

                Debug.Assert(false);
                return InvalidCharChangeType.ReplaceToUnderscore;
            }

            set
            {
                int index = -1;

                switch (value)
                {
                    case InvalidCharChangeType.ReplaceToUnderscore:
                        index = 0;
                        break;
                    case InvalidCharChangeType.RemoveChar:
                        index = 1;
                        break;
                    default:
                        Debug.Assert(false);
                        break;
                }
                comboBox_IllegalLetterProcess.SelectedIndex = index;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public string ItemPastePostfix
        {
            get { return this.tbx_ItemPastePostfix.Text; }
            set { this.tbx_ItemPastePostfix.Text = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public bool EnabledNameDelimiter
        {
            get { return this.chk_NameDelimiter.Checked; }
            set { this.chk_NameDelimiter.Checked = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public string NameDelimiter
        {
            get { return this.tbx_NameDelimiter.Text; }
            set { this.tbx_NameDelimiter.Text = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public string Comment
        {
            get { return CommentTextBox.Text; }
            set { CommentTextBox.Text = value; }
        }

        public string Comment1
        {
            get { return Comment1TextBox.Text; }
            set { Comment1TextBox.Text = value; }
        }

        public string Comment2

        {
            get { return Comment2TextBox.Text; }
            set { Comment2TextBox.Text = value; }
        }

        public string Comment3
        {
            get { return Comment3TextBox.Text; }
            set { Comment3TextBox.Text = value; }
        }

        public string Comment4
        {
            get { return Comment4TextBox.Text; }
            set { Comment4TextBox.Text = value; }
        }

        public string Comment5
        {
            get { return Comment5TextBox.Text; }
            set { Comment5TextBox.Text = value; }
        }

        public string Comment6
        {
            get { return Comment6TextBox.Text; }
            set { Comment6TextBox.Text = value; }
        }

        public string Comment7
        {
            get { return Comment7TextBox.Text; }
            set { Comment7TextBox.Text = value; }
        }

        public string Comment8
        {
            get { return Comment8TextBox.Text; }
            set { Comment8TextBox.Text = value; }
        }

        public string Comment9
        {
            get { return Comment9TextBox.Text; }
            set { Comment9TextBox.Text = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public string ColorComment0
        {
            get { return tbx_Color0.Text; }
            set { tbx_Color0.Text = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public string ColorComment1
        {
            get { return tbx_Color1.Text; }
            set { tbx_Color1.Text = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public string ColorComment2
        {
            get { return tbx_Color2.Text; }
            set { tbx_Color2.Text = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public string ColorComment3
        {
            get { return tbx_Color3.Text; }
            set { tbx_Color3.Text = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public string ColorComment4
        {
            get { return tbx_Color4.Text; }
            set { tbx_Color4.Text = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public string ColorComment5
        {
            get { return tbx_Color5.Text; }
            set { tbx_Color5.Text = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public string ColorComment6
        {
            get { return tbx_Color6.Text; }
            set { tbx_Color6.Text = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public string ColorComment7
        {
            get { return tbx_Color7.Text; }
            set { tbx_Color7.Text = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public string ColorComment8
        {
            get { return tbx_Color8.Text; }
            set { tbx_Color8.Text = value; }
        }

        /// <summary>
        /// ユーザーコマンド1の名前
        /// </summary>
        public string UserCommandName1
        {
            get { return this.userCommandPanel1.CommandName; }
            set { this.userCommandPanel1.CommandName = value; }
        }

        /// <summary>
        /// ユーザーコマンド2の名前
        /// </summary>
        public string UserCommandName2
        {
            get { return this.userCommandPanel2.CommandName; }
            set { this.userCommandPanel2.CommandName = value; }
        }

        /// <summary>
        /// ユーザーコマンド3の名前
        /// </summary>
        public string UserCommandName3
        {
            get { return this.userCommandPanel3.CommandName; }
            set { this.userCommandPanel3.CommandName = value; }
        }

        /// <summary>
        /// ユーザーコマンド4の名前
        /// </summary>
        public string UserCommandName4
        {
            get { return this.userCommandPanel4.CommandName; }
            set { this.userCommandPanel4.CommandName = value; }
        }

        /// <summary>
        /// ユーザーコマンド5の名前
        /// </summary>
        public string UserCommandName5
        {
            get { return this.userCommandPanel5.CommandName; }
            set { this.userCommandPanel5.CommandName = value; }
        }

        /// <summary>
        /// ユーザーコマンド6の名前
        /// </summary>
        public string UserCommandName6
        {
            get { return this.userCommandPanel6.CommandName; }
            set { this.userCommandPanel6.CommandName = value; }
        }

        /// <summary>
        /// ユーザーコマンド7の名前
        /// </summary>
        public string UserCommandName7
        {
            get { return this.userCommandPanel7.CommandName; }
            set { this.userCommandPanel7.CommandName = value; }
        }

        /// <summary>
        /// ユーザーコマンド8の名前
        /// </summary>
        public string UserCommandName8
        {
            get { return this.userCommandPanel8.CommandName; }
            set { this.userCommandPanel8.CommandName = value; }
        }



        /// <summary>
        /// ユーザーコマンド1
        /// </summary>
        public string UserCommand1
        {
            get { return this.userCommandPanel1.Command; }
            set { this.userCommandPanel1.Command = value; }
        }

        /// <summary>
        /// ユーザーコマンド2
        /// </summary>
        public string UserCommand2
        {
            get { return this.userCommandPanel2.Command; }
            set { this.userCommandPanel2.Command = value; }
        }

        /// <summary>
        /// ユーザーコマンド3
        /// </summary>
        public string UserCommand3
        {
            get { return this.userCommandPanel3.Command; }
            set { this.userCommandPanel3.Command = value; }
        }

        /// <summary>
        /// ユーザーコマンド4
        /// </summary>
        public string UserCommand4
        {
            get { return this.userCommandPanel4.Command; }
            set { this.userCommandPanel4.Command = value; }
        }

        /// <summary>
        /// ユーザーコマンド5
        /// </summary>
        public string UserCommand5
        {
            get { return this.userCommandPanel5.Command; }
            set { this.userCommandPanel5.Command = value; }
        }

        /// <summary>
        /// ユーザーコマンド6
        /// </summary>
        public string UserCommand6
        {
            get { return this.userCommandPanel6.Command; }
            set { this.userCommandPanel6.Command = value; }
        }

        /// <summary>
        /// ユーザーコマンド7
        /// </summary>
        public string UserCommand7
        {
            get { return this.userCommandPanel7.Command; }
            set { this.userCommandPanel7.Command = value; }
        }

        /// <summary>
        /// ユーザーコマンド8
        /// </summary>
        public string UserCommand8
        {
            get { return this.userCommandPanel8.Command; }
            set { this.userCommandPanel8.Command = value; }
        }

        /// <summary>
        /// ユーザーコマンド、アイコンファイルパス1
        /// </summary>
        public string UserCommandIconFilePath1
        {
            get { return this.userCommandPanel1.IconFilePath; }
            set { this.userCommandPanel1.IconFilePath = value; }
        }

        /// <summary>
        /// ユーザーコマンド、アイコンファイルパス2
        /// </summary>
        public string UserCommandIconFilePath2
        {
            get { return this.userCommandPanel2.IconFilePath; }
            set { this.userCommandPanel2.IconFilePath = value; }
        }

        /// <summary>
        /// ユーザーコマンド、アイコンファイルパス3
        /// </summary>
        public string UserCommandIconFilePath3
        {
            get { return this.userCommandPanel3.IconFilePath; }
            set { this.userCommandPanel3.IconFilePath = value; }
        }

        /// <summary>
        /// ユーザーコマンド、アイコンファイルパス4
        /// </summary>
        public string UserCommandIconFilePath4
        {
            get { return this.userCommandPanel4.IconFilePath; }
            set { this.userCommandPanel4.IconFilePath = value; }
        }

        /// <summary>
        /// ユーザーコマンド、アイコンファイルパス5
        /// </summary>
        public string UserCommandIconFilePath5
        {
            get { return this.userCommandPanel5.IconFilePath; }
            set { this.userCommandPanel5.IconFilePath = value; }
        }

        /// <summary>
        /// ユーザーコマンド、アイコンファイルパス6
        /// </summary>
        public string UserCommandIconFilePath6
        {
            get { return this.userCommandPanel6.IconFilePath; }
            set { this.userCommandPanel6.IconFilePath = value; }
        }

        /// <summary>
        /// ユーザーコマンド、アイコンファイルパス7
        /// </summary>
        public string UserCommandIconFilePath7
        {
            get { return this.userCommandPanel7.IconFilePath; }
            set { this.userCommandPanel7.IconFilePath = value; }
        }

        /// <summary>
        /// ユーザーコマンド、アイコンファイルパス8
        /// </summary>
        public string UserCommandIconFilePath8
        {
            get { return this.userCommandPanel8.IconFilePath; }
            set { this.userCommandPanel8.IconFilePath = value; }
        }

        /// <summary>
        /// プロジェクトのコメント
        /// </summary>
        public string ProjectComment
        {
            get { return tbx_ProjectComment.Text; }
            set { tbx_ProjectComment.Text = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int SyncPort
        {
            get { return (int)numericUpDown_SndEditSyncPort.Value; }
            set { numericUpDown_SndEditSyncPort.Value = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int FuncPort
        {
            get { return (int)numericUpDown_SndEditFuncPort.Value; }
            set { numericUpDown_SndEditFuncPort.Value = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int CtrlPort
        {
            get { return (int)numericUpDown_SndEditCtrlPort.Value; }
            set { numericUpDown_SndEditCtrlPort.Value = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int SyncChannel
        {
            get { return (int)SndEditChannelSync_NumericUpDown.Value; }
            set { SndEditChannelSync_NumericUpDown.Value = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int FuncChannel
        {
            get { return (int)SndEditChannelFunc_NumericUpDown.Value; }
            set { SndEditChannelFunc_NumericUpDown.Value = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public int CtrlChannel
        {
            get { return (int)SndEditChannelCtrl_NumericUpDown.Value; }
            set { SndEditChannelCtrl_NumericUpDown.Value = value; }
        }

        /// <summary>
        /// 接続、切断の設定の有効無効
        /// </summary>
        public bool EnabledConnectionTimeOut
        {
            get { return EnabledConnectionTimeOut_CheckBox.Checked; }
            set { EnabledConnectionTimeOut_CheckBox.Checked = value; }
        }

        /// <summary>
        /// 接続中に反応がなくなった場合、切断するまでのタイムアウト（秒）です。
        /// </summary>
        public int ConnectionTimeOut
        {
            get { return (int)ConnectionTimeOut_NumericUpDown.Value; }
            set { ConnectionTimeOut_NumericUpDown.Value = value; }
        }

        /// <summary>
        /// 不要な出力ファイルやディレクトリをコンバート時に削除する設定
        /// </summary>
        public bool DoDeleteGarbageOutputBinaries
        {
            get { return DoDeleteGarbageOutputBinaries_CheckBox.Checked; }
            set { DoDeleteGarbageOutputBinaries_CheckBox.Checked = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public void LoadSettings(SoundProjectService projectService)
        {
            if (null == projectService) { throw new ArgumentNullException("projectService"); }

            ProjectFilePath = projectService.ProjectDocument.Resource.Key;

            DoWarnUnreferencedItems = projectService.Project.DoWarnUnreferencedItems;
            DoWarnDisableGroupItemTargets = projectService.Project.DoWarnDisableGroupItemTargets;
            ExcludeStringTable = projectService.Project.ExcludeStringTable;
            IntermediateOutputPath = projectService.Project.IntermediateOutputPath;
            InGameEditCacheOutputPath = projectService.Project.InGameEditCacheOutputPath;
            OutputLabel = projectService.Project.OutputLabel;
            OutputPath = projectService.Project.OutputDirectoryPath;
            StreamPath = projectService.Project.ExternalFileOutputDirectoryPath;
            DoWarnPCBinariesForAACNotFound = projectService.Project.DoWarnPCBinariesForAACNotFound;
            DoDeleteGarbageOutputBinaries = projectService.Project.DoDeleteGarbageOutputBinaries;
            DoUseHardwareOpusDecoder = projectService.Project.DoUseHardwareOpusDecoder;

            OutputUserManagementDirectory = projectService.Project.UserManagementFileOutputDirectoryPath;

            SequenceSound = projectService.Project.SoundArchivePlayerSequenceSoundCount;
            SequenceTrack = projectService.Project.SoundArchivePlayerSequenceTrackCount;
            StreamSound = projectService.Project.SoundArchivePlayerStreamSoundCount;
            StreamChannel = projectService.Project.SoundArchivePlayerStreamChannelCount;
            StreamBufferTimes = projectService.Project.SoundArchivePlayerStreamBufferTimes;
            WaveSound = projectService.Project.SoundArchivePlayerWaveSoundCount;
            PreConvertCommandEnabled = projectService.Project.IsPreConvertCommandsEnabled;
            PostConvertCommandEnabled = projectService.Project.IsPostConvertCommandsEnabled;
            PreConvertCommands = this.StringListToString(projectService.Project.PreConvertCommands);
            PostConvertCommands = this.StringListToString(projectService.Project.PostConvertCommands);
            FileSavePreCommandEnabled = projectService.Project.IsFileSavePreCommandEnabled;
            FileSavePostCommandEnabled = projectService.Project.IsFileSavePostCommandEnabled;
            FileSavePreCommandPath = projectService.Project.FileSavePreCommandPath;
            FileSavePostCommandPath = projectService.Project.FileSavePostCommandPath;

            DismantleSMF = projectService.Project.SmfConvertTimeBase;
            DoUseBuiltInWavePreprocessExe = projectService.Project.DoUseBuiltInWavePreprocessExe;
            WavePreprocessExePath = projectService.Project.WavePreprocessExePath;

            Debug.Assert(null != SoundListOutputSettings, "unexpected error");
            SoundListOutputSettings.Clear();
            foreach (SoundListOutput output in projectService.Project.SoundListOutputSettings.ListOutputs)
            {
                SoundListOutputSettings.Add(output.Clone());
            }

            UpdateSoundListOutputSettingList();

            StreamName = projectService.Project.StreamSoundNamePrefix;
            WaveSoundSetName = projectService.Project.WaveSoundSetNamePrefix;
            WaveSoundCoName = projectService.Project.WaveSoundNamePrefix;
            SeqSetName = projectService.Project.SequenceSoundSetNamePrefix;
            SeqName = projectService.Project.SequenceSoundNamePrefix;
            BankName = projectService.Project.BankNamePrefix;
            PlayerName = projectService.Project.PlayerNamePrefix;
            GroupName = projectService.Project.GroupNamePrefix;
            InstName = projectService.Project.InstrumentNamePrefix;
            WarcName = projectService.Project.WaveArchiveNamePrefix;

            EnablePrefix = projectService.Project.ItemNamingHasPrefix;
            UpperLowerLetterProcess = projectService.Project.ItemNamingCaseChange;
            IllegalLetterProcess = projectService.Project.ItemNamingInvalidCharChange;
            ItemPastePostfix = projectService.Project.ItemPastePostfix;
            EnabledNameDelimiter = projectService.Project.EnabledNameDelimiter;
            NameDelimiter = projectService.Project.NameDelimiter;

            //
            Comment = projectService.Project.CommentColumnText;
            Comment1 = projectService.Project.Comment1ColumnText;
            Comment2 = projectService.Project.Comment2ColumnText;
            Comment3 = projectService.Project.Comment3ColumnText;
            Comment4 = projectService.Project.Comment4ColumnText;
            Comment5 = projectService.Project.Comment5ColumnText;
            Comment6 = projectService.Project.Comment6ColumnText;
            Comment7 = projectService.Project.Comment7ColumnText;
            Comment8 = projectService.Project.Comment8ColumnText;
            Comment9 = projectService.Project.Comment9ColumnText;

            //
            ColorComment0 = projectService.Project.ColorComment0;
            ColorComment1 = projectService.Project.ColorComment1;
            ColorComment2 = projectService.Project.ColorComment2;
            ColorComment3 = projectService.Project.ColorComment3;
            ColorComment4 = projectService.Project.ColorComment4;
            ColorComment5 = projectService.Project.ColorComment5;
            ColorComment6 = projectService.Project.ColorComment6;
            ColorComment7 = projectService.Project.ColorComment7;
            ColorComment8 = projectService.Project.ColorComment8;

            //
            UserCommandName1 = projectService.Project.UserCommands[0].Name;
            UserCommandName2 = projectService.Project.UserCommands[1].Name;
            UserCommandName3 = projectService.Project.UserCommands[2].Name;
            UserCommandName4 = projectService.Project.UserCommands[3].Name;
            UserCommandName5 = projectService.Project.UserCommands[4].Name;
            UserCommandName6 = projectService.Project.UserCommands[5].Name;
            UserCommandName7 = projectService.Project.UserCommands[6].Name;
            UserCommandName8 = projectService.Project.UserCommands[7].Name;

            UserCommand1 = projectService.Project.UserCommands[0].Command;
            UserCommand2 = projectService.Project.UserCommands[1].Command;
            UserCommand3 = projectService.Project.UserCommands[2].Command;
            UserCommand4 = projectService.Project.UserCommands[3].Command;
            UserCommand5 = projectService.Project.UserCommands[4].Command;
            UserCommand6 = projectService.Project.UserCommands[5].Command;
            UserCommand7 = projectService.Project.UserCommands[6].Command;
            UserCommand8 = projectService.Project.UserCommands[7].Command;

            UserCommandIconFilePath1 = projectService.Project.UserCommands[0].IconFilePath;
            UserCommandIconFilePath2 = projectService.Project.UserCommands[1].IconFilePath;
            UserCommandIconFilePath3 = projectService.Project.UserCommands[2].IconFilePath;
            UserCommandIconFilePath4 = projectService.Project.UserCommands[3].IconFilePath;
            UserCommandIconFilePath5 = projectService.Project.UserCommands[4].IconFilePath;
            UserCommandIconFilePath6 = projectService.Project.UserCommands[5].IconFilePath;
            UserCommandIconFilePath7 = projectService.Project.UserCommands[6].IconFilePath;
            UserCommandIconFilePath8 = projectService.Project.UserCommands[7].IconFilePath;

            //
            ProjectComment = projectService.Project.ProjectComment;

            //
            this.settings = projectService.Project.UserDataStructureSettings;

            UserParameterSettingPanel[] panels = new UserParameterSettingPanel[]
                {
                    this.userParameterSettingPanel0,
                    this.userParameterSettingPanel1,
                    this.userParameterSettingPanel2,
                    this.userParameterSettingPanel3,
                };

            duplicatedSettings = new UserParameterStructureSetting[panels.Length];
            for (int index = 0; index < panels.Length; index++)
            {
                duplicatedSettings[index] = this.settings.Settings[index].Clone()
                    as UserParameterStructureSetting;
                panels[index].Setting = duplicatedSettings[index];
                panels[index].ValueChanged += delegate (object sender, EventArgs e)
                    {
                        UpdateControls();
                    };
            }

            SyncPort = projectService.Project.SyncPort;
            FuncPort = projectService.Project.FuncPort;
            CtrlPort = projectService.Project.CtrlPort;

            SyncChannel = projectService.Project.SyncChannel;
            FuncChannel = projectService.Project.FuncChannel;
            CtrlChannel = projectService.Project.CtrlChannel;

            EnabledConnectionTimeOut = projectService.Project.EnabledConnectionTimeOut;
            ConnectionTimeOut = projectService.Project.ConnectionTimeOut;

            //
            string projectDirectory = Path.GetDirectoryName(ProjectFilePath);
            this.userCommandPanel1.CardinalFilePath = projectDirectory;
            this.userCommandPanel2.CardinalFilePath = projectDirectory;
            this.userCommandPanel3.CardinalFilePath = projectDirectory;
            this.userCommandPanel4.CardinalFilePath = projectDirectory;
            this.userCommandPanel5.CardinalFilePath = projectDirectory;
            this.userCommandPanel6.CardinalFilePath = projectDirectory;
            this.userCommandPanel7.CardinalFilePath = projectDirectory;
            this.userCommandPanel8.CardinalFilePath = projectDirectory;

            //
            this.soundArchiveOutputType.Clear();
            foreach (var soundSet in projectService.SoundSets)
            {
                SoundArchiveOutputTypes outputType;
                if (projectService.Project.SoundArchiveOutputTypes.TryGetValue(soundSet, out outputType) == false)
                {
                    outputType = SoundArchiveOutputTypes.SoundArchive;
                }
                this.soundArchiveOutputType.Add(soundSet, outputType);

                var item = new ListViewItem();
                item.Text = soundSet.Name;
                item.Tag = soundSet;
                item.SubItems.Add(this.OutputTypeToText(outputType));
                item.SubItems[0].Tag = ColumnSoundSetName;
                item.SubItems[1].Tag = ColumnOutputType;
                this.listView.Items.Add(item);
            }

            // 統計タブ
            this.EnabledIntegratedLoudnessThreshold = projectService.Project.EnabledIntegratedLoudnessThreshold;
            this.IntegratedLoudnessThreshold = projectService.Project.IntegratedLoudnessThreshold;
            this.SequenceMaxMeasureDuration = projectService.Project.SequenceMaxMeasureDuration;
        }

        private string OutputTypeToText(SoundArchiveOutputTypes value)
        {
            switch (value)
            {
                case SoundArchiveOutputTypes.None: return MessageResource.Label_OutputType_None;
                case SoundArchiveOutputTypes.SoundArchive: return MessageResource.Label_OutputType_SoundArchive;
                case SoundArchiveOutputTypes.AddonSoundArchive: return MessageResource.Label_OutputType_AddonSoundArchive;
            }
            return String.Empty;
        }

        /// <summary>
        ///
        /// </summary>
        public void SaveSettings(SoundProjectService projectService)
        {
            if (null == projectService) { throw new ArgumentNullException("projectService"); }

            if (this.IsModifiedSettings(projectService) == false) return;

            projectService.Project.DoWarnUnreferencedItems = DoWarnUnreferencedItems;
            projectService.Project.DoWarnDisableGroupItemTargets = DoWarnDisableGroupItemTargets;
            projectService.Project.ExcludeStringTable = ExcludeStringTable;
            projectService.Project.IntermediateOutputPath = IntermediateOutputPath;
            projectService.Project.InGameEditCacheOutputPath = InGameEditCacheOutputPath;
            projectService.Project.OutputLabel = OutputLabel;
            projectService.Project.OutputDirectoryPath = OutputPath;
            projectService.Project.ExternalFileOutputDirectoryPath = StreamPath;
            projectService.Project.DoWarnPCBinariesForAACNotFound = DoWarnPCBinariesForAACNotFound;
            projectService.Project.DoDeleteGarbageOutputBinaries = DoDeleteGarbageOutputBinaries;
            projectService.Project.DoUseHardwareOpusDecoder = DoUseHardwareOpusDecoder;

            projectService.Project.UserManagementFileOutputDirectoryPath = OutputUserManagementDirectory;

            projectService.Project.SoundArchivePlayerSequenceSoundCount = SequenceSound;
            projectService.Project.SoundArchivePlayerSequenceTrackCount = SequenceTrack;
            projectService.Project.SoundArchivePlayerStreamSoundCount = StreamSound;
            projectService.Project.SoundArchivePlayerStreamChannelCount = StreamChannel;
            projectService.Project.SoundArchivePlayerStreamBufferTimes = StreamBufferTimes;

            projectService.Project.SoundArchivePlayerWaveSoundCount = WaveSound;
            projectService.Project.IsPreConvertCommandsEnabled = PreConvertCommandEnabled;
            projectService.Project.IsPostConvertCommandsEnabled = PostConvertCommandEnabled;
            projectService.Project.PreConvertCommands = this.StringToStringList(PreConvertCommands);
            projectService.Project.PostConvertCommands = this.StringToStringList(PostConvertCommands);
            projectService.Project.IsFileSavePreCommandEnabled = FileSavePreCommandEnabled;
            projectService.Project.IsFileSavePostCommandEnabled = FileSavePostCommandEnabled;
            projectService.Project.FileSavePreCommandPath = FileSavePreCommandPath;
            projectService.Project.FileSavePostCommandPath = FileSavePostCommandPath;

            projectService.Project.SmfConvertTimeBase = DismantleSMF;
            projectService.Project.DoUseBuiltInWavePreprocessExe = DoUseBuiltInWavePreprocessExe;
            projectService.Project.WavePreprocessExePath = WavePreprocessExePath;

            projectService.Project.SoundListOutputSettings.ListOutputs.Clear();
            foreach (SoundListOutput output in SoundListOutputSettings)
            {
                projectService.Project.SoundListOutputSettings.ListOutputs.Add(output);
            }

            projectService.Project.StreamSoundNamePrefix = StreamName;
            projectService.Project.WaveSoundSetNamePrefix = WaveSoundSetName;
            projectService.Project.WaveSoundNamePrefix = WaveSoundCoName;
            projectService.Project.SequenceSoundSetNamePrefix = SeqSetName;
            projectService.Project.SequenceSoundNamePrefix = SeqName;
            projectService.Project.BankNamePrefix = BankName;
            projectService.Project.PlayerNamePrefix = PlayerName;
            projectService.Project.GroupNamePrefix = GroupName;
            projectService.Project.InstrumentNamePrefix = InstName;
            projectService.Project.WaveArchiveNamePrefix = WarcName;

            projectService.Project.ItemNamingHasPrefix = EnablePrefix;
            projectService.Project.ItemNamingCaseChange = UpperLowerLetterProcess;
            projectService.Project.ItemNamingInvalidCharChange = IllegalLetterProcess;
            projectService.Project.ItemPastePostfix = ItemPastePostfix;
            projectService.Project.EnabledNameDelimiter = EnabledNameDelimiter;
            projectService.Project.NameDelimiter = NameDelimiter;

            projectService.Project.CommentColumnText = Comment;
            projectService.Project.Comment1ColumnText = Comment1;
            projectService.Project.Comment2ColumnText = Comment2;
            projectService.Project.Comment3ColumnText = Comment3;
            projectService.Project.Comment4ColumnText = Comment4;
            projectService.Project.Comment5ColumnText = Comment5;
            projectService.Project.Comment6ColumnText = Comment6;
            projectService.Project.Comment7ColumnText = Comment7;
            projectService.Project.Comment8ColumnText = Comment8;
            projectService.Project.Comment9ColumnText = Comment9;

            //
            projectService.Project.ColorComment0 = ColorComment0;
            projectService.Project.ColorComment1 = ColorComment1;
            projectService.Project.ColorComment2 = ColorComment2;
            projectService.Project.ColorComment3 = ColorComment3;
            projectService.Project.ColorComment4 = ColorComment4;
            projectService.Project.ColorComment5 = ColorComment5;
            projectService.Project.ColorComment6 = ColorComment6;
            projectService.Project.ColorComment7 = ColorComment7;
            projectService.Project.ColorComment8 = ColorComment8;

            //
            projectService.Project.UserCommands[0].Name = UserCommandName1;
            projectService.Project.UserCommands[1].Name = UserCommandName2;
            projectService.Project.UserCommands[2].Name = UserCommandName3;
            projectService.Project.UserCommands[3].Name = UserCommandName4;
            projectService.Project.UserCommands[4].Name = UserCommandName5;
            projectService.Project.UserCommands[5].Name = UserCommandName6;
            projectService.Project.UserCommands[6].Name = UserCommandName7;
            projectService.Project.UserCommands[7].Name = UserCommandName8;

            projectService.Project.UserCommands[0].Command = UserCommand1;
            projectService.Project.UserCommands[1].Command = UserCommand2;
            projectService.Project.UserCommands[2].Command = UserCommand3;
            projectService.Project.UserCommands[3].Command = UserCommand4;
            projectService.Project.UserCommands[4].Command = UserCommand5;
            projectService.Project.UserCommands[5].Command = UserCommand6;
            projectService.Project.UserCommands[6].Command = UserCommand7;
            projectService.Project.UserCommands[7].Command = UserCommand8;

            projectService.Project.UserCommands[0].IconFilePath = UserCommandIconFilePath1;
            projectService.Project.UserCommands[1].IconFilePath = UserCommandIconFilePath2;
            projectService.Project.UserCommands[2].IconFilePath = UserCommandIconFilePath3;
            projectService.Project.UserCommands[3].IconFilePath = UserCommandIconFilePath4;
            projectService.Project.UserCommands[4].IconFilePath = UserCommandIconFilePath5;
            projectService.Project.UserCommands[5].IconFilePath = UserCommandIconFilePath6;
            projectService.Project.UserCommands[6].IconFilePath = UserCommandIconFilePath7;
            projectService.Project.UserCommands[7].IconFilePath = UserCommandIconFilePath8;

            //
            projectService.Project.ProjectComment = ProjectComment;

            //
            for (int index = 0; index < this.settings.Settings.Count; index++)
            {
                this.settings.Settings[index].CopyFrom(duplicatedSettings[index]);
            }

            //
            projectService.ProjectDocument.SetDirty();
            projectService.Project.OnModifiedProjectSetting();

            //
            projectService.Project.SyncPort = SyncPort;
            projectService.Project.FuncPort = FuncPort;
            projectService.Project.CtrlPort = CtrlPort;

            //
            projectService.Project.SyncChannel = SyncChannel;
            projectService.Project.FuncChannel = FuncChannel;
            projectService.Project.CtrlChannel = CtrlChannel;

            projectService.Project.EnabledConnectionTimeOut = EnabledConnectionTimeOut;
            projectService.Project.ConnectionTimeOut = ConnectionTimeOut;

            //
            projectService.Project.SoundArchiveOutputTypes.Clear();
            foreach (var pair in this.soundArchiveOutputType)
            {
                projectService.AddSoundArchiveOutputType(pair.Key, pair.Value);
            }

            // 統計タブ
            projectService.Project.EnabledIntegratedLoudnessThreshold = this.EnabledIntegratedLoudnessThreshold;
            projectService.Project.IntegratedLoudnessThreshold = this.IntegratedLoudnessThreshold;
            projectService.Project.SequenceMaxMeasureDuration = this.SequenceMaxMeasureDuration;
        }

        /// <summary>
        /// ユーザーコマンドタブを選択します。
        /// </summary>
        public void SelectUserCommandTab()
        {
            this.ProjectSettingTab.SelectedTab = UserCommandTabPage;
        }

        /// <summary>
        ///
        /// </summary>
        private bool IsModifiedSettings(SoundProjectService projectService)
        {
            if (null == projectService) { throw new ArgumentNullException("projectService"); }

            if (projectService.Project.DoWarnUnreferencedItems != DoWarnUnreferencedItems ||
                projectService.Project.DoWarnDisableGroupItemTargets != DoWarnDisableGroupItemTargets ||
                projectService.Project.ExcludeStringTable != ExcludeStringTable ||
                projectService.Project.IntermediateOutputPath != IntermediateOutputPath ||
                projectService.Project.DoWarnPCBinariesForAACNotFound != DoWarnPCBinariesForAACNotFound ||
                projectService.Project.DoDeleteGarbageOutputBinaries != DoDeleteGarbageOutputBinaries ||
                projectService.Project.DoUseHardwareOpusDecoder != DoUseHardwareOpusDecoder ||
                projectService.Project.InGameEditCacheOutputPath != InGameEditCacheOutputPath ||
                projectService.Project.OutputLabel != OutputLabel ||
                projectService.Project.OutputDirectoryPath != OutputPath ||
                projectService.Project.ExternalFileOutputDirectoryPath != StreamPath ||
                projectService.Project.UserManagementFileOutputDirectoryPath != OutputUserManagementDirectory ||
                projectService.Project.SoundArchivePlayerSequenceSoundCount != SequenceSound ||
                projectService.Project.SoundArchivePlayerSequenceTrackCount != SequenceTrack ||
                projectService.Project.SoundArchivePlayerStreamSoundCount != StreamSound ||
                projectService.Project.SoundArchivePlayerStreamChannelCount != StreamChannel ||
                projectService.Project.SoundArchivePlayerStreamBufferTimes != StreamBufferTimes ||
                projectService.Project.SoundArchivePlayerWaveSoundCount != WaveSound ||
                projectService.Project.IsPreConvertCommandsEnabled != PreConvertCommandEnabled ||
                projectService.Project.IsPostConvertCommandsEnabled != PostConvertCommandEnabled ||
                this.StringListToString(projectService.Project.PreConvertCommands) != PreConvertCommands ||
                this.StringListToString(projectService.Project.PostConvertCommands) != PostConvertCommands ||
                projectService.Project.IsFileSavePreCommandEnabled != FileSavePreCommandEnabled ||
                projectService.Project.IsFileSavePostCommandEnabled != FileSavePostCommandEnabled ||
                projectService.Project.FileSavePreCommandPath != FileSavePreCommandPath ||
                projectService.Project.FileSavePostCommandPath != FileSavePostCommandPath ||
                projectService.Project.SmfConvertTimeBase != DismantleSMF ||
                projectService.Project.DoUseBuiltInWavePreprocessExe != DoUseBuiltInWavePreprocessExe ||
                projectService.Project.WavePreprocessExePath != WavePreprocessExePath ||
                projectService.Project.EnabledIntegratedLoudnessThreshold != this.EnabledIntegratedLoudnessThreshold ||
                projectService.Project.IntegratedLoudnessThreshold != this.IntegratedLoudnessThreshold ||
                projectService.Project.SequenceMaxMeasureDuration != this.SequenceMaxMeasureDuration)
            {
                return true;
            }

            IList<SoundListOutput> listOutputs =
                projectService.Project.SoundListOutputSettings.ListOutputs;

            if (listOutputs.Count != SoundListOutputSettings.Count) return true;

            for (int i = 0; i < listOutputs.Count; i++)
            {
                if (listOutputs[i].Equals(SoundListOutputSettings[i]) == false)
                {
                    return true;
                }
            }

            if (projectService.Project.StreamSoundNamePrefix != StreamName ||
                projectService.Project.WaveSoundSetNamePrefix != WaveSoundSetName ||
                projectService.Project.WaveSoundNamePrefix != WaveSoundCoName ||
                projectService.Project.SequenceSoundSetNamePrefix != SeqSetName ||
                projectService.Project.SequenceSoundNamePrefix != SeqName ||
                projectService.Project.BankNamePrefix != BankName ||
                projectService.Project.PlayerNamePrefix != PlayerName ||
                projectService.Project.GroupNamePrefix != GroupName ||
                projectService.Project.InstrumentNamePrefix != InstName ||
                projectService.Project.WaveArchiveNamePrefix != WarcName ||
                projectService.Project.ItemNamingHasPrefix != EnablePrefix ||
                projectService.Project.ItemNamingCaseChange != UpperLowerLetterProcess ||
                projectService.Project.ItemNamingInvalidCharChange != IllegalLetterProcess ||
                projectService.Project.ItemPastePostfix != ItemPastePostfix ||
                projectService.Project.EnabledNameDelimiter != EnabledNameDelimiter ||
                projectService.Project.NameDelimiter != NameDelimiter)
            {
                return true;
            }

            if (projectService.Project.CommentColumnText != Comment ||
                projectService.Project.Comment1ColumnText != Comment1 ||
                projectService.Project.Comment2ColumnText != Comment2 ||
                projectService.Project.Comment3ColumnText != Comment3 ||
                projectService.Project.Comment4ColumnText != Comment4 ||
                projectService.Project.Comment5ColumnText != Comment5 ||
                projectService.Project.Comment6ColumnText != Comment6 ||
                projectService.Project.Comment7ColumnText != Comment7 ||
                projectService.Project.Comment8ColumnText != Comment8 ||
                projectService.Project.Comment9ColumnText != Comment9)
            {
                return true;
            }

            if (projectService.Project.ColorComment0 != ColorComment0 ||
                projectService.Project.ColorComment1 != ColorComment1 ||
                projectService.Project.ColorComment2 != ColorComment2 ||
                projectService.Project.ColorComment3 != ColorComment3 ||
                projectService.Project.ColorComment4 != ColorComment4 ||
                projectService.Project.ColorComment5 != ColorComment5 ||
                projectService.Project.ColorComment6 != ColorComment6 ||
                projectService.Project.ColorComment7 != ColorComment7 ||
                projectService.Project.ColorComment8 != ColorComment8)
            {
                return true;
            }

            //
            UserCommand[] userCommands = projectService.Project.UserCommands;
            if (userCommands[0].Name != UserCommandName1 ||
                userCommands[1].Name != UserCommandName2 ||
                userCommands[2].Name != UserCommandName3 ||
                userCommands[3].Name != UserCommandName4 ||
                userCommands[4].Name != UserCommandName5 ||
                userCommands[5].Name != UserCommandName6 ||
                userCommands[6].Name != UserCommandName7 ||
                userCommands[7].Name != UserCommandName8 ||
                userCommands[0].Command != UserCommand1 ||
                userCommands[1].Command != UserCommand2 ||
                userCommands[2].Command != UserCommand3 ||
                userCommands[3].Command != UserCommand4 ||
                userCommands[4].Command != UserCommand5 ||
                userCommands[5].Command != UserCommand6 ||
                userCommands[6].Command != UserCommand7 ||
                userCommands[7].Command != UserCommand8 ||
                userCommands[0].IconFilePath != UserCommandIconFilePath1 ||
                userCommands[1].IconFilePath != UserCommandIconFilePath2 ||
                userCommands[2].IconFilePath != UserCommandIconFilePath3 ||
                userCommands[3].IconFilePath != UserCommandIconFilePath4 ||
                userCommands[4].IconFilePath != UserCommandIconFilePath5 ||
                userCommands[5].IconFilePath != UserCommandIconFilePath6 ||
                userCommands[6].IconFilePath != UserCommandIconFilePath7 ||
                userCommands[7].IconFilePath != UserCommandIconFilePath8)
            {
                return true;
            }

            if (projectService.Project.ProjectComment != ProjectComment)
            {
                return true;
            }

            for (int index = 0; index < this.settings.Settings.Count; index++)
            {
                if (this.settings.Settings[index].Equals(duplicatedSettings[index]) == false)
                {
                    return true;
                }
            }

            if (projectService.Project.SyncPort != SyncPort ||
                projectService.Project.FuncPort != FuncPort ||
                projectService.Project.CtrlPort != CtrlPort ||

                projectService.Project.SyncChannel != SyncChannel ||
                projectService.Project.FuncChannel != FuncChannel ||
                projectService.Project.CtrlChannel != CtrlChannel)
            {
                return true;
            }

            if (projectService.Project.EnabledConnectionTimeOut != EnabledConnectionTimeOut ||
                projectService.Project.ConnectionTimeOut != ConnectionTimeOut)
            {
                return true;
            }

            //
            foreach (var pair in this.soundArchiveOutputType)
            {
                SoundArchiveOutputTypes outputType;
                if (projectService.Project.SoundArchiveOutputTypes.TryGetValue(pair.Key, out outputType) == false)
                {
                    outputType = SoundArchiveOutputTypes.SoundArchive;
                }

                if (outputType != pair.Value)
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        ///
        /// </summary>
        private string StringListToString(List<string> lines)
        {
            Assertion.Argument.NotNull(lines);

            StringBuilder builder = new StringBuilder();

            foreach (string line in lines)
            {
                builder.AppendLine(line);
            }

            return builder.ToString();
        }

        /// <summary>
        ///
        /// </summary>
        private List<string> StringToStringList(string text)
        {
            Assertion.Argument.NotNull(text);

            List<string> lines = new List<string>();

            foreach (string line in
                text.Split(new char[] { '\r', '\n' }, StringSplitOptions.RemoveEmptyEntries))
            {
                lines.Add(line);
            }

            return lines;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnTextChangedName(object sender, EventArgs e)
        {
            UpdateControls();
            UpdateSampleItemName();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private System.Windows.Forms.ErrorProvider _ErrorProvider = new ErrorProvider();

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool CanDecide
        {
            get
            {
                bool stream = InspectName(textBox_Stream);
                bool waveSoundSet = InspectName(textBox_WaveSoundSet);
                bool waveSoundCo = InspectName(textBox_WaveSoundCo);
                bool seqSet = InspectName(textBox_SeqSet);
                bool seq = InspectName(textBox_Seq);
                bool bank = InspectName(textBox_Bank);
                bool player = InspectName(textBox_Player);
                bool group = InspectName(textBox_Group);
                bool inst = InspectName(textBox_Inst);
                bool userParameter = (userParameterSettingPanel0.OverlappedStructure == false &&
                                       userParameterSettingPanel1.OverlappedStructure == false &&
                                       userParameterSettingPanel2.OverlappedStructure == false &&
                                       userParameterSettingPanel3.OverlappedStructure == false);

                if (stream != false &&
                    waveSoundSet != false &&
                    waveSoundCo != false &&
                    seqSet != false &&
                    seq != false &&
                    bank != false &&
                    player != false &&
                    group != false &&
                    inst != false &&
                    userParameter != false &&
                    CanAccept != false)
                {
                    return true;
                }

                return false;
            }
        }

        ///--------------------------------
        /// <summary>
        /// コントロールの更新
        /// </summary>
        private void UpdateControls()
        {
            if (this.ReadOnly == true)
            {
                return;
            }

            OK.Enabled = CanDecide;

            //
            panel_Prefix.Enabled = checkBox_Prefix.Checked;

            this.lbl_NameDelimiter.Enabled = this.chk_NameDelimiter.Checked;
            this.tbx_NameDelimiter.Enabled = this.chk_NameDelimiter.Checked;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void UpdateSampleItemName()
        {
            if (_SuspendUpdateSampleItemName != false)
            {
                return;
            }

            string streamPrefix = textBox_Stream.Text;

            ItemNamingSettings settings = new ItemNamingSettings(
                EnablePrefix, UpperLowerLetterProcess, IllegalLetterProcess,
                EnabledNameDelimiter, NameDelimiter);

            try
            {
                textBox_ItemName.Text = ItemNamer.CreateUniqueNameFromFileName
                    (null, settings, streamPrefix, string.Empty, textBox_FileName.Text);
            }
            catch (Exception)
            {
            }
        }

        ///--------------------------------
        /// <summary>
        /// エラーの表示
        /// </summary>
        protected void ShowError(Control control, string text)
        {
            if (null == control) { throw new ArgumentNullException("control"); }
            if (null == text) { throw new ArgumentNullException("text"); }
            _ErrorProvider.SetError(control, text);
        }

        ///--------------------------------
        /// <summary>
        /// エラーの非表示
        /// </summary>
        protected void HideError(Control control)
        {
            _ErrorProvider.SetError(control, null);
        }

        ///--------------------------------
        /// <summary>
        /// 接頭語の検査
        /// </summary>
        private bool InspectName(TextBox textBox)
        {
            string name = textBox.Text;

            if (name == String.Empty)
            {
                // 空白の接頭語を許可します。
            }
            else
            {
                ValidationResult result = ItemNameValidator.ValidateSoundProjectItemName(name);
                if (!result.IsValid)
                {
                    ShowError(textBox, result.ToString());
                    return false;
                }
            }

            HideError(textBox);
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// サウンドリスト出力設定リストの表示を更新
        /// </summary>
        private void UpdateSoundListOutputSettingList()
        {
            bool settingEmpty = (this.soundListOutputs == null ||
                                  this.soundListOutputs.Count == 0);

            mSoundListOutputSettingTabControl.Visible = !settingEmpty;
            mOutputSoundListDeleteButton.Enabled = !settingEmpty;

            mSoundListOutputSettingListListBox.Items.Clear();
            currentSoundListOutput = null;

            if (this.soundListOutputs.Count > 0)
            {
                Object[] itemArray = new Object[this.soundListOutputs.Count];
                int index = 0;
                foreach (SoundListOutput s in this.soundListOutputs)
                {
                    itemArray[index] = s.Name;
                    index++;
                }
                mSoundListOutputSettingListListBox.Items.AddRange(itemArray);

                mSoundListOutputSettingListListBox.SelectedIndex = 0;
                currentSoundListOutput = this.soundListOutputs[0];

                UpdateSoundListOutputSetting();
            }

        }

        ///--------------------------------
        /// <summary>
        /// サウンドリスト出力設定の表示を更新
        /// </summary>
        private void UpdateSoundListOutputSetting()
        {
            if (currentSoundListOutput == null)
            {

                return;
            }

            SoundListOutput setting = currentSoundListOutput;
            mSoundListOutputSettingListTextBox.Text = setting.Name;
            mSoundListOutputSettingFilePathTextBox.Text = setting.FilePath;
            mSoundListOutputSettingTitleTextBox.Text = setting.Title;
            mSoundListOutputSettingEachSoundSetTableCheckBox.Checked =
                (setting.OutputType == SoundListOutputType.EachSoundSet);
            mSoundListOutputSettingIsIgnoreDisableItemsCheckBox.Checked =
                setting.IsIgnoreDisableItems;

            mOutputListBox.Items.Clear();

            foreach (ListOutput listOutput in setting.ListOutputs)
            {
                if (!this.listTexts.ContainsKey(listOutput.Name))
                {
                    continue;
                }

                string listName = string.Empty;

                OutputListInfo outputListInfo = new OutputListInfo(this.listTexts[listOutput.Name])
                {
                    ItemInfo = listOutput,
                    ListInfoArray = new List<OutputItemInfo>(),
                };

                foreach (ListColumnOutput columnOutput in listOutput.ColumnOutputs)
                {
                    outputListInfo.ListInfoArray.Add(
                        new OutputItemInfo(this.listColumnTexts[columnOutput.Name])
                        {
                            ItemInfo = columnOutput,
                        });
                }

                int index = mOutputListBox.Items.Add(outputListInfo);
                mOutputListBox.SetItemChecked(index, listOutput.IsOutput);
            }

            if (mOutputListBox.Items.Count > 0)
            {
                mOutputListBox.SelectedIndex = 0;
                UpdateListItemListBox(((OutputListInfo)mOutputListBox.Items[0]).ItemInfo.IsOutput);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int DismantleSMF
        {
            get
            {
                switch (comboBox_Dismantle.SelectedIndex)
                {
                    case 0:
                        return 48;
                    case 1:
                        return 96;
                }
                return -1;
            }

            set
            {
                int index = 0;

                switch (value)
                {
                    case 48:
                        index = 0;
                        break;
                    case 96:
                        index = 1;
                        break;
                    default:
                        return;
                }

                comboBox_Dismantle.SelectedIndex = index;
            }
        }

        public bool DoUseBuiltInWavePreprocessExe
        {
            get { return this.useBuiltInSox.Checked; }
            set
            {
                this.useUserSox.Checked = !value;
                this.useBuiltInSox.Checked = value;
                this.UpdateUserSoxPathControls();
            }
        }

        public string WavePreprocessExePath
        {
            get { return this.userSoxPathTextBox.Text; }
            set { this.userSoxPathTextBox.Text = value; }
        }

        public bool DoWarnUnreferencedItems
        {
            get { return this.checkBox_warnUnreferencedItem.Checked; }
            set { this.checkBox_warnUnreferencedItem.Checked = value; }
        }

        public bool DoWarnDisableGroupItemTargets
        {
            get { return this.checkBox_warnDisableGroupItemTargets.Checked; }
            set { this.checkBox_warnDisableGroupItemTargets.Checked = value; }
        }

        public bool ExcludeStringTable
        {
            get { return !this.checkBox_StringTableOutput.Checked; }
            set { this.checkBox_StringTableOutput.Checked = !value; }
        }

        ///--------------------------------
        /// <summary>
        /// 中間ファイルのキャッシュフォルダの取得
        /// </summary>
        public string IntermediateOutputPath
        {
            get { return this.IntermediateOutputPathTextBox.Text; }
            set { this.IntermediateOutputPathTextBox.Text = value; }
        }

        ///--------------------------------
        /// <summary>
        /// インゲーム編集のキャッシュフォルダの取得
        /// </summary>
        public string InGameEditCacheOutputPath
        {
            get { return this.textBox_sre_cache.Text; }
            set { this.textBox_sre_cache.Text = value; }
        }

        ///--------------------------------
        /// <summary>
        /// アイテム別ラベルを出力するかどうかの取得
        /// </summary>
        public bool OutputLabel
        {
            get { return this.checkBox_OutputLabel.Checked; }
            set { this.checkBox_OutputLabel.Checked = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public bool DoWarnPCBinariesForAACNotFound
        {
            get { return this.chk_WarnNotFoundPCBinariesAAC.Checked; }
            set { this.chk_WarnNotFoundPCBinariesAAC.Checked = value; }
        }

        /// <summary>
        /// ターゲットでの Opus のデコードにはハードウェアを使用するように指定する。
        /// </summary>
        public bool DoUseHardwareOpusDecoder
        {
            get
            {
                return this.checkBox_DoUseHardwareOpusDecoder.Checked;
            }
            set
            {
                this.checkBox_DoUseHardwareOpusDecoder.Checked = value;
            }
        }

        /// <summary>
        /// 出力先パスを取得または設定します。
        /// </summary>
        public string OutputPath
        {
            get
            {
                return OutputPathTextBox.Text;
            }
            set
            {
                OutputPathTextBox.Text = value;
            }
        }

        ///--------------------------------
        /// <summary>
        /// ストリームファイルの出力先パスを取得または設定します。
        /// </summary>
        public string StreamPath
        {
            get
            {
                return StreamPathTextBox.Text;
            }
            set
            {
                StreamPathTextBox.Text = value;
            }
        }

        /// <summary>
        /// ユーザー管理ファイルの出力先ディレクトリを取得/設定します。
        /// </summary>
        public string OutputUserManagementDirectory
        {
            get
            {
                return this.tbx_OutputUserManagementDirectory.Text;
            }
            set
            {
                this.tbx_OutputUserManagementDirectory.Text = value;
            }
        }

        const int SequenceSoundMin = 0;
        const int SequenceSoundMax = 999;

        /// <summary>
        /// シーケンスサウンドの数を取得または設定します。
        /// </summary>
        public int SequenceSound
        {
            get
            {
                try
                {
                    return int.Parse(SequenceSoundTextBox.Text);
                }
                catch
                {
                    Debug.Assert(false);
                    return 0;
                }
            }
            set
            {
                if (value < SequenceSoundMin)
                    value = SequenceSoundMin;
                if (value > SequenceSoundMax)
                    value = SequenceSoundMax;
                SequenceSoundTextBox.Text = value.ToString();
            }
        }
        const int SequenceTrackMin = 0;
        const int SequenceTrackMax = 999;
        /// <summary>
        /// シーケンストラックの数を取得または設定します。
        /// </summary>
        public int SequenceTrack
        {
            get
            {
                try
                {
                    return int.Parse(SequenceTrackTextBox.Text);
                }
                catch
                {
                    Debug.Assert(false);
                    return 0;
                }
            }
            set
            {
                if (value < SequenceTrackMin)
                    value = SequenceTrackMin;
                if (value > SequenceTrackMax)
                    value = SequenceTrackMax;
                SequenceTrackTextBox.Text = value.ToString();
            }
        }
        const int StreamSoundMin = 0;
        const int StreamSoundMax = 999;
        /// <summary>
        /// ストリームサウンドの数を取得または設定します。
        /// </summary>
        public int StreamSound
        {
            get
            {
                try
                {
                    return int.Parse(StreamSoundTextBox.Text);
                }
                catch
                {
                    Debug.Assert(false);
                    return 0;
                }
            }
            set
            {
                if (value < StreamSoundMin)
                    value = StreamSoundMin;
                if (value > StreamSoundMax)
                    value = StreamSoundMax;
                StreamSoundTextBox.Text = value.ToString();
            }
        }
        const int StreamChannelMin = 0;
        const int StreamChannelMax = 32;
        /// <summary>
        /// ストリームチャンネルの数を取得または設定します。
        /// </summary>
        public int StreamChannel
        {
            get
            {
                try
                {
                    return int.Parse(StreamChannelTextbox.Text);
                }
                catch
                {
                    Debug.Assert(false);
                    return 0;
                }
            }
            set
            {
                if (value < StreamChannelMin)
                    value = StreamChannelMin;
                if (value > StreamChannelMax)
                    value = StreamChannelMax;
                StreamChannelTextbox.Text = value.ToString();
            }
        }
        /// <summary>
        /// ストリームバッファデフォルトサイズを取得します。
        /// </summary>
        public int StreamBufferDefalutSize
        {
            get
            {
                return SoundProject.SoundArchivePlayerStreamBufferDefalutSize;
            }
        }
        /// <summary>
        /// ストリームバッファサイズを取得します。
        /// </summary>
        public int StreamBufferSize
        {
            get
            {
                return (SoundProject.SoundArchivePlayerStreamBufferDefalutSize +
                        SoundProject.SoundArchivePlayerStreamBufferExtensionSize);
            }
        }
        const int StreamBufferTimesMin = 1;
        const int StreamBufferTimesMax = 4;
        /// <summary>
        /// ストリームバッファサイズの倍率を取得または設定します。
        /// </summary>
        public int StreamBufferTimes
        {
            get
            {
                return comboBox_SoundArchiveStreamBufferTimes.SelectedIndex + 1;
            }
            set
            {
                if (value < StreamBufferTimesMin)
                    value = StreamBufferTimesMin;
                if (value > StreamBufferTimesMax)
                    value = StreamBufferTimesMax;
                comboBox_SoundArchiveStreamBufferTimes.SelectedIndex = value - 1;
            }
        }
        const int WaveSoundMin = 0;
        const int WaveSoundMax = 999;
        /// <summary>
        /// ウェーブサウンドの数を取得または設定します。
        /// </summary>
        public int WaveSound
        {
            get
            {
                try
                {
                    return int.Parse(WaveSoundTextBox.Text);
                }
                catch
                {
                    return 0;
                }
            }
            set
            {
                if (value < WaveSoundMin)
                    value = WaveSoundMin;
                if (value > WaveSoundMax)
                    value = WaveSoundMax;
                WaveSoundTextBox.Text = value.ToString();
            }
        }
        /// <summary>
        /// コンバート前コマンドの実行設定を取得または設定します。
        /// </summary>
        public bool PreConvertCommandEnabled
        {
            get
            {
                return checkBox_PreConvertCommand.Checked;
            }
            set
            {
                checkBox_PreConvertCommand.Checked = value;
            }
        }
        /// <summary>
        /// コンバート後コマンドの実行設定を取得または設定します。
        /// </summary>
        public bool PostConvertCommandEnabled
        {
            get
            {
                return checkBox_PostConvertCommand.Checked;
            }
            set
            {
                checkBox_PostConvertCommand.Checked = value;
            }
        }
        /// <summary>
        /// コンバート前に実行されるコマンドを取得または設定します。
        /// </summary>
        public string PreConvertCommands
        {
            get { return PreConvertCommandListTextBox.Text; }
            set { PreConvertCommandListTextBox.Text = value; }
        }
        /// <summary>
        /// コンバート後に実行されるコマンドを取得または設定します。
        /// </summary>
        public string PostConvertCommands
        {
            get { return PostConvertCommandListTextBox.Text; }
            set { PostConvertCommandListTextBox.Text = value; }
        }


        /// <summary>
        /// ファイル保存前コマンドの実行設定を取得または設定します。
        /// </summary>
        public bool FileSavePreCommandEnabled
        {
            get { return FileSavePreCommandEnabled_checkBox.Checked; }
            set { FileSavePreCommandEnabled_checkBox.Checked = value; }
        }
        /// <summary>
        /// ファイル保存後コマンドの実行設定を取得または設定します。
        /// </summary>
        public bool FileSavePostCommandEnabled
        {
            get { return FileSavePostCommandEnabled_checkBox.Checked; }
            set { FileSavePostCommandEnabled_checkBox.Checked = value; }
        }
        /// <summary>
        /// ファイル保存前に実行されるコマンドを取得または設定します。
        /// </summary>
        public string FileSavePreCommandPath
        {
            get { return FileSavePreCommandPath_textBox.Text; }
            set { FileSavePreCommandPath_textBox.Text = value; }
        }
        /// <summary>
        /// ファイル保存後に実行されるコマンドを取得または設定します。
        /// </summary>
        public string FileSavePostCommandPath
        {
            get { return FileSavePostCommandPath_textBox.Text; }
            set { FileSavePostCommandPath_textBox.Text = value; }
        }


        /// <summary>
        /// 選択されているプロジェクト設定タブのインデックスを取得または設定します。
        /// </summary>
        public int SelectedProjectSettingTabIndex
        {
            get { return this.ProjectSettingTab.SelectedIndex; }
            set
            {
                if (0 == this.ProjectSettingTab.TabCount)
                {
                    value = -1;
                }
                else if (value < 0 || this.ProjectSettingTab.TabCount <= value)
                {
                    value = 0;
                }
                this.ProjectSettingTab.SelectedIndex = value;
            }
        }

        /// <summary>
        /// 選択されているHTMLサウンドリスト設定タブのインデックスを取得または設定します。
        /// </summary>
        public int SelectedHTMLSoundListTabIndex
        {
            get { return this.mSoundListOutputSettingTabControl.SelectedIndex; }
            set
            {
                if (0 == this.mSoundListOutputSettingTabControl.TabCount)
                {
                    value = -1;
                }
                else if (value < 0 || this.mSoundListOutputSettingTabControl.TabCount <= value)
                {
                    value = 0;
                }
                this.mSoundListOutputSettingTabControl.SelectedIndex = value;
            }
        }

        /// <summary>
        /// 選択されているアイテムを取得または設定します。
        /// </summary>
        public int SelectedHTMLSoundListListBoxIndex
        {
            get
            {
                if (isCancel == true)
                {
                    return this.saveSelectedHTMLSoundListListBoxIndex;
                }
                else
                {
                    return this.mSoundListOutputSettingListListBox.SelectedIndex;
                }
            }
            set
            {
                if (0 == this.mSoundListOutputSettingListListBox.Items.Count)
                {
                    value = -1;
                }
                else if (value < 0 || this.mSoundListOutputSettingListListBox.Items.Count <= value)
                {
                    value = 0;
                }
                this.mSoundListOutputSettingListListBox.SelectedIndex = value;
                this.saveSelectedHTMLSoundListListBoxIndex = value;
                this.saveHTMLSoundListListBoxItemsCount = this.mSoundListOutputSettingListListBox.Items.Count;
            }
        }

        // <summary>
        // 「最大平均ラウドネス値を超える場合は警告する」の有効無効を設定または取得します。
        // </summary>
        public bool EnabledIntegratedLoudnessThreshold
        {
            get
            {
                return this.checkBox_EnabledIntegratedLoudnessThreshold.Checked;
            }
            set
            {
                this.checkBox_EnabledIntegratedLoudnessThreshold.Checked = value;
                label_IntegratedLoudnessThresholdDocument.Enabled = value;
                textBox_IntegratedLoudnessThreshold.Enabled = value;
                label_LKFS.Enabled = value;
                label_IntegratedLoudnessThreshold.Enabled = value;
            }
        }

        // <summary>
        // 最大平均ラウドネス値を設定または取得します。
        // </summary>
        public float IntegratedLoudnessThreshold
        {
            get
            {
                float number;

                if (float.TryParse(this.textBox_IntegratedLoudnessThreshold.Text, out number) == true)
                {
                    return number;
                }

                return 0;
            }
            set
            {
                this.textBox_IntegratedLoudnessThreshold.Text = string.Format("{0:F1}", value);
            }
        }

        // <summary>
        // シーケンスサウンドの最大計測時間
        // </summary>
        public int SequenceMaxMeasureDuration
        {
            get
            {
                int number;

                if (int.TryParse(this.textBox_SequenceMaxMeasureDuration.Text, out number) == true)
                {
                    return number;
                }

                return 60 * 3; // デフォルト 3分
            }
            set
            {
                this.textBox_SequenceMaxMeasureDuration.Text = value.ToString();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnShown(object sender, EventArgs e)
        {
            _SuspendUpdateSampleItemName = false;
            UpdateSampleItemName();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnHelp(object sender, EventArgs e)
        {
            ShowHelpHandler.Execute(FormsApplication.Instance.ProjectSettingsHelpURL);
        }

        private void StreamPathReference_Click(object sender, System.EventArgs e)
        {
            FolderBrowserDialog f = new FolderBrowserDialog();
            DialogResult r = f.ShowDialog(this);
            if (r == DialogResult.OK)
            {
                StreamPathTextBox.Text = f.SelectedPath;
            }
        }

        bool ValidateNum(Control control, int min, int max)
        {
            if (null == control) { throw new ArgumentNullException("control"); }

            try
            {

                IntParameterValue value = new IntParameterValue(min, min, max);
                value.Parse(control.Text);

                ValidationResult result = value.Validate();

                if (!result.IsValid)
                {
                    ShowError(control, result.ToString());
                    return false;
                }

            }
            catch
            {
                ShowError(control,
                           string.Format(NwSoundResources.MessageResource.Message_OutOfRange, min, max));
                return false;
            }

            HideError(control);
            return true;
        }

        bool err2, err3, err4, err5, err6, err7;
        bool CanAccept
        {
            get
            {
                return err2 && err3 && err4 && err5 && err6 && err7;
            }
        }

        private void SequenceSoundTextBox_TextChanged(object sender, System.EventArgs e)
        {
            err2 = ValidateNum(SequenceSoundTextBox, SequenceSoundMin, SequenceSoundMax);
            UpdateControls();
        }

        private void SequenceTrackTextBox_TextChanged(object sender, System.EventArgs e)
        {
            err3 = ValidateNum(SequenceTrackTextBox, SequenceTrackMin, SequenceTrackMax);
            UpdateControls();
        }

        private void StreamSoundTextBox_TextChanged(object sender, System.EventArgs e)
        {
            err4 = ValidateNum(StreamSoundTextBox, StreamSoundMin, StreamSoundMax);
            UpdateControls();
        }

        private void StreamChannelTextbox_TextChanged(object sender, System.EventArgs e)
        {
            err5 = ValidateNum(StreamChannelTextbox, StreamChannelMin, StreamChannelMax);
            this.UpdateSoundArchiveStreamDisplay();
            UpdateControls();
        }

        private void WaveSoundTextBox_TextChanged(object sender, System.EventArgs e)
        {
            err6 = ValidateNum(WaveSoundTextBox, WaveSoundMin, WaveSoundMax);
            UpdateControls();
        }

        ///
        private void OutputPath_TextChanged(object sender, EventArgs e)
        {
#if false
            if( FilePathValidator.ValidateRelativePathIsCurrent( OutputPathTextBox.Text) == true )
            {
                ShowError( OutputPathTextBox,
                           NwSoundResources.MessageResource.Message_InvalidFilePath);
                err7 = false;
            }
            else
            {
                HideError( OutputPathTextBox);
                err7 = true;
            }
            OK.Enabled = CanAccept;
#else
            err7 = true;
#endif
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnSelectedIndexChangedDismantleSMF(object sender, EventArgs e)
        {
        }

        private string CreateUniqueSettingName()
        {
            int index = 1;
            while (true)
            {
                bool flag = true;
                string name = "Setting " + index.ToString();
                foreach (SoundListOutput setting in soundListOutputs)
                {
                    if (name == setting.Name)
                    {
                        flag = false;
                        break;
                    }
                }
                if (flag)
                    return name;
                index++;
            }
            //            return String.Empty;
        }

        ///--------------------------------
        /// <summary>
        /// サウンドリスト出力設定の新規作成ボタンをクリック
        /// </summary>
        private void mOutputSoundListCreateButton_Click(object sender, EventArgs e)
        {
            SoundListOutputSettingCreateDialog dialog = new SoundListOutputSettingCreateDialog(ProjectFilePath);

            string settingName = CreateUniqueSettingName();
            string outputFileName = String.Format
                ("{0}_{1}", Path.GetFileNameWithoutExtension(ProjectFilePath), settingName);
            string outputFilePath = Path.ChangeExtension(outputFileName, ".html");

            dialog.SettingName = settingName;
            dialog.OutputFilePath = outputFilePath;

            if (dialog.ShowDialog(this) == DialogResult.OK)
            {
                SoundListOutput setting = dialog.NewSoundListOutput(
                    Path.GetFileNameWithoutExtension(this.ProjectFilePath)
                    );

                soundListOutputs.Add(setting);
                int index = mSoundListOutputSettingListListBox.Items.Add(setting.Name);

                currentSoundListOutput = soundListOutputs[index];
                mSoundListOutputSettingListListBox.SelectedIndex = index;

                mSoundListOutputSettingTabControl.Visible = true;
                mOutputSoundListDeleteButton.Enabled = true;
            }
        }

        ///--------------------------------
        /// <summary>
        /// サウンドリスト出力設定の削除ボタンをクリック
        /// </summary>
        private void mOutputSoundListDeleteButton_Click(object sender, EventArgs e)
        {
            int index = mSoundListOutputSettingListListBox.SelectedIndex;
            if (0 <= index)
            {
                mSoundListOutputSettingListListBox.Items.Remove(soundListOutputs[index].Name);
                soundListOutputs.Remove(soundListOutputs[index]);
                currentSoundListOutput = null;
                if (0 < mSoundListOutputSettingListListBox.Items.Count)
                {
                    if (mSoundListOutputSettingListListBox.Items.Count <= index)
                    {
                        index = mSoundListOutputSettingListListBox.Items.Count - 1;
                    }
                    mSoundListOutputSettingListListBox.SelectedIndex = index;
                }
                else
                {
                    UpdateSoundListOutputSettingList();
                }
            }
        }

        ///--------------------------------
        /// <summary>
        /// 出力設定名の変更
        /// </summary>
        private void mSoundListOutputSettingListTextBox_TextChanged(object sender, EventArgs e)
        {
            int index = mSoundListOutputSettingListListBox.SelectedIndex;
            if (0 <= index)
            {
                soundListOutputs[index].Name = mSoundListOutputSettingListTextBox.Text;
                mSoundListOutputSettingListListBox.Items[index] =
                    mSoundListOutputSettingListTextBox.Text;
            }
        }

        ///--------------------------------
        /// <summary>
        /// サウンドリスト出力設定リストのコンボボックスで選択
        /// </summary>
        private void mSoundListOutputSettingListListBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            int selectedIndex = mSoundListOutputSettingListListBox.SelectedIndex;
            if (selectedIndex < 0 || selectedIndex > soundListOutputs.Count - 1)
                return;

            currentSoundListOutput = soundListOutputs[selectedIndex];
            UpdateSoundListOutputSetting();
        }

        ///--------------------------------
        /// <summary>
        /// HTMLサウンドリスト出力ファイルパスの検証
        /// </summary>
        private void mSoundListOutputSettingFilePathTextBox_Validating(object sender, CancelEventArgs e)
        {
            try
            {
                string path = mSoundListOutputSettingFilePathTextBox.Text;
                if (Path.IsPathRooted(mSoundListOutputSettingFilePathTextBox.Text) == true)
                {
                    path = Path.GetDirectoryName(mSoundListOutputSettingFilePathTextBox.Text);
                }
                else
                {
                    string dir = Path.GetDirectoryName(this.ProjectFilePath);
                    path = Path.GetDirectoryName(Path.Combine(dir, path).GetFullPath());
                }
#if false
                if (Directory.Exists(path) == false)
                {
                    throw new ApplicationException();
                }
#endif
                // 禁則文字チェック
                Path.GetFullPath(mSoundListOutputSettingFilePathTextBox.Text);

                string fileName = Path.GetFileName(mSoundListOutputSettingFilePathTextBox.Text);

                if (fileName.Length == 0)
                {
                    throw new ApplicationException();
                }
            }
            catch
            {
                ApplicationBase.Instance.UIService.ShowMessageBox(MessageResource.Message_InvalidOutputFilePath);
                e.Cancel = true;
#if false
                int index = mSoundListOutputSettingListComboBox.SelectedIndex;
                SoundListOutput setting = soundListOutputs[index];
                mSoundListOutputSettingFilePathTextBox.Text = Path.Combine(
                    Path.GetDirectoryName(this.ProjectFilePath),
                    setting.FilePath
                    ).GetFullPath();
#endif
            }
        }

        ///--------------------------------
        /// <summary>
        /// HTMLサウンドリスト出力ファイルパス参照ボタンをクリック
        /// </summary>
        private void mSoundListOutputSettingFilePathButton_Click(object sender, EventArgs e)
        {
            string projectPath = Path.GetDirectoryName(this.ProjectFilePath);
            SaveFileDialog fileDialog = new SaveFileDialog();

            if (Path.IsPathRooted(mSoundListOutputSettingFilePathTextBox.Text) == true)
            {
                fileDialog.InitialDirectory = Path.GetDirectoryName(mSoundListOutputSettingFilePathTextBox.Text);
            }
            else
            {
                string fullPath = Path.Combine(projectPath, mSoundListOutputSettingFilePathTextBox.Text);
                fullPath = Path.GetFullPath(fullPath);
                if (Directory.Exists(Path.GetDirectoryName(fullPath)) == true)
                {
                    fileDialog.InitialDirectory = Path.GetDirectoryName(fullPath);
                }
                else
                {
                    fileDialog.InitialDirectory = projectPath;
                }
            }

            fileDialog.FileName = Path.GetFileName(mSoundListOutputSettingFilePathTextBox.Text);
            fileDialog.Filter = "HTML files (*.html;*.htm)|*.html;*.htm|All files (*.*)|*.*";
            fileDialog.DefaultExt = "html";
            fileDialog.OverwritePrompt = false;
            fileDialog.Title = MessageResource.DialogTitle_OutputFilePath;

            if (fileDialog.ShowDialog(this) == DialogResult.OK)
            {
                string filePath = PathEx.MakeRelative(fileDialog.FileName, projectPath);
                mSoundListOutputSettingFilePathTextBox.Text = filePath.Replace("/", "\\");

                int index = mSoundListOutputSettingListListBox.SelectedIndex;
                SoundListOutput setting = soundListOutputs[index];

                setting.FilePath = mSoundListOutputSettingFilePathTextBox.Text;
            }
        }

        ///--------------------------------
        /// <summary>
        /// タブ切り替え時の検証
        /// </summary>
        private void mSoundListOutputSettingTabControl_Validated(object sender, EventArgs e)
        {
            if (currentSoundListOutput == null)
                return;

            this.currentSoundListOutput.FilePath = mSoundListOutputSettingFilePathTextBox.Text;
            this.currentSoundListOutput.Title = mSoundListOutputSettingTitleTextBox.Text;
            this.currentSoundListOutput.OutputType = mSoundListOutputSettingEachSoundSetTableCheckBox.Checked ?
                SoundListOutputType.EachSoundSet : SoundListOutputType.EntireSoundProject;
            this.currentSoundListOutput.IsIgnoreDisableItems = mSoundListOutputSettingIsIgnoreDisableItemsCheckBox.Checked;

            this.currentSoundListOutput.ListOutputs.Clear();

            int index = 0;
            foreach (OutputListInfo info in mOutputListBox.Items)
            {
                this.currentSoundListOutput.ListOutputs.Add(info.ItemInfo);
                info.ItemInfo.IsOutput = mOutputListBox.GetItemChecked(index);
                index++;
            }
        }

        private void mOutputList_ItemCheck(object sender, ItemCheckEventArgs e)
        {
            UpdateListItemListBox(e.NewValue == CheckState.Checked);
            UpdateListMoveButton();
        }

        ///--------------------------------
        /// <summary>
        /// 出力するリストの選択変更
        /// </summary>
        bool mOutputList_SelectedIndexChanged_Enable = true;
        private void mOutputList_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (!mOutputList_SelectedIndexChanged_Enable)
                return;

            mCurrentSoundListOutput = null;

            int selectedIndex = mOutputListBox.SelectedIndex;
            if (selectedIndex < 0)
                return;
            System.Diagnostics.Debug.WriteLine("テキスト : " + selectedIndex.ToString());
            mCurrentSoundListOutput = (OutputListInfo)mOutputListBox.Items[selectedIndex];

            mListItemLabel.Text = String.Format(MessageResource.Label_HtmlSoundListDescription,
                                                 mCurrentSoundListOutput.ToString());

            UpdateListItemListBox(mOutputListBox.GetItemChecked(selectedIndex));

            UpdateListMoveButton();
        }

        ///--------------------------------
        /// <summary>
        /// 出力するリスト項目のリストボックス更新
        /// </summary>
        private void UpdateListItemListBox(bool enable)
        {
            mListItemListBox.Items.Clear();

            mListItemListBox.Enabled = enable;

            if (mCurrentSoundListOutput == null || this.currentSoundListOutput == null)
                return;

            // ListBoxModelをソート


            foreach (OutputItemInfo info in mCurrentSoundListOutput.ListInfoArray)
            {
                int index = mListItemListBox.Items.Add(info);
                mListItemListBox.SetItemChecked(index, info.ItemInfo.IsOutput);
            }

            UpdateItemListMoveButton();
        }

        // 出力するリストの「上／下へ移動ボタン」の更新
        private void UpdateListMoveButton()
        {
            bool isSelected = (mOutputListBox.SelectedIndex >= 0);

            mListMoveUpButton.Enabled = isSelected;
            mListMoveDownButton.Enabled = isSelected;
        }

        private void UpdateItemListMoveButton()
        {
            bool isListEnable = mOutputListBox.Enabled;
            bool isSelected = (mListItemListBox.SelectedIndex >= 0);

            bool enabled = isListEnable && isSelected;

            mItemMoveUpButton.Enabled = enabled;
            mItemMoveDownButton.Enabled = enabled;
        }

        private void SwapListItem(int selectedIndex, int newIndex)
        {
            mOutputList_SelectedIndexChanged_Enable = false;

            bool isChecked = mOutputListBox.GetItemChecked(selectedIndex);
            Object item = mOutputListBox.Items[selectedIndex];
            mOutputListBox.Items[selectedIndex] = mOutputListBox.Items[newIndex];
            mOutputListBox.SetItemChecked(selectedIndex, mOutputListBox.GetItemChecked(newIndex));
            mOutputListBox.Items[newIndex] = item;
            mOutputListBox.SetItemChecked(newIndex, isChecked);

            mOutputListBox.SelectedIndex = newIndex;

            mOutputList_SelectedIndexChanged_Enable = true;
        }

        private void mListMoveUpButton_Click(object sender, EventArgs e)
        {
            int selectedIndex = mOutputListBox.SelectedIndex;
            if (selectedIndex < 0)
                return;
            if (selectedIndex == 0)
                return;

            int newIndex = selectedIndex - 1;
            SwapListItem(selectedIndex, newIndex);
            UpdateListMoveButton();
        }

        private void mListMoveDownButton_Click(object sender, EventArgs e)
        {
            int selectedIndex = mOutputListBox.SelectedIndex;
            if (selectedIndex < 0)
                return;
            if (selectedIndex == mOutputListBox.Items.Count - 1)
                return;

            int newIndex = selectedIndex + 1;
            SwapListItem(selectedIndex, newIndex);
            UpdateListMoveButton();
        }

        private void SwapItemListItem(int selectedIndex, int newIndex)
        {
            mListItemListBox_SelectedIndexChanged_Enable = false;

            bool isChecked = mListItemListBox.GetItemChecked(selectedIndex);
            Object item = mListItemListBox.Items[selectedIndex];
            mListItemListBox.Items[selectedIndex] = mListItemListBox.Items[newIndex];
            mListItemListBox.SetItemChecked(selectedIndex, mListItemListBox.GetItemChecked(newIndex));
            mListItemListBox.Items[newIndex] = item;
            mListItemListBox.SetItemChecked(newIndex, isChecked);

            mListItemListBox.SelectedIndex = newIndex;

            mListItemListBox_SelectedIndexChanged_Enable = true;
        }

        private void mItemMoveUpButton_Click(object sender, EventArgs e)
        {
            int selectedIndex = mListItemListBox.SelectedIndex;
            if (selectedIndex < 0)
                return;
            if (selectedIndex == 0)
                return;

            int newIndex = selectedIndex - 1;
            SwapItemListItem(selectedIndex, newIndex);
            UpdateItemListMoveButton();
            UpdateListItemListBoxData();
        }

        private void mItemMoveDownButton_Click(object sender, EventArgs e)
        {
            int selectedIndex = mListItemListBox.SelectedIndex;
            if (selectedIndex < 0)
                return;
            if (selectedIndex == mListItemListBox.Items.Count - 1)
                return;

            int newIndex = selectedIndex + 1;
            SwapItemListItem(selectedIndex, newIndex);
            UpdateItemListMoveButton();
            UpdateListItemListBoxData();
        }

        ///--------------------------------
        /// <summary>
        /// 出力するリスト項目のリストボックスで選択変更
        /// </summary>
        bool mListItemListBox_SelectedIndexChanged_Enable = true;
        private void mListItemListBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (!mListItemListBox_SelectedIndexChanged_Enable)
                return;

            UpdateItemListMoveButton();
        }


        ///--------------------------------
        /// <summary>
        /// 出力するリスト項目のリストボックスを設定データに反映
        /// </summary>
        private void UpdateListItemListBoxData()
        {
            if (mCurrentSoundListOutput == null)
                return;
            if (currentSoundListOutput == null)
                return;

            SoundListOutput setting = currentSoundListOutput;

            this.mCurrentSoundListOutput.ListInfoArray.Clear();             // ListBoxControlModel
            this.mCurrentSoundListOutput.ItemInfo.ColumnOutputs.Clear();    // Model

            int index = 0;
            foreach (OutputItemInfo info in mListItemListBox.Items)
            {
                this.mCurrentSoundListOutput.ListInfoArray.Add(info);
                this.mCurrentSoundListOutput.ItemInfo.ColumnOutputs.Add(info.ItemInfo);

                info.ItemInfo.IsOutput = mListItemListBox.GetItemChecked(index);
                index++;
            }

            // ★多分不要
            /*
            setting.SoundListColumnInfo.Index = mSoundListIndexInfo.ItemInfo;
            setting.SoundListColumnInfo.Name = mSoundListNameInfo.ItemInfo;
            setting.SoundListColumnInfo.Type = mSoundListTypeInfo.ItemInfo;
            setting.SoundListColumnInfo.Volume = mSoundListVolumeInfo.ItemInfo;
            setting.SoundListColumnInfo.Player = mSoundListPlayerInfo.ItemInfo;
            setting.SoundListColumnInfo.ActorPlayer = mSoundListActorPlayerInfo.ItemInfo;
            setting.SoundListColumnInfo.PlayerPrio = mSoundListPlayerPrioInfo.ItemInfo;
            setting.SoundListColumnInfo.Comment = mSoundListCommentInfo.ItemInfo;

            setting.GroupListColumnInfo.Index = mGroupListIndexInfo.ItemInfo;
            setting.GroupListColumnInfo.Name = mGroupListNameInfo.ItemInfo;
            setting.GroupListColumnInfo.Comment = mGroupListCommentInfo.ItemInfo;

            setting.PlayerListColumnInfo.Index = mPlayerListIndexInfo.ItemInfo;
            setting.PlayerListColumnInfo.Name = mPlayerListNameInfo.ItemInfo;
            setting.PlayerListColumnInfo.SoundLimit = mPlayerListSoundLimitInfo.ItemInfo;
            setting.PlayerListColumnInfo.Comment = mPlayerListCommentInfo.ItemInfo;

            setting.BankListColumnInfo.Index = mBankListIndexInfo.ItemInfo;
            setting.BankListColumnInfo.Name = mBankListNameInfo.ItemInfo;
            setting.BankListColumnInfo.Comment = mBankListCommentInfo.ItemInfo;
             */
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void UpdateSoundArchiveStreamDisplay()
        {
            int size = StreamBufferSize * StreamChannel * StreamBufferTimes;

            this.label_SoundArchiveStreamBufferSizeValue.Text =
                string.Format("{0:#,0} Byte ({1:d} KB)", size, size / 1024);

            int aChannelBufferSize = StreamBufferDefalutSize * StreamBufferTimes;

            // 32kHz * [ピッチ] * [1サンプルあたりのバイト量]
            float time = ((float)aChannelBufferSize) / (32000 * 1 * 8 / 14);
            this.label_SoundArchiveStreamBufferConversionADPCMValue.Text =
                string.Format("{0:f1}", time);

            // 32kHz * [ピッチ] * [1サンプルあたりのバイト量]
            time = ((float)aChannelBufferSize) / (32000 * 1 * 2);
            this.label_SoundArchiveStreamBufferConversionPCM16Value.Text =
                string.Format("{0:f1}", time);
        }


        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void mListItemListBox_Validated(object sender, EventArgs e)
        {
            UpdateListItemListBoxData();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnCheckedChangedEnablePrefix(object sender, EventArgs e)
        {
            UpdateControls();
            UpdateSampleItemName();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnUpperLowerProcessChanged(object sender, EventArgs e)
        {
            UpdateControls();
            UpdateSampleItemName();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnIllegalLetterProcessChanged(object sender, EventArgs e)
        {
            UpdateControls();
            UpdateSampleItemName();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnCheckedChangedEnableNameDelimiter(object sender, EventArgs e)
        {
            UpdateControls();
            UpdateSampleItemName();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnTextChangedNameDelimiter(object sender, EventArgs e)
        {
            UpdateControls();
            UpdateSampleItemName();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnClick_CancelButton(object sender, EventArgs e)
        {
            if (this.mSoundListOutputSettingListListBox.Items.Count !=
                this.saveHTMLSoundListListBoxItemsCount)
            {
                isCancel = true;
            }
        }

        private void OnClickButtonSndEditDefaultPort(object sender, EventArgs e)
        {
            numericUpDown_SndEditSyncPort.Value = 54086;
            numericUpDown_SndEditFuncPort.Value = 54087;
            numericUpDown_SndEditCtrlPort.Value = 54088;
        }

        private void OnClick_SndEditChannelDefault(object sender, EventArgs e)
        {
            this.SndEditChannelSync_NumericUpDown.Value = 0x0A;
            this.SndEditChannelFunc_NumericUpDown.Value = 0x0B;
            this.SndEditChannelCtrl_NumericUpDown.Value = 0x0C;
        }

        private void OnSelectedIndexChanged_comboBox_SoundArchiveStreamBufferTimes(object sender, EventArgs e)
        {
            this.UpdateSoundArchiveStreamDisplay();
        }

        private void OnFileSavePreCommandPathOpenFile(object sender, EventArgs e)
        {
            OpenFileDialog dialog = this.CreateOpenFileDialog(this.FileSavePreCommandPath_textBox.Text);
            if (dialog.ShowDialog() == DialogResult.OK)
            {
                this.FileSavePreCommandPath_textBox.Text = dialog.FileName;
            }
        }

        private void OnFileSavePostCommandPathOpenFile(object sender, EventArgs e)
        {
            OpenFileDialog dialog = this.CreateOpenFileDialog(this.FileSavePostCommandPath_textBox.Text);
            if (dialog.ShowDialog() == DialogResult.OK)
            {
                this.FileSavePostCommandPath_textBox.Text = dialog.FileName;
            }
        }

        private void referSoxPathButton_Click(object sender, EventArgs e)
        {
            var initialDirectory = string.IsNullOrEmpty(this.userSoxPathTextBox.Text)
                ? string.Empty
                : Path.GetDirectoryName(this.userSoxPathTextBox.Text);

            if (!Path.IsPathRooted(initialDirectory))
            {
                var projectDirPath = Path.GetDirectoryName(this.ProjectFilePath);
                initialDirectory = Path.GetDirectoryName(
                    PathUtility.GetFullPath(Path.Combine(projectDirPath, initialDirectory)));
            }

            var dialog = new OpenFileDialog()
            {
                Title = Resources.MessageResource.DialogTitle_SelectSoxPathDialog,
                InitialDirectory = initialDirectory,
            };

            if (dialog.ShowDialog() != DialogResult.OK)
            {
                return;
            }

            this.userSoxPathTextBox.Text = dialog.FileName;
        }

        private void useUserSox_CheckedChanged(object sender, EventArgs e)
        {
            this.UpdateUserSoxPathControls();
        }

        private void UpdateUserSoxPathControls()
        {
            this.userSoxPathTextBox.Enabled = this.useUserSox.Checked;
            this.referSoxPathButton.Enabled = this.useUserSox.Checked;
        }

        private OpenFileDialog CreateOpenFileDialog(string path)
        {
            OpenFileDialog dialog = new OpenFileDialog();

            dialog.RestoreDirectory = false;
            dialog.Filter = "Program files (*.exe;*.bat;*.com)|*.exe;*.bat;*.com|All files (*.*)|*.*";
            if (string.IsNullOrEmpty(path) == false)
            {
                dialog.InitialDirectory = Path.GetDirectoryName(path);
            }

            return dialog;
        }

        private void OnTextChangedIntegratedLoudnessThreshold(object sender, EventArgs e)
        {
            try
            {
                FloatParameterValue value = new FloatParameterValue();
                value.Parse(this.textBox_IntegratedLoudnessThreshold.Text);

                ValidationResult result = value.Validate();

                if (result.IsValid == false)
                {
                    this.ShowError(this.textBox_IntegratedLoudnessThreshold, result.ToString());
                    return;
                }
            }
            catch
            {
                this.ShowError(this.textBox_IntegratedLoudnessThreshold, MessageResource.Message_InvalidValue);
                return;
            }

            this.HideError(this.textBox_IntegratedLoudnessThreshold);

            return;
        }

        private void OnTextChangedSequenceMaxMeasureDuration(object sender, EventArgs e)
        {
            try
            {
                IntParameterValue value = new IntParameterValue();
                value.Parse(this.textBox_SequenceMaxMeasureDuration.Text);

                ValidationResult result = value.Validate();

                if (result.IsValid == false)
                {
                    this.ShowError(this.textBox_SequenceMaxMeasureDuration, result.ToString());
                    return;
                }
            }
            catch
            {
                this.ShowError(this.textBox_SequenceMaxMeasureDuration, MessageResource.Message_InvalidValue);
                return;
            }

            this.HideError(this.textBox_SequenceMaxMeasureDuration);

            return;
        }

        private void OnClick_EnabledIntegratedLoudnessThreshold(object sender, EventArgs e)
        {
            this.EnabledIntegratedLoudnessThreshold = this.checkBox_EnabledIntegratedLoudnessThreshold.Checked;
        }

        private void OnMouseDown_ListView(object sender, MouseEventArgs e)
        {
            this.listViewHitTestInfo = this.listView.HitTest(e.X, e.Y);
        }

        private void OnMouseUp_ListView(object sender, MouseEventArgs e)
        {
            if (this.listViewHitTestInfo != null &&
                this.listViewHitTestInfo.Item != null &&
                this.listViewHitTestInfo.Item.Selected == true)
            {
                var result = this.listView.HitTest(e.X, e.Y);
                if (this.listViewHitTestInfo.SubItem == result.SubItem && (string)result.SubItem.Tag == ColumnOutputType)
                {
                    var soundSet = result.Item.Tag as SoundSet;
                    var value = this.soundArchiveOutputType[soundSet];

                    var index = Enum.GetValues(typeof(SoundArchiveOutputTypes))
                        .Cast<SoundArchiveOutputTypes>()
                        .ToList()
                        .IndexOf(value);

                    this.ShowInplaceComboBox(result.SubItem.Bounds, index, this.listView);
                    this.editingSoundSet = soundSet;
                }
            }
        }

        private void CheckedChangedEnabledConnection(object sender, EventArgs e)
        {
            this.UpdateConnectionTimeOutEnabled();
        }

        private void UpdateConnectionTimeOutEnabled()
        {
            bool enabled = this.EnabledConnectionTimeOut_CheckBox.Checked;

            this.ConnectionTimeOut_Label.Enabled = enabled;
            this.ConnectionTimeOutInfo_Label.Enabled = enabled;
            this.ConnectionTimeOut_NumericUpDown.Enabled = enabled;
        }

        private void ShowInplaceComboBox(Rectangle bounds, int selectedIndex, SoundArchiveListView parent)
        {
            var inplaceComboBox = new ComboBox();

            inplaceComboBox.DropDownStyle = ComboBoxStyle.DropDownList;
            inplaceComboBox.Leave += (s, e) =>
                {
                    var comboBox = s as ComboBox;

                    if (this.editingSoundSet != null)
                    {
                        var value = (SoundArchiveOutputTypes)Enum.GetValues(typeof(SoundArchiveOutputTypes))
                            .Cast<SoundArchiveOutputTypes>()
                            .ToArray()
                            .GetValue(comboBox.SelectedIndex);

                        this.soundArchiveOutputType[this.editingSoundSet] = value;

                        var item = this.listView.Items
                            .Cast<ListViewItem>()
                            .Where(i => i.Tag == this.editingSoundSet)
                            .First();

                        for (int index = 0; index < item.SubItems.Count; index++)
                        {
                            if ((string)item.SubItems[index].Tag == ColumnOutputType)
                            {
                                item.SubItems[1].Text = this.OutputTypeToText(value);
                            }
                        }
                    }

                    this.editingSoundSet = null;

                    comboBox.Visible = false;
                    comboBox.Parent = null;
                };

            Enum.GetValues(typeof(SoundArchiveOutputTypes))
                .Cast<SoundArchiveOutputTypes>()
                .ToList()
                .ForEach(v => inplaceComboBox.Items.Add(this.OutputTypeToText(v)));

            inplaceComboBox.Bounds = bounds;
            inplaceComboBox.SelectedIndex = selectedIndex;
            inplaceComboBox.Parent = parent;
            inplaceComboBox.BringToFront();
            inplaceComboBox.Visible = true;
            inplaceComboBox.Focus();
            inplaceComboBox.DroppedDown = true;
        }
    }

    ///
    public class HTMLSoundListPathValidator
    {
        public static bool Validate(string cardinalFilePath, string filePath)
        {
            try
            {
                string directory = null;

                if (Path.IsPathRooted(filePath) == true)
                {
                    directory = Path.GetDirectoryName(filePath);
                }
                else
                {
                    string cardinalDirectory = Path.GetDirectoryName(cardinalFilePath);
                    directory = Path.GetDirectoryName
                        (Path.Combine(cardinalDirectory, filePath).GetFullPath());
                }
#if false
                if( Directory.Exists( directory) == false )
                {
                    throw new ApplicationException();
                }
#endif
                // 禁則文字チェック
                Path.GetFullPath(filePath);

                string fileName = Path.GetFileName(filePath);

                if (fileName.Length == 0)
                {
                    throw new ApplicationException();
                }

                return true;
            }
            catch
            {
                return false;
            }
        }
    }

    public class SoundArchiveListView : System.Windows.Forms.ListView
    {
        private System.ComponentModel.Container components = null;

        public SoundArchiveListView()
        {
            InitializeComponent();
        }

        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                if (components != null)
                {
                    components.Dispose();
                }
            }
            base.Dispose(disposing);
        }

        private void InitializeComponent()
        {
            components = new System.ComponentModel.Container();
        }

        private const int WM_HSCROLL = 0x114;
        private const int WM_VSCROLL = 0x115;

        protected override void WndProc(ref Message msg)
        {
            switch (msg.Msg)
            {
                case WM_HSCROLL:
                case WM_VSCROLL:
                    this.Focus();
                    break;
            }
            base.WndProc(ref msg);
        }
    }
}

