﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections;
    using System.ComponentModel;
    using System.Drawing;
    using System.IO;
    using System.Linq;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Framework.Utilities;

    ///--------------------------------------------------------------------------
    /// <summary>
    /// CreateSoundSetDialog の概要の説明です。
    /// </summary>
    public partial class CreateSoundSetDialog : CreateProjectItemDialog
    {
        ///--------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CreateSoundSetDialog(SoundProjectService projectService) : base(projectService)
        {
            InitializeComponent();
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private string FilePath
        {
            get
            {
                return SoundSetFileQuester.CreateFilePath
                    (textBox_Directory.Text, textBox_Name.Text);
            }
        }

        /// <summary>
        /// サウンドセットドキュメントを作成します。
        /// </summary>
        public DocumentReference CreateSoundSetDocument()
        {
            if (DialogResult != DialogResult.OK) { return null; }

            DocumentReference documentRef = ApplicationBase.Instance.DocumentService.CreateDocument(
                ApplicationBase.Instance.Traits.IntermediateOutputTraits.SoundSetDocumentTypeName);
            documentRef.Document.Resource = new FileResource(FilePath);


            SoundSet soundSet = new SoundSet();
            (documentRef.Document as SoundSetDocument).SoundSet = soundSet;
            soundSet.Name = textBox_Name.Text;

            soundSet.Children.Add(new StreamSoundPack());
            soundSet.Children.Add(new WaveSoundSetPack());
            soundSet.Children.Add(new SequenceSoundPack());
            soundSet.Children.Add(new SequenceSoundSetPack());
            soundSet.Children.Add(new SoundSetBankPack());
            soundSet.Children.Add(new WaveArchivePack());
            soundSet.Children.Add(new GroupPack());
            soundSet.Children.Add(new PlayerPack());

            if (SaveHelper.Save(() => FormsApplication.Instance.DocumentService.SaveDocument(documentRef.Document), new string[] { FilePath }) == false)
            {
                documentRef.Close();
                documentRef = null;
            }

            return documentRef;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override Control GetAcceptButton()
        {
            return this.button_OK;
        }


        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override bool CanCreate
        {
            get
            {
                bool name = InspectName();
                bool directory = InspectDirectory();

                if (name == false || directory == false)
                {
                    return false;
                }

                try
                {
                    //既に同じパスのサウンドセットが登録されているのか？
                    if (ContainsSoundSet(FilePath) != false)
                    {
                        MessageBox.Show(MessageResource.Message_SoundSetAlreadyExisting);
                        return false;
                    }

                    //
                    if (File.Exists(FilePath) != false)
                    {
                        string message = MessageResource.Message_ConfirmOverwriteFile;
                        if (InquireYesNo(message) != AppMessageBoxResult.OK)
                        {
                            return false;
                        }
                    }
                    return true;
                }
                catch (UriFormatException)
                {
                    ShowError(textBox_Directory, MessageResource.Message_DirectoryNotFound);
                }
                catch (Exception)
                {
                }
                return false;
            }
        }

        private bool ContainsSoundSet(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            return this.ProjectService.ContainsSoundSetDocument(filePath);
        }

        ///--------------------------------
        /// <summary>
        /// 名前の検査
        /// </summary>
        private bool InspectName()
        {
            string name = textBox_Name.Text;

            if (name == String.Empty)
            {
                ShowError(textBox_Name, MessageResource.Message_InputFileName);
                return false;
            }

            ValidationResult result = FilePathValidator.ValidateFileName(name);
            if (!result.IsValid)
            {
                ShowError(textBox_Name, result.ToString());
                return false;
            }

            HideError(textBox_Name);
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// ディレクトリの検査
        /// </summary>
        private bool InspectDirectory()
        {
            string directory = textBox_Directory.Text;

            if (directory == String.Empty)
            {
                ShowError(textBox_Directory, MessageResource.Message_InputPath);
                return false;
            }

            if (Directory.Exists(directory) == false)
            {
                ShowError(textBox_Directory, MessageResource.Message_DirectoryNotFound);
                return false;
            }

            HideError(textBox_Directory);
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// パスの参照
        /// </summary>
        private void OnReference(object sender, System.EventArgs e)
        {
            FolderBrowserDialog dialog = new FolderBrowserDialog();

            dialog.SelectedPath = Path.GetDirectoryName(ProjectService.ProjectDocument.Resource.Key);

            if (dialog.ShowDialog() != DialogResult.OK)
            {
                return;
            }

            textBox_Directory.Text = dialog.SelectedPath;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnClosing(object sender, FormClosingEventArgs e)
        {
            if (DialogResult == DialogResult.OK)
            {
                if (ProjectFilePathUtility.ConfirmFilePath(FilePath) == false)
                {
                    e.Cancel = true;
                    return;
                }
            }
        }

        private void OnTextChanged(object sender, EventArgs e)
        {
            UpdateControls();
        }
    }
}
