﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.IO;
    using System.Diagnostics;
    using System.Drawing;
    using System.Collections;
    using System.ComponentModel;
    using System.Windows.Forms;
    using System.Resources;
    using System.Reflection;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Framework.Utilities;
    using CoreResources = NintendoWare.SoundFoundation.Resources;

    ///--------------------------------------------------------------------------
    /// <summary>
    /// CreateBankDialog の概要の説明です。
    /// </summary>
    public partial class CreateBankDialog : CreateSoundSetItemDialog
    {
        private DocumentService _documentService;
        private readonly SoundSetBankBase _bank = ApplicationBase.Instance.CreateComponentService.Create<SoundSetBankBase>();

        ///--------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CreateBankDialog(DocumentService documentService, SoundProjectService projectService)
            : base(projectService)
        {
            if (null == documentService) { throw new ArgumentNullException("documentService"); }
            _documentService = documentService;

            InitializeComponent();

            //
            SetUniqueNameToTextBox(textBox_Label, projectService.Project,
                                    ProjectService.ComponentDictionary,
                                    ProjectService.Project.BankNamePrefix);

            //
            radioButton_NewBankFile.Checked = true;

            InspectNewBankFileName();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool DoCreateBankFile
        {
            get { return radioButton_NewBankFile.Checked != false ? true : false; }
        }

        ///--------------------------------
        /// <summary>
        /// ラベルの取得
        /// </summary>
        private string Label
        {
            get { return textBox_Label.Text; }
        }

        ///--------------------------------
        /// <summary>
        /// ファイルパスの取得
        /// </summary>
        private string FilePath
        {
            get
            {
                if (DoCreateBankFile != false)
                {
                    return NewBankFilePath;
                }
                return textBox_PathSpecifyBankFile.Text;
            }
        }

        /// <summary>
        /// 作成したバンクを取得します。
        /// </summary>
        public SoundSetBankBase Bank
        {
            get
            {
                _bank.Name = Label;
                _bank.FilePath = FilePath;

                return _bank;
            }
        }

        ///--------------------------------
        /// <summary>
        /// 新規バンクファイルのファイルパスの取得
        /// </summary>
        private string NewBankFilePath
        {
            get
            {
                string filePath = Path.Combine(textBox_DirectoryNewBankFile.Text,
                                                 textBox_FileNameNewBankFile.Text);
                filePath += "." +
                    ApplicationBase.Instance.Traits.IntermediateOutputTraits.BankFileExtension;

                return filePath.GetFullPath();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override bool CanCreate
        {
            get
            {
                bool properLabel = false;
                bool properNewBankFile = false;
                bool properSpecifyBankFile = false;

                properLabel = InspectLabel();
                properNewBankFile = (InspectNewBankFileName() != false &&
                                                            InspectNewBankFileDirectory() != false ?
                                                            true : false);
                properSpecifyBankFile = InspectSpecifyBankFilePath();

                //
                if (properLabel == false)
                {
                    return false;
                }

                if (DoCreateBankFile != false)
                {
                    if (properNewBankFile == false) { return false; }
                }
                else
                {
                    if (properSpecifyBankFile == false) { return false; }
                }

                return true;
            }
        }

        ///--------------------------------
        /// <summary>
        /// バンクファイルの新規/既存の切り替え
        /// </summary>
        private void OnCheckedChangedBankFile(object sender, System.EventArgs e)
        {
            if (radioButton_NewBankFile.Checked != false)
            {
                panel_NewBankFile.Enabled = true;
                panel_SpecifyBankFile.Enabled = false;
            }

            if (radioButton_SpecifyBankFile.Checked != false)
            {
                panel_NewBankFile.Enabled = false;
                panel_SpecifyBankFile.Enabled = true;
            }

            InspectNewBankFileName();
            InspectNewBankFileDirectory();
            InspectSpecifyBankFilePath();
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        /// 新規バンクファイルのディレクトリの選択
        /// </summary>
        private void OnClickReferenceNewBankDirectory(object sender, System.EventArgs e)
        {
            FolderBrowserDialog dialog = new FolderBrowserDialog();

            dialog.SelectedPath = Path.GetDirectoryName(ProjectService.ProjectDocument.Resource.Key);

            if (dialog.ShowDialog() != DialogResult.OK)
            {
                return;
            }

            textBox_DirectoryNewBankFile.Text = dialog.SelectedPath;
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        /// 既存のバンクファイルの選択
        /// </summary>
        private void OnClickReferenceSpecifyBankFile(object sender, System.EventArgs e)
        {
            string filePath = null;
            DocumentReference reference = null;

            try
            {
                if ((filePath = BankFileQuester.QueryLoad()) == null)
                {
                    return;
                }

                reference = _documentService.OpenDocument(new FileResource(filePath));
                if (!(reference.Document is BankDocument))
                {
                    throw new ApplicationException(
                        string.Format(CoreResources.MessageResource.Message_UnsupportedFileFormat, filePath));
                }

                textBox_PathSpecifyBankFile.Text = filePath;

                UpdateControls();
            }
            catch (Exception ex)
            {
                ShowErrorText(textBox_PathSpecifyBankFile, ex.Message);
            }
            finally
            {
                if (null != reference)
                {
                    reference.Close();
                }
            }
        }

        ///--------------------------------
        /// <summary>
        /// ラベルの検査
        /// </summary>
        private bool InspectLabel()
        {
            return ValidateName(textBox_Label.Text, textBox_Label);
        }

        ///--------------------------------
        /// <summary>
        /// 新規バンクファイルのファイル名の検査
        /// </summary>
        private bool InspectNewBankFileName()
        {
            TextBox textBox = textBox_FileNameNewBankFile;

            if (DoCreateBankFile != false)
            {

                if (textBox.Text.Length <= 0)
                {
                    button_CreateLabel.Enabled = false;
                    ShowError(textBox, MessageResource.Message_InputFileName);
                    return false;
                }

                ValidationResult result = FilePathValidator.ValidateFileName(textBox.Text);
                if (!result.IsValid)
                {
                    button_CreateLabel.Enabled = false;
                    ShowError(textBox, result.ToString());
                    return false;
                }

                button_CreateLabel.Enabled = true;
            }

            HideError(textBox);
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// 新規バンクファイルのディレクトリの検査
        /// </summary>
        private bool InspectNewBankFileDirectory()
        {
            if (DoCreateBankFile != false)
            {
                if (Directory.Exists(textBox_DirectoryNewBankFile.Text) == false)
                {
                    ShowError(textBox_DirectoryNewBankFile, MessageResource.Message_DirectoryNotFound);
                    return false;
                }
            }

            HideError(textBox_DirectoryNewBankFile);
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// 既存バンクファイルの検査
        /// </summary>
        private bool InspectSpecifyBankFilePath()
        {
            if (DoCreateBankFile == false)
            {
                if (File.Exists(textBox_PathSpecifyBankFile.Text) == false)
                {
                    button_CreateLabel.Enabled = false;
                    ShowError(textBox_PathSpecifyBankFile, MessageResource.Message_FileNotFound);
                    return false;
                }

                button_CreateLabel.Enabled = true;
            }

            HideError(textBox_PathSpecifyBankFile);
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// 確定に使用するボタンの取得
        /// </summary>
        protected override Control GetAcceptButton()
        {
            return button_OK;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnTextChanged(object sender, System.EventArgs e)
        {
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnValidatedLabel(object sender, System.EventArgs e)
        {
            InspectLabel();
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnValidatedFileNameNewBankFile(object sender, System.EventArgs e)
        {
            InspectNewBankFileName();
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnValidatedDirectoryNewBankFile(object sender, System.EventArgs e)
        {
            InspectNewBankFileDirectory();
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnValidatedSpecifyBankFile(object sender, System.EventArgs e)
        {
            InspectSpecifyBankFilePath();
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnClosing(object sender, FormClosingEventArgs e)
        {
            if (DialogResult == DialogResult.OK)
            {

                if (ProjectFilePathUtility.ConfirmFilePath(FilePath) == false)
                {
                    e.Cancel = true;
                    return;
                }

                //新規にバンクファイルを作成するのか？
                //既にファイルが存在するならば、上書きしてよいのか問い合わせる
                if (DoCreateBankFile != false)
                {
                    if (File.Exists(FilePath) != false)
                    {
                        if (InquireYesNo(MessageResource.Message_ConfirmOverwriteFile) != AppMessageBoxResult.OK)
                        {
                            e.Cancel = true;
                        }
                    }
                }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnCreateLabel(object sender, EventArgs e)
        {
            string fileName = String.Empty;

            if (radioButton_NewBankFile.Checked != false)
            {
                fileName = textBox_FileNameNewBankFile.Text;
            }

            if (radioButton_SpecifyBankFile.Checked != false)
            {
                fileName = Path.GetFileNameWithoutExtension(textBox_PathSpecifyBankFile.Text);
            }

            textBox_Label.Text = CreateSoundSetItemNameFromFilePath(
                                    ProjectService.Project.BankNamePrefix, fileName);
        }
    }
}
