﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using NintendoWare.SoundFoundation.Core.Parameters;
using NintendoWare.SoundFoundation.Projects;
using NintendoWare.SoundFoundation.Windows.Forms;
using NintendoWare.SoundMaker.Framework.Resources;
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Imaging;
using System.Windows.Forms;
using System.Windows.Forms.VisualStyles;

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    /// <summary>
    /// ユーザーパラメータ
    /// </summary>
    public class ListPartUserParameterDrawer : ListPartULongDrawer
    {
        private Brush disabledTextBrush = null;

        /// <summary>
        ///
        /// </summary>
        public ListPartUserParameterDrawer()
        {
            this.disabledTextBrush = new SolidBrush(Color.FromArgb(255, 192, 192, 192));
        }

        /// <summary>
        ///
        /// </summary>
        public override string Name
        {
            get { return "UserParameter"; }
        }

        /// <summary>
        ///
        /// </summary>
        public override void Draw(ListDrawDescriptor desc)
        {
            if (Enableds != null &&
                Enableds[UserParameterSettingWatcher.ToIndex(desc.Name)] == false)
            {
                desc.ContentDrawer.DrawString
                    (desc, desc.Parameter.ToString(),
                      desc.Font, this.disabledTextBrush, desc.Bounds, Format);
            }
            else
            {
                base.Draw(desc);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public bool[] Enableds { get; set; }
    }

    /// <summary>
    /// ユーザーパラメータ（整数）
    /// </summary>
    public class ListPartUserParameterIntegerDrawer : ListPartIntegerDrawer
    {
        private Brush disabledTextBrush = null;

        /// <summary>
        ///
        /// </summary>
        public ListPartUserParameterIntegerDrawer()
        {
            this.disabledTextBrush = new SolidBrush(Color.FromArgb(255, 192, 192, 192));
        }

        /// <summary>
        ///
        /// </summary>
        public override string Name
        {
            get { return "UserParameterInteger"; }
        }

        /// <summary>
        ///
        /// </summary>
        public override void Draw(ListDrawDescriptor desc)
        {
            string name = UserParameterSettingWatcher.GetUserParameterName(desc.Name);
            if (Enableds != null &&
                Enableds[UserParameterSettingWatcher.ToIndex(name)] == false)
            {
                desc.ContentDrawer.DrawString
                    (desc, desc.Parameter.ToString(),
                      desc.Font, this.disabledTextBrush, desc.Bounds, Format);
            }
            else
            {
                base.Draw(desc);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public bool[] Enableds { get; set; }
    }

    /// <summary>
    /// ユーザーパラメータ（符号無し整数）
    /// </summary>
    public class ListPartUserParameterULongDrawer : ListPartULongDrawer
    {
        private Brush disabledTextBrush = null;

        /// <summary>
        ///
        /// </summary>
        public ListPartUserParameterULongDrawer()
        {
            this.disabledTextBrush = new SolidBrush(Color.FromArgb(255, 192, 192, 192));
        }

        /// <summary>
        ///
        /// </summary>
        public override string Name
        {
            get { return "UserParameterULong"; }
        }

        /// <summary>
        ///
        /// </summary>
        public override void Draw(ListDrawDescriptor desc)
        {
            string name = UserParameterSettingWatcher.GetUserParameterName(desc.Name);
            if (Enableds != null &&
                Enableds[UserParameterSettingWatcher.ToIndex(name)] == false)
            {
                desc.ContentDrawer.DrawString
                    (desc, desc.Parameter.ToString(),
                      desc.Font, this.disabledTextBrush, desc.Bounds, Format);
            }
            else
            {
                base.Draw(desc);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public bool[] Enableds { get; set; }
    }

    /// <summary>
    /// ユーザーパラメータ（少数）
    /// </summary>
    public class ListPartUserParameterFloatDrawer : ListPartFloatDrawer
    {
        private Brush disabledTextBrush = null;

        /// <summary>
        ///
        /// </summary>
        public ListPartUserParameterFloatDrawer()
        {
            this.disabledTextBrush = new SolidBrush(Color.FromArgb(255, 192, 192, 192));
        }

        /// <summary>
        ///
        /// </summary>
        public override string Name
        {
            get { return "UserParameterFloat"; }
        }

        /// <summary>
        ///
        /// </summary>
        public override void Draw(ListDrawDescriptor desc)
        {
            string name = UserParameterSettingWatcher.GetUserParameterName(desc.Name);
            if (Enableds != null &&
                Enableds[UserParameterSettingWatcher.ToIndex(name)] == false)
            {
                desc.ContentDrawer.DrawString
                    (desc, desc.Parameter.ToString(),
                      desc.Font, this.disabledTextBrush, desc.Bounds, Format);
            }
            else
            {
                base.Draw(desc);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public bool[] Enableds { get; set; }
    }

    /// <summary>
    /// ユーザーパラメータ（Boolean）
    /// </summary>
    public class ListPartUserParameterBooleanDrawer : ListPartCheckBoxDrawer
    {
        private ListPartTextDrawer textDrawer = new ListPartTextDrawer();

        /// <summary>
        ///
        /// </summary>
        public ListPartUserParameterBooleanDrawer()
        {
        }

        /// <summary>
        ///
        /// </summary>
        public override string Name
        {
            get { return "UserParameterBoolean"; }
        }

        /// <summary>
        ///
        /// </summary>
        public override void Draw(ListDrawDescriptor desc)
        {
            BoolParameterValue parameter = desc.Parameter as BoolParameterValue;
            if (parameter == null &&
                desc.Parameter is TextParameterValue)
            {
                this.textDrawer.Draw(desc);
                return;
            }

            ButtonState state = ButtonState.Normal;
            string name = UserParameterSettingWatcher.GetUserParameterName(desc.Name);
            if (Enableds != null &&
                Enableds[UserParameterSettingWatcher.ToIndex(name)] == false)
            {
                state |= ButtonState.Inactive;
            }
            if (parameter != null && (bool)parameter != false)
            {
                state |= ButtonState.Checked;
            }
            state = GetButtonState(desc, state);

            ControlPaint.DrawCheckBox(desc.Gc, desc.Bounds, state);
        }

        /// <summary>
        ///
        /// </summary>
        public bool[] Enableds { get; set; }
    }

    /// <summary>
    /// ユーザーパラメータ（AssistDropDown）
    /// </summary>
    public class ListPartUserParameterAssistDropDownDrawer : ListPartTextDrawer
    {
        private Brush disabledTextBrush = null;

        /// <summary>
        ///
        /// </summary>
        public ListPartUserParameterAssistDropDownDrawer()
        {
            this.disabledTextBrush = new SolidBrush(Color.FromArgb(255, 192, 192, 192));
        }

        /// <summary>
        ///
        /// </summary>
        public override string Name
        {
            get { return "UserParameterAssistDropDown"; }
        }

        /// <summary>
        ///
        /// </summary>
        public override void Draw(ListDrawDescriptor desc)
        {
            string name = UserParameterSettingWatcher.GetUserParameterName(desc.Name);
            if (Enableds != null &&
                Enableds[UserParameterSettingWatcher.ToIndex(name)] == false)
            {
                desc.ContentDrawer.DrawString
                    (desc, desc.Parameter.ToString(),
                      desc.Font, this.disabledTextBrush, desc.Bounds, Format);
            }
            else
            {
                base.Draw(desc);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public bool[] Enableds { get; set; }
    }

    /// <summary>
    /// プレビュープレイ
    /// </summary>
    public class ListPartPreviewPlayDrawer : IListPartDrawer
    {
        /// <summary>
        ///
        /// </summary>
        public string Name
        {
            get { return "PreviewPlay"; }
        }

        /// <summary>
        ///
        /// </summary>
        public void Draw(ListDrawDescriptor desc)
        {
            Image image = ImageResource.BitmapIconPlaySmall;

            CommonListItem item = desc.Item as CommonListItem;
            if (item == null ||
                IsTarget(item.Target) == false)
            {
                return;
            }

            if (image != null)
            {
                Rectangle bounds = desc.Bounds;
                if (VisualStyleRenderer.IsSupported)
                {
                    VisualStyleRenderer renderer =
                        new VisualStyleRenderer(VisualStyleElement.Button.PushButton.Normal);
                    bounds = new Rectangle(bounds.X, bounds.Y, bounds.Width + 1, bounds.Height + 1);
                    renderer.DrawBackground(desc.Gc, bounds);
                }
                else
                {
                    ControlPaint.DrawButton(desc.Gc, bounds, ButtonState.Normal);
                }
                ImageAttributes attr = new ImageAttributes();
                attr.SetColorKey(Color.Magenta, Color.Magenta);
                bounds = new Rectangle(
                                       bounds.X + (bounds.Width - image.Width) / 2,
                                       bounds.Y + (bounds.Height - image.Height) / 2,
                                       image.Width,
                                       image.Height
                                       );
                desc.Gc.DrawImage(
                                  image,
                                  bounds,
                                  0, 0, image.Width, image.Height,
                                  GraphicsUnit.Pixel,
                                  attr
                                  );
            }
        }

        /// <summary>
        ///
        /// </summary>
        public int DrawWidth(ListDrawDescriptor desc)
        {
            return desc.Bounds.Width + 2;
        }

        /// <summary>
        /// 再生ボタンを表示する対象なのか調べます。
        /// </summary>
        private bool IsTarget(Component component)
        {
            return (component is StreamSoundBase ||
                     component is WaveSoundBase ||
                     component is SequenceSoundBase) ? true : false;
        }
    }

    /// <summary>
    /// ミュート
    /// </summary>
    public class ListPartPreviewMuteDrawer : IListPartDrawer
    {
        /// <summary>
        ///
        /// </summary>
        public virtual string Name
        {
            get { return "PreviewMute"; }
        }

        /// <summary>
        ///
        /// </summary>
        public void Draw(ListDrawDescriptor desc)
        {
            CommonListItem item = desc.Item as CommonListItem;
            if (item == null ||
                IsTarget(item.Target) == false)
            {
                return;
            }

            //IParameterValue parameterValue = item.Target.Parameters["PreviewMute"];
            bool value = GetValue(item);
            Image image = GetImage(value);
            Rectangle bounds = desc.Bounds;
            ImageAttributes attr = new ImageAttributes();
            attr.SetColorKey(Color.Magenta, Color.Magenta);
            bounds = new Rectangle(bounds.X + (bounds.Width - image.Width) / 2,
                                    bounds.Y + (bounds.Height - image.Height) / 2,
                                    image.Width,
                                    image.Height);
            desc.Gc.DrawImage(image,
                               bounds,
                               0, 0, image.Width, image.Height,
                               GraphicsUnit.Pixel,
                               attr);
        }

        /// <summary>
        ///
        /// </summary>
        public int DrawWidth(ListDrawDescriptor desc)
        {
            return desc.Bounds.Width + 2;
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual bool GetValue(CommonListItem item)
        {
            IParameterValue value = item.Target.Parameters["PreviewMute"];
            return (bool)value.Value;
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual Image GetImage(bool value)
        {
            return (value != false ?
                     ImageResource.BitmapIconPreviewMuteEnable :
                     ImageResource.BitmapIconPreviewMuteDisable);
        }

        /// <summary>
        ///
        /// </summary>
        private bool IsTarget(Component component)
        {
            return (component is StreamSoundTrackBase) ? true : false;
        }
    }

    /// <summary>
    /// ソロ再生
    /// </summary>
    public class ListPartPreviewSoloPlayDrawer : ListPartPreviewMuteDrawer
    {
        /// <summary>
        ///
        /// </summary>
        public override string Name
        {
            get { return "PreviewSoloPlay"; }
        }

        /// <summary>
        ///
        /// </summary>
        protected override bool GetValue(CommonListItem item)
        {
            IParameterValue value = item.Target.Parameters["PreviewSoloPlay"];
            return (bool)value.Value;
        }

        /// <summary>
        ///
        /// </summary>
        protected override Image GetImage(bool value)
        {
            return (value != false ?
                     ImageResource.BitmapIconPreviewSoloPlayEnable :
                     ImageResource.BitmapIconPreviewSoloPlayDisable);
        }
    }

    /// <summary>
    /// 行ヘッダ
    /// </summary>
    public class ListPartCommonRowHeaderDrawer : IListPartDrawer, IListItemToolTipProvider
    {
        private static StringFormat _Format = new StringFormat();
        private static Dictionary<Type, Image> _Images = null;
        private static Image _DisabledImage = null;
        private static Image _ErrorImage = null;
        private static Image _ErrorSmallImage = null;
        private static Image _WarningSmallImage = null;

        /// <summary>
        ///
        /// </summary>
        static ListPartCommonRowHeaderDrawer()
        {
            _Format.Trimming = StringTrimming.EllipsisCharacter;
            _Format.LineAlignment = StringAlignment.Far;
            _Format.Alignment = StringAlignment.Far;
            _Format.FormatFlags = StringFormatFlags.NoWrap;

            //
            _Images = new Dictionary<Type, Image>();
            //_Images.Add(typeof(StreamSoundBase), ImageResource.BitmapIconStreamSound);
            _Images.Add(typeof(WaveSoundSetBase), ImageResource.BitmapIconWaveSoundSet);
            //_Images.Add(typeof(WaveSoundBase), ImageResource.BitmapIconWaveSound);
            _Images.Add(typeof(SequenceSoundSetBase), ImageResource.BitmapIconSequenceSoundSet);
            //_Images.Add(typeof(SequenceSoundBase), ImageResource.BitmapIconSequenceSound);
            //_Images.Add(typeof(SoundSetBankBase), ImageResource.BitmapIconBank);
            _Images.Add(typeof(WaveArchiveBase), ImageResource.BitmapIconWaveArchive);
            _Images.Add(typeof(GroupBase), ImageResource.BitmapIconGroup);
            _Images.Add(typeof(Instrument), ImageResource.BitmapIconInstrument);
            _Images.Add(typeof(PlayerBase), ImageResource.BitmapIconPlayer);
            _Images.Add(typeof(ImaginaryInstrument), ImageResource.BitmapIconInstrument);
            //_Images.Add(typeof(StreamSoundTrackBase), ImageResource.BitmapIconStreamSound);

            _Images.Add(typeof(StreamSoundPack), ImageResource.BitmapIconStreamSoundFolder);
            _Images.Add(typeof(WaveSoundSetPack), ImageResource.BitmapIconWaveSoundSetFolder);
            _Images.Add(typeof(SequenceSoundPack), ImageResource.BitmapIconSequenceSoundFolder);
            _Images.Add(typeof(SequenceSoundSetPack), ImageResource.BitmapIconSequenceSoundSetFolder);
            _Images.Add(typeof(SoundSetBankPack), ImageResource.BitmapIconBankFolder);
            _Images.Add(typeof(WaveArchivePack), ImageResource.BitmapIconWaveArchiveFolder);
            _Images.Add(typeof(GroupPack), ImageResource.BitmapIconGroupFolder);
            _Images.Add(typeof(PlayerPack), ImageResource.BitmapIconPlayerFolder);

            _DisabledImage = ImageResource.PngOverlayDisabled;
            _ErrorImage = ImageResource.BitmapIconItemError;
            _ErrorSmallImage = ImageResource.PngOverlayItemErrorSmall;
            _WarningSmallImage = ImageResource.PngOverlayWarningFindResultSmall;
        }

        /// <summary>
        ///
        /// </summary>
        public static void DrawItem(DrawItemEventArgs e, Font font, Component component, string text)
        {
            Graphics gc = e.Graphics;
            Brush brush = new SolidBrush(e.ForeColor);
            ImageAttributes attr = null;
            Rectangle bounds = e.Bounds;
            int textHeight = 0;
            Image image = ListPartCommonRowHeaderDrawer.GetImageStatic(component);
            int width = 16;
            int height = 16;
            int iconSpan = 0;

            e.DrawBackground();

            if (image != null)
            {
                attr = new ImageAttributes();
                attr.SetColorKey(Color.Magenta, Color.Magenta);

                gc.DrawImage(image, new Rectangle(bounds.X, bounds.Y, width, height),
                              0, 0, width, height,
                              GraphicsUnit.Pixel, attr);

                iconSpan = image.Width;
            }

            textHeight = (int)gc.MeasureString(text, font).Height;
            bounds.X += iconSpan;
            bounds.Y += (bounds.Height - textHeight) / 2;

            gc.DrawString(text, font, brush, bounds);
            e.DrawFocusRectangle();
        }

        /// <summary>
        ///
        /// </summary>
        public ListPartCommonRowHeaderDrawer()
        {
            ShowRowNumber = true;
        }

        /// <summary>
        ///
        /// </summary>
        public string Name
        {
            get { return "RowHeader"; }
        }

        /// <summary>
        ///
        /// </summary>
        public bool ShowRowNumber { get; set; }

        /// <summary>
        ///
        /// </summary>
        public void Draw(ListDrawDescriptor desc)
        {
            Image image = null;
            int width = 16;
            int height = 16;
            int baseX = 2;
            int baseY = desc.Bounds.Y + (desc.Bounds.Height / 2) - (height / 2);
            Component target = GetTarget(desc);

            ImageAttributes attr = new ImageAttributes();
            attr.SetColorKey(Color.Magenta, Color.Magenta);

            if ((image = GetImage(desc, target)) != null)
            {
                desc.Gc.DrawImage(image, new Rectangle(baseX, baseY, width, height),
                                   0, 0, width, height,
                                   GraphicsUnit.Pixel, attr);
            }

            //
            int ox = baseX + (int)((double)width * 0.4);
            int oy = baseY + (int)((double)height * 0.4);

            // 無効、エラー、警告の優先順位で条件を満たすアイコンを一つだけ描画します。
            // アイコンは重ねて描画することはしません。
            if (DoShowDisabledIcon(desc) == true)
            {
                desc.Gc.DrawImage(_DisabledImage, new Rectangle(ox, oy, width, height),
                                  0, 0, width, height,
                                  GraphicsUnit.Pixel, attr);
            }
            else if (DoShowErrorIcon(desc) == true)
            {
                desc.Gc.DrawImage(_ErrorSmallImage, new Rectangle(ox, oy, width, height),
                                  0, 0, width, height,
                                  GraphicsUnit.Pixel, attr);
            }
            else if (this.DoShowWarningIcon(desc) == true)
            {
                desc.Gc.DrawImage(_WarningSmallImage, new Rectangle(ox, oy, width, height),
                                  0, 0, width, height,
                                  GraphicsUnit.Pixel, attr);
            }

            //
            if (desc.Parameter != null && desc.Parameter.ToString() != "*")
            {
                desc.Gc.DrawString(desc.Parameter.ToString(),
                                    desc.Font, desc.TextBrush, desc.Bounds, _Format);
                return;
            }

            //
            if (ShowRowNumber != false)
            {
                desc.Gc.DrawString(desc.Index.ToString(),
                                    desc.Font, desc.TextBrush, desc.Bounds, _Format);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public int DrawWidth(ListDrawDescriptor desc)
        {
            string valueString = desc.Parameter.ToString();

            return (int)(desc.Gc.MeasureString(valueString, desc.Font).Width) + 2;
        }

        /// <summary>
        ///
        /// </summary>
        protected static Dictionary<Type, Image> Images
        {
            get
            {
                return _Images;
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected static Image GetImageStatic(Component component)
        {
            if (component == null)
            {
                return null;
            }

            if (component is StreamSoundBase)
            {
                return ImageResource.BitmapIconStreamSound;
            }

            if (component is StreamSoundTrackBase)
            {
                return ImageResource.BitmapIconStreamSoundTrack;
            }

            if (component is WaveSoundBase)
            {
                return ImageResource.BitmapIconWaveSound;
            }

            if (component is SequenceSoundBase)
            {
                return ImageResource.BitmapIconSequenceSound;
            }

            if (component is SoundSetBankBase)
            {
                return ImageResource.BitmapIconBank;
            }

            //
            Type componentType = component.GetType();

            if (_Images.ContainsKey(componentType) == false)
            {
                return null;
            }

            return _Images[componentType];
        }

        /// <summary>
        ///
        /// </summary>
        protected SoundProjectService ProjectService
        {
            get
            {
                return ApplicationBase.Instance.ProjectService;
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual Image GetImage(ListDrawDescriptor desc, Component component)
        {
            return GetImageStatic(component);
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual Component GetTarget(ListDrawDescriptor desc)
        {
            CommonListItem item = desc.Item as CommonListItem;
            if (item != null)
            {
                return item.Target;
            }

            ParameterListItem paraitem = desc.Item as ParameterListItem;
            ComponentReference reference = (ComponentReference)paraitem.Target;
            string name = reference.TargetName;
            if (ProjectService.ComponentDictionary.Contains(name) != false)
            {
                return (Component)ProjectService.ComponentDictionary[name][0];
            }
            return null;
        }

        protected virtual bool DoShowDisabledIcon(ListDrawDescriptor desc)
        {
            return CommonListDecorationEvaluator.GetEvaluator(desc.Item)?.DoShowDisabledIcon(null, desc.Item) ?? false;
        }

        protected virtual bool DoShowErrorIcon(ListDrawDescriptor desc)
        {
            return CommonListDecorationEvaluator.GetEvaluator(desc.Item)?.DoShowErrorIcon(null, desc.Item) ?? false;
        }

        protected virtual bool DoShowWarningIcon(ListDrawDescriptor desc)
        {
            return CommonListDecorationEvaluator.GetEvaluator(desc.Item)?.DoShowWarningIcon(null, desc.Item) ?? false;
        }

        ListItemToolTipInfo IListItemToolTipProvider.GetToolTip(string name, IListItem item)
        {
            return CommonListDecorationEvaluator.GetEvaluator(item)?.GetToolTip(null, item);
        }
    }

    /// <summary>
    /// テキスト
    /// </summary>
    public class ListPartTextDrawer : IListPartDrawer
    {
        private static Brush NATextBrush = SystemBrushes.GrayText;
        private static StringFormat _Format = new StringFormat();

        /// <summary>
        ///
        /// </summary>
        static ListPartTextDrawer()
        {
            _Format.Trimming = StringTrimming.EllipsisCharacter;
            _Format.LineAlignment = StringAlignment.Far;
            _Format.Alignment = StringAlignment.Near;
            _Format.FormatFlags = StringFormatFlags.NoWrap;
        }

        /// <summary>
        ///
        /// </summary>
        public virtual string Name
        {
            get { return "Text"; }
        }

        /// <summary>
        ///
        /// </summary>
        public virtual void Draw(ListDrawDescriptor desc)
        {
            ListTextDrawerArgument arg = desc.DrawerArgument as ListTextDrawerArgument;

            if (arg != null)
            {
                _Format.LineAlignment = arg.LineAlignment;
                _Format.Alignment = arg.Alignment;
            }
            else
            {
                _Format.LineAlignment = StringAlignment.Far;
                _Format.Alignment = StringAlignment.Near;
            }

            Brush textBrush = desc.TextBrush;
            if (desc.Parameter is TextWithNAParameterValue)
            {
                TextWithNAParameterValue parameter = desc.Parameter as TextWithNAParameterValue;
                if (parameter.Available == false)
                {
                    textBrush = NATextBrush;
                }
            }

            desc.ContentDrawer.DrawString(desc, desc.Parameter.ToString(), desc.Font, textBrush, desc.Bounds, _Format);
        }

        /// <summary>
        ///
        /// </summary>
        public int DrawWidth(ListDrawDescriptor desc)
        {
            string valueString = desc.Parameter.ToString();

            return (int)(desc.Gc.MeasureString(valueString, desc.Font).Width) + 2;
        }

        /// <summary>
        ///
        /// </summary>
        protected static StringFormat Format
        {
            get
            {
                return _Format;
            }
        }
    }

    /// <summary>
    /// 整数
    /// </summary>
    public class ListPartIntegerDrawer : IListPartDrawer
    {
        private static Brush NATextBrush = SystemBrushes.GrayText;
        private static StringFormat _Format = new StringFormat();

        /// <summary>
        ///
        /// </summary>
        static ListPartIntegerDrawer()
        {
            _Format.Trimming = StringTrimming.EllipsisCharacter;
            _Format.LineAlignment = StringAlignment.Far;
            _Format.Alignment = StringAlignment.Far;
            _Format.FormatFlags = StringFormatFlags.NoWrap;
        }

        /// <summary>
        ///
        /// </summary>
        public virtual string Name
        {
            get { return "Integer"; }
        }

        /// <summary>
        ///
        /// </summary>
        public virtual void Draw(ListDrawDescriptor desc)
        {
            Component target = null;
            Brush textBrush = desc.TextBrush;

            if (desc.Item != null)
            {
                if (desc.Item is ComponentListItem)
                {
                    ComponentListItem item = desc.Item as ComponentListItem;
                    if (item != null)
                    {
                        target = item.Target;
                    }
                }
            }

            //
            if (desc.Parameter is TextWithNAParameterValue)
            {
                TextWithNAParameterValue parameter = desc.Parameter as TextWithNAParameterValue;
                if (parameter.Available == false)
                {
                    textBrush = NATextBrush;
                }
            }

            //
            desc.ContentDrawer.DrawString
                (desc, desc.Parameter.ToString(),
                  desc.Font, textBrush, desc.Bounds, _Format);
        }

        /// <summary>
        ///
        /// </summary>
        public int DrawWidth(ListDrawDescriptor desc)
        {
            string valueString = desc.Parameter.ToString();

            return (int)(desc.Gc.MeasureString(valueString, desc.Font).Width) + 2;
        }

        /// <summary>
        ///
        /// </summary>
        protected static StringFormat Format
        {
            get
            {
                return _Format;
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class CommonDrawParameter
    {
        private static Color _RowColor0 = Color.FromArgb(0, 0, 0, 0);
        private static Color _RowColor1 = Color.FromArgb(255, 255, 170, 170);
        private static Color _RowColor2 = Color.FromArgb(255, 255, 224, 128);
        private static Color _RowColor3 = Color.FromArgb(255, 255, 255, 170);
        private static Color _RowColor4 = Color.FromArgb(255, 208, 255, 192);
        private static Color _RowColor5 = Color.FromArgb(255, 192, 236, 255);
        private static Color _RowColor6 = Color.FromArgb(255, 180, 221, 255);
        private static Color _RowColor7 = Color.FromArgb(255, 216, 216, 255);
        private static Color _RowColor8 = Color.FromArgb(255, 255, 221, 255);

        private static Color[] _RowColors = {
            _RowColor0,
            _RowColor1,
            _RowColor2,
            _RowColor3,
            _RowColor4,
            _RowColor5,
            _RowColor6,
            _RowColor7,
            _RowColor8,
        };

        private static Brush[] _RowBrushs = null;
        private static Brush[] _RowDarkBrushs = null;

        static CommonDrawParameter()
        {
            int count = RowColors.Length;

            _RowBrushs = new Brush[count];
            _RowDarkBrushs = new Brush[count];

            for (int index = 0; index < count; index++)
            {
                Color rowColor = RowColors[index];
                _RowBrushs[index] = new SolidBrush(rowColor);
                _RowDarkBrushs[index] = new SolidBrush(ControlPaint.Dark(rowColor));
            }
        }

        ///
        public static Color[] RowColors
        {
            get { return _RowColors; }
        }

        ///
        public static Brush[] RowBrushs
        {
            get { return _RowBrushs; }
        }

        ///
        public static Brush[] RowDarkBrushs
        {
            get { return _RowDarkBrushs; }
        }

        /// <summary>
        ///
        /// </summary>
        public static int GetColorIndex(IListItem origItem)
        {
            ComponentListItem item = origItem as ComponentListItem;

            if (item != null &&
                item.Target is GroupItemBase &&
                null != ((GroupItemBase)item.Target).Target)
            {
                return ((GroupItemBase)item.Target).Target.ColorIndex;
            }

            return (item != null && item.Target != null) ? item.Target.ColorIndex : 0;
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class CommonListContentDrawer : ListContentDrawer
    {
        private const int ImageWidth = 16;
        private const int ImageHeight = 16;
        private static readonly Image _ErrorImage = ImageResource.BitmapIconItemError;
        private static readonly Image _WarningImage = ImageResource.BitmapIconWarningFindResult;
        private static readonly Image _DisabledImage = ImageResource.BitmapIconItemDisabled;

        public CommonListContentDrawer()
        {
        }

        /// <summary>
        ///
        /// </summary>
        public override void DrawString(ListDrawDescriptor desc, string text, Font font, Brush brush, Rectangle bounds, StringFormat format)
        {
            if (DoShowDisabledIcon(desc))
            {
                DrawIcon(desc, ref bounds, _DisabledImage);
            }
            else if (DoShowErrorIcon(desc))
            {
                DrawIcon(desc, ref bounds, _ErrorImage);
            }
            else if (DoShowWarningIcon(desc))
            {
                DrawIcon(desc, ref bounds, _WarningImage);
            }

            int colorIndex = CommonDrawParameter.GetColorIndex(desc.Item);

            if (colorIndex >= 1 && colorIndex <= 8 && desc.Selected != false)
            {
                DrawOutlineText(desc.Gc, text, font,
                                 CommonDrawParameter.RowDarkBrushs[colorIndex],
                                 bounds, format);
            }
            else
            {
                base.DrawString(desc, text, font, brush, bounds, format);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void DrawOutlineText(Graphics g, string text, Font font, Brush brush, Rectangle rect, StringFormat format)
        {
            Rectangle workRect = rect;

            // 左上
            workRect.Offset(-1, -1);
            g.DrawString(text, font, brush, workRect, format);

            // 上
            workRect.Offset(1, 0);
            g.DrawString(text, font, brush, workRect, format);

            // 右上
            workRect.Offset(1, 0);
            g.DrawString(text, font, brush, workRect, format);

            // 右
            workRect.Offset(0, 1);
            g.DrawString(text, font, brush, workRect, format);

            // 右下
            workRect.Offset(0, 1);
            g.DrawString(text, font, brush, workRect, format);

            // 下
            workRect.Offset(-1, 0);
            g.DrawString(text, font, brush, workRect, format);

            // 左下
            workRect.Offset(-1, 0);
            g.DrawString(text, font, brush, workRect, format);

            // 左
            workRect.Offset(0, -1);
            g.DrawString(text, font, brush, workRect, format);

            // 中央
            g.DrawString(text, font, Brushes.WhiteSmoke, rect, format);
        }

        /// <summary>
        ///
        /// </summary>
        private void DrawIcon(ListDrawDescriptor desc, ref Rectangle bounds, Image image)
        {
            int x = bounds.X + bounds.Width - 16;
            int y = bounds.Y + (bounds.Height / 2) - (ImageHeight / 2);

            var attr = new ImageAttributes();
            attr.SetColorKey(Color.Magenta, Color.Magenta);

            desc.Gc.DrawImage(
                image,
                new Rectangle(x, y, ImageWidth, ImageHeight),
                0,
                0,
                ImageWidth,
                ImageHeight,
                GraphicsUnit.Pixel,
                attr);

            // アイコンを表示した分、文字列の描画範囲を狭めます。
            bounds.Width -= 16;
        }

        private bool DoShowDisabledIcon(ListDrawDescriptor desc)
        {
            return CommonListDecorationEvaluator.GetEvaluator(desc.Item)?.DoShowDisabledIcon(desc.Name, desc.Item) ?? false;
        }

        private bool DoShowErrorIcon(ListDrawDescriptor desc)
        {
            return CommonListDecorationEvaluator.GetEvaluator(desc.Item)?.DoShowErrorIcon(desc.Name, desc.Item) ?? false;
        }

        private bool DoShowWarningIcon(ListDrawDescriptor desc)
        {
            return CommonListDecorationEvaluator.GetEvaluator(desc.Item)?.DoShowWarningIcon(desc.Name, desc.Item) ?? false;
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class CommonListBackgroundDrawer : ListBackgroundDrawer
    {
        private static Brush _HeaderBrush = null;
        private static Brush _HeaderBrushFtSt = null;
        private static Brush _HeaderBrushFfSt = null;

        private Color parameterBarColor;
        private Color selectedParameterBarColor;
        private Brush parameterBarBrush = null;
        private Brush selectedParameterBarBrush = null;

        private HashSet<string> excludeParameterBar = new HashSet<string>();

        ///
        static CommonListBackgroundDrawer()
        {
            Color baseColor1;
            Color baseColor2;

            //
            baseColor1 = SystemColors.Control;
            baseColor2 = SystemColors.ControlDark;

            Color headerColor = Color.FromArgb
                ((baseColor1.R * 192 / 255) + (baseColor2.R * (255 - 192) / 255),
                 (baseColor1.G * 192 / 255) + (baseColor2.G * (255 - 192) / 255),
                 (baseColor1.B * 192 / 255) + (baseColor2.B * (255 - 192) / 255));

            //
            baseColor1 = Color.FromArgb(SystemColors.Highlight.R * 7 / 8,
                                         SystemColors.Highlight.G * 7 / 8,
                                         SystemColors.Highlight.B * 7 / 8);
            baseColor2 = headerColor;

            Color headerFtSt = Color.FromArgb
                ((baseColor1.R * 224 / 255) + (baseColor2.R * (255 - 224) / 255),
                 (baseColor1.G * 224 / 255) + (baseColor2.G * (255 - 224) / 255),
                 (baseColor1.B * 224 / 255) + (baseColor2.B * (255 - 224) / 255));

            //
            baseColor1 = SystemColors.ControlDark;
            baseColor2 = headerColor;

            Color headerFfSt = Color.FromArgb
                ((baseColor1.R * 224 / 255) + (baseColor2.R * (255 - 224) / 255),
                 (baseColor1.G * 224 / 255) + (baseColor2.G * (255 - 224) / 255),
                 (baseColor1.B * 224 / 255) + (baseColor2.B * (255 - 224) / 255));

            //
            _HeaderBrush = new SolidBrush(headerColor);
            _HeaderBrushFtSt = new SolidBrush(headerFtSt);
            _HeaderBrushFfSt = new SolidBrush(headerFfSt);
        }

        /// <summary>
        ///
        /// </summary>
        public CommonListBackgroundDrawer()
        {
            //
            this.excludeParameterBar.Add(ProjectParameterNames.DataSize);
            this.excludeParameterBar.Add(ProjectParameterNames.StreamSoundTrack.TrackNo);
        }

        /// <summary>
        ///
        /// </summary>
        public bool ShowParameterBar { get; set; }

        /// <summary>
        ///
        /// </summary>
        public int ParameterBarSize { get; set; }

        /// <summary>
        ///
        /// </summary>
        public Color ParameterBarColor
        {
            get
            {
                return this.parameterBarColor;
            }
            set
            {
                this.parameterBarColor = value;
                this.parameterBarBrush = new SolidBrush(value);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public Color SelectedParameterBarColor
        {
            get
            {
                return this.selectedParameterBarColor;
            }
            set
            {
                this.selectedParameterBarColor = value;
                this.selectedParameterBarBrush = new SolidBrush(value);
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected override void DrawInternal(ListDrawDescriptor desc)
        {
            Brush brush = null;

            if (desc.Name == "RowHeader")
            {
                if (desc.Selected != false)
                {
                    brush = desc.Focused != false ? _HeaderBrushFtSt : _HeaderBrushFfSt;
                }
                else
                {
                    brush = _HeaderBrush;
                }

                desc.Gc.FillRectangle(brush, desc.Bounds);
                return;
            }

            //
            int colorIndex = CommonDrawParameter.GetColorIndex(desc.Item);

            if (colorIndex >= 1 && colorIndex <= 8 && desc.Selected == false)
            {
                desc.Gc.FillRectangle(CommonDrawParameter.RowBrushs[colorIndex], desc.Bounds);
            }
            else
            {

                base.DrawInternal(desc);
            }

            //
            if (ShowParameterBar != false)
            {
                bool draw = false;
                int barWidth = 0;

                if (this.excludeParameterBar.Contains(desc.Name) == false)
                {
                    if (desc.Parameter is IntParameterValue)
                    {
                        var parameter = desc.Parameter as IntParameterValue;
                        int valueRange = parameter.MaxValue - parameter.MinValue;

                        if (valueRange > 0)
                        {
                            barWidth = desc.Bounds.Width *
                                (parameter.Value - parameter.MinValue) / valueRange;
                            draw = true;
                        }
                    }

                    if (desc.Parameter is FloatParameterValue)
                    {
                        var parameter = desc.Parameter as FloatParameterValue;
                        float valueRange = parameter.MaxValue - parameter.MinValue;

                        if (valueRange > 0.0F)
                        {
                            barWidth = (int)(desc.Bounds.Width * parameter.Value / valueRange);
                            draw = true;
                        }
                    }

                    if (draw != false)
                    {
                        int barHeight = (int)((float)Math.Max(desc.Bounds.Height - 4, 4) *
                                               ((float)ParameterBarSize / 100.0F));
                        int barY = desc.Bounds.Bottom - barHeight - 2;

                        brush = desc.Selected == false ?
                            this.parameterBarBrush :
                            this.selectedParameterBarBrush;

                        desc.Gc.FillRectangle(
                                              brush,
                                              desc.Bounds.Left,
                                              barY, //desc.Bounds.Top + 2,
                                              barWidth - 2,
                                              barHeight); //Math.Max(desc.Bounds.Height - 4, 4));
                    }
                }
            }
        }
    }
}
