﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections;
    using System.Drawing;
    using System.Windows.Forms;

    /// <summary>
    /// AdsrControl の概要の説明です。
    /// </summary>
    public partial class AdsrControl : UserControl
    {
        HScrollBar scrollBar;
        Panel panel;
        int attack = 127;
        int hold = 0;
        int decay = 127;
        int sustain = 127;
        int release = 127;
        ArrayList apoints;
        int attackStep;
        int decayStep;
        int releaseStep;
        bool updateLock = false;

        const int SUSTAIN_STEP = 60;
        const int STEP_PIXEL = 1;
        const int UPDATE_TIME = 2; // 曲線の粗さ
        const int UPDATE_MSEC = 3;
        const int SCALE_Y = 1000;
        const float MSEC_PER_STEP = 3;

        public AdsrControl()
        {
            UpdatePoints();

            panel = new Panel();
            panel.Parent = this;
            panel.Dock = DockStyle.Fill;
            panel.BorderStyle = BorderStyle.Fixed3D;
            panel.Paint += new PaintEventHandler(PanelOnPaint);
            panel.EnabledChanged += new EventHandler(PanelOnEnabledChanged);
            panel.SizeChanged += new EventHandler(PanelOnSizeChanged);
            panel.Enabled = true;

            scrollBar = new HScrollBar();
            scrollBar.Parent = this;
            scrollBar.Dock = DockStyle.Bottom;
            scrollBar.Scroll += new ScrollEventHandler(ScrollBarOnScroll);

            SetStyle(ControlStyles.AllPaintingInWmPaint
                      | ControlStyles.DoubleBuffer, true);
        }

        public int Attack
        {
            set
            {
                attack = value;
                UpdatePoints();
                UpdateScrollBar();
                panel.Invalidate();
            }
            get { return attack; }
        }
        public int Hold
        {
            set
            {
                hold = value;
                UpdatePoints();
                UpdateScrollBar();
                panel.Invalidate();
            }
            get { return hold; }
        }
        public int Decay
        {
            set
            {
                decay = value;
                UpdatePoints();
                UpdateScrollBar();
                panel.Invalidate();
            }
            get { return decay; }
        }
        public int Sustain
        {
            set
            {
                sustain = value;
                UpdatePoints();
                UpdateScrollBar();
                panel.Invalidate();
            }
            get { return sustain; }
        }
        public int Release
        {
            set
            {
                release = value;
                UpdatePoints();
                UpdateScrollBar();
                panel.Invalidate();
            }
            get { return release; }
        }

        public float AttackTime
        {
            get { return attackStep * MSEC_PER_STEP; }
        }
        public int HoldTime
        {
            get
            {
                int msec = hold;
                msec++;
                msec *= msec;
                msec /= 4;
                return msec;
            }
        }
        public float DecayTime
        {
            get { return decayStep * MSEC_PER_STEP; }
        }
        public float ReleaseTime
        {
            get { return releaseStep * MSEC_PER_STEP; }
        }
        public void LockWindowUpdate(bool bLock)
        {
            updateLock = bLock;
            if (!bLock)
            {
                UpdatePoints();
                UpdateScrollBar();
                panel.Invalidate();
            }
        }

        void PanelOnEnabledChanged(object sender, EventArgs e)
        {
            if (panel.Enabled)
            {
                panel.ForeColor = SystemColors.WindowText;
                panel.BackColor = SystemColors.Window;
            }
            else
            {
                panel.ForeColor = SystemColors.GrayText;
                panel.BackColor = SystemColors.Control;
            }
        }

        void PanelOnSizeChanged(object sender, EventArgs e)
        {
            UpdateScrollBar();
            panel.Invalidate();
        }

        void ScrollBarOnScroll(object sender, ScrollEventArgs e)
        {
            panel.Invalidate();
        }

        void UpdateScrollBar()
        {
            if (updateLock)
            {
                return;
            }

            Point lastPoint = (Point)apoints[apoints.Count - 1];
            int step = lastPoint.X;
            int maximum = step * STEP_PIXEL;
            int length = panel.ClientSize.Width;
            int value = 0;

            if (scrollBar.Value + length > maximum)
            {
                if ((value = maximum - length) < 0)
                {
                    value = 0;
                }
                scrollBar.Value = value;
            }

            scrollBar.Maximum = maximum;
            scrollBar.LargeChange = length;
            scrollBar.Enabled = (scrollBar.Maximum > scrollBar.LargeChange);
        }

        int GetNextEnvPointY(AdsrEnvelope env, int msec)
        {
            env.Update(msec);
            double volume = Math.Pow(10, env.Current / 200.0);

            return (int)((1.0 - volume) * SCALE_Y + 0.5);
        }

        void UpdatePoints()
        {
            if (updateLock)
            {
                return;
            }

            apoints = new ArrayList();

            AdsrEnvelope env = new AdsrEnvelope();
            env.Attack = attack;
            env.Hold = hold;
            env.Decay = decay;
            env.Sustain = sustain;
            env.Release = release;

            int pointX = 0;
            int pointY = 0;
            apoints.Add(new Point(pointX, SCALE_Y - 1));

            // 無効値のチェック
            if (0 > attack ||
                0 > hold ||
                0 > decay ||
                0 > sustain ||
                0 > release)
            {
                apoints.Add(new Point(pointX, SCALE_Y - 1));
                attackStep = 0;
                releaseStep = 0;
                return;
            }

            // アタック
            int prevCount = apoints.Count;
            while (env.Status == AdsrEnvelopeStatus.Attack)
            {
                pointY = GetNextEnvPointY(env, UPDATE_MSEC);
                apoints.Add(new Point(pointX, pointY));
                pointX += STEP_PIXEL * UPDATE_TIME;
            }
            attackStep = (apoints.Count - prevCount) - 1;

            // ホールド
            while (env.Status == AdsrEnvelopeStatus.Hold)
            {
                pointY = GetNextEnvPointY(env, UPDATE_MSEC);
                apoints.Add(new Point(pointX, pointY));
                pointX += STEP_PIXEL * UPDATE_TIME;
            }

            // ディケイ
            prevCount = apoints.Count;
            while (env.Status != AdsrEnvelopeStatus.Sustain)
            {
                pointY = GetNextEnvPointY(env, UPDATE_MSEC);
                apoints.Add(new Point(pointX, pointY));
                pointX += STEP_PIXEL * UPDATE_TIME;
            }
            if (prevCount == apoints.Count)
            {
                decayStep = 0;
            }
            else
            {
                decayStep = (apoints.Count - prevCount) - 1;
            }

            // サスティン
            apoints.Add(new Point(pointX + SUSTAIN_STEP * STEP_PIXEL, pointY));
            pointX += SUSTAIN_STEP * STEP_PIXEL;
            pointX += STEP_PIXEL * UPDATE_TIME;

            // リリース
            env.NoteOff();
            prevCount = apoints.Count;
            while (!env.IsFinishedRelease)
            {
                pointY = GetNextEnvPointY(env, UPDATE_MSEC);
                apoints.Add(new Point(pointX, pointY));
                pointX += STEP_PIXEL * UPDATE_TIME;
            }
            releaseStep = Math.Max((apoints.Count - prevCount) - 1, 0);
        }

        void PanelOnPaint(object sender, PaintEventArgs e)
        {
            if (panel.ClientSize.Height <= 1) return;

            Graphics gfx = e.Graphics;

            int begin = 0;
            int end = apoints.Count - 1;
            for (int i = 0; i < apoints.Count; i++)
            {
                Point point = (Point)apoints[i];
                if (point.X < scrollBar.Value)
                {
                    begin = Math.Max(i, begin);
                }
                else if (point.X > scrollBar.Value + panel.ClientSize.Width)
                {
                    end = Math.Min(i, end);
                }
            }

            Point[] points = new Point[end - begin + 1];
            apoints.CopyTo(begin, points, 0, points.Length);

            gfx.ScaleTransform(1.0f, (panel.ClientSize.Height - 1) / (float)SCALE_Y);
            gfx.TranslateTransform(-scrollBar.Value, 0);
            gfx.DrawLines(new Pen(panel.ForeColor), points);
        }
    }

}
