﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------


namespace NintendoWare.SoundMaker.Framework.Windows.Forms.CommandHandlers
{
    using System;
    using System.Diagnostics;
    using System.Drawing;
    using System.IO;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// ユーザーコマンドを実行します。
    /// </summary>
    public class UserCommandHandler : CommandHandler
    {
        private string commandID = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="commandID">処理対象コマンドID。</param>
        /// <param name="helpFilePath">ヘルプファイルパス。</param>
        public UserCommandHandler(string commandID)
        {
            if (null == commandID) { throw new ArgumentNullException("commandID"); }
            this.commandID = commandID;
        }

        /// <summary>
        /// コマンドハンドラが実行可能なコマンドを取得します。
        /// </summary>
        public override string TargetCommandID
        {
            get { return this.commandID; }
        }

        /// <summary>
        /// コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <returns>コマンドの状態。</returns>
        protected override CommandStatus QueryStatusInternal(Command command, IQueryCommandParameter parameters)
        {
            UserCommand userCommand = GetUserCommand(parameters);
            if (userCommand == null ||
                userCommand.Command == String.Empty)
            {
                return CommandStatus.Unsupported;
            }

            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        protected override bool ExecuteInternal(Command command, IQueryCommandParameter parameters)
        {
            UserCommand userCommand = GetUserCommand(parameters);
            if (userCommand == null ||
                userCommand.Command == String.Empty)
            {
                return false;
            }

            try
            {
                SoundProjectDocument projectDocument =
                    ApplicationBase.Instance.ProjectService.ProjectDocument;
                string filePath = projectDocument.Resource.Key;
                string commandText = Environment.ExpandEnvironmentVariables(userCommand.Command);

                ProcessStartInfo startInfo = new ProcessStartInfo(commandText);
                //startInfo.Arguments = null;
                startInfo.WorkingDirectory = Path.GetDirectoryName(filePath);

                Process process = Process.Start(startInfo);
                //process.WaitForExit();
                return true;
            }
            catch
            {
                return false;
            }
        }

        /// <summary>
        /// ユーザーコマンドを取得します。
        /// </summary>
        private UserCommand GetUserCommand(IQueryCommandParameter parameters)
        {
            string parameterName = "Index";

            if (parameters.ContainsParameter(parameterName) == false)
            {
                Debug.Assert(false, "Not contains index parameter.");
                return null;
            }

            try
            {
                int index = int.Parse((string)parameters.GetParameter(parameterName));
                SoundProject soundProject = ApplicationBase.Instance.ProjectService.Project;
                if (soundProject == null)
                {
                    return null;
                }

                UserCommand userCommand = soundProject.UserCommands[index];
                return userCommand;
            }
            catch
            {
                return null;
            }
        }
    }
}
