﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.CommandHandlers
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.Utilities;
    using CoreResources = NintendoWare.SoundFoundation.Resources;

    /// <summary>
    /// 既存のサウンドセットを追加します。
    /// </summary>
    public class QueryAndAddExistingSoundSetHandler : AddSoundSetHandler
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="queryTarget">対象の問い合わせ先。</param>
        public QueryAndAddExistingSoundSetHandler(IQueryCommandParameter queryParameter)
            : base(ProjectCommands.AddExistingSoundSet.ID, false, queryParameter) { }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        protected override bool ExecuteInternal(Command command, IQueryCommandParameter parameters)
        {
            DocumentReference documentRef = null;
            string filePath = SoundSetFileQuester.QueryLoad();
            if (null == filePath) { return false; }

            List<DocumentReference> docRefList = null;
            try
            {
                // ファイルのバージョンを確認します。
                bool oldFileAllForceSave = false;
                bool isReadOnly = false;
                List<string> newVersionFiles = new List<string>();
                List<string> oldVersionFiles = new List<string>();
                docRefList = FileVersionUtility.GetDifferentVersionFilesWithSoundSetFile(filePath, ref newVersionFiles, ref oldVersionFiles);

                if (0 < newVersionFiles.Count)
                {
                    string message = string.Format("{0}\n{1}", CoreResources.MessageResource.Message_UnsupportedFileVersion, filePath);
                    ApplicationBase.Instance.UIService.ShowMessageBox(message);

                    return false;
                }
                else if (0 < oldVersionFiles.Count)
                {
                    UpgradeFileDialog dialog = new UpgradeFileDialog();
                    switch (dialog.ShowDialog())
                    {
                        case DialogResult.Cancel:
                            return false;

                        case DialogResult.No: // 読み取り専用で開く
                            isReadOnly = true;
                            break;

                        case DialogResult.OK:
                            oldFileAllForceSave = true;
                            isReadOnly = false;
                            break;
                    }
                }

                documentRef = Application.DocumentService.OpenDocument(new FileResource(filePath));

                if (null == documentRef ||
                    documentRef.Document is SoundSetDocument == false ||
                    documentRef.Document.TypeName != Application.Traits.IntermediateOutputTraits.SoundSetDocumentTypeName)
                {
                    ApplicationBase.Instance.UIService.ShowMessageBox(
                                                                      string.Format(CoreResources.MessageResource.Message_UnsupportedFileFormat,
                                                                                    filePath));
                    if (documentRef != null)
                    {
                        documentRef.Close();
                    }

                    return false;
                }

                // 旧バージョンを全て保存する指示があったなら保存する。
                if (oldFileAllForceSave == true)
                {
                    SaveHelper.SaveForce(oldVersionFiles.ToArray());
                }

                // 読み取り専用をクリア
                (documentRef.Document as SoundSetDocument).IsReadOnly = false;

                if (isReadOnly == true)
                {
                    DocumentService documentService = ApplicationBase.Instance.DocumentService;

                    // 旧バージョンのみ読み取り専用に設定
                    foreach (string oldFilePath in oldVersionFiles)
                    {
                        SoundDocument document = documentService.Documents.Where(a => a.Resource.Key == oldFilePath).FirstOrDefault() as SoundDocument;

                        if (document != null)
                        {
                            document.IsReadOnly = true;
                        }
                    }
                }
            }
            catch (FileNotSupportedVersionException e)
            {
                ApplicationBase.Instance.UIService.ShowMessageBox(string.Format("{0}\n{1}",
                                                                                e.Message,
                                                                                e.FilePath));
                if (documentRef != null)
                {
                    documentRef.Close();
                }

                return false;
            }
            catch (Exception e)
            {
                string message = e.Message;
                if (string.IsNullOrEmpty(message) == true)
                {
                    message = Resources.MessageResource.Message_UnknownError;
                }
                ApplicationBase.Instance.UIService.ShowMessageBox(message);

                if (documentRef != null)
                {
                    documentRef.Close();
                }

                return false;
            }
            finally
            {
                if (docRefList != null)
                {
                    foreach (DocumentReference docRef in docRefList)
                    {
                        docRef.Close();
                    }
                }
            }

            try
            {
                return Execute(parameters, documentRef);
            }
            finally
            {
                if (documentRef != null)
                {
                    documentRef.Close();
                }
            }
        }
    }
}
