﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.CommandHandlers
{
    using System;
    using System.Linq;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;

    /// <summary>
    /// 新しいバンクを作成します。
    /// </summary>
    public class QueryAndAddBankHandler : AddSoundSetItemHandler
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="insert">挿入する場合は false、追加する場合は false。</param>
        /// <param name="queryTarget">対象の問い合わせ先。</param>
        public QueryAndAddBankHandler(bool insert, IQueryCommandParameter queryParameter)
            : base(insert ? ProjectCommands.InsertBank.ID : ProjectCommands.AddBank.ID,
                    insert, queryParameter)
        { }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        protected override bool ExecuteInternal(Command command, IQueryCommandParameter parameters)
        {
            CreateBankDialog form = new CreateBankDialog(Application.DocumentService, Application.ProjectService);
            if (form.ShowDialog() != DialogResult.OK) { return false; }

            if (form.DoCreateBankFile != false)
            {
                if (!NewBank(form.Bank)) { return false; }
            }

            if (!Execute(parameters, form.Bank)) { return false; }

            (Application.UIService as FormsUIService).MainWindow.ShowPageByComponent(form.Bank);
            return true;
        }

        /// <summary>
        /// アイテムの追加先を取得します。
        /// </summary>
        /// <param name="queryTarget">対象の問い合わせ先。</param>
        /// <returns>アイテムの追加先。</returns>
        protected override Component GetInsertParentComponent(IQueryCommandParameter queryParameter)
        {
            SoundSetDocument targetDocument = GetTargetDocument(queryParameter);
            if (null == targetDocument) { return null; }

            return (from Component component in targetDocument.SoundSet.Children
                    where component is SoundSetBankPack
                    select component as SoundSetBankPack).FirstOrDefault();
        }

        /// <summary>
        /// 新しいバンクを作成します。
        /// </summary>
        /// <param name="bank">サウンドセットバンク。</param>
        private bool NewBank(SoundSetBankBase bank)
        {
            bool result = true;

            if (null == bank) { throw new ArgumentNullException("bank"); }
            if (null == bank.FilePath) { throw new ArgumentException("bank.FilePath must not be null."); }

            BankService bankService = new BankService(
                Application.DocumentService,
                Application.Traits.IntermediateOutputTraits);

            try
            {
                bankService.Create(bank.FilePath);
                result = SaveHelper.Save(() => Application.DocumentService.SaveDocument(bankService.BankDocument), new string[] { bank.FilePath });
            }
            catch
            {
                ApplicationBase.Instance.UIService.ShowMessageBox(
                    Resources.MessageResource.Message_UnknownError);
                return false;
            }
            finally
            {
                bankService.Close();
            }

            return result;
        }
    }
}
