﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms.CommandHandlers
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Text.RegularExpressions;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Commands;

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class OpenItemFolderHandler : CommandHandler
    {
        ///--------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public OpenItemFolderHandler(IQueryCommandParameter queryParameter) :
            base(queryParameter)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override string TargetCommandID
        {
            get { return FileCommands.OpenItemFolder.ID; }
        }

        ///--------------------------------
        /// <summary>
        /// コマンドを実行できるかどうか調べる
        /// </summary>
        protected override CommandStatus QueryStatusInternal(Command command, IQueryCommandParameter parameters)
        {
            Component[] components = GetTargetComponents(parameters);
            SoundProjectService projectService = GetTargetProjectService(parameters);

            if (components.Length != 1)
            {
                return CommandStatus.SupportedAndVisible;
            }

            foreach (Component component in components)
            {
                string filePath = GetFilePath(component, projectService);
                if (null == filePath) { continue; }
                return CommandStatus.SupportedAndEnabledAndVisible;
            }

            return CommandStatus.SupportedAndVisible;
        }

        ///--------------------------------
        /// <summary>
        /// コマンドの実行
        /// </summary>
        protected override bool ExecuteInternal(Command command, IQueryCommandParameter parameters)
        {
            Component[] components = GetTargetComponents(parameters);
            SoundProjectService projectService = GetTargetProjectService(parameters);

            foreach (Component component in components)
            {
                string filePath = GetFilePath(component, projectService);
                if (null == filePath) { continue; }

                try
                {
                    Execute(component, filePath);
                }
                catch (DirectoryNotFoundException)
                {
                    ApplicationBase.Instance.UIService.ShowMessageBox(
                        Resources.MessageResource.Message_DirectoryNotFound);
                }
                catch (FileNotFoundException e)
                {
                    ApplicationBase.Instance.UIService.ShowMessageBox(
                       string.Format(Resources.MessageResource.Message_FileNotFound,
                                     e.FileName));
                }
                catch (Exception e)
                {
                    ApplicationBase.Instance.UIService.ShowMessageBox(e.Message);
                    //Resources.MessageResource.Message_UnknownError);
                }
            }

            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual void Execute(Component component, string filePath)
        {
            Process.Start("explorer.exe", "/e,/select, " + filePath);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private SoundProjectService GetTargetProjectService(IQueryCommandParameter parameters)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }

            if (!parameters.ContainsParameter(CommandParameterNames.TargetComponentService))
            {
                return null;
            }

            return parameters.GetParameter(CommandParameterNames.TargetComponentService)
                   as SoundProjectService;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private SoundDocument GetTargetDocument(IQueryCommandParameter parameters)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }

            if (!parameters.ContainsParameter(CommandParameterNames.TargetDocuments))
            {
                return null;
            }

            return (parameters.GetParameter(CommandParameterNames.TargetDocuments)
                     as IEnumerable<SoundDocument>).FirstOrDefault();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private Component[] GetTargetComponents(IQueryCommandParameter parameters)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }

            if (!parameters.ContainsParameter(CommandParameterNames.TargetComponents))
            {
                return null;
            }

            return (parameters.GetParameter(CommandParameterNames.TargetComponents)
                     as IEnumerable<Component>).ToArray();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private string GetFilePath(Component component, SoundProjectService projectService)
        {
            Debug.Assert(null != component, "invalid argument");

            if (component is SoundProject)
            {
                if (null == projectService) { return null; }
                if (null == projectService.ProjectDocument) { return null; }
                return projectService.ProjectDocument.Resource.Key;
            }

            if (component is SoundSet)
            {
                if (null == projectService) { return null; }
                return GetSoundSetFilePath(projectService, component as SoundSet);
            }

            if (component is SoundSetItemPack)
            {
                if (null == projectService) { return null; }
                if (!(component.Parent is SoundSet)) { return null; }
                return GetSoundSetFilePath(projectService, component.Parent as SoundSet);
            }

            if (component is StreamSoundBase)
            {
                if (component.Children.Count <= 0) { return null; }
                component = component.Children[0];
            }

            if (component is Instrument)
            {
                if (component.Children.Count <= 0) { return null; }
                component = component.Children[0].Children[0];
            }

            if (component is KeyRegion)
            {
                if (component.Children.Count <= 0) { return null; }
                component = component.Children[0];
            }

            //
            if (component.Parameters.ContainsKey(ProjectParameterNames.FilePath) == false)
            {
                return null;
            }

            return (string)component.Parameters[ProjectParameterNames.FilePath].Value;
        }

        private string GetSoundSetFilePath(SoundProjectService projectService, SoundSet soundSet)
        {
            Debug.Assert(null != projectService, "invalid argument");
            Debug.Assert(null != soundSet, "invalid argument");

            foreach (SoundSetDocument soundSetDocument in projectService.SoundSetDocuments)
            {
                if (soundSetDocument.SoundSet != soundSet) { continue; }
                return soundSetDocument.Resource.Key;
            }

            return null;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// 外部エディタで参照ファイルを開きます。
    /// </summary>
    public class ExecuteItemFolderHandler : OpenItemFolderHandler
    {
        private enum FileType
        {
            Unknown,
            Text,
            Midi,
            Wav,
        }

        ///--------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ExecuteItemFolderHandler(IQueryCommandParameter queryParameter)
            : base(queryParameter)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override string TargetCommandID
        {
            get { return FileCommands.ExecuteItemFile.ID; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void Execute(Component component, string filePath)
        {
            OpenHelper.FileOpenPreCommand(filePath);

            if (component is SoundSetBankBase)
            {
                FormsApplication.Instance.UIService.MainWindow.OpenFile
                    ((SoundSetBankBase)component);
            }
            else
            {
                var fileType = GetFileType(filePath);
                var command = GetCommandString(fileType);
                var args = GetArgumentsString(fileType);

                if (component is SequenceSoundBase && fileType == FileType.Text)
                {
                    uint lineNum = GetLineNumberFromFile(filePath, (component as SequenceSoundBase).StartPosition);
                    args = GetArgs(args, filePath, lineNum);
                }
                else
                {
                    args = GetArgs(args, filePath);
                }

                OpenItemFile(filePath, command, args);
            }
        }

        // TODO: (SIGLO-64975) SampleMapCtrl がコマンドバインディングに対応したら削除する
        /// <summary>
        /// 外部エディタで指定のファイルを開きます。
        /// </summary>
        /// <param name="filePath"></param>
        public static void Execute(string filePath)
        {
            if (string.IsNullOrEmpty(filePath))
            {
                return;
            }

            OpenHelper.FileOpenPreCommand(filePath);

            var fileType = GetFileType(filePath);
            var command = GetCommandString(fileType);
            var args = GetArgumentsString(fileType);

            args = GetArgs(args, filePath);
            OpenItemFile(filePath, command, args);
        }

        private static string GetCommandString(FileType fileType)
        {
            switch (fileType)
            {
                case FileType.Text:
                    if (FormsApplication.Instance.AppConfiguration.Options.Application.General.EditorUseEditor == false)
                    {
                        return "notepad.exe";
                    }
                    else
                    {
                        return FormsApplication.Instance.AppConfiguration.Options.Application.General.EditorCommand;
                    }

                case FileType.Midi:
                    return FormsApplication.Instance.AppConfiguration.Options.Application.General.MidiEditorCommand;

                case FileType.Wav:
                    return FormsApplication.Instance.AppConfiguration.Options.Application.General.WavEditorCommand;

                default:
                    return null;
            }
        }

        private static string GetArgumentsString(FileType fileType)
        {
            switch (fileType)
            {
                case FileType.Text:
                    if (FormsApplication.Instance.AppConfiguration.Options.Application.General.EditorUseEditor == false)
                    {
                        return "$(file)";
                    }
                    else
                    {
                        return FormsApplication.Instance.AppConfiguration.Options.Application.General.EditorArguments;
                    }

                case FileType.Midi:
                    return FormsApplication.Instance.AppConfiguration.Options.Application.General.MidiEditorArguments;

                case FileType.Wav:
                    return FormsApplication.Instance.AppConfiguration.Options.Application.General.WavEditorArguments;

                default:
                    return null;
            }
        }

        private static FileType GetFileType(string filePath)
        {
            string extension = Path.GetExtension(filePath).ToLower();
            switch (extension)
            {
                case ".fseq":
                    return FileType.Text;

                case ".mid":
                    return FileType.Midi;

                case ".aac":
                case ".aif":
                case ".aiff":
                case ".opus":
                case ".wav":
                    return FileType.Wav;

                default:
                    return FileType.Unknown;
            }
        }

        private static void OpenItemFile(string filePath, string command, string args)
        {
            try
            {
                if (string.IsNullOrEmpty(command))
                {
                    // 関連付けられたアプリケーションで開きます。
                    Process.Start(filePath);
                }
                else
                {
                    Process.Start(command, args);
                }
            }
            catch (Exception e)
            {
                Debug.WriteLine($"{e.ToString()}");
                ApplicationBase.Instance.UIService.ShowMessageBox(Resources.MessageResource.Message_EditorNotFound, Resources.MessageResource.Message_UnknownError);
            }
        }

        private static uint GetLineNumberFromFile(string filePath, string label)
        {
            uint lineNum = 1;

            try
            {
                if (File.Exists(filePath) == true)
                {
                    using (StreamReader reader = new StreamReader(filePath))
                    {
                        Regex regex =
                            new Regex("^(\\s*[a-zA-Z_][a-zA-Z0-9_]*\\s*:)*\\s*" + label + "\\s*:");
                        while (reader.EndOfStream == false)
                        {
                            string line = reader.ReadLine();
                            if (regex.IsMatch(line) == true)
                            {
                                return lineNum;
                            }
                            ++lineNum;
                        }

                        return 1;
                    }
                }
            }
            catch
            {
            }

            return 1;
        }

        private static string GetArgs(string args, string filePath, uint lineNum = 0)
        {
            if (string.IsNullOrEmpty(args))
            {
                return string.Empty;
            }
            args = args.Replace("$(file)", filePath);
            args = args.Replace("$(line)", lineNum.ToString());

            return args;
        }
    }
}
