﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Runtime.InteropServices;
using System.Windows.Forms;
using Microsoft.Win32;
using NintendoWare.SoundFoundation.CommandHandlers;
using NintendoWare.SoundFoundation.Commands;
using NintendoWare.SoundMaker.Framework.Resources;

namespace NintendoWare.SoundMaker.Framework.Windows.Forms.CommandHandlers
{
    /// <summary>
    /// ヘルプを表示します。
    /// </summary>
    public class ShowHelpHandler : CommandHandler
    {
        private string _commandID;
        private string _helpURL;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="commandID">処理対象コマンドID。</param>
        /// <param name="helpFilePath">ヘルプファイルパス。</param>
        public ShowHelpHandler(string commandID, string helpURL)
        {
            if (null == commandID) { throw new ArgumentNullException("commandID"); }
            if (null == helpURL) { throw new ArgumentNullException("helpURL"); }

            _commandID = commandID;
            _helpURL = helpURL;
        }

        /// <summary>
        /// コマンドハンドラが実行可能なコマンドを取得します。
        /// </summary>
        public override string TargetCommandID
        {
            get { return _commandID; }
        }

        /// <summary>
        /// ヘルプを表示します。
        /// </summary>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        public static bool Execute(string url)
        {
            string browser = GetDefaultBrowserExePath();

            try
            {
                if (string.IsNullOrEmpty(browser) == true || File.Exists(browser) == false)
                {
                    // ブラウザが存在しません。ヘルプを表示するにはブラウザが必要です。
                    MessageBox.Show(MessageResource.Message_ErrorNoHelpBrowser, MessageResource.Message_Warning, MessageBoxButtons.OK, MessageBoxIcon.Warning);
                }
                else
                {
                    // ヘルプをブラウザで表示します。
                    System.Diagnostics.Process.Start(browser, url);
                }
            }
            catch
            {
                // ヘルプの表示に失敗しました。ブラウザが動作するか確認してください。
                MessageBox.Show(MessageResource.Message_ErrorHelpDisplay, MessageResource.Message_Warning, MessageBoxButtons.OK, MessageBoxIcon.Warning);
            }

            return true;
        }

        /// <summary>
        /// コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <returns>コマンドの状態。</returns>
        protected override CommandStatus QueryStatusInternal(Command command, IQueryCommandParameter parameters)
        {
            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        protected override bool ExecuteInternal(Command command, IQueryCommandParameter parameters)
        {
            Execute(_helpURL);
            return true;
        }

        /// <summary>
        /// 既定ブラウザのexeファイルパスを取得します. （ソース EffectMaker:HelpManager.cs:140 から）
        /// </summary>
        /// <returns>既定ブラウザのexeファイルパス</returns>
        private static string GetDefaultBrowserExePath()
        {
            const string userChoice = @"Software\Microsoft\Windows\Shell\Associations\UrlAssociations\http\UserChoice";
            try
            {
                using (var key = Registry.CurrentUser.OpenSubKey(userChoice))
                {
                    if (key != null)
                    {
                        var progIdValue = key.GetValue("ProgId");
                        if (progIdValue != null)
                        {
                            using (var pathKey = Registry.ClassesRoot.OpenSubKey(progIdValue.ToString() + @"\shell\open\command"))
                            {
                                if (pathKey != null)
                                {
                                    var command = pathKey.GetValue(null).ToString();

                                    // 引数がついているかもしれないのでコマンド部分だけ取得
                                    int argc;
                                    var argv = CommandLineToArgvW(command, out argc);
                                    if (argv != IntPtr.Zero)
                                    {
                                        try
                                        {
                                            // 0 番目の文字列を取得
                                            var arg = Marshal.ReadIntPtr(argv, 0);
                                            return Marshal.PtrToStringUni(arg);
                                        }
                                        finally
                                        {
                                            Marshal.FreeHGlobal(argv);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
            catch
            {
            }

            return string.Empty;
        }

        /// <summary>
        /// Unicode コマンドライン文字列から Unicode 版の argv と argc の値を取得します。
        /// </summary>
        /// <param name="lpCmdLine">コマンドライン文字列へのポインタ</param>
        /// <param name="pNumArgs">引数の数を受け取る変数へのポインタ</param>
        /// <returns>
        /// 関数が成功すると、組み立てられた引数リストへの NULL 以外のポインタが返ります。
        /// この引数リストは、Unicode ワイド文字引数文字列の集合です。
        /// 関数が失敗すると、NULL が返ります。
        /// </returns>
        [DllImport("shell32.dll", SetLastError = true)]
        private static extern IntPtr CommandLineToArgvW([MarshalAs(UnmanagedType.LPWStr)] string lpCmdLine, out int pNumArgs);
    }
}
