﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Utilities
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.Windows;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;

    /// <summary>
    ///
    /// </summary>
    public static class GroupUtility
    {
        /// <summary>
        ///
        /// </summary>
        public static GroupItemBase[] GetRemoveGroupItem(GroupBase targetGroup, Component component)
        {
            HashSet<GroupItemBase> hashSet = new HashSet<GroupItemBase>();

            if (component is SequenceSoundSetBase)
            {
                // 削除対象のグループアイテムを取得します。。
                // 登録しようとするグループのシーケンスサウンドセットに含まれている
                // シーケンスサウンドが対象です。
                foreach (GroupItemBase inGroupItem in targetGroup.Children)
                {
                    if (inGroupItem.Target.Parent == component)
                    {
                        hashSet.Add(inGroupItem);
                    }
                }

                // 削除するグループアイテムがあった場合にはユーザーに問い合わせます。
                // 削除しない場合でも、登録しようとするグループアイテムの登録処理は継続します。
                if (hashSet.Count > 0)
                {
                    string message = String.Format
                        (Resources.MessageResource.Message_ConfirmRemoveGroupItem,
                          component.Name);

                    if (ApplicationBase.Instance.UIService.ShowMessageBox
                        (message,
                          null,
                          AppMessageBoxButton.YesNo,
                          AppMessageBoxImage.Question,
                          AppMessageBoxResult.Yes) != AppMessageBoxResult.Yes)
                    {
                        hashSet.Clear();
                    }
                }
            }

            return hashSet.ToArray();
        }

        /// <summary>
        ///
        /// </summary>
        public static bool AddToGroup(SoundSetDocument targetDocument, GroupBase targetGroup, GroupItemBase insertTargetGroupItem, GroupItemBase[] groupItems)
        {
            HashSet<GroupItemBase> hashSet = new HashSet<GroupItemBase>();

            foreach (GroupItemBase groupItem in groupItems)
            {
                if (groupItem.Target is SequenceSoundSetBase)
                {
                    // 削除対象のグループアイテムを取得します。。
                    // 登録しようとするグループのシーケンスサウンドセットに含まれている
                    // シーケンスサウンドが対象です。
                    foreach (GroupItemBase inGroupItem in targetGroup.Children)
                    {
                        if (inGroupItem.Target.Parent == groupItem.Target)
                        {
                            hashSet.Add(inGroupItem);
                        }
                    }

                    // 削除するグループアイテムがあった場合にはユーザーに問い合わせます。
                    // 削除しない場合でも、登録しようとするグループアイテムの登録処理は継続します。
                    if (hashSet.Count > 0)
                    {
                        string message = String.Format
                            (Resources.MessageResource.Message_ConfirmRemoveGroupItem,
                              groupItem.Target.Name);

                        if (ApplicationBase.Instance.UIService.ShowMessageBox
                            (message,
                              null,
                              AppMessageBoxButton.YesNo,
                              AppMessageBoxImage.Question,
                              AppMessageBoxResult.Yes) != AppMessageBoxResult.Yes)
                        {
                            hashSet.Clear();
                        }
                    }
                }
            }

            if (null == targetDocument) { throw new ArgumentNullException("document"); }
            if (null == targetGroup) { throw new ArgumentNullException("group"); }

            try
            {
                targetDocument.OperationHistory.BeginTransaction();

                var operations = new List<Operation>()
                {
                    new InsertComponentOperation( targetGroup, insertTargetGroupItem, groupItems)
                };

                var removeGroupItems = hashSet.ToArray();
                if (removeGroupItems.Length > 0)
                {
                    operations.Add(new RemoveComponentOperation(removeGroupItems));
                }

                foreach (Operation operation in operations)
                {
                    if (operation.Execute() == false)
                    {
                        targetDocument.OperationHistory.CancelTransaction();
                        return false;
                    }
                    targetDocument.OperationHistory.AddOperation(operation);
                }

                targetDocument.OperationHistory.EndTransaction();
                return true;
            }
            catch
            {
                targetDocument.OperationHistory.CancelTransaction();
                return false;
            }
        }

        public static bool ContainsGroup(GroupBase[] groups, Component component)
        {
            foreach (GroupBase group in groups)
            {
                foreach (GroupItemBase item in group.Children)
                {
                    if (item.Target == component)
                    {
                        return true;
                    }
                }
            }

            return false;
        }

        public static bool DoReferred(GroupItemBase groupItem, SoundSetItem item)
        {
            return groupItem.TargetItemReference == item.Name && groupItem.Target == item;
        }
    }
}
