﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.IO;
using System.Text;

namespace NintendoWare.SoundMaker.Framework.Preview.Communications
{
    public abstract class CommConnection
    {
        /// <summary>
        /// 接続します。
        /// </summary>
        public abstract bool Connect();

        /// <summary>
        /// 接続中か調べます。
        /// </summary>
        public abstract bool IsConnected { get; }

        /// <summary>
        /// 切断します。
        /// </summary>
        public abstract void Disconnect();

        /// <summary>
        /// 切断します。
        /// </summary>
        public abstract void Disconnect(int timeout);

        /// <summary>
        ///
        /// </summary>
        public abstract void SendPacket(CommPacket packet);
    }

    //-----------------------------------------------------------------------
    /// <summary>
    /// ProtocolSoundReader
    /// short,ushort,int,uint,floatのエンディアンを変換し、読み込みます
    /// </summary>
    public class ProtocolSoundReader : BinaryReader
    {
        /// <summary>コンストラクタ</summary>
        public ProtocolSoundReader(Stream input) : base(input) { }
        /// <summary>コンストラクタ</summary>
        public ProtocolSoundReader(Stream input, Encoding encoding) :
            base(input, encoding)
        { }

#if false
        /// <summary>short読み込み</summary>
        public override short ReadInt16(){
            return IPAddress.NetworkToHostOrder(base.ReadInt16());
        }

        /// <summary>ushort読み込み</summary>
        public override ushort ReadUInt16(){
            short result = (short)base.ReadUInt16();
            return (ushort)IPAddress.NetworkToHostOrder(result);
        }

        /// <summary>int読み込み</summary>
        public override int ReadInt32(){
            return IPAddress.NetworkToHostOrder(base.ReadInt32());
        }

        /// <summary>uint読み込み</summary>
        public override uint ReadUInt32(){
            int result = (int)base.ReadUInt32();
            return (uint)IPAddress.NetworkToHostOrder(result);
        }

        /// <summary>float読み込み</summary>
        public override float ReadSingle(){
            // ネットワークとエンディアンが違うことが前提になっています
            byte[] floatBytes = BitConverter.GetBytes(base.ReadSingle());
            Array.Reverse(floatBytes);
            return BitConverter.ToSingle(floatBytes, 0);
        }
#endif
    }

    //-----------------------------------------------------------------------
    /// <summary>
    /// ProtocolSoundWriter
    /// short,ushort,int,uint,floatのエンディアンを変換し、書き込みます
    /// </summary>
    public class ProtocolSoundWriter : BinaryWriter
    {

        /// <summary>コンストラクタ</summary>
        public ProtocolSoundWriter() : base() { }
        /// <summary>コンストラクタ</summary>
        public ProtocolSoundWriter(Stream input) : base(input) { }
        /// <summary>コンストラクタ</summary>
        public ProtocolSoundWriter(Stream input, Encoding encoding) :
            base(input, encoding)
        { }

#if false
        /// <summary>short書き出し</summary>
        public override void Write(short value){
            base.Write(IPAddress.HostToNetworkOrder(value));
        }

        /// <summary>ushort書き出し</summary>
        public override void Write(ushort value){
            base.Write((ushort)IPAddress.HostToNetworkOrder((short)value));
        }

        /// <summary>int書き出し</summary>
        public override void Write(int value){
            base.Write(IPAddress.HostToNetworkOrder(value));
        }

        /// <summary>uint書き出し</summary>
        public override void Write(uint value){
            base.Write((uint)IPAddress.HostToNetworkOrder((int)value));
        }

        /// <summary>float書き出し</summary>
        public override void Write(float value){
            // ネットワークとエンディアンが違うことが前提になっています
            byte[] floatBytes = BitConverter.GetBytes(value);
            Array.Reverse(floatBytes);
            base.Write(BitConverter.ToSingle(floatBytes, 0));
        }
#endif
    }

#if false
    //-----------------------------------------------------------------------
    /** <summary>サウンドエラー</summary> */
    public class SoundError {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
        0+4;
        /** <summary> 4B: エラーコード</summary> */
        public uint errorCode;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(BinaryReader reader){
            errorCode = reader.ReadUInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(BinaryWriter writer){
            writer.Write(errorCode);
        }
    }

    //-----------------------------------------------------------------------
    /** <summary></summary> */
    public class ToolSoundSystemParameters
    {
      /// <summary>構造体にした場合のサイズ</summary>
      public const int StructSize =
        0 + 4;
      /** <summary> 4B: 出力モード</summary> */
      public ToolSoundOutputMode outputMode;

      /// <summary>読み込みメソッド</summary>
      public virtual void Read(ProtocolSoundReader reader)
      {
        outputMode = (ToolSoundOutputMode)reader.ReadInt32();
      }

      /// <summary>書き出しメソッド</summary>
      public virtual void Write(ProtocolSoundWriter writer)
      {
        writer.Write((int)outputMode);
      }
    }

    //-----------------------------------------------------------------------
    /** <summary>アニメーションコンポーネントの種類</summary> */
    public enum ToolSoundAnimComponentType
    {
      /** <summary>3Dモデル</summary> */
      AnimComponentType_3DModel = 1,
      /** <summary>アクション</summary> */
      AnimComponentType_Action = 2,
      /** <summary>アニメーションサウンド</summary> */
      AnimComponentType_AnimSound = 3,
    }

    //-----------------------------------------------------------------------
    /** <summary>データの読み込みエラー種別</summary> */
    public enum Err3DLoadCode
    {
      /** <summary>ファイルのロードに失敗しました。        </summary> */
      ERR_3D_LOAD_FILE_NOT_FOUND = 0,
      /** <summary>メモリのアロケーションに失敗しました。  </summary> */
      ERR_3D_LOAD_MEMORY_ALLOCATE = 1,
      /** <summary>有効なデータではありませんでした。      </summary> */
      ERR_3D_LOAD_INVALID_DATA = 2,
      /** <summary>MCS通信で異常が発生しました。           </summary> */
      ERR_3D_LOAD_MCS_STREAM_FAIL = 3,
    }

    //-----------------------------------------------------------------------
    /** <summary>アニメーションコンポーネント識別子</summary> */
    public class ToolSoundAnimComponentID {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
        0+4+4+4+4;
        /** <summary> 4B: コンポーネントの種類</summary> */
        public ToolSoundAnimComponentType type;
        /** <summary> 4B: モデルID</summary> */
        public uint modelID;
        /** <summary> 4B: アクションID（指定しない場合は -1）</summary> */
        public uint actionID;
        /** <summary> 4B: アニメーションサウンドID（指定しない場合は -1）</summary> */
        public uint animSoundID;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            type = (ToolSoundAnimComponentType)reader.ReadInt32();
            modelID = reader.ReadUInt32();
            actionID = reader.ReadUInt32();
            animSoundID = reader.ReadUInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write((int)type);
            writer.Write(modelID);
            writer.Write(actionID);
            writer.Write(animSoundID);
        }
    }

    //-----------------------------------------------------------------------
    /** <summary>
              未定義コマンドエラーメッセージ

              受け取った未定義コマンドをcommandパラメータで返します。
          </summary> */
    public class ErrSystemUnknownCommand
    {
      /// <summary>構造体にした場合のサイズ</summary>
      public const int StructSize =
        0 + 2;
      /** <summary>  2B: 受け取った未定義コマンド  </summary> */
      public ushort command;

      /// <summary>読み込みメソッド</summary>
      public virtual void Read(ProtocolSoundReader reader)
      {
        command = reader.ReadUInt16();
      }

      /// <summary>書き出しメソッド</summary>
      public virtual void Write(ProtocolSoundWriter writer)
      {
        writer.Write(command);
      }
    }

    //-----------------------------------------------------------------------
    /** <summary>
           バージョン要求応答メッセージ
              ツールのプロトコルバージョンを受け取り、ビューアのプロトコルバージョンを返します。
          </summary> */
    public class ToolSystemVersion
    {
      /// <summary>構造体にした場合のサイズ</summary>
      public const int StructSize =
        0 + 4;
      /** <summary>4B: バージョン番号</summary> */
      public uint version;

      /// <summary>読み込みメソッド</summary>
      public virtual void Read(ProtocolSoundReader reader)
      {
        version = reader.ReadUInt32();
      }

      /// <summary>書き出しメソッド</summary>
      public virtual void Write(ProtocolSoundWriter writer)
      {
        writer.Write(version);
      }
    }

    //-----------------------------------------------------------------------
    /** <summary>
          同期メッセージ

                Viewerは、この同期コマンドを受け取ると、指定されたtagName
                名をエコーした応答コマンドをアプリケーションに返信します。

                tagNameは､ツール側が自由に設定できます。
          </summary> */
    public class ToolSystemSync
    {
      /// <summary>構造体にした場合のサイズ</summary>
      public const int StructSize =
        0 + 4;
      /** <summary>4B: タグ名</summary> */
      public uint tagName;

      /// <summary>読み込みメソッド</summary>
      public virtual void Read(ProtocolSoundReader reader)
      {
        tagName = reader.ReadUInt32();
      }

      /// <summary>書き出しメソッド</summary>
      public virtual void Write(ProtocolSoundWriter writer)
      {
        writer.Write(tagName);
      }
    }

    //-----------------------------------------------------------------------
    /** <summary>
              データの読み込みエラー
          </summary> */
    public class Err3DLoad
    {
      /// <summary>構造体にした場合のサイズ</summary>
      public const int StructSize =
        0 + 4;
      /** <summary>  4B: エラーコード          </summary> */
      public Err3DLoadCode code;

      /// <summary>読み込みメソッド</summary>
      public virtual void Read(ProtocolSoundReader reader)
      {
        code = (Err3DLoadCode)reader.ReadInt32();
      }

      /// <summary>書き出しメソッド</summary>
      public virtual void Write(ProtocolSoundWriter writer)
      {
        writer.Write((int)code);
      }
    }

    //-----------------------------------------------------------------------
    /** <summary>アニメーションコンポーネント追加情報</summary> */
    public class ToolSoundAnimComponentAddInfo
    {
      /// <summary>構造体にした場合のサイズ</summary>
      public const int StructSize =
        0 + ToolSoundAnimComponentID.StructSize + 1 * 260 + 4;
      /** <summary> 16B: 対象ID</summary> */
      public ToolSoundAnimComponentID id = new ToolSoundAnimComponentID();
      /** <summary>260B: ファイル名</summary> */
      public byte[] fileName = new byte[260];
      /** <summary>  4B: キャッシュを使用しない場合は 1, キャッシュを使用する場合は 0</summary> */
      public int reload;

      /// <summary>読み込みメソッド</summary>
      public virtual void Read(ProtocolSoundReader reader)
      {
        id.Read(reader);
        for (int i = 0; i < 260; i++)
        {
          fileName[i] = reader.ReadByte();
        }
        reload = reader.ReadInt32();
      }

      /// <summary>書き出しメソッド</summary>
      public virtual void Write(ProtocolSoundWriter writer)
      {
        id.Write(writer);
        for (int i = 0; i < 260; i++)
        {
          writer.Write(fileName[i]);
        }
        writer.Write(reload);
      }
    }

    //-----------------------------------------------------------------------
    /** <summary>サウンドアーカイブ読み込みメッセージ </summary> */
    public class ToolSoundLoadSar
    {
      /// <summary>構造体にした場合のサイズ</summary>
      public const int StructSize =
        0 + 1 * 260;
      /** <summary>260B: PC上のデータのパス名</summary> */
      public byte[] pathName = new byte[260];

      /// <summary>読み込みメソッド</summary>
      public virtual void Read(ProtocolSoundReader reader)
      {
        for (int i = 0; i < 260; i++)
        {
          pathName[i] = reader.ReadByte();
        }
      }

      /// <summary>書き出しメソッド</summary>
      public virtual void Write(ProtocolSoundWriter writer)
      {
        for (int i = 0; i < 260; i++)
        {
          writer.Write(pathName[i]);
        }
      }
    }

    //-----------------------------------------------------------------------
    /** <summary>サウンドアーカイブ読み込み応答メッセージ</summary> */
    public class ToolSoundLoadSarReply
    {
      /// <summary>構造体にした場合のサイズ</summary>
      public const int StructSize =
        0 + 4;
      /** <summary>4B: 未使用</summary> */
      public uint id;

      /// <summary>読み込みメソッド</summary>
      public virtual void Read(ProtocolSoundReader reader)
      {
        id = reader.ReadUInt32();
      }

      /// <summary>書き出しメソッド</summary>
      public virtual void Write(ProtocolSoundWriter writer)
      {
        writer.Write(id);
      }
    }

    //-----------------------------------------------------------------------
    /** <summary>サウンドアーカイブを閉じる応答メッセージ</summary> */
    public class ToolSoundCloseSarReply
    {
      /// <summary>構造体にした場合のサイズ</summary>
      public const int StructSize =
        0 + 4;
      /** <summary>4B: 未使用</summary> */
      public uint result;

      /// <summary>読み込みメソッド</summary>
      public virtual void Read(ProtocolSoundReader reader)
      {
        result = reader.ReadUInt32();
      }

      /// <summary>書き出しメソッド</summary>
      public virtual void Write(ProtocolSoundWriter writer)
      {
        writer.Write(result);
      }
    }

    //-----------------------------------------------------------------------
    /** <summary>
              アニメーションフレーム取得応答メッセージ
          </summary> */
    public class ToolSoundQueryAnimFrameReply
    {
      /// <summary>構造体にした場合のサイズ</summary>
      public const int StructSize =
        0 + 4;
      /** <summary>4B: アニメーションのフレーム</summary> */
      public float frame;

      /// <summary>読み込みメソッド</summary>
      public virtual void Read(ProtocolSoundReader reader)
      {
        frame = reader.ReadSingle();
      }

      /// <summary>書き出しメソッド</summary>
      public virtual void Write(ProtocolSoundWriter writer)
      {
        writer.Write(frame);
      }
    }

    //-----------------------------------------------------------------------
    /** <summary>
              ループ状態設定メッセージ
          </summary> */
    public class ToolSoundSetLoop
    {
      /// <summary>構造体にした場合のサイズ</summary>
      public const int StructSize =
        0 + 4;
      /** <summary>4B: ループ状態</summary> */
      public int loopFlag;

      /// <summary>読み込みメソッド</summary>
      public virtual void Read(ProtocolSoundReader reader)
      {
        loopFlag = reader.ReadInt32();
      }

      /// <summary>書き出しメソッド</summary>
      public virtual void Write(ProtocolSoundWriter writer)
      {
        writer.Write(loopFlag);
      }

    }

    //-----------------------------------------------------------------------
    /** <summary>
        アニメーションフレーム設定メッセージ
        </summary> */
    public class ToolSoundAnimFrame
    {
      /// <summary>構造体にした場合のサイズ</summary>
      public const int StructSize =
        0 + 4 + 4 + 4;
      /** <summary>4B: アニメーションのフレーム</summary> */
      public float frame;
      /** <summary>4B: イベントを処理するかどうか</summary> */
      public int doEvent;
      /** <summary>4B: 同期をとるかどうか</summary> */
      public int sync;

      /// <summary>読み込みメソッド</summary>
      public virtual void Read(ProtocolSoundReader reader)
      {
        frame = reader.ReadSingle();
        doEvent = reader.ReadInt32();
        sync = reader.ReadInt32();
      }

      /// <summary>書き出しメソッド</summary>
      public virtual void Write(ProtocolSoundWriter writer)
      {
        writer.Write(frame);
        writer.Write(doEvent);
        writer.Write(sync);
      }
    }

    //-----------------------------------------------------------------------
    /** <summary>アニメーションフレーム設定応答メッセージ</summary> */
    public class ToolSoundAnimFrameReply
    {
      /// <summary>構造体にした場合のサイズ</summary>
      public const int StructSize =
        0 + 4;
      /** <summary>4B: 未使用</summary> */
      public uint result;

      /// <summary>読み込みメソッド</summary>
      public virtual void Read(ProtocolSoundReader reader)
      {
        result = reader.ReadUInt32();
      }

      /// <summary>書き出しメソッド</summary>
      public virtual void Write(ProtocolSoundWriter writer)
      {
        writer.Write(result);
      }
    }

    //-----------------------------------------------------------------------
    /** <summary>
              アニメーション再生レート設定メッセージ
          </summary> */
    public class ToolSoundAnimRate
    {
      /// <summary>構造体にした場合のサイズ</summary>
      public const int StructSize =
        0 + 4;
      /** <summary>4B: アニメーションの再生レート</summary> */
      public float rate;

      /// <summary>読み込みメソッド</summary>
      public virtual void Read(ProtocolSoundReader reader)
      {
        rate = reader.ReadSingle();
      }

      /// <summary>書き出しメソッド</summary>
      public virtual void Write(ProtocolSoundWriter writer)
      {
        writer.Write(rate);
      }
    }

    //-----------------------------------------------------------------------
    /** <summary>
              アニメーション再生範囲設定メッセージ
          </summary> */
    public class ToolSoundAnimRange
    {
      /// <summary>構造体にした場合のサイズ</summary>
      public const int StructSize =
        0 + 4 + 4;
      /** <summary>4B: ループ開始位置</summary> */
      public float start;
      /** <summary>4B: ループ終了位置</summary> */
      public float end;

      /// <summary>読み込みメソッド</summary>
      public virtual void Read(ProtocolSoundReader reader)
      {
        start = reader.ReadSingle();
        end = reader.ReadSingle();
      }

      /// <summary>書き出しメソッド</summary>
      public virtual void Write(ProtocolSoundWriter writer)
      {
        writer.Write(start);
        writer.Write(end);
      }

    }

    //-----------------------------------------------------------------------
    /** <summary>
              アニメーション再生範囲設定応答メッセージ
          </summary> */
    public class ToolSoundAnimRangeReply
    {
      /// <summary>構造体にした場合のサイズ</summary>
      public const int StructSize =
        0 + 4;
      /** <summary>4B: 設定変更後の現在のフレーム</summary> */
      public float frame;

      /// <summary>読み込みメソッド</summary>
      public virtual void Read(ProtocolSoundReader reader)
      {
        frame = reader.ReadSingle();
      }

      /// <summary>書き出しメソッド</summary>
      public virtual void Write(ProtocolSoundWriter writer)
      {
        writer.Write(frame);
      }

    }

    //-----------------------------------------------------------------------
    /** <summary>
              アニメーションサウンドのイベント更新コマンド
          </summary> */
    public class ToolSoundModifierEvent
    {
      /// <summary>構造体にした場合のサイズ</summary>
      public const int StructSize =
        0 + 4 + 4 + 4 + 1 * 260 + 1 + 1 + 1 + 1 + 4 + 4 + 4 + 1 + 1 + 1 + 1 + 1 + 1;
      /** <summary>4B: 更新なら0、削除なら1</summary> */
      public int isDelete;
      /** <summary>4B: インデックス</summary> */
      public int index;
      /** <summary>4B: サウンドID</summary> */
      public uint soundId;
      /** <summary>260B: 名前</summary> */
      public byte[] soundName = new byte[260];
      /** <summary>1B: ボリューム</summary> */
      public byte volume;
      /** <summary>1B: パディング</summary> */
      public byte padding1;
      /** <summary>1B: パディング</summary> */
      public byte padding2;
      /** <summary>1B: パディング</summary> */
      public byte padding3;
      /** <summary>4B: ピッチ</summary> */
      public float pitch;
      /** <summary>4B: 開始フレーム</summary> */
      public int startFrame;
      /** <summary>4B: 終了フレーム</summary> */
      public int endFrame;
      /** <summary>1B: フレーム処理のオプションフラグ</summary> */
      public byte frameFlag;
      /** <summary>1B: 再生するループ数</summary> */
      public sbyte loopOffset;
      /** <summary>1B: loopCount以降で再生するループ間隔</summary> */
      public byte loopInterval;
      /** <summary>1B: 再生方向</summary> */
      public byte playDirection;
      /** <summary>1B: 再生速度反映用シーケンス変数</summary> */
      public byte playSpeedSequenceVariable;
      /** <summary>1B: パディング</summary> */
      public byte padding4;

      /// <summary>読み込みメソッド</summary>
      public virtual void Read(ProtocolSoundReader reader)
      {
        isDelete = reader.ReadInt32();
        index = reader.ReadInt32();
        soundId = reader.ReadUInt32();
        for (int i = 0; i < 260; i++)
        {
          soundName[i] = reader.ReadByte();
        }
        volume = reader.ReadByte();
        padding1 = reader.ReadByte();
        padding2 = reader.ReadByte();
        padding3 = reader.ReadByte();
        pitch = reader.ReadSingle();
        startFrame = reader.ReadInt32();
        endFrame = reader.ReadInt32();
        frameFlag = reader.ReadByte();
        loopOffset = reader.ReadSByte();
        loopInterval = reader.ReadByte();
        playDirection = reader.ReadByte();
        playSpeedSequenceVariable = reader.ReadByte();
        padding4 = reader.ReadByte();
      }

      /// <summary>書き出しメソッド</summary>
      public virtual void Write(ProtocolSoundWriter writer)
      {
        writer.Write(isDelete);
        writer.Write(index);
        writer.Write(soundId);
        for (int i = 0; i < 260; i++)
        {
          writer.Write(soundName[i]);
        }
        writer.Write(volume);
        writer.Write(padding1);
        writer.Write(padding2);
        writer.Write(padding3);
        writer.Write(pitch);
        writer.Write(startFrame);
        writer.Write(endFrame);
        writer.Write(frameFlag);
        writer.Write(loopOffset);
        writer.Write(loopInterval);
        writer.Write(playDirection);
        writer.Write(playSpeedSequenceVariable);
        writer.Write(padding4);
      }

    }

    //-----------------------------------------------------------------------
    /** <summary>サウンドの状態</summary> */
    public enum ToolSoundOutputMode {
        /** <summary>無効値</summary> */
        OutputMode_Invalid =  0,
        /** <summary>モノラル</summary> */
        OutputMode_Monoral =  1,
        /** <summary>ステレオ</summary> */
        OutputMode_Stereo = 2,
        /** <summary>サラウンド</summary> */
        OutputMode_Surround = 3,
        /** <summary>DPL2</summary> */
        OutputMode_Dpl2 = 4,
    }
#endif

    ///
    public class Constants
    {

        //-------------------------------------------------------------------
        /*
                バージョン情報の定義
                バージョン番号の運用は各モジュールとツールでの運用ルールにあわせてください
             */
        /** <summary>プロトコルバージョン</summary> */
        public const uint PROTOCOL_VERSION = 0x00010002;

        //-------------------------------------------------------------------
        /* メッセージのフラグとマスクの定義 */
        /** <summary>エラー用フラグ        </summary> */
        public const ushort HEADER_ERROR_FLAG = 0x8000;
        /** <summary>要返信メッセージフラグ</summary> */
        public const ushort HEADER_REPLY_FLAG = 0x4000;
        /** <summary>カテゴリ用マスク      </summary> */
        public const ushort HEADER_CATEGORY_MASK = 0x3F00;
        /** <summary>コマンド用マスク      </summary> */
        public const ushort HEADER_COMMAND_MASK = 0x00FF;

        //-------------------------------------------------------------------
        /*
                カテゴリの定義
             */
        /** <summary>システム関連カテゴリ </summary> */
        public const ushort CATEGORY_SYSTEM = 0x0000;
        /** <summary>3Dモデル関連カテゴリ </summary> */
        public const ushort CATEGORY_3D = 0x0100;
        /** <summary>サウンド関連カテゴリ </summary> */
        public const ushort CATEGORY_SOUND = 0x0300;
        /** <summary>MCS通信テストカテゴリ</summary> */
        public const ushort CATEGORY_TEST = 0x3F00;

        //-------------------------------------------------------------------
        /* エラーコード用フラグの定義 */
        /** <summary>エラーフラグ</summary> */
        public const uint SOUND_ERRORFLAG = 0x80000000;

        //-------------------------------------------------------------------
        /* エラーコード用カテゴリの定義 */
        /** <summary>一般</summary> */
        public const uint SOUNDCATEGORY_GENERAL = 0x00000000;
        /** <summary>サウンド</summary> */
        public const uint SOUNDCATEGORY_SOUND = 0x00010000;
        /** <summary>アニメーションサウンド</summary> */
        public const uint SOUNDCATEGORY_ANIMSOUND = 0x00020000;

        //-------------------------------------------------------------------
        /* システムエラーカテゴリのメッセージ */
        /** <summary>未定義コマンドエラー</summary> */
        public const ushort ERR_SYSTEM_UNKNOWN_COMMAND = 0x0 + CATEGORY_SYSTEM + HEADER_ERROR_FLAG;

        //-------------------------------------------------------------------
        /* データエラーカテゴリのメッセージ */
        /** <summary>データロードエラー        </summary> */
        public const ushort ERR_3D_LOAD = 0x0 + CATEGORY_3D + HEADER_ERROR_FLAG;

        //-------------------------------------------------------------------
        /* サウンドカテゴリのメッセージ */
        /** <summary>サウンドエラー</summary> */
        public const ushort ERR_SOUND = 0x0 + CATEGORY_SOUND + HEADER_ERROR_FLAG;

        //-------------------------------------------------------------------
        /* サウンドエラーコード */
        /** <summary>成功。</summary> */
        public const uint SOUND_S_OK = 0x0000;

        /** <summary>エラーが発生しました。        </summary> */
        public const uint SOUND_E_FAILED = 0x0000 + SOUNDCATEGORY_GENERAL + SOUND_ERRORFLAG;
        /** <summary>ファイルが見つかりません。    </summary> */
        public const uint SOUND_E_FILENOTFOUND = 0x0001 + SOUNDCATEGORY_GENERAL + SOUND_ERRORFLAG;
        /** <summary>メモリが不足しています。      </summary> */
        public const uint SOUND_E_OUTOFMEMORY = 0x0002 + SOUNDCATEGORY_GENERAL + SOUND_ERRORFLAG;
        /** <summary>無効なデータです。            </summary> */
        public const uint SOUND_E_INVALIDDATA = 0x0003 + SOUNDCATEGORY_GENERAL + SOUND_ERRORFLAG;
        /** <summary>MCS通信で異常が発生しました。 </summary> */
        public const uint SOUND_E_TRANSFERFAILED = 0x0004 + SOUNDCATEGORY_GENERAL + SOUND_ERRORFLAG;
        /** <summary>無効なアニメーションIDです。      </summary> */
        public const uint SOUND_E_ANIM_INVALIDCOMPONENTID = 0x0001 + SOUNDCATEGORY_ANIMSOUND + SOUND_ERRORFLAG;
        /** <summary>モデルキャッシュが見つかりません。</summary> */
        public const uint SOUND_E_ANIM_MODELCACHENOTFOUND = 0x0002 + SOUNDCATEGORY_ANIMSOUND + SOUND_ERRORFLAG;
        /** <summary>アクションキャッシュが見つかりません。</summary> */
        public const uint SOUND_E_ANIM_ACTIONCACHENOTFOUND = 0x0003 + SOUNDCATEGORY_ANIMSOUND + SOUND_ERRORFLAG;

        //-------------------------------------------------------------------
        /* システム関連カテゴリのメッセージ */
        /** <summary>バージョン要求コマンド</summary> */
        public const ushort TOOL_SYSTEM_VERSION = 0x00 + CATEGORY_SYSTEM + HEADER_REPLY_FLAG;
        /** <summary>同期用コマンド</summary> */
        public const ushort TOOL_SYSTEM_SYNC = 0x01 + CATEGORY_SYSTEM + HEADER_REPLY_FLAG;
        /** <summary>モジュールリセットコマンド</summary> */
        public const ushort TOOL_SYSTEM_RESET = 0x02 + CATEGORY_SYSTEM;

        //-------------------------------------------------------------------
        /* 3Dモデル関連カテゴリのメッセージ */
        /** <summary>モデルデータの読み込みコマンド</summary> */
        public const ushort TOOL_3D_LOAD_MODEL = 0x00 + CATEGORY_3D + HEADER_REPLY_FLAG;
        /** <summary>ロードされているモデルリスト取得コマンド</summary> */
        public const ushort TOOL_3D_GET_MODEL_LIST = 0x01 + CATEGORY_3D + HEADER_REPLY_FLAG;
        /** <summary>モデルのノードリスト取得コマンド</summary> */
        public const ushort TOOL_3D_GET_NODE_LIST = 0x02 + CATEGORY_3D + HEADER_REPLY_FLAG;

        //-------------------------------------------------------------------
        /* サウンド関連カテゴリのメッセージ */
        /** <summary>サウンドシステム関連メッセージ範囲</summary> */
        public const ushort TOOL_SOUND_SYSTEMBEGIN = 0x00 + CATEGORY_SOUND;
        /** <summary>サウンドシステム関連メッセージ範囲</summary> */
        public const ushort TOOL_SOUND_SYSTEMEND = 0x39 + CATEGORY_SOUND;
        /** <summary>サウンドモードへの強制切り替えコマンド</summary> */
        public const ushort TOOL_SOUND_FOCUS = 0x01 + CATEGORY_SOUND;
        /** <summary>エコーコマンド</summary> */
        public const ushort TOOL_SOUND_ECHO = 0x02 + CATEGORY_SOUND + HEADER_REPLY_FLAG;
        /** <summary>全発音停止</summary> */
        public const ushort TOOL_SOUND_ALLSTOP = 0x03 + CATEGORY_SOUND;
        /** <summary>サウンドシステム設定</summary> */
        public const ushort TOOL_SOUND_SET_SYSTEMPARAMETERS = 0x04 + CATEGORY_SOUND;

        /** <summary>サウンド関連メッセージ範囲</summary> */
        public const ushort TOOL_SOUND_SOUNDBEGIN = 0x40 + CATEGORY_SOUND;
        /** <summary>サウンド関連メッセージ範囲</summary> */
        public const ushort TOOL_SOUND_SOUNDEND = 0x79 + CATEGORY_SOUND;
        /** <summary>サウンドアーカイブの読み込みコマンド</summary> */
        public const ushort TOOL_SOUND_LOADSAR = 0x41 + CATEGORY_SOUND + HEADER_REPLY_FLAG;
        /** <summary>サウンドラベルの設定</summary> */
        public const ushort TOOL_SOUND_SET_SOUNDLABELS = 0x42 + CATEGORY_SOUND;
        /** <summary>サウンド状態の取得</summary> */
        public const ushort TOOL_SOUND_GET_SOUNDINFORMATIONS = 0x43 + CATEGORY_SOUND;
        /** <summary>サウンド情報の表示</summary> */
        /** public const ushort TOOL_SOUND_SHOWSOUNDINFORMATION = 0x44 + CATEGORY_SOUND; */
        /** <summary>再生コマンド</summary> */
        public const ushort TOOL_SOUND_PLAY = 0x45 + CATEGORY_SOUND;
        /** <summary>一時停止コマンド</summary> */
        public const ushort TOOL_SOUND_PAUSE = 0x46 + CATEGORY_SOUND;
        /** <summary>停止コマンド</summary> */
        public const ushort TOOL_SOUND_STOP = 0x47 + CATEGORY_SOUND;
        /** <summary>エフェクトパラメータの転送 PC→Viewer</summary> */
        public const ushort TOOL_SOUND_SET_EFFECTPARAMETERS = 0x48 + CATEGORY_SOUND;
        /** <summary>エフェクトパラメータの転送 PC←Viewer</summary> */
        public const ushort TOOL_SOUND_GET_EFFECTPARAMETERS = 0x49 + CATEGORY_SOUND + HEADER_REPLY_FLAG;
        /** <summary>シーケンス変数の転送 PC→Viewer</summary> */
        public const ushort TOOL_SOUND_SET_SEQVARIABLES = 0x50 + CATEGORY_SOUND;
        /** <summary>シーケンス変数の転送 PC←Viewer</summary> */
        public const ushort TOOL_SOUND_GET_SEQVARIABLES = 0x51 + CATEGORY_SOUND;
        /** <summary>プレビュープレイヤーパラメータの転送 PC→Viewer</summary> */
        public const ushort TOOL_SOUND_SET_PREVIEWPLAYERPARAMETER = 0x52 + CATEGORY_SOUND;

        /** <summary>サウンドアーカイブファイルを閉じる</summary> */
        public const ushort TOOL_SOUND_CLOSESAR = 0x53 + CATEGORY_SOUND + HEADER_REPLY_FLAG;

        /** <summary>サウンド情報の編集</summary> */
        public const ushort TOOL_SOUND_EDITSOUNDINFO = 0x54 + CATEGORY_SOUND;

        //-------------------------------------------------------------------
        /** <summary>アニメーションサウンド関連メッセージ範囲</summary> */
        public const ushort TOOL_SOUND_ANIM_SOUNDBEGIN = 0x80 + CATEGORY_SOUND;
        /** <summary>アニメーションサウンド関連メッセージ範囲</summary> */
        public const ushort TOOL_SOUND_ANIM_SOUNDEND = 0xB9 + CATEGORY_SOUND;
        /** <summary>3Dモデルデータの読み込みコマンド</summary> */
        public const ushort TOOL_SOUND_ANIM_LOAD3DMODEL = 0x81 + CATEGORY_SOUND + HEADER_REPLY_FLAG;
        /** <summary>アニメーションデータの読み込みコマンド</summary> */
        public const ushort TOOL_SOUND_ANIM_LOADANIMATION = 0x82 + CATEGORY_SOUND + HEADER_REPLY_FLAG;
        /** <summary>アニメーションサウンドデータの読み込みコマンド</summary> */
        public const ushort TOOL_SOUND_ANIM_LOADANIMSOUND = 0x83 + CATEGORY_SOUND + HEADER_REPLY_FLAG;
        /** <summary>アニメーションキャッシュの破棄コマンド</summary> */
        /*未使用*/
        public const ushort TOOL_SOUND_ANIM_DISPOSECACHE = 0x84 + CATEGORY_SOUND + HEADER_REPLY_FLAG;
        /** <summary>アクティブアニメーションの切り替え</summary> */
        /*未使用*/
        public const ushort TOOL_SOUND_ANIM_ACTIVATEANIMATION = 0x85 + CATEGORY_SOUND + HEADER_REPLY_FLAG;
        /** <summary>アニメーションフレーム取得コマンド</summary> */
        public const ushort TOOL_SOUND_ANIM_QUERYFRAME = 0x86 + CATEGORY_SOUND + HEADER_REPLY_FLAG;
        /** <summary>アニメーションフレーム設定コマンド</summary> */
        public const ushort TOOL_SOUND_ANIM_SETFRAME = 0x87 + CATEGORY_SOUND + HEADER_REPLY_FLAG;
        /** <summary>アニメーション再生</summary> */
        public const ushort TOOL_SOUND_ANIM_PLAY = 0x88 + CATEGORY_SOUND;
        /** <summary>アニメーション停止</summary> */
        public const ushort TOOL_SOUND_ANIM_STOP = 0x89 + CATEGORY_SOUND;
        /** <summary>アニメーション再生レート設定コマンド</summary> */
        public const ushort TOOL_SOUND_ANIM_SETRATE = 0x90 + CATEGORY_SOUND;
        /** <summary>ループ状態設定コマンド</summary> */
        public const ushort TOOL_SOUND_ANIM_SETLOOP = 0x91 + CATEGORY_SOUND;
        /** <summary>アニメーション再生範囲設定コマンド</summary> */
        public const ushort TOOL_SOUND_ANIM_SETRANGE = 0x92 + CATEGORY_SOUND + HEADER_REPLY_FLAG;
        /** <summary>アニメーションサウンドのイベント更新コマンド</summary> */
        public const ushort TOOL_SOUND_ANIM_MODIFIEREVENT = 0x93 + CATEGORY_SOUND;

        //-------------------------------------------------------------------
        /** リアルタイム編集関連メッセージ */

        public const uint SRE_CATEGORY_MASK = 0x00ff0000;
        public const uint SRE_MESSAGE_MASK = 0x00007fff;
        public const ushort SRE_MESSAGE_REPLY_FLAG_MASK = 0x8000;

        public const ushort SRE_MESSAGE_REPLY_FLAG = SRE_MESSAGE_REPLY_FLAG_MASK;

        public const uint SRE_CATEGORY_SYSTEM = 0x00010000;
        public const uint SRE_CATEGORY_SNDEDIT = 0x00020000;
        public const uint SRE_CATEGORY_SNDCTRL = 0x00030000;

        // 生存確認のメッセージです。
        // 実機から定期的に送信されてきます。
        // SRE_CATEGORY_SYSTEM
        public const ushort SRE_SYNC = 0x00;
        public const ushort SRE_SYNC_REPLY = SRE_SYNC + SRE_MESSAGE_REPLY_FLAG;

        // 有効/無効アイテムの問い合わせメッセージです。
        // 実機から定期的に送信されてきます。
        // SRE_CATEGORY_SYSTEM
        public const ushort SRE_QUERY_ITEMS = 0x01;
        public const ushort SRE_QUERY_ITEMS_REPLY = SRE_QUERY_ITEMS + SRE_MESSAGE_REPLY_FLAG;


        //public const ushort SRE_QUERY_INVALID_ITEMS = 0x01; // + SRE_CATEGORY_REALTIMEEDIT
        //public const ushort SRE_QUERY_INVALID_ITEMS_REPLY = SRE_QUERY_INVALID_ITEMS+ SRE_MESSAGE_REPLY_FLAG;

        // アイテム情報の要求メッセージです。
        // SRE_CATEGORY_SNDEDIT
        public const ushort SNDEDIT_GET_ITEM_EDIT_INFO = 0x00;
        public const ushort SNDEDIT_GET_ITEM_EDIT_INFO_REPLY = (SNDEDIT_GET_ITEM_EDIT_INFO +
                                                                 SRE_MESSAGE_REPLY_FLAG);

        // アイテムのキャッシュ開始メッセージです。
        // SRE_CATEGORY_SNDEDIT
        public const ushort SNDEDIT_BEGIN_ITEM_CACHE = 0x01;
        public const ushort SNDEDIT_BEGIN_ITEM_CACHE_REPLY = (SNDEDIT_BEGIN_ITEM_CACHE +
                                                                 SRE_MESSAGE_REPLY_FLAG);

        // アイテムのキャッシュ終了メッセージです。
        // SRE_CATEGORY_SNDEDIT
        public const ushort SNDEDIT_END_ITEM_CACHE = 0x02;
        public const ushort SNDEDIT_END_ITEM_CACHE_REPLY = (SNDEDIT_END_ITEM_CACHE +
                                                                 SRE_MESSAGE_REPLY_FLAG);

        // サウンド再生メッセージです。
        // SRE_CATEGORY_SNDCTRL
        public const ushort SNDCTRL_PLAY_SOUND = 0x00;

        // サウンド停止メッセージです。
        // SRE_CATEGORY_SNDCTRL
        public const ushort SNDCTRL_STOP_SOUND = 0x01;

        // サウンド一時停止メッセージです。
        // SRE_CATEGORY_SNDCTRL
        public const ushort SNDCTRL_PAUSE_SOUND = 0x02;

        // サウンドハンドル情報更新メッセージです。
        // SRE_CATEGORY_SNDCTRL
        public const ushort SNDCTRL_UPDATE_SOUND_HANDLE_INFO = 0x80;
    }
}
