﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Xml;
using System.Xml.Schema;
using System.Xml.Serialization;
using NintendoWare.SoundMaker.Framework.Resources;
using Schemas = NintendoWare.SoundMaker.Framework.Configurations.Schemas;

namespace NintendoWare.SoundMaker.Framework
{
    /// <summary>
    /// コンフィギュレーション管理の基本クラスです。
    /// </summary>
    public abstract class ConfigurationBase
    {
        // コンフィギュレーション情報
        private ConfigurationRoot _root = new ConfigurationRoot();

        private string _loadingFilePath = string.Empty;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="platform">対象プラットフォーム。</param>
        /// <param name="version">コンフィギュレーションファイルのバージョン。</param>
        public ConfigurationBase(string platform, string version)
        {
            if (null == platform) { throw new ArgumentNullException("platform"); }
            if (null == version) { throw new ArgumentNullException("version"); }
            _root.Platform = platform;
            _root.Version = version;
        }

        public event EventHandler Saving;

        protected abstract string RootElementName { get; }

        /// <summary>
        /// コンフィギュレーションファイルを読み込みます。
        /// </summary>
        /// <param name="filePath">入力ファイルパス。</param>
        /// <returns>成功した場合は true、失敗した場合は false。</returns>
        public bool Load(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            bool succeeded = true;

            try
            {

                string basePath = Path.GetDirectoryName(filePath);

                using (XmlReader reader = XmlTextReader.Create(filePath))
                {

                    XmlDocument document = new XmlDocument();
                    document.Load(reader);

                    if (null == document.DocumentElement)
                    {
                        throw new Exception();
                    }

                    ConfigurationRoot root = ReadStartConfigurationRoot(document.DocumentElement as XmlElement);

                    if (!CanLoadConfiguration(root, this._root))
                    {
                        succeeded = true;
                    }
                    else
                    {

                        foreach (XmlNode xmlNode in document.DocumentElement.ChildNodes)
                        {

                            if (xmlNode is XmlElement)
                            {
                                ReadConfigurationElement(xmlNode as XmlElement, basePath);
                            }

                        }

                    }

                }

                // 検証に失敗したらデフォルト設定にする
                if (!Validate())
                {
                    throw new Exception();
                }

            }
            catch (FileNotFoundException)
            {
                succeeded = false;
            }
            catch
            {
                succeeded = false;
            }

            LoadDefaults(!succeeded);

            _loadingFilePath = filePath;

            return succeeded;
        }

        /// <summary>
        /// デフォルト設定を読み込みます。
        /// </summary>
        /// <param name="forced">現在の設定を全て破棄する場合は true、読み込み済みの設定を残す場合は false。</param>
        public void LoadDefaults(bool forced)
        {
            LoadDefaultsInternal(CreateReaderSettings(), forced);
        }

        /// <summary>
        /// 現在の設定をコンフィギュレーションファイルに保存します。
        /// </summary>
        public void Save()
        {
            if (0 == _loadingFilePath.Length) { throw new ApplicationException("configuration not loaded"); }

            OnSaving(EventArgs.Empty);

            Save(_loadingFilePath);
        }

        /// <summary>
        /// 現在の設定をコンフィギュレーションファイルに保存します。
        /// </summary>
        /// <param name="filePath">出力ファイルパス。</param>
        public void Save(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            try
            {

                string basePath = Path.GetDirectoryName(filePath);

                XmlWriterSettings settings = new XmlWriterSettings()
                {
                    Encoding = Encoding.UTF8,
                    Indent = true,
                    IndentChars = "  ",
                };

                using (XmlWriter writer = XmlTextWriter.Create(filePath, settings))
                {

                    writer.WriteStartDocument();
                    WriteStartConfigurationRoot(writer);

                    WriteConfigurationElements(writer, basePath);

                    writer.WriteEndElement();
                    writer.WriteEndDocument();

                }

            }
            catch
            {
                // ★
            }
        }

        /// <summary>
        /// デフォルト設定を読み込みます。
        /// </summary>
        /// <param name="settings">XmlReader 設定。</param>
        /// <param name="forced">現在の設定を全て破棄する場合は true、読み込み済みの設定を残す場合は false。</param>
        protected abstract void LoadDefaultsInternal(XmlReaderSettings settings, bool forced);

        /// <summary>
        /// スキーマセットを作成します。
        /// </summary>
        /// <returns>スキーマセット。</returns>
        protected virtual XmlSchemaSet CreateSchemaSet()
        {
            return new XmlSchemaSet();
        }

        /// <summary>
        /// XmlReader 設定を作成します。
        /// </summary>
        /// <returns>XmlReader 設定。</returns>
        protected XmlReaderSettings CreateReaderSettings()
        {
            return new XmlReaderSettings()
            {
                ValidationType = ValidationType.Schema,
                Schemas = CreateSchemaSet(),
            };
        }

        /// <summary>
        /// コンフィギュレーションを読み込めるかどうかを調べます。
        /// </summary>
        /// <param name="readingInfo">読み込もうとしているコンフィギュレーションの情報を指定します。</param>
        /// <param name="readableInfo">読み込み可能なコンフィギュレーションの情報を指定します。</param>
        /// <returns>読み込める場合は true、読み込めない場合は false。</returns>
        protected virtual bool CanLoadConfiguration(ConfigurationRoot readingInfo, ConfigurationRoot readableInfo)
        {
            return readingInfo.Platform == readableInfo.Platform;
        }

        /// <summary>
        /// コンフィギュレーション要素を読み込みます。
        /// </summary>
        /// <param name="xmlElement">コンフィギュレーション要素。</param>
        /// <param name="basePath">基準パス。</param>
        protected abstract void ReadConfigurationElement(XmlElement xmlElement, string basePath);

        /// <summary>
        /// コンフィギュレーション要素を書き出します。
        /// </summary>
        /// <param name="reader">XmlReader。</param>
        protected abstract void WriteConfigurationElements(XmlWriter writer, string basePath);

        /// <summary>
        /// 読み込んだコンフィギュレーションを検証します。
        /// </summary>
        /// <returns>読み込んだ結果が正常な場合は true、それ以外の場合は false。</returns>
        protected virtual bool Validate()
        {
            return true;
        }

        protected TConfigurationPart ReadConfigurationPart<TConfigurationPart>(string xmlText)
            where TConfigurationPart : class
        {
            return ReadConfigurationPart<TConfigurationPart>(xmlText, null);
        }

        protected TConfigurationPart ReadConfigurationPart<TConfigurationPart>(string xmlText, XmlReaderSettings settings)
            where TConfigurationPart : class
        {
            if (null == xmlText) { throw new ArgumentNullException("xmlText"); }

            XmlReader reader;

            if (null == settings)
            {
                reader = XmlTextReader.Create(new StringReader(xmlText));
            }
            else
            {
                reader = XmlTextReader.Create(new StringReader(xmlText), settings);
            }

            return ReadConfigurationPart<TConfigurationPart>(reader);
        }

        protected TConfigurationPart ReadConfigurationPart<TConfigurationPart>(XmlReader reader)
            where TConfigurationPart : class
        {
            if (null == reader) { throw new ArgumentNullException("reader"); }
            return new XmlSerializer(typeof(TConfigurationPart)).Deserialize(reader) as TConfigurationPart;
        }

        protected void WriteConfigurationPart<TConfigurationPart>(XmlWriter writer, object target)
            where TConfigurationPart : class
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }
            if (null == target) { throw new ArgumentNullException("target"); }
            new XmlSerializer(typeof(TConfigurationPart)).Serialize(writer, target);
        }

        /// <summary>
        /// コンフィギュレーションルートを書き出します。
        /// </summary>
        /// <param name="writer">XmlWriter。</param>
        protected virtual void WriteStartConfigurationRoot(XmlWriter writer)
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }

            writer.WriteStartElement(RootElementName);

            writer.WriteAttributeString("Platform", _root.Platform);
            writer.WriteAttributeString("Version", _root.Version);

            writer.WriteAttributeString("xmlns", "xsi", null, "http://www.w3.org/2001/XMLSchema-instance");
            writer.WriteAttributeString("xmlns", "xs", null, "http://www.w3.org/2001/XMLSchema");
        }

        protected virtual void OnSaving(EventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (null != Saving)
            {
                Saving(this, e);
            }
        }

        /// <summary>
        /// コンフィギュレーションルートを読み込みます。
        /// </summary>
        /// <param name="reader">XmlReader。</param>
        /// <returns>ルート要素の情報。</returns>
        private ConfigurationRoot ReadStartConfigurationRoot(XmlElement xmlRoot)
        {
            if (null == xmlRoot) { throw new ArgumentNullException("xmlRoot"); }

            string platform = xmlRoot.GetAttribute("Platform");
            string version = xmlRoot.GetAttribute("Version");

            if (null == _root.Platform || null == _root.Version)
            {
                throw new ApplicationException("invalid configuration file.");
            }

            return new ConfigurationRoot
            {
                Platform = platform,
                Version = version,
            };
        }
    }

    /// <summary>
    /// コンフィギュレーションファイルのルート情報
    /// </summary>
    public class ConfigurationRoot
    {
        private string _platform = string.Empty;
        private string _version = "0.0.0.0";

        /// <summary>
        /// 対象プラットフォームを取得または設定します。
        /// </summary>
        public string Platform
        {
            get { return _platform; }
            set
            {
                if (null == value) { throw new ArgumentNullException("value"); }
                _platform = value;
            }
        }

        /// <summary>
        /// バージョン情報を取得または設定します。
        /// </summary>
        public string Version
        {
            get { return _version; }
            set
            {
                if (null == value) { throw new ArgumentNullException("value"); }

                if (!Regex.IsMatch(value, @"^\d+\.\d+\.\d+.\d+$"))
                {
                    throw new ArgumentException("invalid version string.");
                }

                _version = value;
            }
        }
    }
}
