﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.CommandHandlers
{
    using System;
    using System.IO;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;
    using CoreResources = NintendoWare.SoundFoundation.Resources;

    /// <summary>
    /// サウンドセットを開きます。
    /// </summary>
    public abstract class ImportSoundSetHandler : CommandHandler
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="queryTarget">対象の問い合わせ先。</param>
        protected ImportSoundSetHandler(IQueryCommandParameter queryParameter) :
            base(queryParameter)
        {
        }

        /// <summary>
        /// コマンドハンドラが実行可能なコマンドIDを取得します。
        /// </summary>
        public override string TargetCommandID
        {
            get { return FileCommands.ImportSoundSet.ID; }
        }

        /// <summary>
        /// アプリケーションを取得します。
        /// </summary>
        protected ApplicationBase Application
        {
            get { return ApplicationBase.Instance; }
        }

        /// <summary>
        /// File.ImportSoundSet コマンドを実行します。
        /// </summary>
        /// <param name="projectService">プロジェクトサービス。</param>
        /// <param name="filePath">ファイルパス。</param>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        protected bool ExecuteInternal(SoundProjectService projectService, string filePath)
        {
            FileResource fileResource = new FileResource(filePath);

            string ext =
               FormsApplication.Instance.Traits.IntermediateOutputTraits.SoundSetFileExtension;
            string path = Path.GetDirectoryName(filePath);
            string logFilePath =
                Path.GetFileNameWithoutExtension(filePath) + "." + ext + ".import.txt";
            logFilePath = Path.Combine(path, logFilePath);
            ImportLog.Open(logFilePath);

            try
            {
                ImportLog.WriteLine(
                                    CoreResources.MessageResource.Message_ImportFileRead,
                                    filePath
                                    );
                using (DocumentReference documentRef =
                       ApplicationBase.Instance.DocumentService.OpenDocument(fileResource))
                {
                    if (documentRef == null)
                    {
                        throw new FileNotSupportedException(filePath);
                    }

                    SoundIntermediateOutputTraits traits =
                        ApplicationBase.Instance.Traits.IntermediateOutputTraits;
                    string typeName = traits.SoundSetDocumentTypeName;
                    Document document = documentRef.Document;
                    using (
                           DocumentReference soundSetDocumentRef =
                           ApplicationBase.Instance.DocumentImportService.Import(
                                                                                 document,
                                                                                 typeName
                                                                                 )
                           )
                    {
                        if (soundSetDocumentRef == null)
                        {
                            return true;
                        }
                        else
                        {
                            AddSoundSetHandler.Execute(projectService, soundSetDocumentRef, projectService.Project);
                        }
                    }
                }
            }
            catch (Exception e)
            {
                string msg = e.Message.Replace("\r\n", "\n");
                msg = msg.Replace("\r", "\n");
                msg = msg.Replace("\n", "\r\n");
                ImportLog.WriteLine(msg);
                if (e.Data.Contains("ImportFilePath") == true)
                {
                    ImportLog.Error((string)(e.Data["ImportFilePath"]));
                }
                else
                {
                    ImportLog.Error(filePath);
                }
                ImportLog.WriteLine(CoreResources.MessageResource.Message_ImportEnd);
                ImportLog.Close();
                ApplicationBase.Instance.DocumentImportService.OnImportError();
                return true;
            }
            finally
            {
                ImportLog.Close();
            }

            return true;
        }

        /// <summary>
        /// コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <returns>コマンドの状態。</returns>
        protected override CommandStatus QueryStatusInternal(
                                                             Command command,
                                                             IQueryCommandParameter parameters
                                                             )
        {
            SoundProjectService projectService = GetTargetProjectService(parameters);
            return (null != projectService && null != projectService.Project) ?
                CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// 対象プロジェクトを取得します。
        /// </summary>
        protected SoundProjectService GetTargetProjectService(IQueryCommandParameter parameters)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }

            if (!parameters.ContainsParameter(CommandParameterNames.TargetComponentService))
            {
                return null;
            }

            return parameters.GetParameter(CommandParameterNames.TargetComponentService)
                as SoundProjectService;
        }
    }
}
