﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Data;
using System.IO;
using System.Text;
using System.Windows.Forms;
using NintendoWare.SoundMaker.Preview;
using NintendoWare.SoundFoundation.Core.Drawing;
using NintendoWare.SoundFoundation.Windows.Forms;
using NintendoWare.SoundFoundation.Windows.Forms.Windowless;
using NintendoWare.SoundMaker.Framework.Windows;
using NintendoWare.SoundMaker.Framework.Windows.Forms;
using FrameworkResources = NintendoWare.SoundMaker.Framework.Resources;
using MCS = NintendoWare.SoundMaker.Preview.MCS;

namespace NintendoWare.SoundMaker.Windows.Forms
{
    public partial class SequenceVariablePanel : DockingPage, IToolWindowPage, IKeyPreviewReceiver
    {
        public const string PageName = "SequenceVariable";

        public SequenceVariablePanel()
            : base(PageName)
        {
            InitializeComponent();
            InitializeMenu();
            InitializeVariablesList();

            Text = FrameworkResources.MessageResource.ToolWindowName_SequenceVariable;
            Image = FrameworkResources.ImageResource.
                    BitmapIconSeqVariableWindow.MakeNewTransparent(UIServiceBase.TransparentColor);

            DockVisible = false;

            // 初期サイズを設定する
            FloatingBounds = new Rectangle(-1, -1,
                                            Width + (SystemInformation.FrameBorderSize.Width * 2),
                                            Height + SystemInformation.CaptionHeight + SystemInformation.FrameBorderSize.Height);
        }

        ///
        public SoundSetPanelCommon SoundSetPanel
        {
            get;
            set;
        }

        #region ** プロパティ

        protected PreviewPlayerManager PreviewPlayerManager
        {
            get { return PreviewManager.Instance.PreviewPlayerManager; }
        }

        #endregion

        #region ** イベントハンドラ

        private void OnProjectOpened(object sender, EventArgs e)
        {
            // ツールバーを初期化する
            _toolStripSeqVariable.Enabled = true;
            UpdateToolStrip();
        }

        private void OnProjectClosing(object sender, EventArgs e)
        {
            DisableToolStrip();
        }

        private void OnViewerSynchronousChanged(object sender, EventArgs e)
        {
            // 非表示の場合は同期しない
            if (!Visible) { return; }

            SynchronizeAllVariables();
        }

        /// <summary>
        /// プレビュープレイヤーの選択が変更されると発生します。
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnPreviewPlayerSelectionChanged(object sender, EventArgs e)
        {
            // シーケンス変数の更新ターゲットを設定する
            SetTargetPreviewPlayer(PreviewPlayerManager.SelectedPreviewPlayer);
        }

        /// <summary>
        /// シーケンス変数の現在値が変更されると発生します。
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnVariableCurrentValueChanged(object sender, EventArgs e)
        {
            _variablesEdit.UpdateCurrentValue();
        }

        private void OnAutoWriteItemClicked(object sender, ToolStripItemClickedEventArgs e)
        {
            PreviewPlayerManager.Settings.AutoWriteVariables = (bool)e.ClickedItem.Tag;

            // コントロールを更新する
            UpdateToolStrip();
        }

        private void OnSyncTypeItemClicked(object sender, ToolStripItemClickedEventArgs e)
        {
            PreviewPlayerManager.Settings.VariablesSyncType = (SeqVariableSyncType)e.ClickedItem.Tag;

            // コントロールを更新する
            UpdateToolStrip();
        }

        private void OnVariablesTreeSelectionChanged(object sender, TreeViewEventArgs e)
        {
            TreeView target = sender as TreeView;
            Debug.Assert(null != target, "unexpected error");

            if (null == target.SelectedNode) { return; }


            VariablesTreeItem targetItem = target.SelectedNode as VariablesTreeItem;
            Debug.Assert(null != targetItem, "unexpected error");

            SetEditor(targetItem.Editor);
        }

        #endregion

        #region ** イベントハンドラのオーバーライド

        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);

            // イベントハンドラを登録する
            FormsApplicationCommon.Instance.ProjectService.Opened += OnProjectOpened;
            FormsApplicationCommon.Instance.ProjectService.Closing += OnProjectClosing;
            PreviewPlayerManager.ViewerSynchronousChanged += OnViewerSynchronousChanged;
            PreviewPlayerManager.PreviewPlayerSelectChanged += OnPreviewPlayerSelectionChanged;

            // ツールバーを無効化する
            DisableToolStrip();

            // 対象シーケンス変数を更新する
            if (null != PreviewPlayerManager.SelectedPreviewPlayer)
            {
                SetTargetPreviewPlayer(PreviewPlayerManager.SelectedPreviewPlayer);
            }

            // すでに開かれている場合は、ツールバーを更新する
            if (null != FormsApplicationCommon.Instance.ProjectService.ProjectDocument)
            {
                _toolStripSeqVariable.Enabled = true;
                UpdateToolStrip();
            }
        }

        protected override void OnHandleDestroyed(EventArgs e)
        {
            // イベントハンドラの登録を解除する
            PreviewPlayerManager.ViewerSynchronousChanged -= OnViewerSynchronousChanged;
            PreviewPlayerManager.PreviewPlayerSelectChanged -= OnPreviewPlayerSelectionChanged;

            base.OnHandleDestroyed(e);
        }

        protected override void OnActiveChanged(EventArgs e)
        {
            if (!Active)
            {

                // 更新対象から外す
                if (null != PreviewPlayerManager && null != PreviewPlayerManager.Settings)
                {
                    PreviewPlayerManager.Settings.SyncTargetVariables = null;
                }

                return;

            }

            Debug.Assert(null != PreviewPlayerManager, "unexpected error");

            // 対象シーケンス変数を更新する
            if (null != PreviewPlayerManager.SelectedPreviewPlayer)
            {
                SetTargetPreviewPlayer(PreviewPlayerManager.SelectedPreviewPlayer);
            }

            base.OnActiveChanged(e);
        }

        protected override void OnVisibleChanged(EventArgs e)
        {
            if (Visible)
            {
                SynchronizeAllVariables();
            }

            base.OnVisibleChanged(e);
        }

        #endregion

        #region ** メソッド

        private void InitializeMenu()
        {
            ToolStripMenuItem menuItemAutoWriteOn = new System.Windows.Forms.ToolStripMenuItem();
            ToolStripMenuItem menuItemAutoWriteOff = new System.Windows.Forms.ToolStripMenuItem();

            //
            // menuItemAutoWriteOn
            //
            menuItemAutoWriteOn.Name = "menuItemAutoWriteOn";
            menuItemAutoWriteOn.Tag = true;
            menuItemAutoWriteOn.Text = Resources.MessageResource.MenuText_SequenceVariable_AutoWriteOn;
            //
            // menuItemAutoWriteOff
            //
            menuItemAutoWriteOff.Name = "menuItemAutoWriteOff";
            menuItemAutoWriteOff.Tag = false;
            menuItemAutoWriteOff.Text = Resources.MessageResource.MenuText_SequenceVariable_AutoWriteOff;

            // DropDownメニューに追加する
            _toolButtonAutoWrite.DropDownItems.AddRange(
                new System.Windows.Forms.ToolStripItem[] {
                    menuItemAutoWriteOn,
                    menuItemAutoWriteOff
                });


            ToolStripMenuItem menuItemSync50ms = new System.Windows.Forms.ToolStripMenuItem();
            ToolStripMenuItem menuItemSync100ms = new System.Windows.Forms.ToolStripMenuItem();
            ToolStripMenuItem menuItemSync500ms = new System.Windows.Forms.ToolStripMenuItem();
            ToolStripMenuItem menuItemSync1000ms = new System.Windows.Forms.ToolStripMenuItem();

            //
            // menuItemSync50ms
            //
            menuItemSync50ms.Name = "menuItemSync50ms";
            menuItemSync50ms.Tag = SeqVariableSyncType.Auto50ms;
            menuItemSync50ms.Text = Resources.MessageResource.MenuText_SequenceVariable_SyncAuto50ms;
            //
            // menuItemSync100ms
            //
            menuItemSync100ms.Name = "menuItemSync100ms";
            menuItemSync100ms.Tag = SeqVariableSyncType.Auto100ms;
            menuItemSync100ms.Text = Resources.MessageResource.MenuText_SequenceVariable_SyncAuto100ms;
            //
            // menuItemSync500ms
            //
            menuItemSync500ms.Name = "menuItemSync500ms";
            menuItemSync500ms.Tag = SeqVariableSyncType.Auto500ms;
            menuItemSync500ms.Text = Resources.MessageResource.MenuText_SequenceVariable_SyncAuto500ms;
            //
            // menuItemSync1000ms
            //
            menuItemSync1000ms.Name = "menuItemSync1000ms";
            menuItemSync1000ms.Tag = SeqVariableSyncType.Auto1000ms;
            menuItemSync1000ms.Text = Resources.MessageResource.MenuText_SequenceVariable_SyncAuto1000ms;

            // DropDownメニューに追加する
            _toolButtonSync.DropDownItems.AddRange(
                new System.Windows.Forms.ToolStripItem[] {
                    menuItemSync50ms,
                    menuItemSync100ms,
                    menuItemSync500ms,
                    menuItemSync1000ms
                });
        }

        private void InitializeVariablesList()
        {
            string trackTypeName = Resources.MessageResource.Label_SeqVariable_Track;

            VariablesTreeItem[] trackItems = {  new VariablesTreeItem( new SeqTrackVariableEditor( 0 ) ),
                                                new VariablesTreeItem( new SeqTrackVariableEditor( 1 ) ),
                                                new VariablesTreeItem( new SeqTrackVariableEditor( 2 ) ),
                                                new VariablesTreeItem( new SeqTrackVariableEditor( 3 ) ),
                                                new VariablesTreeItem( new SeqTrackVariableEditor( 4 ) ),
                                                new VariablesTreeItem( new SeqTrackVariableEditor( 5 ) ),
                                                new VariablesTreeItem( new SeqTrackVariableEditor( 6 ) ),
                                                new VariablesTreeItem( new SeqTrackVariableEditor( 7 ) ),
                                                new VariablesTreeItem( new SeqTrackVariableEditor( 8 ) ),
                                                new VariablesTreeItem( new SeqTrackVariableEditor( 9 ) ),
                                                new VariablesTreeItem( new SeqTrackVariableEditor( 10 ) ),
                                                new VariablesTreeItem( new SeqTrackVariableEditor( 11 ) ),
                                                new VariablesTreeItem( new SeqTrackVariableEditor( 12 ) ),
                                                new VariablesTreeItem( new SeqTrackVariableEditor( 13 ) ),
                                                new VariablesTreeItem( new SeqTrackVariableEditor( 14 ) ),
                                                new VariablesTreeItem( new SeqTrackVariableEditor( 15 ) ) };

            TreeNode[] items = {    new VariablesTreeItem( new SeqGlobalVariableEditor( PreviewPlayerManager ) ),
                                    new VariablesTreeItem( new SeqLocalVariableEditor() ),
                                    new VariablesTreeItem( new SeqVariableEditor( trackTypeName ), trackItems, true ) };

            _variablesTree.Nodes.AddRange(items);

            _variablesTree.SelectedNode = _variablesTree.Nodes[1];
        }

        private void SetEditor(SeqVariableEditor targetEditer)
        {
            if (_variablesEdit.Editor == (targetEditer as ISeqVariableEdit)) { return; }

            targetEditer.TargetPreviewPlayer = PreviewPlayerManager.SelectedPreviewPlayer;
            _variablesEdit.Editor = targetEditer;

            UpdateEdit();
        }

        private void SetTargetPreviewPlayer(PreviewPlayer targetPreviewPlayer)
        {
            if (null == targetPreviewPlayer) { throw new ArgumentNullException("targetPreviewPlayer"); }
            if (null == _variablesEdit.Editor) { return; }

            // 対象プレビュープレイヤーを変更する
            (_variablesEdit.Editor as SeqVariableEditor).TargetPreviewPlayer = targetPreviewPlayer;

            UpdateEdit();
        }

        private void SynchronizeAllVariables()
        {
            PreviewPlayerManager.SetAllVariables();
        }

        private void UpdateEdit()
        {
            // 値を更新する
            _variablesEdit.UpdateEditValue();
            _variablesEdit.UpdateCurrentValue();

            // 変数タイトルラベルを更新する
            Debug.Assert(null != _variablesEdit, "unexpected error");
            _labelVariables.Text = _variablesEdit.TargetName;

            // シーケンス変数の更新ターゲットを設定する
            if (null != PreviewPlayerManager.Settings.SyncTargetVariables)
            {
                PreviewPlayerManager.Settings.SyncTargetVariables.ValueChanged -= OnVariableCurrentValueChanged;
            }

            if (null == _variablesEdit.Editor.VariablesSet)
            {
                PreviewPlayerManager.Settings.SyncTargetVariables = null;
                _variablesEdit.Visible = false;
            }
            else
            {
                PreviewPlayerManager.Settings.SyncTargetVariables = _variablesEdit.Editor.VariablesSet.Current;
                PreviewPlayerManager.Settings.SyncTargetVariables.ValueChanged += OnVariableCurrentValueChanged;
                _variablesEdit.Visible = true;
            }
        }

        private void UpdateToolStrip()
        {
            ToolStripMenuItem targetAutoWrite = null;
            ToolStripMenuItem targetSync = null;

            // チェックを切り替える
            foreach (ToolStripMenuItem item in _toolButtonAutoWrite.DropDownItems)
            {

                if (PreviewPlayerManager.Settings.AutoWriteVariables == (bool)item.Tag)
                {
                    item.Checked = true;
                    targetAutoWrite = item;
                }
                else
                {
                    item.Checked = false;
                }

            }
            foreach (ToolStripMenuItem item in _toolButtonSync.DropDownItems)
            {

                if (PreviewPlayerManager.Settings.VariablesSyncType == (SeqVariableSyncType)item.Tag)
                {
                    item.Checked = true;
                    targetSync = item;
                }
                else
                {
                    item.Checked = false;
                }

            }

            // 選択文字列を差し替える
            _toolButtonAutoWrite.Text = targetAutoWrite.Text;
            _toolButtonSync.Text = (null == targetSync) ? "-" : targetSync.Text;

            _variablesEdit.UpdateControls();
        }

        private void DisableToolStrip()
        {
            _toolStripSeqVariable.Enabled = false;
        }

        #endregion

        ///
        void IKeyPreviewReceiver.KeyDown(KeyEventArgs e)
        {
            switch (e.KeyCode)
            {
                case Keys.F9:
                    if (SoundSetPanel != null)
                    {
                        SoundSetPanel.Play();
                    }
                    break;

                case Keys.F10:
                    if (SoundSetPanel != null)
                    {
                        SoundSetPanel.StopAll();
                    }
                    break;
            }
        }

        //
        void IKeyPreviewReceiver.KeyUp(KeyEventArgs e)
        {
        }

        #region ** 内部クラス

        #region ** リストアイテム

        private class VariablesTreeItem : TreeNode
        {
            SeqVariableEditor _editor;

            public VariablesTreeItem(SeqVariableEditor editor)
                : base(editor.TypeName)
            {
                if (null == editor) { throw new ArgumentNullException("editor"); }
                _editor = editor;
            }

            public VariablesTreeItem(SeqVariableEditor editor, VariablesTreeItem[] children, bool expand)
                : base(editor.TypeName, children)
            {
                if (null == editor) { throw new ArgumentNullException("editor"); }
                _editor = editor;

                if (expand)
                {
                    Expand();
                }
            }

            #region ** プロパティ

            public SeqVariableEditor Editor
            {
                get { return _editor; }
            }

            #endregion
        }

        #endregion

        #region ** 各シーケンス変数用のエディタ

        private class SeqVariableEditor : ISeqVariableEdit
        {
            #region ** パラメータ

            private string _typeName;
            private PreviewPlayerManager _previewPlayerManager;
            private PreviewPlayer _targetPreviewPlayer;

            #endregion

            public SeqVariableEditor() { }

            public SeqVariableEditor(string typeName)
            {
                _typeName = typeName;
            }

            public SeqVariableEditor(PreviewPlayerManager playerManager)
            {
                if (null == playerManager) { throw new ArgumentNullException("playerManager"); }
                _previewPlayerManager = playerManager;
            }

            #region ** プロパティ

            public PreviewPlayerManager PreviewPlayerManager
            {
                get { return _previewPlayerManager; }
                set { _previewPlayerManager = value; }
            }

            public PreviewPlayer TargetPreviewPlayer
            {
                get { return _targetPreviewPlayer; }
                set { _targetPreviewPlayer = value; }
            }

            #endregion

            #region ** ISeqVariableEdit プロパティの実装

            public virtual string TypeName { get { return _typeName; } }
            public virtual string TargetName { get { return string.Empty; } }

            public virtual SeqVariableContainerSet VariablesSet { get { return null; } }

            #endregion

            #region ** ISeqVariableEdit メソッドの実装

            public virtual void ApplyVariables() { }

            #endregion
        }

        private class SeqGlobalVariableEditor : SeqVariableEditor
        {
            public SeqGlobalVariableEditor(PreviewPlayerManager playerManager) : base(playerManager) { }

            #region ** ISeqVariableEdit プロパティの実装

            public override string TypeName
            {
                get { return Resources.MessageResource.Label_SeqVariable_Global; }
            }

            public override string TargetName
            {
                get { return Resources.MessageResource.Label_SeqVariable_PathGlobal; }
            }

            public override SeqVariableContainerSet VariablesSet
            {
                get { return PreviewPlayerManager.GlobalVariables; }
            }

            #endregion

            #region ** ISeqVariableEdit メソッドの実装

            public override void ApplyVariables()
            {
                PreviewPlayerManager.SetGlobalVariables();
            }

            #endregion
        }

        private class SeqLocalVariableEditor : SeqVariableEditor
        {
            #region ** ISeqVariableEdit プロパティの実装

            public override string TypeName
            {
                get { return Resources.MessageResource.Label_SeqVariable_Local; }
            }

            public override string TargetName
            {
                get
                {
                    return string.Format(
                                Resources.MessageResource.Label_SeqVariable_PathLocal,
                                VariablesSet.Current.ID.PreviewPlayerIndex);
                }
            }

            public override SeqVariableContainerSet VariablesSet
            {
                get
                {
                    Debug.Assert(null != TargetPreviewPlayer, "unexpected error");
                    return TargetPreviewPlayer.LocalVariables;
                }
            }

            #endregion

            #region ** ISeqVariableEdit メソッドの実装

            public override void ApplyVariables()
            {
                TargetPreviewPlayer.SetLocalVariables();
            }

            #endregion
        }

        private class SeqTrackVariableEditor : SeqVariableEditor
        {
            #region ** パラメータ

            private uint _trackNo;	// トラック番号

            #endregion

            public SeqTrackVariableEditor(uint trackNo)
            {
                _trackNo = trackNo;
            }

            #region ** ISeqVariableEdit プロパティの実装

            public override string TypeName
            {
                get
                {
                    return Resources.MessageResource.Label_SeqVariable_Track + _trackNo.ToString();
                }
            }

            public override string TargetName
            {
                get
                {
                    return string.Format(
                        Resources.MessageResource.Label_SeqVariable_PathTrack,
                        VariablesSet.Current.ID.PreviewPlayerIndex, _trackNo);
                }
            }

            public override SeqVariableContainerSet VariablesSet
            {
                get
                {
                    Debug.Assert(null != TargetPreviewPlayer, "unexpected error");
                    return TargetPreviewPlayer.TrackVariables.Items[(int)_trackNo];
                }
            }

            #endregion

            #region ** ISeqVariableEdit メソッドの実装

            public override void ApplyVariables()
            {
                TargetPreviewPlayer.SetTrackVariables(_trackNo);
            }

            #endregion
        }

        #endregion

        #endregion
    }
}
