﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundMaker.Windows.Forms
{
    using System;
    using System.Collections.ObjectModel;
    using System.ComponentModel;
    using System.Diagnostics;
    using System.Drawing;
    using System.Linq;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Core.Drawing;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Preview.Communications;
    using NintendoWare.SoundMaker.Framework.Windows;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;
    using NintendoWare.SoundMaker.Preview;
    using NintendoWare.SoundMaker.Preview.Communications;
    using FrameworkResources = NintendoWare.SoundMaker.Framework.Resources;
    using MCS = NintendoWare.SoundMaker.Preview.MCS;
    using Win32 = NintendoWare.SoundFoundation.Core.Win32;

    public partial class PreviewPlayerPanel :
        DockingPage, IToolWindowPage, ICommandTarget
    {
        public const string PageName = "PreviewPlayer";

        private CommandBindingCollection commandBindings = new CommandBindingCollection();

        private PreviewPlayerControlCollection _previewPlayers = new PreviewPlayerControlCollection();
        private PreviewPlayerControl _selectedControl = null;

        public PreviewPlayerPanel()
            : base(PageName)
        {
            InitializeComponent();

            Text = FrameworkResources.MessageResource.ToolWindowName_PreviewPlayer;
            Image = FrameworkResources.ImageResource.
                    BitmapIconPreviewPlayerWindow.MakeNewTransparent(UIServiceBase.TransparentColor);

            toolStripButtonDelete.Image = FrameworkResources.ImageResource.BitmapIconDeleteFilterText;

            // 初期サイズを設定する
            FloatingBounds = new Rectangle(-1, -1,
                                            Width + (SystemInformation.FrameBorderSize.Width * 2),
                                            Height + SystemInformation.CaptionHeight + SystemInformation.FrameBorderSize.Height);

            Initialize();

            //
            this.CommunicationService.ViewerConnectionChanged +=
                delegate (object sender, ConnectionChangedEventArgs e)
                    {
                        foreach (PreviewPlayerControl control in this._previewPlayers)
                        {
                            control.UpdateIcon();
                        }
                    };

            // ToolStrip のアイテムの Tag に設定されたコマンド ID を Command インスタンスに変換しておきます。
            // (Tag が設定されていないものはパネル内のローカルな処理で Click イベントで実装します。）
            foreach (ToolStripItem item in toolStrip.Items)
            {
                var commandID = item.Tag as string;
                if (commandID != null)
                {
                    if (FormsApplication.Instance.CommandService.Commands.Contains(commandID))
                    {
                        var command = FormsApplication.Instance.CommandService.Commands[commandID];
                        item.Tag = command;
                        item.Image = command.Image;
                    }
                }
            }

            // ToolStrip のアイテムが押されたらコマンドを発行します。
            toolStrip.ItemClicked += (sender, args) =>
            {
                var command = args.ClickedItem.Tag as Command;
                if (command != null)
                {
                    FormsApplication.Instance.ExecuteCommand(command);
                }
            };
        }

        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);

            UpdateToolStripStates();
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 選択中のアイテムを取得または設定します。
        /// </summary>
        [Category("Behavior")]
        [Browsable(false)]
        [ReadOnly(true)]
        public PreviewPlayerControl SelectedItem
        {
            get { return _selectedControl; }
        }

        [Category("Behavior")]
        [ReadOnly(true)]
        public uint ItemCount
        {
            get { return (uint)PreviewPlayerManager.PreviewPlayers.Count; }
        }

        [Browsable(false)]
        [ReadOnly(true)]
        public bool CanPlayAll
        {
            get
            {
                foreach (PreviewPlayerControl control in _previewPlayers)
                {
                    if (control.PreviewPlayer.CanPlay)
                    {
                        return true;
                    }
                }

                return false;
            }
        }

        [Browsable(false)]
        [ReadOnly(true)]
        public bool CanPauseAll
        {
            get
            {
                foreach (PreviewPlayerControl control in _previewPlayers)
                {
                    if (control.PreviewPlayer.CanPause) { return true; }
                }

                return false;
            }
        }

        [Browsable(false)]
        [ReadOnly(true)]
        public bool CanStopAll
        {
            get
            {
                foreach (PreviewPlayerControl control in _previewPlayers)
                {
                    if (control.PreviewPlayer.CanStop) { return true; }
                }

                return false;
            }
        }

        [Browsable(false)]
        [ReadOnly(true)]
        private bool CanDelete
        {
            get
            {
                return _previewPlayers.Any(it => it.Item != null);
            }
        }

        [Browsable(false)]
        [ReadOnly(true)]
        private PreviewPlayerManager PreviewPlayerManager
        {
            get { return PreviewManager.Instance.PreviewPlayerManager; }
        }

        [Browsable(false)]
        [ReadOnly(true)]
        private PreviewPlayer SelectedPreviewPlayer
        {
            get { return PreviewPlayerManager.SelectedPreviewPlayer; }
        }

        [Browsable(false)]
        [ReadOnly(true)]
        private NTabControl OwnerTabControl
        {
            get { return Parent as NTabControl; }
        }

        private CommunicationService CommunicationService
        {
            get
            {
                return FormsApplicationCommon.Instance.CommunicationService;
            }
        }

        //-----------------------------------------------------------------

        public void Initialize()
        {
            Debug.Assert(null != PreviewPlayerManager, "unexpected Error");
            if (PreviewPlayerManager.PreviewPlayers.Count == 0) { throw new Exception(); }

            PreviewPlayerManager.PreviewPlayerSelectChanged += OnPreviewPlayerSelectChanged;
            FormsApplication.Instance.ProjectService.ComponentsAdded += OnComponentsChanged;
            FormsApplication.Instance.ProjectService.ComponentsRemoved += OnComponentsChanged;

            SuspendLayout();

            // 新しいコントロールコレクションを作成する
            PreviewPlayerControlCollection oldCollection = DetachContainer();
            PreviewPlayerControlCollection newCollection = new PreviewPlayerControlCollection();

            int controlTop = 0;

            for (int index = 0; index < PreviewPlayerManager.PreviewPlayers.Count; index++)
            {

                PreviewPlayerControl newPreviewPlayer;

                if (null != oldCollection && index < oldCollection.Count)
                {
                    newPreviewPlayer = oldCollection[index];
                }
                else
                {

                    newPreviewPlayer = new PreviewPlayerControl(PreviewPlayerManager.PreviewPlayers[index]);

                    newPreviewPlayer.Anchor = AnchorStyles.Top | AnchorStyles.Left | AnchorStyles.Right;
                    newPreviewPlayer.AllowDrop = true;
                    newPreviewPlayer.Name = "PreviewPlayer" + index.ToString();
                    newPreviewPlayer.Location = new Point(0, controlTop);
                    newPreviewPlayer.Size = new Size(panelPreviewPlayers.ClientRectangle.Width, newPreviewPlayer.Height);
                    newPreviewPlayer.TabStop = false;
                    newPreviewPlayer.TabIndex = index;

                    newPreviewPlayer.SelectedChanged += OnPreviewPlayerSelectedChanged;
                    newPreviewPlayer.DialogKeyProcessed += OnDialogKeyProcessed;
                    newPreviewPlayer.PreviewPlayer.SoundChanged += OnPreviewPlayerItemChanged;

                }

                panelPreviewPlayers.Controls.Add(newPreviewPlayer);
                newCollection.Add(newPreviewPlayer);

                controlTop += newPreviewPlayer.Height - 1; // ボーターの１ピクセル分を重ねます。

            }

            AttachContainer(newCollection);

            this.ResumeLayout();

            this.InitializeCommandBindings();
        }

        /// <summary>
        ///
        /// </summary>
        public PreviewPlayer GetPreviewPlayer(Sound sound)
        {
            foreach (PreviewPlayerControl control in this._previewPlayers)
            {
                if (control.PreviewPlayer.Item == sound)
                {
                    return control.PreviewPlayer;
                }
            }
            return null;
        }

        /// <summary>
        ///
        /// </summary>
        public void SelectItem(uint targetIndex)
        {
            if (null != _selectedControl)
            {
                _selectedControl.Selected = false;
            }

            if (targetIndex < _previewPlayers.Count)
            {
                _selectedControl = _previewPlayers[(int)targetIndex];
                _selectedControl.Selected = true;
            }

            UpdateToolStripStates();
        }

        public void UpdateItems()
        {
            foreach (PreviewPlayerControl control in _previewPlayers)
            {
                control.UpdateItem();
            }
        }

        private void UpdateToolStripStates()
        {
            foreach (ToolStripItem item in toolStrip.Items)
            {
                var command = item.Tag as Command;
                if (command != null)
                {
                    item.Enabled = FormsApplication.Instance.QueryCommandStatus(command).IsEnabled();
                }
            }

            toolStripButtonDelete.Enabled = CanDelete;
        }

        /// <summary>
        /// 登録されているサウンドを全て、同時に再生開始します。
        /// </summary>
        public void PlayAll()
        {
            FormsApplicationCommon.Instance.LoudnessService.Stop();

            foreach (var control in _previewPlayers)
            {
                if (control.PreviewPlayer.CanPlay == false)
                {
                    continue;
                }

                var sound = control.Item;
                if (sound == null)
                {
                    continue;
                }

                PreviewPlayerOperator.CheckExceptionForPlay(() =>
                {
                    try
                    {
                        // ストリームサウンドの場合は Mute, Solo 設定をプレビューに反映させます。
                        // （ラウドネス計算など、他のプレビュー再生に作用しないようにするため、設定は再生開始後ただちに削除します。）
                        PreviewStreamSound.AddPlayMuteParameter(sound as StreamSoundBase);
                        control.PreviewPlayer.Play();
                    }
                    catch
                    {
                        control.Item = null;
                        throw;
                    }
                    finally
                    {
                        PreviewStreamSound.RemovePlayMuteParameter(sound as StreamSoundBase);
                    }
                });
            }
        }

        /// <summary>
        /// 一時停止
        /// </summary>
        public void PauseAll()
        {
            bool pause = false;

            // 再生中のサウンドがあった場合は、一時停止する
            // 一時停止中のみの場合は、再生を再開する（停止中のサウンドは対象外）
            foreach (PreviewPlayerControl control in _previewPlayers)
            {

                if (control.PreviewPlayer.State == PreviewPlayerState.Preparing ||
                    control.PreviewPlayer.State == PreviewPlayerState.Playing)
                {
                    pause = true;
                    break;
                }

            }

            foreach (PreviewPlayerControl control in _previewPlayers)
            {

                switch (control.PreviewPlayer.State)
                {
                    case PreviewPlayerState.Paused:
                        if (pause) { break; }
                        control.PreviewPlayer.Pause();
                        break;

                    case PreviewPlayerState.Preparing:
                    case PreviewPlayerState.Playing:
                        if (!pause) { break; }
                        control.PreviewPlayer.Pause();
                        break;
                }

            }
        }

        /// <summary>
        /// 停止
        /// </summary>
        public void StopAll()
        {
            foreach (PreviewPlayerControl control in _previewPlayers)
            {

                if (control.PreviewPlayer.CanStop)
                {
                    control.PreviewPlayer.Stop();
                }

            }
        }

        /// <summary>
        /// 指定コマンドを処理するコマンドターゲットを検索します。
        /// </summary>
        /// <param name="command">処理するコマンド。</param>
        /// <returns>コマンドターゲット。</returns>
        ICommandTarget ICommandTarget.FindTarget(Command command)
        {
            if (!this.commandBindings.Contains(command.ID))
            {
                return null;
            }

            return this;
        }

        /// <summary>
        /// 指定コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <param name="command">実行するコマンド。</param>
        /// <returns>コマンドの状態。</returns>
        CommandStatus ICommandTarget.QueryStatus(Command command)
        {
            if (!this.commandBindings.Contains(command.ID))
            {
                return CommandStatus.Unsupported;
            }

            return this.commandBindings[command.ID].QueryStatus(command);
        }

        /// <summary>
        /// 指定コマンドを実行します。
        /// </summary>
        /// <param name="command">実行するコマンド。</param>
        bool ICommandTarget.Execute(Command command)
        {
            if (!this.commandBindings.Contains(command.ID))
            {
                return false;
            }

            return this.commandBindings[command.ID].Execute(command);
        }

        //-----------------------------------------------------------------

        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                case Win32.WM.WM_SETFOCUS:
                    if (null != SelectedItem)
                    {
                        SelectedItem.Focus();
                    }
                    return;
            }

            base.WndProc(ref m);
        }

        protected override void OnWindowActiveChanged(EventArgs e)
        {
            base.OnWindowActiveChanged(e);

            if (this.SelectedItem != null)
            {
                this.SelectedItem.SetActive(this.WindowActive);
            }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// コマンドの関連付けを初期化します。
        /// </summary>
        private void InitializeCommandBindings()
        {
            this.commandBindings.Add(
                new CommandBinding(this, EditCommands.Delete.ID, QueryStatusDelete, ExecuteDelete));
        }

        private void AttachContainer(PreviewPlayerControlCollection newContainer)
        {
            Debug.Assert(null != newContainer);

            _previewPlayers = newContainer;

            SelectItem(PreviewPlayerManager.SelectedPreviewPlayer.Index);

            foreach (var control in _previewPlayers)
            {
                control.PreviewPlayer.PlayerStateChanged += OnPreviewPlayerStateChanged;
            }

            UpdateToolStripStates();
        }

        private PreviewPlayerControlCollection DetachContainer()
        {
            Debug.Assert(null != _previewPlayers);

            foreach (PreviewPlayerControl item in _previewPlayers)
            {
                if (null == item) continue;
                panelPreviewPlayers.Controls.Remove(item);
                item.PreviewPlayer.PlayerStateChanged -= OnPreviewPlayerStateChanged;
            }

            PreviewPlayerControlCollection oldContainer = _previewPlayers;
            _previewPlayers = new PreviewPlayerControlCollection();

            return oldContainer;
        }

        private void UpdateColors()
        {
            foreach (PreviewPlayerControl control in _previewPlayers)
            {
                control.UpdateColors(this.WindowActive);
            }
        }

        private void SelectPrevItem()
        {
            if (ItemCount == 0) { return; }
            if (null == SelectedItem) { return; }
            if (SelectedItem.Index == 0) { return; }

            SelectItem(SelectedItem.Index - 1);
        }

        private void SelectNextItem()
        {
            if (ItemCount == 0) { return; }
            if (null == SelectedItem) { return; }
            if (SelectedItem.Index + 1 >= ItemCount) { return; }

            SelectItem(SelectedItem.Index + 1);
        }

        private void SelectFirstItem()
        {
            if (ItemCount == 0) { return; }
            SelectItem(0);
        }

        private void SelectLastItem()
        {
            if (ItemCount == 0) { return; }
            SelectItem(ItemCount - 1);
        }

        private CommandStatus QueryStatusDelete(Command command)
        {
            return CanDelete ? CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        private bool ExecuteDelete(Command command)
        {
            this.ResetSelectedItem();
            return true;
        }

        private void ResetSelectedItem()
        {
            this.SelectedItem.Item = null;
        }

        private void ResetAllItems()
        {
            foreach (var control in _previewPlayers)
            {
                control.Item = null;
            }
        }

        //-----------------------------------------------------------------

        private bool OnDialogKeyProcessed(object sender, KeyEventArgs e)
        {
            switch (e.KeyData)
            {
                case Keys.Up:
                    SelectPrevItem();
                    return true;

                case Keys.Down:
                    SelectNextItem();
                    return true;

                case Keys.Home:
                    SelectFirstItem();
                    return true;

                case Keys.End:
                    SelectLastItem();
                    return true;

                case Keys.Tab:
                    ProcessDialogKey(Keys.Tab);
                    return true;

                case Keys.F9:
                    if (this._selectedControl != null)
                    {
                        this._selectedControl.Play();
                        return true;
                    }
                    break;
            }

            return false;
        }

        private void OnPreviewPlayerItemChanged(object sender, EventArgs e)
        {
            FormsApplication.Instance.UIService.MainWindow.BuildCommandUI();
        }

        private void OnPreviewPlayerSelectChanged(object sender, EventArgs e)
        {
            SelectItem(PreviewPlayerManager.SelectedPreviewPlayerIndex);
        }

        private void OnPreviewPlayerStateChanged(object sender, EventArgs e)
        {
            UpdateToolStripStates();
        }

        private void OnComponentsChanged(object sender, EventArgs e)
        {
            UpdateItems();
        }

        private void OnPreviewPlayerSelectedChanged(object sender, EventArgs e)
        {
            PreviewPlayerControl target = sender as PreviewPlayerControl;
            Debug.Assert(null != target);

            if (!target.Selected) { return; }

            // プレビュープレイヤーの選択を変更する
            PreviewPlayerManager.SelectedPreviewPlayerIndex = target.PreviewPlayer.Index;

            FormsApplication.Instance.UIService.MainWindow.BuildCommandUI();
        }

        private void toolStripButtonDelete_Click(object sender, EventArgs e)
        {
            // NOTE:
            // ToolStrip の[削除]ボタンは本パネル内のアイテムにのみ作用するので Click イベントで実装します。

            ResetAllItems();
        }
    }

    #region ** コレクションクラス

    public class PreviewPlayerControlCollection : Collection<PreviewPlayerControl> { }

    #endregion
}
