﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using NintendoWare.SoundMaker.Configurations;
using NintendoWare.SoundMaker.Preview;
using NintendoWare.SoundMaker.Preview.Configurations;
using NintendoWare.SoundMaker.Configurations.Schemas;
using NintendoWare.SoundFoundation.Core.Drawing;
using NintendoWare.SoundFoundation.Windows.Forms;
using NintendoWare.SoundMaker.Framework.Windows;
using NintendoWare.SoundMaker.Framework.Windows.Forms;
using FrameworkResources = NintendoWare.SoundMaker.Framework.Resources;

namespace NintendoWare.SoundMaker.Windows.Forms
{
    public partial class EffectPanel : DockingPage, IToolWindowPage
    {
        public const string PageName = "Effect";

        public EffectPanel()
            : base(PageName)
        {
            InitializeComponent();

            // テキストとアイコンをロードする
            Text = FrameworkResources.MessageResource.ToolWindowName_Effect;
            Image = FrameworkResources.ImageResource.
                BitmapIconEffectWindow.MakeNewTransparent(UIServiceBase.TransparentColor);

            DockVisible = false;

            // 初期サイズを設定する
            FloatingBounds = new Rectangle(-1, -1,
                                            Width + (SystemInformation.FrameBorderSize.Width * 2),
                                            Height + SystemInformation.CaptionHeight + SystemInformation.FrameBorderSize.Height);

            effectControlPane_TV.ChangePresets += ChangePresets;
            effectControlPane_DRC0.ChangePresets += ChangePresets;
        }

        #region ** プロパティ

        private EffectControlPane CurrnetEffectControlPane
        {
            get
            {
                if (tabColtrol_Device.SelectedTab == tabPage_TV)
                {
                    return effectControlPane_TV;
                }
                else if (tabColtrol_Device.SelectedTab == tabPage_DRC0)
                {
                    return effectControlPane_DRC0;
                }
                else
                {
                    Debug.Assert(false);
                    return effectControlPane_TV;
                }
            }
        }

        private EffectManager EffectManager
        {
            get { return PreviewManager.Instance.EffectManager; }
        }

        private PreviewPlayerManager PreviewPlayerManager
        {
            get { return PreviewManager.Instance.PreviewPlayerManager; }
        }

        private PresetConfigurationCommon PresetConfiguration
        {
            get { return FormsApplicationCommon.Instance.PresetConfiguration; }
        }

        #endregion

        #region ** イベントハンドラ

        private void OnProjectOpened(object sender, EventArgs e)
        {
            // プリセットを初期化する
            effectControlPane_TV.UpdatePresets();
            effectControlPane_DRC0.UpdatePresets();

            effectControlPane_TV.ResetItemText();
            effectControlPane_DRC0.ResetItemText();
        }

        private void OnProjectClosing(object sender, EventArgs e)
        {
            // プリセットをクリアする
            effectControlPane_TV.ClearPresets();
            effectControlPane_DRC0.ClearPresets();

            effectControlPane_TV.ResetItemText();
            effectControlPane_DRC0.ResetItemText();
        }

        private void OnViewerSynchronousChanged(object sender, EventArgs e)
        {
            // 表示中の場合だけ同期する
            if (Visible == true)
            {
                CurrnetEffectControlPane.Synchronize();
            }
        }

        private void OnEffectParameterChanged(object sender, EventArgs e)
        {
            // 全エフェクトパラメータをViewerに送信する

            // 表示中の場合だけ同期する
            if (Visible == true)
            {
                CurrnetEffectControlPane.Synchronize();
            }
        }

        #endregion

        #region ** イベントハンドラのオーバーライド

        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);

            // イベントハンドラを設定する
            EffectManager.EffectParameterChanged += OnEffectParameterChanged;
            FormsApplicationCommon.Instance.ProjectService.Opened += OnProjectOpened;
            FormsApplicationCommon.Instance.ProjectService.Closing += OnProjectClosing;
            PreviewPlayerManager.ViewerSynchronousChanged += OnViewerSynchronousChanged;

            // 各ペインを初期化する
            effectControlPane_TV.PreviewPlayerManager = PreviewPlayerManager;
            effectControlPane_TV.PresetConfiguration = PresetConfiguration;
            effectControlPane_TV.Initialize(EffectManager.DeviceTypeToUint(DeviceType.TV),
                                            EffectManager.ParameterSets(DeviceType.TV));

            effectControlPane_DRC0.PreviewPlayerManager = PreviewPlayerManager;
            effectControlPane_DRC0.PresetConfiguration = PresetConfiguration;
            effectControlPane_DRC0.Initialize(EffectManager.DeviceTypeToUint(DeviceType.DRC0),
                                              EffectManager.ParameterSets(DeviceType.DRC0));

            // すでに開かれている場合は、プリセットを初期化する
            if (null != FormsApplicationCommon.Instance.ProjectService.ProjectDocument)
            {
                effectControlPane_TV.UpdatePresets();
                effectControlPane_DRC0.UpdatePresets();
            }
        }

        protected override void OnHandleDestroyed(EventArgs e)
        {
            // イベントハンドラの登録を解除する
            EffectManager.EffectParameterChanged -= OnEffectParameterChanged;
            PreviewPlayerManager.ViewerSynchronousChanged -= OnViewerSynchronousChanged;

            base.OnHandleDestroyed(e);
        }

        protected override void OnVisibleChanged(EventArgs e)
        {
            base.OnVisibleChanged(e);

            // 表示中の場合だけ同期する
            if (Visible == true)
            {
                CurrnetEffectControlPane.Synchronize();
            }
        }

        #endregion

        private void OnSelected(object sender, TabControlEventArgs e)
        {
            // 表示中の場合だけ同期する
            if (Visible == true)
            {
                CurrnetEffectControlPane.Synchronize();
            }
        }

        private void ChangePresets(object sender, EventArgs e)
        {
            effectControlPane_TV.UpdatePresets();
            effectControlPane_DRC0.UpdatePresets();
        }
    }
}
