﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Drawing;
    using System.IO;
    using System.Linq;
    using System.Reflection;
    using System.Windows.Forms;
    using Nintendo.AudioToolkit.DomainModels;
    using Nintendo.AudioToolkit.DataModels;
    using Nintendo.AudioToolkit.Operations;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Configurations;
    using NintendoWare.SoundMaker.Framework;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.FileFormats;
    using NintendoWare.SoundMaker.Framework.Projects;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.Windows.Forms.CommandHandlers;
    using NintendoWare.SoundMaker.Preview;
    using NintendoWare.SoundMaker.Preview.Service;
    using NintendoWare.SoundMaker.Resources;
    using NintendoWare.SoundMaker.Windows.Forms.CommandHandlers;
    using FrameworkResources = NintendoWare.SoundMaker.Framework.Resources;

    /// <summary>
    ///
    /// </summary>
    public class SoundSetPanelCommon : SoundSetPanel, IParameterPanelKeyPreviewListener
    {
        private const string EnableTextSequenceToWaveFileName = "EnableTextSequenceToWaveFile";

        private bool readonlyPreviewPlayFlag = false;
        private readonly CommonListFileWatcher _listFileWatcher;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public SoundSetPanelCommon(DocumentReference soundSetDocReference)
            : base(soundSetDocReference)
        {
            base.PreviewPlay += PreviewPlayEventListener.OnPreviewPlay;
            base.PreviewMute += PreviewPlayEventListener.OnPreviewMute;
            base.PreviewSoloPlay += PreviewPlayEventListener.OnPreviewSoloPlay;
            this.ListDecorationEvaluator.AddDoShowErrorIconDelegate(ProjectParameterNames.WaveEncoding, this.DoShowErrorIconWaveEncoding);
            this.ListDecorationEvaluator.AddDoShowErrorIconDelegate(ProjectParameterNames.SampleRate, this.DoShowErrorIconSampleRate);
            this.ListDecorationEvaluator.AddDoShowWarningIconDelegate(ProjectParameterNames.IntegratedLoudness, this.DoShowWarningIconIntegratedLoudness);
            this.ListCtrlMain.ItemDoubleClicked += OnItemDoubleClicked;
            this.ListCtrlSub.ItemDoubleClicked += OnItemDoubleClicked;
            this.ListCtrlMain.PartDrawers.Replace(new SoundSetPanelCommonListPartCheckBoxDrawer());
            this.ListCtrlSub.PartDrawers.Replace(new SoundSetPanelCommonListPartCheckBoxDrawer());

            Application.LoudnessService.Measuring += this.OnMeasuringLoudness;
            Application.LoudnessService.Measured += this.OnMeasuredLoudness;

            _listFileWatcher = new CommonListFileWatcher(this, this.ListControls);
            _listFileWatcher.FileChanged += (sender, args) => this.RedrawPanel();
        }

        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                _listFileWatcher.Dispose();
            }

            base.Dispose(disposing);
        }

        /// <summary>
        ///
        /// </summary>
        public override Component[] TargetComponents
        {
            get
            {
                Component[] components = base.TargetComponents;

                if (components.Length == 0)
                {
                    if (SoundSetDocument.IsReadOnly == true)
                    {
                        if (this.readonlyPreviewPlayFlag == true &&
                            null != ActiveListCtrl &&
                            ActiveListCtrl is ListCtrl == true)
                        {
                            components = ActiveListCtrl.GetComponentsBySelectedItem();
                        }
                    }
                }

                return components;
            }
        }

        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);

            _listFileWatcher.Start();
        }

        /// <summary>
        ///
        /// </summary>
        public override void Activated()
        {
            base.Activated();

            SequenceVariablePanel.SoundSetPanel = this;
        }

        /// <summary>
        ///
        /// </summary>
        public override void OnClosed()
        {
            base.OnClosed();
            if (SequenceVariablePanel.SoundSetPanel == this)
            {
                SequenceVariablePanel.SoundSetPanel = null;
            }

            _listFileWatcher.Stop();
        }

        /// <summary>
        ///
        /// </summary>
        public void Play()
        {
            Sound[] sounds = GetSelectedSoundsFromListControl();
            if (sounds.Length > 0)
            {
                Play(sounds[0]);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void StopAll()
        {
            Application.StopAll();
        }

        /// <summary>
        ///
        /// </summary>
        void IParameterPanelKeyPreviewListener.KeyDown(KeyEventArgs e)
        {
            if (e.KeyCode == Keys.F9)
            {
                Play();
            }
        }

        /// <summary>
        ///
        /// </summary>
        void IParameterPanelKeyPreviewListener.KeyUp(KeyEventArgs e)
        {
            if (e.KeyCode == Keys.F10)
            {
                StopAll();
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected SequenceVariablePanel SequenceVariablePanel
        {
            get
            {
                return FormsApplication.Instance.UIService.
                    MainWindow.ToolPages[SequenceVariablePanel.PageName] as SequenceVariablePanel;
            }
        }

        /// <summary>
        /// リストコントロールを作成します。
        /// </summary>
        protected override CommonListCtrl CreateListCtrl()
        {
            CommonListCtrl listCtrl = new ListCtrlCommon();

            return listCtrl;
        }

        /// <summary>
        ///
        /// </summary>
        protected override CommonListAdapter CreateListAdapter()
        {
            CommonListAdapter commonListAdapter = new CommonListAdapterCommon(ListDecorationEvaluator);

            commonListAdapter.Setters.Add
                (ProjectParameterNames.SndEdit,
                  SndEdit.SetSndEdit);

            commonListAdapter.ListItemCreator = delegate (Component component)
                {
                    CommonListItem item = new CommonListItem(component);
                    item.ConstValueGetters.Add("SndEditState", SndEdit.GetSndEditState);
                    item.ConstValueGetters.Add(ProjectParameterNames.FilePath, UseWaveSoundResourceConstValueGetter);
                    return item;
                };

            return commonListAdapter;
        }

        /// <summary>
        /// "貼り付け"の実行
        /// </summary>
        protected override bool Paste()
        {
            return base.Paste();
        }

        /// <summary>
        /// コマンドの関連付けを初期化します。
        /// </summary>
        protected override void InitializeCommandBindings()
        {
            base.InitializeCommandBindings();

            // 編集関連コマンドハンドラ
            if (AppConfigurationCommon.EnabledWaveSoundEdit == true)
            {
                BindCommand(new CommandBinding(this, EditCommands.UseWaveSoundWaveFile.ID,
                                                QueryStatusUseWaveSoundWaveFile, ExecuteUseWaveSoundWaveFile));
                BindCommand(new CommandBinding(this, EditCommands.UseWaveSoundResource.ID,
                                                QueryStatusUseWaveSoundResource, ExecuteUseWaveSoundResource));
            }

            // プレビュー関連コマンドハンドラ
            BindCommand(new CommandBinding(this, PreviewCommands.Play.ID,
                                            QueryStatusPlay, ExecutePlay));
            BindCommand(new CommandBinding(this, PreviewCommands.Play2.ID,
                                            QueryStatusPlay, ExecutePlay));
            BindCommand(new CommandBinding(this, PreviewCommands.PlayKeySpaceWithShift.ID,
                                            QueryStatusPlay, ExecutePlayKeySpaceWithShift));

            BindCommand(new CommandBinding(this, PreviewCommands.RegisterOnPreviewBank.ID,
                                            QueryStatusRegisterOnPreviewBank,
                                            ExecuteRegisterOnPreviewBank));
            BindCommand(new CommandBinding(this, new QueryAndOutputWaveFileHandler(this)));
            BindCommand(new CommandBinding(this, new QueryAndMeasureIntegratedLoudnessHandler(this)));

            if (IsTextSequenceToWaveFileEnabled())
            {
                BindCommand(new CommandBinding(
                    this,
                    new WaveConvertHandler(this, message => this.MainWindow.UpdateStatusText(message), false)
                    ));
                BindCommand(new CommandBinding(
                    this,
                    new WaveConvertHandler(this, message => this.MainWindow.UpdateStatusText(message), true)
                    ));
                BindCommand(new CommandBinding(
                    this, new OpenTextSequenceForWaveHandler(this)
                    ));
            }

            // プロジェクト関連コマンドハンドラ
            BindCommand(new CommandBinding(this, new QueryAndConvertSmfFileHandlerCommon(this)));
            BindCommand(new CommandBinding(this,
                        new QueryAndAddStreamSoundHandler(true, this, ApplicationBase.DefaultPlayerName)));
            BindCommand(new CommandBinding(this,
                                           new QueryAndAddStreamSoundTrackHandlerCommon(true, this)));

            BindCommand(new CommandBinding(this,
                        new QueryAndAddStreamSoundHandler(false, this, ApplicationBase.DefaultPlayerName)));
            BindCommand(new CommandBinding(this,
                                           new QueryAndAddStreamSoundTrackHandlerCommon(false, this)));

            // サウンドリスト関連コマンドハンドラ
            BindCommand(new CommandBinding(this,
                                           new SoundListQueryAndAddStreamSoundHandler(true, this, ApplicationBase.DefaultPlayerName)));
            BindCommand(new CommandBinding(this,
                                           new SoundListQueryAndAddStreamSoundTrackHandlerCommon(true, this)));

            BindCommand(new CommandBinding(this,
                                           new SoundListQueryAndAddStreamSoundHandler(false, this, ApplicationBase.DefaultPlayerName)));
            BindCommand(new CommandBinding(this,
                                           new SoundListQueryAndAddStreamSoundTrackHandlerCommon(false, this)));
        }

        protected override bool ValidateSoundSetItemForDragDrop(Component parent, Component component, out string message)
        {
            message = string.Empty;

            if (component is StreamSoundBase)
            {
                StreamSoundBase streamSound = component as StreamSoundBase;
                StreamSoundTrackBase track = streamSound.Children[0] as StreamSoundTrackBase;
                if (AACUtil.IsAACFile(track.FilePath) == true || FileUtil.IsOpusFile(track.FilePath) == true)
                {
                    ValidationResult result = track.Parameters.GetValue(ProjectParameterNames.FilePath).Validate();
                    if (result.IsValid == false)
                    {
                        message = result.ToString();
                        return false;
                    }
                }
            }
            else if (component is WaveSoundBase)
            {
                WaveSoundBase waveSound = component as WaveSoundBase;
                if (AACUtil.IsAACFile(waveSound.FilePath) == true)
                {
                    ValidationResult result = waveSound.Parameters.GetValue(ProjectParameterNames.FilePath).Validate();
                    if (result.IsValid == false)
                    {
                        message = result.ToString();
                        return false;
                    }
                }
            }

            return true;
        }

        protected override Component CreateSoundSetItemForDragDrop(Component component, string filePath, int channelCount)
        {
            if (component is StreamSoundPack == true && FileUtil.IsOpusFile(filePath) == true)
            {
                StreamSoundTrackBase streamSoundTrack = this.CreateComponentService.Create<StreamSoundTrackBase>();
                streamSoundTrack.FilePath = filePath;

                StreamSoundBase streamSound = this.CreateComponentService.Create<StreamSoundBase>();
                streamSound.Name = ItemNamer.CreateUniqueNameFromFileName(ProjectService.ComponentDictionary, new ItemNamingSettings(ProjectService.Project), ProjectService.Project.StreamSoundNamePrefix, String.Empty, filePath);
                streamSound.PlayerReference = ProjectService.Players.Length > 0 ? ProjectService.Players[0].Name : ApplicationBase.DefaultPlayerName;
                streamSound.Encoding = WaveEncoding.NoConvert;
                streamSound.Children.Add(streamSoundTrack);

                return streamSound;
            }

            return base.CreateSoundSetItemForDragDrop(component, filePath, channelCount);
        }

        /// <summary>
        /// リストコントロール Main の縦スクロールボックスが移動した時に呼ばれます。
        /// </summary>
        protected override void OnListCtrlMainVerticalScrollValueChanged(object sender, EventArgs e)
        {
            if (Application.AppConfiguration.Options.Application.Statistics.EnabledAutoMeasureLoudness == true)
            {
                // ListCtrlMain に表示されている未計測のサウンドを計測します。
                Application.LoudnessService.MeasureLowPriority(this.GetUnmeasuredSounds(this.ListCtrlMain.GetDisplayedItems()));
            }
        }

        /// <summary>
        /// リストコントロール Sub の縦スクロールボックスが移動した時に呼ばれます。
        /// </summary>
        protected override void OnListCtrlSubVerticalScrollValueChanged(object sender, EventArgs e)
        {
            if (Application.AppConfiguration.Options.Application.Statistics.EnabledAutoMeasureLoudness == true)
            {
                // ListCtrlSub に表示されている未計測のサウンドを計測します。
                Application.LoudnessService.MeasureLowPriority(this.GetUnmeasuredSounds(this.ListCtrlSub.GetDisplayedItems()));
            }
        }

        /// <summary>
        /// パネルが表示された時に呼ばれます。
        /// </summary>
        protected override void OnShowedPanel(object sender, EventArgs e)
        {
            base.OnShowedPanel(sender, e);

            if (Application.AppConfiguration.Options.Application.Statistics.EnabledAutoMeasureLoudness == true)
            {
                // ListCtrlMain に表示されている未計測のサウンドを計測します。
                Application.LoudnessService.MeasureLowPriority(this.GetUnmeasuredSounds(this.ListCtrlMain.GetDisplayedItems()));

                // ListCtrlSub に表示されている未計測のサウンドを計測します。
                Application.LoudnessService.MeasureLowPriority(this.GetUnmeasuredSounds(this.ListCtrlSub.GetDisplayedItems()));
            }
        }

        // <summary>
        // IListItem から未計測のサウンドを取得します。
        // </summary>
        private IEnumerable<Sound> GetUnmeasuredSounds(IEnumerable<IListItem> items)
        {
            return items
                .Cast<CommonListItem>()
                .Select(i => i.Target)
                .OfType<Sound>()
                .Where(s => s.IntegratedLoudnessStatus == IntegratedLoudnessStatus.Unmeasured); // 未計測なサウンド
        }

        /// <summary>
        /// 圧縮形式のエラーアイコンの表示非表示判定メソッドです。
        /// </summary>
        private string DoShowErrorIconWaveEncoding(Component component)
        {
            if (component is StreamSound == true || component is WaveSound == true)
            {
                var waveEncoding = (WaveEncoding)component.Parameters[ProjectParameterNames.WaveEncoding].Value;
                if (this.Application.CurrentSoundMakerPlugin.IsValidateWaveEncodingValue(waveEncoding) == false)
                {
                    // 波形のエンコード形式が不正です。
                    return string.Format(MessageResource.ErrorMessage_InvalidWaveEncoding, waveEncoding);
                }
            }

            return null;
        }

        /// <summary>
        /// サンプルレートのエラーアイコンの表示非表示判定メソッドです。
        /// </summary>
        private string DoShowErrorIconSampleRate(Component component)
        {
            if (component.Parameters.ContainsKey(ProjectParameterNames.IsResampleEnabled) == false ||
                (bool)component.Parameters[ProjectParameterNames.IsResampleEnabled].Value == false)
            {
                return null;
            }

            if (component is StreamSound == true)
            {
                // トラックを抽出します。
                var tracks = component.Children.Cast<StreamSoundTrack>();

                // 波形ファイルの情報がなければ更新します。
                tracks
                    .Where(t => t.WaveFile == null)
                    .ToList()
                    .ForEach(t => t.UpdateWaveFile());

                // １つでもダウンサンプルではなくアップサンプルになっていたらエラーにします。
                if (tracks
                    .Where(t => t.WaveFile != null)
                    .Any(t => t.WaveFile.SampleRate < (component as StreamSound).SampleRate))
                {
                    // アップサンプルはサポートされません。
                    return MessageResource.ErrorMessage_UpsampleNotSupported;
                }

                return null;
            }

            if (component is WaveSound == true)
            {
                var waveSound = component as WaveSound;

                if (waveSound.WaveFile == null)
                {
                    waveSound.UpdateWaveFile();
                }

                // ダウンサンプルではなくアップサンプルになっていたらエラーにします。
                if (waveSound.WaveFile != null && waveSound.WaveFile.SampleRate < waveSound.SampleRate)
                {
                    // アップサンプルはサポートされません。
                    return MessageResource.ErrorMessage_UpsampleNotSupported;
                }

                return null;
            }

            return null;
        }

        /// <summary>
        /// 平均ラウドネス値のワーニングアイコンの表示非表示判定メソッドです。
        /// </summary>
        private string DoShowWarningIconIntegratedLoudness(Component component)
        {
            if (component is Sound == true)
            {
                var sound = (Sound)component;

                switch (sound.IntegratedLoudnessStatus)
                {
                    case IntegratedLoudnessStatus.Error:
                        // メッセージはセルに表示されるのでアイコンだけ表示します。
                        return string.Empty;

                    case IntegratedLoudnessStatus.Measured:
                        if (Application.ProjectService.Project.EnabledIntegratedLoudnessThreshold == true)
                        {
                            if (sound.IntegratedLoudness > Application.ProjectService.Project.IntegratedLoudnessThreshold)
                            {
                                return MessageResource.ErrorMessage_IntegratedLoudnessExceedThreshold;
                            }
                        }
                        break;
                }
            }

            return null;
        }

        /// <summary>
        ///
        /// </summary>
        private FormsApplicationCommon Application
        {
            get
            {
                return FormsApplication.Instance as FormsApplicationCommon;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private Component[] targetComponents
        {
            get
            {
                this.readonlyPreviewPlayFlag = true;

                Component[] components = TargetComponents;

                this.readonlyPreviewPlayFlag = false;

                return components;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private CommandStatus QueryStatusPlay(Command command)
        {
            if (null == PreviewPlayerOperator.SelectedPreviewPlayer)
            {
                return CommandStatus.SupportedAndVisible;
            }
            if (1 != this.targetComponents.Length) { return CommandStatus.SupportedAndVisible; }
            return (this.targetComponents[0] is Sound ||
                    this.targetComponents[0] is StreamSoundTrackBase) ?
                CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// "再生"の実行です。
        /// </summary>
        private bool ExecutePlay(Command command)
        {
            if (OptionHelper.FunctionOfKeySpaceIsToggle != false &&
                PreviewPlayerOperator.SelectedPreviewPlayer.Playing != false)
            {
                StopAll();
            }
            else
            {
                Play(this.targetComponents[0]);
            }

            return true;
        }

        /// <summary>
        /// "再生"の実行です。
        /// ショートカットキー、"Space + Shift"用です。
        /// </summary>
        private bool ExecutePlayKeySpaceWithShift(Command command)
        {
            if (OptionHelper.FunctionOfKeySpaceWithShiftIsToggle != false &&
                PreviewPlayerOperator.SelectedPreviewPlayer.Playing != false)
            {
                StopAll();
            }
            else
            {
                Play(this.targetComponents[0]);
            }
            return true;
        }

        /// <summary>
        /// 再生します。
        /// </summary>
        private void Play(Component component)
        {
            PreviewPlayerOperator.Play(component);
        }

        /// <summary>
        /// プレビューバンクに登録コマンドが実行可能かどうかを調べます。
        /// </summary>
        private CommandStatus QueryStatusRegisterOnPreviewBank(Command command)
        {
            Component[] components = TargetComponents;

            if (components.Length != 1)
            {
                return CommandStatus.SupportedAndVisible;
            }

            if (components[0] is SequenceSoundBase || components[0] is SoundSetBankBase)
            {
                return CommandStatus.SupportedAndEnabledAndVisible;
            }

            return CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// プレビューバンクに登録コマンドを実行します。
        /// </summary>
        private bool ExecuteRegisterOnPreviewBank(Command command)
        {
            Component[] components = TargetComponents;

            if (components.Length != 1)
            {
                return false;
            }

            PreviewBankPanel previewBankPanel = this.MainWindow.ToolPages[PreviewBankPanel.PageName] as PreviewBankPanel;

            if (components[0] is SoundSetBankBase)
            {
                previewBankPanel.GetItem(0).Attach(components[0] as SoundSetBankBase);
            }
            else if (components[0] is SequenceSoundBase)
            {
                previewBankPanel.Attach(components[0] as SequenceSoundBase);
            }
            else
            {
                return false;
            }

            this.MainWindow.ActivateToolPage(previewBankPanel);

            return true;
        }

        /// <summary>
        /// ウェーブサウンドで波形ファイルを使用するコマンドが実行可能か調べます。
        /// </summary>
        private CommandStatus QueryStatusUseWaveSoundWaveFile(Command command)
        {
            Component[] components = TargetComponents;

            if (components.Length <= 0 || components[0] is WaveSound == false)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        /// <summary>
        /// ウェーブサウンドで波形ファイルを使用するコマンドを実行します。
        /// </summary>
        private bool ExecuteUseWaveSoundWaveFile(Command command)
        {
            Component[] components = TargetComponents;

            if (components.Length <= 0 || components[0] is WaveSound == false)
            {
                return false;
            }

            try
            {
                SoundSetDocument.OperationHistory.BeginTransaction();
                foreach (WaveSound waveSound in components)
                {
                    WaveSound capturedWaveSound = waveSound; // Delegate Captured Variable
                    WaveSoundEditWindow capturedWaveSoundEditWindow = (MainWindow as MainWindowCommon).WaveSoundEditWindow; // Delegate Captured Variable
                    Operation operation = ActionOperation.CreateActionOperation<WaveSoundResource>(capturedWaveSound.WaveSoundResource, null, delegate (WaveSoundResource v) { capturedWaveSound.WaveSoundResource = v; if (capturedWaveSoundEditWindow.WaveSound == capturedWaveSound) { capturedWaveSoundEditWindow.WaveSound = capturedWaveSound; } });
                    SoundSetDocument.OperationHistory.AddOperation(operation);
                    if (operation.Execute() == false)
                    {
                        SoundSetDocument.OperationHistory.CancelTransaction();
                        return false;
                    }
                }
                SoundSetDocument.OperationHistory.EndTransaction();
            }
            catch
            {
                SoundSetDocument.OperationHistory.CancelTransaction();
                return false;
            }

            if (components.Length == 1)
            {
                (MainWindow as MainWindowCommon).WaveSoundEditWindow.WaveSound = (components[0] as WaveSound);
            }
            else
            {
                (MainWindow as MainWindowCommon).WaveSoundEditWindow.WaveSound = null;
            }

            return true;
        }

        /// <summary>
        /// ウェーブサウンドリソースを使用するコマンドが実行可能か調べます。
        /// </summary>
        private CommandStatus QueryStatusUseWaveSoundResource(Command command)
        {
            Component[] components = TargetComponents;

            if (components.Length <= 0 || components[0] is WaveSound == false)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        /// <summary>
        /// ウェーブサウンドリソースを使用するコマンドを実行します。
        /// </summary>
        private bool ExecuteUseWaveSoundResource(Command command)
        {
            Component[] components = TargetComponents;

            if (components.Length <= 0 || components[0] is WaveSound == false)
            {
                return false;
            }

            try
            {
                SoundSetDocument.OperationHistory.BeginTransaction();
                foreach (WaveSound waveSound in components)
                {
                    if (waveSound.WaveSoundResource == null)
                    {
                        WaveSoundResourceData waveSoundResourceData = new WaveSoundResourceData();
                        waveSoundResourceData.TrackResources.Add(new WaveSoundTrackResourceData());
                        WaveSoundResource waveSoundResource = new WaveSoundResource(waveSoundResourceData, (Application.ProjectService as SoundProjectServiceCommon).GetOperationExecutor(this.SoundSetDocument));

                        WaveSound capturedWaveSound = waveSound; // Delegate Captured Variable
                        WaveSoundEditWindow capturedWaveSoundEditWindow = (MainWindow as MainWindowCommon).WaveSoundEditWindow; // Delegate Captured Variable
                        Operation operation = ActionOperation.CreateActionOperation<WaveSoundResource>(null, waveSoundResource, delegate (WaveSoundResource v) { capturedWaveSound.WaveSoundResource = v; if (capturedWaveSoundEditWindow.WaveSound == capturedWaveSound) { capturedWaveSoundEditWindow.WaveSound = capturedWaveSound; } });
                        SoundSetDocument.OperationHistory.AddOperation(operation);
                        if (operation.Execute() == false)
                        {
                            SoundSetDocument.OperationHistory.CancelTransaction();
                            return false;
                        }
                    }
                }
                SoundSetDocument.OperationHistory.EndTransaction();
            }
            catch
            {
                SoundSetDocument.OperationHistory.CancelTransaction();
                return false;
            }

            if (components.Length == 1)
            {
                (MainWindow as MainWindowCommon).WaveSoundEditWindow.WaveSound = (components[0] as WaveSound);
            }
            else
            {
                (MainWindow as MainWindowCommon).WaveSoundEditWindow.WaveSound = null;
            }

            return true;
        }

        private bool IsTextSequenceToWaveFileEnabled()
        {
            var enableTextSequenceFilePath = Path.Combine(
                Path.GetDirectoryName(Assembly.GetEntryAssembly().Location),
                EnableTextSequenceToWaveFileName
                );
            return File.Exists(enableTextSequenceFilePath);
        }

        private IConstParameterValue UseWaveSoundResourceConstValueGetter(Component component, string name, ref bool cancel)
        {
            if (component is WaveSound == true &&
                name == ProjectParameterNames.FilePath &&
                (component as WaveSound).WaveSoundResource != null)
            {
                return new TextParameterValue("*");
            }

            cancel = true;
            return null;
        }

        private void OnItemDoubleClicked(object sender, ListItemDoubleClickedEventArgs e)
        {
            ComponentListItem item = e.Item as ComponentListItem;
            if (null == item) { throw new ArgumentNullException("item"); }

            if (item.Target is WaveSound)
            {
                (MainWindow as MainWindowCommon).ShowWaveSoundEditorWindow(item.Target as WaveSound);
            }
        }

        private void OnMeasuringLoudness(object sender, MeasureLoudnessEventArgs e)
        {
            foreach (Sound sound in e.Sounds)
            {
                // 状態を計測中にする。
                sound.IntegratedLoudnessStatus = IntegratedLoudnessStatus.Measuring;
            }

            this.RedrawPanel();
        }

        private void OnMeasuredLoudness(object sender, MeasureLoudnessEventArgs e)
        {
            for (int i = 0; i < e.Sounds.Length; i++)
            {
                if (e.IsSuccess == true) // 計測が成功した場合
                {
                    // 状態を計測済にする。
                    e.Sounds[i].IntegratedLoudnessStatus = IntegratedLoudnessStatus.Measured;
                    // 平均ラウドネス値を設定する。
                    e.Sounds[i].IntegratedLoudness = e.IntegratedLoudness[i];
                }
                else if (e.IsCancel == true) // 計測が中断した場合
                {
                    // 状態を未計測にする。
                    e.Sounds[i].IntegratedLoudnessStatus = IntegratedLoudnessStatus.Unmeasured;
                    // 平均ラウドネス値を初期化する。
                    e.Sounds[i].IntegratedLoudness = 0;
                }
                else // 計測でエラーの場合
                {
                    // 状態を計測エラーにする。
                    e.Sounds[i].IntegratedLoudnessStatus = IntegratedLoudnessStatus.Error;
                    // 平均ラウドネス値を初期化する。
                    e.Sounds[i].IntegratedLoudness = 0;
                }
            }

            this.RedrawPanel();
        }

        /// <summary>
        /// チェックボックスの有効無効の描画をします。
        /// </summary>
        public class SoundSetPanelCommonListPartCheckBoxDrawer : ListPartCheckBoxDrawer
        {
            protected override ButtonState GetButtonState(ListDrawDescriptor desc, ButtonState state)
            {
                if (desc.Name == ListTraitsCommon.ColumnName_IsResampleEnabled ||
                    desc.Name == ListTraitsCommon.ColumnName_IsDownMixEnabled)
                {
                    Component component = (desc.Item as CommonListItem).Target;

                    if (component is StreamSoundBase == true &&
                        component.Children.OfType<StreamSoundTrackBase>().Any(t => FileUtil.IsOpusFile(t.FilePath)) == true)
                    {
                        // opus ファイルの場合はダウンサンプルチェックボックスを無効にします。
                        state |= ButtonState.Inactive;
                    }
                }

                return state;
            }
        }
    }
}
