﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundMaker.Windows.Forms
{
    using System;
    using System.Diagnostics;
    using System.Drawing;
    using System.IO;
    using System.Windows.Forms;
    using System.Xml;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.Windows;
    using NintendoWare.SoundMaker.Preview;
    using NintendoWare.ToolDevelopmentKit;
    using FrameworkResources = NintendoWare.SoundMaker.Framework.Resources;
    using Win32 = NintendoWare.SoundFoundation.Core.Win32;

    /// <summary>
    /// プレビューバンクコントロール
    /// </summary>
    public partial class PreviewBankControl : NUserControl
    {
        public delegate bool DialogKeyEventHandler(object sender, KeyEventArgs e);

        private const string ClipboardDataName = "NintendoWareSoundMakerData";

        private PreviewBank previewBank;

        private Color headerForeColor;
        private Color headerBackColor;
        private Color headerForeColorSelected;
        private Color headerBackColorSelected;
        private Color headerForeColorSelectedNoFocus;
        private Color headerBackColorSelectedNoFocus;

        private Color itemForeColor;
        private Color itemBackColor;
        private Color itemForeColorSelected;
        private Color itemBackColorSelected;
        private Color itemForeColorSelectedNoFocus;
        private Color itemBackColorSelectedNoFocus;

        private string targetItemName = string.Empty;
        private SoundSetBankBase targetItem = null;

        private bool isSelected = false;
        private bool isFocused = false;

        public PreviewBankControl(PreviewBank bank)
        {
            Ensure.Argument.NotNull(bank);
            InitializeComponent();

            // インデックスを設定する（以後、読み取り専用）
            this.previewBank = bank;
            this.indexLabel.Text = bank.Index.ToString();

            // イベントハンドラを設定する
            bank.ItemChanged += OnPreviewBankItemChanged;

            this.ProjectService.Closed += OnProjectClosed;

            FormsApplicationCommon.Instance.AppActivated += OnAppActivated;
            FormsApplicationCommon.Instance.AppDeactivated += OnAppDeactivated;
        }

        //-----------------------------------------------------------------

        public event EventHandler SelectedChanged;

        public event DialogKeyEventHandler DialogKeyProcessed;

        //-----------------------------------------------------------------

        /// <summary>
        /// プレビューバンクのインデックス
        /// </summary>
        public uint Index
        {
            get { return this.previewBank.Index; }
        }

        /// <summary>
        /// プレビューバンクを取得します。
        /// </summary>
        public PreviewBank PreviewBank
        {
            get { return this.previewBank; }
        }

        /// <summary>
        /// 関連付けられたアイテム
        /// </summary>
        public SoundSetBankBase Item
        {
            get { return this.previewBank.Item; }
            set
            {
                if (null == value)
                {
                    this.Reset();
                }
                else
                {
                    this.Attach(value);
                }
            }
        }

        /// <summary>
        /// 選択状態
        /// </summary>
        public bool Selected
        {
            get { return isSelected; }
            set
            {
                if (value == isSelected)
                {
                    return;
                }

                this.UpdateState(value);

                if (!(isFocused) && value)
                {
                    this.Focus();
                }

                this.UpdateColors();
            }
        }

        private SoundProjectService ProjectService
        {
            get { return FormsApplicationCommon.Instance.ProjectService; }
        }

        private Images.IconID IconID
        {
            get
            {
                if (null == Item)
                {
                    return Images.IconID.None;
                }

                return Images.IconID.Bank;
            }
        }

        private Rectangle ImageRect
        {
            get
            {
                return new Rectangle(
                    this.nameLabel.Left - 3 - 16,
                    (this.ClientRectangle.Height - 16) / 2,
                    16, 16);
            }
        }

        private ImageList IconImages
        {
            get { return Images.IconImages; }
        }

        //-----------------------------------------------------------------

        public void Attach(string itemName)
        {
            this.targetItemName = (null == itemName) ? string.Empty : itemName;

            this.Attach(GetValidItem(itemName, null));
        }

        public void Attach(SoundSetBankBase item)
        {
            if (this.targetItem == item)
            {
                return;
            }

            this.previewBank.Item = item;
        }

        public void UpdateItem()
        {
            this.UpdateItem(true);
        }

        public void UpdateItem(bool checkItem)
        {
            if (checkItem)
            {
                this.Attach(
                    this.GetValidItem(this.targetItemName, this.Item)
                    );
            }

            if (this.targetItem != this.previewBank.Item)
            {
                if (null != this.targetItem)
                {
                    this.targetItem.Parameters.ParameterValueChanged -= OnItemParameterChanged;
                }

                this.targetItem = this.previewBank.Item;

                if (null != this.targetItem)
                {
                    this.targetItem.Parameters.ParameterValueChanged += OnItemParameterChanged;
                }
            }

            this.UpdateName();
            this.Invalidate(ImageRect);
        }

        public void UpdateColors()
        {
            this.UpdateColors(true);
        }

        public void UpdateColors(bool isActive)
        {
            // 選択色 or 通常色 の設定
            if (this.isSelected)
            {
                if (this.isFocused && isActive)
                {
                    this.indexLabel.ForeColor = this.headerForeColorSelected;
                    this.itemHeader.BackColor = this.headerBackColorSelected;
                    this.nameLabel.ForeColor = this.itemForeColorSelected;
                    this.BackColor = this.itemBackColorSelected;
                }
                else
                {
                    this.indexLabel.ForeColor = this.headerForeColorSelectedNoFocus;
                    this.itemHeader.BackColor = this.headerBackColorSelectedNoFocus;
                    this.nameLabel.ForeColor = this.itemForeColorSelectedNoFocus;
                    this.BackColor = this.itemBackColorSelectedNoFocus;
                }
            }
            else
            {
                this.indexLabel.ForeColor = this.headerForeColor;
                this.itemHeader.BackColor = this.headerBackColor;
                this.nameLabel.ForeColor = this.itemForeColor;
                this.BackColor = this.itemBackColor;
            }

            this.Invalidate(true);
        }

        public void SetActive(bool isActive)
        {
            if (this.isFocused == isActive)
            {
                return;
            }

            this.isFocused = isActive;
            this.UpdateColors();
        }

        //-----------------------------------------------------------------

        protected override bool ProcessDialogKey(Keys keyData)
        {
            if (null != DialogKeyProcessed)
            {
                bool br = DialogKeyProcessed(this, new KeyEventArgs(keyData));
                if (br) { return true; }
            }

            // プレビューバンクが無効な場合は、←→で次のアイテムに選択が移るのを防ぐ
            if (!this.previewBank.IsEnabled)
            {
                switch (keyData)
                {
                    case Keys.Left:
                    case Keys.Right:
                        return true;
                }
            }

            return base.ProcessDialogKey(keyData);
        }

        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                case Win32.WM.WM_MOUSEACTIVATE:
                    // コントロール上をクリックされたらフォーカスを設定する
                    Focus();
                    m.Result = (IntPtr)Win32.MA.MA_ACTIVATE;
                    return;
            }

            base.WndProc(ref m);
        }

        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);

            InitializeColor();

            FormsApplicationCommon.Instance.UIService.MainWindow.ProjectReloaded += OnProjectReloaded;
        }

        protected override void OnEnter(EventArgs e)
        {
            base.OnEnter(e);
            isFocused = true;

            UpdateState(true);
            UpdateColors();
        }

        protected override void OnLeave(EventArgs e)
        {
            base.OnLeave(e);
            isFocused = false;

            UpdateColors();
        }

        protected override void OnPaintBackground(PaintEventArgs e) { }

        protected override void OnSystemColorsChanged(EventArgs e)
        {
            base.OnSystemColorsChanged(e);

            InitializeColor();
        }

        //-----------------------------------------------------------------

        private void InitializeColor()
        {
            this.headerForeColor = SystemColors.WindowText;
            this.headerBackColor = SystemColors.Control;
            this.itemForeColor = SystemColors.WindowText;
            this.itemBackColor = SystemColors.Window;

            this.headerForeColorSelected = SystemColors.HighlightText;
            this.headerBackColorSelected =
                    Color.FromArgb(
                         SystemColors.Highlight.R * 7 / 8,
                         SystemColors.Highlight.G * 7 / 8,
                         SystemColors.Highlight.B * 7 / 8
                         );
            this.itemForeColorSelected = SystemColors.HighlightText;
            this.itemBackColorSelected =
                    Color.FromArgb(
                         (SystemColors.Highlight.R * 192 / 255) + (Color.White.R * (255 - 192) / 255),
                         (SystemColors.Highlight.G * 192 / 255) + (Color.White.G * (255 - 192) / 255),
                         (SystemColors.Highlight.B * 192 / 255) + (Color.White.B * (255 - 192) / 255)
                         );

            this.headerForeColorSelectedNoFocus = SystemColors.HighlightText;
            this.headerBackColorSelectedNoFocus = SystemColors.ControlDark;
            this.itemForeColorSelectedNoFocus = SystemColors.HighlightText;
            this.itemBackColorSelectedNoFocus =
                    Color.FromArgb(
                         (SystemColors.ControlDark.R * 160 / 255) + (Color.White.R * (255 - 160) / 255),
                         (SystemColors.ControlDark.G * 160 / 255) + (Color.White.G * (255 - 160) / 255),
                         (SystemColors.ControlDark.B * 160 / 255) + (Color.White.B * (255 - 160) / 255)
                         );

            this.UpdateColors();
        }

        private void Reset()
        {
            this.Attach((SoundSetBankBase)null);

            this.targetItemName = string.Empty;
            this.nameLabel.Text = string.Empty;
        }

        private SoundSetBankBase GetValidItem(string itemName, SoundSetBankBase currentItem)
        {
            Assertion.Argument.NotNull(itemName);

            if (!ProjectService.ComponentDictionary.Contains(itemName))
            {
                return null;
            }

            Component[] components = ProjectService.ComponentDictionary[itemName];

            foreach (Component component in components)
            {
                if (component == currentItem)
                {
                    return currentItem;
                }
            }

            foreach (Component component in components)
            {
                if (component is SoundSetBankBase)
                {
                    return component as SoundSetBankBase;
                }
            }

            return null;
        }

        private void UpdateName()
        {
            if (null == this.previewBank.Item) { return; }

            // ラベルが変更されたら、Viewerに送信する
            if (this.previewBank.Item.Name != nameLabel.Text)
            {
                // TODO : ★未実装
                //MCS.MCSSound.Instance.SetSoundLabel(this.previewBank);
            }

            this.targetItemName = this.previewBank.Item.Name;
            this.nameLabel.Text = targetItemName;
        }

        private void UpdateState(bool newState)
        {
            if (isSelected == newState) { return; }

            isSelected = newState;

            if (null != SelectedChanged)
            {
                SelectedChanged(this, new EventArgs());
            }
        }

        /// <summary>
        /// XMLから対象アイテム名を取得します
        /// </summary>
        /// <param name="xmlDocument">検索対象のXML</param>
        /// <returns>対象アイテム名。</returns>
        private string GetTargetItemName(XmlDocument xmlDocument)
        {
            if (null == xmlDocument) { return null; }

            XmlElement xmlSoundSetBank = xmlDocument.DocumentElement.SelectSingleNode("Items/Bank[1]") as XmlElement;

            if (null != xmlSoundSetBank)
            {
                return GetTargetItemName(xmlSoundSetBank);
            }

            XmlElement xmlSequenceSound = xmlDocument.DocumentElement.SelectSingleNode("Items/SequenceSound[1]") as XmlElement;

            if (null != xmlSequenceSound)
            {
                return GetTargetItemName(xmlSequenceSound);
            }

            return null;
        }

        /// <summary>
        /// XMLから対象アイテム名を取得します
        /// </summary>
        /// <param name="xmlDocument">検索対象のXML</param>
        /// <returns>対象アイテム名。</returns>
        private string GetTargetItemName(XmlElement xmlItem)
        {
            if (null == xmlItem) { return null; }

            XmlElement xmlParameters = xmlItem.SelectSingleNode("Parameters") as XmlElement;
            if (null == xmlParameters) { return null; }

            foreach (XmlElement xmlParameter in xmlParameters.GetElementsByTagName("Parameter"))
            {
                string parameterName = xmlParameter.GetAttribute("Name");
                if (null == parameterName || parameterName != "Name") { continue; }

                string parameterValue = xmlParameter.GetAttribute("Value");
                if (null == parameterValue || 0 == parameterValue.Length) { continue; }

                if (!ProjectService.ComponentDictionary.Contains(parameterValue)) { return null; }

                return parameterValue;
            }

            return null;
        }

        //-----------------------------------------------------------------

        private void OnPaint(object sender, PaintEventArgs e)
        {
            // グリッドを描画する
            e.Graphics.DrawLine(SystemPens.ControlDark, ClientRectangle.Left, ClientRectangle.Bottom - 1,
                                                         ClientRectangle.Right, ClientRectangle.Bottom - 1);

            // アイコンを描画する
            if (Images.IconID.None == IconID) { return; }
            IconImages.Draw(e.Graphics, ImageRect.Location, (int)IconID);
        }

        private void OnAppActivated(object sender, EventArgs e)
        {
            isFocused = ContainsFocus;
            UpdateColors();
        }

        private void OnAppDeactivated(object sender, EventArgs e)
        {
            isFocused = false;
            UpdateColors();
        }

        private void OnPreviewBankItemChanged(object sender, EventArgs e)
        {
            UpdateItem(false);
        }

        private void OnItemParameterChanged(object sender, ParameterEventArgs e)
        {
            if (e.Key != ProjectParameterNames.Name) { return; }

            UpdateName();
        }

        private void OnProjectReloaded(object sender, EventArgs e)
        {
            UpdateItem();
        }

        private void OnProjectClosed(object sender, EventArgs e)
        {
            UpdateItem();
        }

        private void OnDragEnter(object sender, DragEventArgs e)
        {
            if (FormsApplicationCommon.Instance.UIService.MainWindow.CanDropFiles(e))
            {
                e.Effect = DragDropEffects.Copy;
                return;
            }

            byte[] data = null;

            if (!e.Data.GetDataPresent(ClipboardDataName)) { return; }
            if ((data = e.Data.GetData(ClipboardDataName) as byte[]) == null) { return; }

            XmlDocument xmlDoc = new XmlDocument();
            xmlDoc.Load(new MemoryStream(data));

            if (GetTargetItemName(xmlDoc) != null)
            {
                e.Effect = DragDropEffects.Link;
            }
            else
            {
                e.Effect = DragDropEffects.None;
            }
        }

        private void OnDragOver(object sender, DragEventArgs e)
        {
            OnDragEnter(sender, e);
        }

        private void OnDragDrop(object sender, DragEventArgs e)
        {
            if (e.Data.GetDataPresent(DataFormats.FileDrop))
            {
                if (FormsApplicationCommon.Instance.UIService.MainWindow.CanDropFiles(e))
                {
                    string[] filePaths = e.Data.GetData(DataFormats.FileDrop, false) as string[];
                    FormsApplicationCommon.Instance.UIService.MainWindow.DropFiles(filePaths);
                    return;
                }
            }

            byte[] data = null;

            if (!e.Data.GetDataPresent(ClipboardDataName)) { return; }
            if ((data = e.Data.GetData(ClipboardDataName) as byte[]) == null) { return; }

            XmlDocument xmlDoc = new XmlDocument();
            xmlDoc.Load(new MemoryStream(data));

            string itemName = this.GetTargetItemName(xmlDoc);
            SoundSetBankBase soundSetBank = this.GetValidItem(itemName, null);

            if (soundSetBank == null)
            {
                if (this.Parent is PreviewBankPanel)
                {
                    (this.Parent as PreviewBankPanel).Attach(itemName);
                }
            }
            else
            {
                this.Attach(soundSetBank);
            }
        }

        private void OnChildlenEnter(object sender, EventArgs e)
        {
            isFocused = true;
            UpdateColors();
        }

        private void OnChildlenLeave(object sender, EventArgs e)
        {
            isFocused = false;
            UpdateColors();
        }

        //-----------------------------------------------------------------

        private static class Images
        {
            public enum IconID
            {
                None = -1,
                Stream = 0,
                WaveSound,
                WaveSoundCo,
                Seq,
                SeqSet,
                Bank,
                WaveArchive,
                Group,
                Player,
                SoundSet,
                Instrument
            }

            private static ImageList _iconImages;

            static Images()
            {
                // アイコンイメージ
                _iconImages = new ImageList();
                _iconImages.ColorDepth = ColorDepth.Depth24Bit;
                _iconImages.ImageSize = new Size(16, 16);

                _iconImages.Images.Add(FrameworkResources.ImageResource.BitmapIconStreamSound, UIServiceBase.TransparentColor);
                _iconImages.Images.Add(FrameworkResources.ImageResource.BitmapIconWaveSound, UIServiceBase.TransparentColor);
                _iconImages.Images.Add(FrameworkResources.ImageResource.BitmapIconWaveSound, UIServiceBase.TransparentColor);
                _iconImages.Images.Add(FrameworkResources.ImageResource.BitmapIconSequenceSound, UIServiceBase.TransparentColor);
                _iconImages.Images.Add(FrameworkResources.ImageResource.BitmapIconSequenceSoundSet, UIServiceBase.TransparentColor);
                _iconImages.Images.Add(FrameworkResources.ImageResource.BitmapIconBank, UIServiceBase.TransparentColor);
                _iconImages.Images.Add(FrameworkResources.ImageResource.BitmapIconWaveArchive, UIServiceBase.TransparentColor);
                _iconImages.Images.Add(FrameworkResources.ImageResource.BitmapIconGroup, UIServiceBase.TransparentColor);
                _iconImages.Images.Add(FrameworkResources.ImageResource.BitmapIconPlayer, UIServiceBase.TransparentColor);
                _iconImages.Images.Add(FrameworkResources.ImageResource.BitmapIconSoundSet, UIServiceBase.TransparentColor);
                _iconImages.Images.Add(FrameworkResources.ImageResource.BitmapIconInstrument, UIServiceBase.TransparentColor);
            }

            public static ImageList IconImages
            {
                get { return _iconImages; }
            }
        }
    }
}
