﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Windows.Forms
{
    using System;
    using System.Collections;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Reflection;
    using System.Runtime.InteropServices;
    using System.Text;
    using System.Threading;
    using System.Windows.Forms;
    using System.Xml;
    using NintendoWare.SoundFoundation;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Conversion;
    using NintendoWare.SoundFoundation.Conversion.NintendoWareBinary;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Core.Reflection;
    using NintendoWare.SoundFoundation.Core.Wave;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.FileFormats.Audio;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareIntermediate;
    using NintendoWare.SoundFoundation.FileFormats.Wave;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Commands;
    using NintendoWare.SoundMaker.Configurations;
    using NintendoWare.SoundMaker.Configurations.Schemas;
    using NintendoWare.SoundMaker.FileFormats;
    using NintendoWare.SoundMaker.Framework;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.FileFormats;
    using NintendoWare.SoundMaker.Framework.Preview.Communications;
    using NintendoWare.SoundMaker.Framework.Windows;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;
    using NintendoWare.SoundMaker.Preview.Communications;
    using NintendoWare.SoundMaker.Preview.MCS;
    using NintendoWare.SoundMaker.Preview.Service;
    using NintendoWare.SoundMaker.Resources;
    using NintendoWare.SoundMakerPlugin;
    using NintendoWare.ToolDevelopmentKit.Collections;
    using NintendoWare.SoundMaker;
    using NintendoWare.SoundMaker.Windows.Forms.CommandHandlers;

    using NW4R.ProtocolSound;
    using CommFunc = NintendoWare.SoundMaker.Framework.Preview.Communications.Func;
    using CommSync = NintendoWare.SoundMaker.Framework.Preview.Communications.Sync;
    using CommTool = NintendoWare.SoundMaker.Framework.Preview.Communications.Tool;
    using FrameworkResources = NintendoWare.SoundMaker.Framework.Resources;
    using Midi = NintendoWare.SoundFoundation.Core.Midi;

    /// <summary>
    /// WindowsFormsベースのCommonアプリケーションクラスです。
    /// </summary>
    public class FormsApplicationCommon : FormsApplication
    {
        /// <summary>
        ///
        /// </summary>
        [DllImport("Imm32.dll")]
        public static extern IntPtr ImmGetContext(IntPtr hWnd);

        [DllImport("Imm32.dll")]
        public static extern bool ImmGetOpenStatus(IntPtr hIMC);

        [DllImport("Imm32.dll")]
        public static extern int ImmGetCompositionString(IntPtr hIMC, int dwIndex, StringBuilder lpBuf, int dwBufLen);

        [DllImport("Imm32.dll")]
        public static extern bool ImmReleaseContext(IntPtr hWnd, IntPtr hIMC);

        ///
        private const string BinarySoundArchiveExtension = "." + SoundFileExtensions.SoundArchiveBinaryFile;
        private const string BinaryStreamSoundExtension = "." + SoundFileExtensions.StreamSoundBinaryFile;
        private const string BinaryAACExtension = ".aac";
        private const string BinaryOpusExtension = ".opus";

        private PresetConfigurationCommon _presetConfiguration = new PresetConfigurationCommon();

        // HACK : ★MidiDeviceモデルを作成したい
        // HACK : ★MidiManagerはPreviewに含めたい
        private IMidiManager _midiManager;
        private IntPtr sequencePlayerForMidiKeyboard;
        private HashSet<IntPtr> sequencePlayerForMidiSequencer = new HashSet<IntPtr>();
        private PreviewNotePlayer _notePlayer = new PreviewNotePlayer();

        private BankDocument _previewBankDocument;
        private Instrument _previewInstrument;

        private bool _isCopySoundArchiveFile = true;

        private CommunicatingDialog _communiatingDialog;
        private bool connectedToSoundPlayer = false;
        private CommunicationService communicationService;
        private Action<TargetPlayerRunArgument> runPcOrTargetPlayerDelegate = null;

        private enum SoundPlayerCtrlIndex : int
        {
            SoundPlayerCtrl_0 = 0,
            //SoundPlayerCtrl_1,
        }

        /// <summary>
        /// コンストラクタ。
        /// Singleton のため外部からの生成は禁止されています。
        /// </summary>
        private FormsApplicationCommon()
            : base(FormsApplicationCommon.CreateTraits())
        {
            this.communicationService = this.CreateService(typeof(CommunicationService)) as CommunicationService;
            this.LoudnessService = this.CreateService(typeof(LoudnessService)) as LoudnessService;
            this.LoudnessService.Measuring += this.OnMeasuringLoudness;
            this.LoudnessService.Measured += this.OnMeasuredLoudness;
            this.StatisticsService = this.CreateService(typeof(StatisticsService)) as StatisticsService;
            this.StatisticsService.ComponentDictionary = this.ProjectService.ComponentDictionary;
            this.StatisticsService.GetHashCodeFunc = this.LoudnessService.GetHashCode;

            this.ProjectService.ComponentsAdded += this.OnComponentsAdded;
            this.ProjectService.ComponentsRemoved += this.OnComponentsRemoved;
            this.ProjectService.ProjectReloading += this.OnProjectReloading;
            this.ProjectService.ProjectReloaded += this.OnProjectReloaded;
            this.ProjectService.Closing += this.OnProjectClosing;
            this.ProjectService.SoundSetDocumentOpened += this.OnSoundSetDocumentOpened;

            this.DocumentService.DocumentSaved += this.OnDocumentSaved;

            // (8KB) * 5block
            SoundProject.SoundArchivePlayerStreamBufferDefalutSize = (8 * 1024) * 5;
            // (48KHz * 3ms * 8倍ピッチ * 2byte(PCM16)) * 5block
            SoundProject.SoundArchivePlayerStreamBufferExtensionSize = (48000 * 3 / 1000 * 8 * 2) * 5;

            SoundMakerPluginManager.Instance.PlatformChanged += this.OnPlatformChanged;
            PlatformChange();

            //
            SplashWindow.BackgroundBitmapImage = ImageResource.Logo;
            Assembly assembly = Assembly.GetExecutingAssembly();
            SplashWindow.VersionInfo = FileVersionInfo.GetVersionInfo(assembly.Location);
            SplashWindow.ShowWindow();

            //
            Preview.PreviewManager.Instance.Initialize();
            ComponentConfigurationCommon.Instance.Initialize();

            //
            Dictionary<string, Type[]> textTypeTable = ClipboardService.TextTypeTable;
            textTypeTable.Remove("StreamSound");
            textTypeTable.Remove("StreamSoundTrack");
            textTypeTable.Remove("SequenceSound");
            textTypeTable.Remove("WaveSound");
            textTypeTable.Remove("Bank");

            textTypeTable.Add("StreamSound", new Type[] { typeof(StreamSound) });
            textTypeTable.Add("StreamSoundTrack", new Type[] { typeof(StreamSoundTrack) });
            textTypeTable.Add("SequenceSound", new Type[] { typeof(SequenceSound) });
            textTypeTable.Add("WaveSound", new Type[] { typeof(WaveSound) });
            textTypeTable.Add("Bank", new Type[] { typeof(SoundSetBank) });

            // クリップボード関係
            Dictionary<Type, Type> typeChildrenTypeTable = ClipboardService.TypeChildrenTypeTable;
            typeChildrenTypeTable.Remove(typeof(StreamSoundPack));
            typeChildrenTypeTable.Remove(typeof(SequenceSoundPack));
            typeChildrenTypeTable.Remove(typeof(SoundSetBankPack));
            typeChildrenTypeTable.Remove(typeof(StreamSoundBase));
            typeChildrenTypeTable.Remove(typeof(SequenceSoundSetBase));
            typeChildrenTypeTable.Remove(typeof(WaveSoundSetBase));

            typeChildrenTypeTable.Add(typeof(StreamSoundPack), typeof(StreamSound));
            typeChildrenTypeTable.Add(typeof(SequenceSoundPack), typeof(SequenceSound));
            typeChildrenTypeTable.Add(typeof(SoundSetBankPack), typeof(SoundSetBank));
            typeChildrenTypeTable.Add(typeof(StreamSound), typeof(StreamSoundTrack));
            typeChildrenTypeTable.Add(typeof(SequenceSoundSetBase), typeof(SequenceSound));
            typeChildrenTypeTable.Add(typeof(WaveSoundSetBase), typeof(WaveSound));

            WaveFileReader.CreateWaveFileOtherReader = delegate (string filePath)
            {
                try
                {
                    if (String.Compare(Path.GetExtension(filePath), ".opus", true) == 0)
                    {
                        var reader = new WaveFileOpusReader();
                        reader.Open(filePath); // Opus ファイルなのか一度開いてみる、
                        reader.Close();        // 確認しただけなので閉じておく。

                        return reader;
                    }
                }
                catch { }
                return null;           // Opus ファイルではない場合。
            };

            FileUtil.EnabledOpus = true; // Opus 対応を有効にします。

            FilePathParameterValue.Validator =
                delegate (string value)
                {
                    ValidationResult result = FilePathValidator.ValidateFilePath(value);
                    // 現在は aac のみ（mp4なし）なので ADTS のチェックをします。
                    if (result.IsValid == true &&
                        AACUtil.IsAACFile(value) == true &&
                        File.Exists(value) == true)
                    {
                        using (Stream stream = File.OpenRead(value))
                        {
                            AdtsHeader adtsHeader = AdtsHeader.Parse(stream);
                            if (adtsHeader != null && adtsHeader.IsValid() == false)
                            {
                                result = new ValidationResult(false, MessageResource.Message_NotAACFileFormat);
                            }
                        }
                    }

                    return result;
                };

            // 検索関係の設定を行ないます。
            FindResultPanel2.ListControlCreator = delegate ()
                {
                    return new ListCtrlCommon();
                };

            ProjectSettingsDialog.ShowHiddenFunction = AppConfigurationCommon.IsHiddenFunctionEnabled();
            ProjectSettingsDialog.InitializeInternal = delegate (ProjectSettingsDialog dialog)
            {
                if (this.MainWindow.IsPlatformCafe() == false)
                {
                    dialog.HideSndEditTab();
                    dialog.HideAAC();
                }

                dialog.HideSndEditChannelTab();

                if (AppConfigurationCommon.EnabledPrefetch == false)
                {
                    dialog.HideStreamBufferTimes();
                }
            };

            OutputWaveFileDialog.InitializeInternal = delegate (OutputWaveFileDialog dialog)
            {
                if (AppConfigurationCommon.Enabled48kHz == true)
                {
                    dialog.ShowSamplingRatePanel();
                }
            };

            FindResultSettings.ReferenceParentParameterNames.Add
                (ProjectParameterNamesCommon.RemoteFilter);
            FindResultSettings.ReferenceParentParameterNames.Add
                (ProjectParameterNamesCommon.Sends.AuxCSend);

            FindResultSettings.ExtendHeaderColumnDatas.Add
                (new FindResultSettings.ExtendHeaderColumnData()
                {
                    PreviousColumnName = ProjectParameterNames.StreamSoundTrack.Sends.AuxBSend,
                    HeaderColumnData = new HeaderColumnData()
                    {
                        Name = ProjectParameterNamesCommon.StreamSoundTrack.Sends.AuxCSend,
                        Text = MessageResource.HeaderText_TrackAuxCSend,
                        Length = 80,
                        Visible = true,
                    },
                });

            FindResultSettings.ReplacedParameterNames.Add
                (ProjectParameterNamesCommon.StreamSoundTrack.Sends.AuxCSend,
                  ProjectParameterNamesCommon.Sends.AuxCSend);

            FindResultSettings.ExtendListColumnStyles.Add
                (new ListColumnStyle(ListTraitsCommon.ColumnName_TrackAuxCSend,
                                       "Integer",
                                       typeof(int)));

            //
            AnotherPlatformProjectFileQuester.RequestFilter +=
                delegate (RequestFilterEventArgs e)
                {
                    e.Filter = Resources.MessageResource.FileFilter_ProjectFilesforOtherPlatforms +
                    " (*.cspj;*.rspj)|*.cspj;*.rspj" +
                    FileQuester.AllFilesFilter;
                };

            // 接続ポートの保存を設定
            SoundProjectParameterTranslateSndEdit.SndEditParamNames = new string[]
            {
                ProjectParameterNames.SndEditSetting.SyncPort,
                ProjectParameterNames.SndEditSetting.FuncPort,
                ProjectParameterNames.SndEditSetting.CtrlPort,
            };

            SoundProjectTranslatorSettings.EnabledSoundArchiveOutputType = true;

            // リアルタイム編集を有効にします。
            this.RealtimeEditService.Attach(ProjectService, ConvertService, BankServices);

            SoundEditInfoCreator.SendsGetter = delegate (Sends sends)
                {
                    SendsParam sendsParam = new SendsParam();
                    SendsCommon sendsCommon = sends as SendsCommon;
                    Debug.Assert(sendsCommon != null, "Sends is not SendsCommon");

                    sendsParam.mainSend = (byte)sendsCommon.MainSend;
                    sendsParam.fxSends[0] = (byte)sendsCommon.AuxASend;
                    sendsParam.fxSends[1] = (byte)sendsCommon.AuxBSend;
                    sendsParam.fxSends[2] = (byte)sendsCommon.AuxCSend;
                    return sendsParam;
                };

            //
            AnotherPlatformBankFileQuester.RequestFilter +=
                delegate (RequestFilterEventArgs e)
                {
                    e.Filter = FrameworkResources.MessageResource.FileFilter_BankFilesforOtherPlatforms +
                    " (*.cbnk;*.rbnk;*.bnk)|*.cbnk;*.rbnk;*.bnk" +
                    FileQuester.AllFilesFilter;
                };

            //
            AnotherPlatformSoundSetFileQuester.RequestFilter +=
                delegate (RequestFilterEventArgs e)
                {
                    e.Filter = FrameworkResources.MessageResource.FileFilter_SoundSetFilesforOtherPlatforms +
                    " (*.csst;*.rsst)|*.csst;*.rsst" +
                    FileQuester.AllFilesFilter;
                };

            //
            UnhandleExceptionHelper.Initialize();

            ProjectService.ProjectDocumentOpened += (s, e) =>
                {
                    MainWindow.ProjectPanel.UpdateProjectCtrl();
                    MainWindow.SoundSetPanels.ToList().ForEach(p => p.UpdateTabImage());

                    WaveFileManager.Instance.PreprocessExePath = this.GetWavePreprocessExeFullPath();
                };

            DocumentService.DocumentOpening += (s, e) =>
                {
                    if (e.FilePath != null)
                    {
                        OpenHelper.FileOpenPreCommand(e.FilePath);
                    }
                };

            // コンポーネントのパラメータにユーザー指定の初期値を設定するようにします。
            this.CreateComponentService.EnabledDefaultParameter = true;
        }

        /// <summary>
        /// アプリケーションのインスタンスを取得します。
        /// </summary>
        public static new FormsApplicationCommon Instance
        {
            get
            {
                if (null == FormsApplication.Instance)
                {
                    // コンストラクタで AppliationBase.Instance が設定される。
                    new FormsApplicationCommon();
                }

                return FormsApplication.Instance as FormsApplicationCommon;
            }
        }

        /// <summary>
        /// アプリケーション設定を取得します。
        /// </summary>
        public new AppConfigurationCommon AppConfiguration
        {
            get { return base.AppConfiguration as AppConfigurationCommon; }
        }

        /// <summary>
        /// プロジェクト設定を取得します。
        /// </summary>
        public new SoundProjectConfigurationCommon ProjectConfiguration
        {
            get { return base.ProjectConfiguration as SoundProjectConfigurationCommon; }
        }

        /// <summary>
        /// プリセット設定を取得します。
        /// </summary>
        public PresetConfigurationCommon PresetConfiguration
        {
            get { return _presetConfiguration; }
        }

        /// <summary>
        /// SoundPlayer ヘルプファイルのパスを取得します。
        /// </summary>
        public string SoundPlayerHelpFilePath
        {
            get
            {
                return PathUtilityCommon.SoundPlayerHelpFilePath;
            }
        }

        /// <summary>
        /// AtkPlayer ヘルプファイルの Url を取得します。
        /// </summary>
        public string AtkPlayerHelpURL
        {
            get
            {
                return PathUtilityCommon.AtkPlayerHelpURL;
            }
        }

        /// <summary>
        /// 現在選択されているプラットフォームのインスタンスを取得します。
        /// </summary>
        public ISoundMakerPlugin CurrentSoundMakerPlugin
        {
            get
            {
                return SoundMakerPluginManager.Instance.CurrentSoundMakerPlugin;
            }
        }

        /// <summary>
        /// CommunicationService を取得します。
        /// </summary>
        public CommunicationService CommunicationService
        {
            get { return this.communicationService; }
        }

        /// <summary>
        /// LoudnessService を取得します。
        /// </summary>
        public LoudnessService LoudnessService
        {
            get;
        }

        /// <summary>
        /// StatisticsService を取得します。
        /// </summary>
        public StatisticsService StatisticsService
        {
            get;
        }

        /// <summary>
        /// 現在選択されているプラットフォームが Generic かどうかを取得します。
        /// </summary>
        public bool IsCurrentSoundPluginGeneric()
        {
            return this.CurrentSoundMakerPlugin.Platform == "Generic";
        }

        /// <summary>
        /// プレビュー対象のバンクドキュメントを取得します。
        /// </summary>
        public BankDocument PreviewBankDocument
        {
            get { return _previewBankDocument; }
        }

        /// <summary>
        /// プレビュー対象のインストルメントを取得します。
        /// </summary>
        public Instrument PreviewInstrument
        {
            get { return _previewInstrument; }
        }

        /// <summary>
        /// NotePlayer を取得します。（★Preview と統合したい）
        /// </summary>
        public PreviewNotePlayer NotePlayer
        {
            get { return _notePlayer; }
        }

        /// <summary>
        /// 拡張コマンドの列挙子を返します。
        /// </summary>
        protected override IEnumerable<Command> ExtendedCommands
        {
            get
            {
                return PreviewCommonCommands.Commands
                    .Concat(SndEditCommonCommands.Commands)
                    .Concat(base.ExtendedCommands)
                    .ToArray();
            }
        }

        /// <summary>
        /// コンバート結果のインストール先フォルダパスを取得します。
        /// </summary>
        protected override string InstallBinaryFilesDirectoryPath
        {
            get
            {
                return PathUtilityCommon.SoundPlayerTempDirectory;
            }
        }

        private MainWindowCommon MainWindow
        {
            get
            {
                return UIService.MainWindow as MainWindowCommon;
            }
        }

        ///
        private string ContentDirectory
        {
            get
            {
                return PathUtilityCommon.SoundPlayerContentsDirectory;
            }
        }

        ///
        private PreviewPlayerPanel PreviewPlayerPanel
        {
            get
            {
                return this.MainWindow.ToolPages[PreviewPlayerPanel.PageName] as PreviewPlayerPanel;
            }
        }

        private PreviewBankPanel PreviewBankPanel
        {
            get
            {
                return this.MainWindow.ToolPages[PreviewBankPanel.PageName] as PreviewBankPanel;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private string PresetConfigurationFilePath
        {
            get
            {
                if (null == ProjectService.ProjectDocument) { return null; }
                return ProjectService.ProjectFilePath + ".presets";
            }
        }

        private bool IsTargetPC
        {
            get
            {
                return this.communicationService.IsTargetPC;
            }
        }

        private bool IsTargetSoundPlayer
        {
            get
            {
                return this.MainWindow.IsConnectTargetSoundPlayer;
            }
        }

        public void StartupWaveOutput()
        {
            ShutdownWaveOutput();

            this.CurrentSoundMakerPlugin.InitializeAudioDevice(AppConfiguration.Options.Application.Sound.WaveOutDeviceName);
        }

        public void ShutdownWaveOutput()
        {
            // NOTE: nn::audio をシャットダウンすることはないが、過去の SoundMaker のロジックを踏襲するために、関数は残しておく
        }

        public void StartupMidiInputDevices()
        {
            this.ShutdownMidiInputDevices();

            this._midiManager = this.CurrentSoundMakerPlugin.CreateMidiManager();
            this._midiManager.Setup(OnNoteOn, null);

            this.sequencePlayerForMidiKeyboard = IntPtr.Zero;

            XmlMidiDevice midiKeyboardDeviceXml =
                this.AppConfiguration.Options.Application.Midi.MidiKeyboardInputDevice;

            if (midiKeyboardDeviceXml != null)
            {
                if (this._midiManager.OpenMidiIn((uint)midiKeyboardDeviceXml.DeviceID))
                {
                    this.sequencePlayerForMidiKeyboard =
                        this._midiManager.GetSequenceSoundPlayerHandle(
                            (uint)midiKeyboardDeviceXml.DeviceID
                            );
                }

                this.ProgramChange(_previewBankDocument, _previewInstrument);
            }

            XmlMidiDevice[] midiSequencerDevicesXml =
                this.AppConfiguration.Options.Application.Midi.MidiSequencerInputDevices;

            if (midiSequencerDevicesXml != null)
            {
                foreach (XmlMidiDevice device in midiSequencerDevicesXml)
                {
                    if (midiKeyboardDeviceXml == null ||
                        midiKeyboardDeviceXml.DeviceID != device.DeviceID)
                    {
                        if (!this._midiManager.OpenMidiIn((uint)device.DeviceID))
                        {
                            continue;
                        }
                    }

                    this.sequencePlayerForMidiSequencer.Add(
                        this._midiManager.GetSequenceSoundPlayerHandle(
                            (uint)device.DeviceID
                        ));
                }
            }
        }

        public void ShutdownMidiInputDevices()
        {
            this.sequencePlayerForMidiKeyboard = IntPtr.Zero;
            this.sequencePlayerForMidiSequencer.Clear();

            if (this._midiManager != null)
            {
                this._midiManager.Shutdown();
                this._midiManager = null;
            }
        }

        public void ProgramChange(BankDocument bankDocument, Instrument instrument)
        {
            bool changed = (_previewBankDocument != bankDocument || _previewInstrument != instrument);

            _previewBankDocument = bankDocument;
            _previewInstrument = instrument;

            if (null != bankDocument && null != instrument)
            {
                this._midiManager.ProgramChange(0, instrument.ProgramNo);

                XmlOptionsApplicationMidi midiXml = AppConfiguration.Options.Application.Midi;

                if (midiXml.MidiKeyboardInputDevice != null)
                {
                    this._midiManager.ProgramChange(
                        (uint)midiXml.MidiKeyboardInputDevice.DeviceID, 0, instrument.ProgramNo);
                }
            }

            if (changed && null != ProgramChanged)
            {
                ProgramChanged(this, EventArgs.Empty);
            }
        }

        public void NoteOn(int programNo, int key, int velocity)
        {
            this._midiManager.ProgramChange(0, programNo);
            NoteOn(key, velocity);
        }

        public void NoteOn(int key, int velocity)
        {
            this.LoudnessService.Stop();
            this._midiManager.SendMidiMessage(0x90, key, velocity);
        }

        public void NoteOff(int key)
        {
            this._midiManager.SendMidiMessage(0x80, key, 0);
        }

        public bool IsOutputSounds()
        {
            bool result = this.CurrentSoundMakerPlugin.RuntimeSoundSystem_GetActiveVoiceCount() > 0;

            // ラウドネス計測中かどうか
            if (this.LoudnessService.IsMeasuring == true)
            {
#if DEBUG
                // デバッグしやすいようにラウドネス計測中でも result を返すようにします。
#else
                // ラウドネス計測中は常に false を返します。
                return false;
#endif
            }

            return result;
        }

        public bool IsReceiveMidiSignal(int duration)
        {
            return this._midiManager.IsActive((uint)duration);
        }

        /// <summary>
        ///
        /// </summary>
        public void StopAll()
        {
            PreviewPlayerPanel.StopAll();
            _notePlayer.Stop();
        }

        public void Play(Component component, uint playerIndex)
        {
            PreviewPlayerPanel.SelectItem(playerIndex);
            PreviewPlayerOperator.Play(component);
        }

        public CommandStatus CanRunTargetPlayer()
        {
            if (this.CurrentSoundMakerPlugin.TargetPlayer == null)
            {
                return CommandStatus.Supported;
            }

            if (null == ProjectService.ProjectDocument) { return CommandStatus.SupportedAndVisible; }
            if (!QueryCommandStatus(ProjectCommands.Convert).IsEnabled()) { return CommandStatus.SupportedAndVisible; }

            if (this.communicationService.IsConnected == true ||
                this.communicationService.IsConnecting == true)
            {
                if (this.IsTargetPC == false && this.IsTargetSoundPlayer == true)
                {
                    return CommandStatus.SupportedAndVisible;
                }
            }

            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        public CommandStatus CanRunPCSoundPlayer()
        {
            if (null == ProjectService.ProjectDocument) { return CommandStatus.SupportedAndVisible; }
            if (!QueryCommandStatus(ProjectCommands.Convert).IsEnabled()) { return CommandStatus.SupportedAndVisible; }

            if (this.communicationService.IsConnected == true ||
                this.communicationService.IsConnecting == true)
            {
                if (this.IsTargetPC == true && this.IsTargetSoundPlayer == true)
                {
                    return CommandStatus.SupportedAndVisible;
                }
            }
            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        public CommandStatus CanRunPCAtkPlayer()
        {
            if (null == ProjectService.ProjectDocument) { return CommandStatus.SupportedAndVisible; }
            if (!QueryCommandStatus(ProjectCommands.Convert).IsEnabled()) { return CommandStatus.SupportedAndVisible; }

            if (this.communicationService.IsConnected == true ||
                this.communicationService.IsConnecting == true)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        ///
        public CommandStatus CanConnectToTarget()
        {
            if (this.communicationService.IsConnected ||
                this.communicationService.IsConnectedSndEdit)
            {
                return CommandStatus.Supported;
            }

            if (this.communicationService.IsConnecting ||
                this.communicationService.IsConnectingSndEdit)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        ///
        public CommandStatus CanDisconnectFromTarget()
        {
            if (this.communicationService.IsConnected ||
                this.communicationService.IsConnectedSndEdit)
            {
                return CommandStatus.SupportedAndEnabledAndVisible;
            }
            else
            {
                return CommandStatus.Supported;
            }
        }

        ///
        public bool ConnectToTarget()
        {
            if (Process.GetProcessesByName("NintendoTargetManager").Count() == 0)
            {
                UIService.ShowMessageBox(Resources.MessageResource.Message_NotRunningTargetManager);
                return false;
            }

            bool result = false;

            try
            {
                string platform = this.MainWindow.SelectedPlatform;
                string targetName = this.MainWindow.SelectedTargetName;

                // 1秒間隔で2回、接続を試みるようにします。
                int retryCount = 2;

                while (retryCount > 0)
                {
                    Debug.WriteLine(retryCount);
                    result = this.communicationService.Connect(platform, targetName);
                    if (result == true)
                    {
                        break;
                    }
                    retryCount--;
                    Thread.Sleep(1000);
                }
            }
            catch
            {
            }

            if (!result)
            {
                UIService.ShowMessageBox(Resources.MessageResource.Message_FailedToConnectApp);
                return false;
            }

            return true;
        }

        ///
        public bool DisconnectFromTarget()
        {
            this.communicationService.Disconnect();
            return true;
        }

        ///
        public bool RunPCSoundPlayer()
        {
            string caption = MessageResource.Command_Preview_RunPCSoundPlayer_Name;
            string message = MessageResource.Message_RunPCSoundPlayer;

            if (MessageBox.Show(message, caption, MessageBoxButtons.OKCancel) == DialogResult.OK)
            {
                this.runPcOrTargetPlayerDelegate = this.RunPCSoundPlayerCallback;

                CommandUri uri = CommandUri.Create(
                                     ProjectCommands.Convert.ID,
                                     new KeyValuePair<string, string>[] {
                                         new KeyValuePair<string, string>(
                                             CommandParameterNames.InstallDirectory,
                                             InstallBinaryFilesDirectoryPath ) }
                                     );

                return ExecuteCommand(new Command(uri.Uri, string.Empty));
            }

            return false;
        }

        public bool RunPCAtkPlayer()
        {
            string caption = MessageResource.Command_Preview_RunPCAtkPlayer_Name;
            string message = MessageResource.Message_RunPCAtkPlayer;

            if (MessageBox.Show(message, caption, MessageBoxButtons.OKCancel) == DialogResult.OK)
            {
                this.runPcOrTargetPlayerDelegate = this.RunPCAtkPlayerCallback;

                CommandUri uri = CommandUri.Create(ProjectCommands.Convert.ID, new KeyValuePair<string, string>[] { });

                return ExecuteCommand(new Command(uri.Uri, string.Empty));
            }

            return false;
        }

        ///
        public bool RunTargetPlayer()
        {
            ITargetPlayer targetPlayer = this.CurrentSoundMakerPlugin.TargetPlayer;

            string caption = targetPlayer.UIConfirmRunTitle;
            string message = targetPlayer.UIConfirmRunMessage;

            if (MessageBox.Show(message, caption, MessageBoxButtons.OKCancel) == DialogResult.OK)
            {
                try
                {
                    // コンバート前に停止しておかないとバイナリファイルを上書きできない
                    targetPlayer.Shutdown();

                    this.runPcOrTargetPlayerDelegate = targetPlayer.Run;

                    CommandUri uri = CommandUri.Create(
                                         ProjectCommands.Convert.ID,
                                         new KeyValuePair<string, string>[] {
                                             new KeyValuePair<string, string>(
                                             CommandParameterNames.InstallDirectory,
                                             InstallBinaryFilesDirectoryPath ) }
                                     );

                    return ExecuteCommand(new Command(uri.Uri, string.Empty));
                }
                catch (Exception e)
                {
                    UIService.ShowMessageBox(e.Message);
                }
            }

            return false;
        }

        ///
        public bool ConnectToPCViewer()
        {
            bool result = false;

            try
            {
                // 1秒間隔で2回、接続を試みるようにします。
                int retryCount = 2;

                while (retryCount > 0)
                {
                    result = this.communicationService.Connect("Generic", "PCSoundPlayer"); // 今は決め打ち
                    if (result == true)
                    {
                        break;
                    }
                    retryCount--;
                    Thread.Sleep(1000);
                }
            }
            catch
            {
            }

            if (!result)
            {
                UIService.ShowMessageBox(Resources.MessageResource.Message_FailedToConnectApp);
                return false;
            }

            return true;
        }

        ///
        public CommandStatus CanConnectToPCViewer()
        {
            if (this.IsCurrentSoundPluginGeneric() == false)
            {
                return CommandStatus.Supported;
            }

            if (this.communicationService.IsConnected ||
                this.communicationService.IsConnectedSndEdit)
            {
                if (this.IsTargetPC == true)
                {
                    return CommandStatus.Supported;
                }
                else
                {
                    return CommandStatus.SupportedAndVisible;
                }
            }

            if (this.communicationService.IsConnecting ||
                this.communicationService.IsConnectingSndEdit)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        public CommandStatus CanDisconnectFromPCViewer()
        {
            if (this.IsCurrentSoundPluginGeneric() == false)
            {
                return CommandStatus.Supported;
            }

            if (this.communicationService.IsConnected ||
                this.communicationService.IsConnectedSndEdit)
            {
                if (this.IsTargetPC == true)
                {
                    return CommandStatus.SupportedAndEnabledAndVisible;
                }
                else
                {
                    return CommandStatus.Supported;
                }
            }

            return CommandStatus.Supported;
        }

        ///
        public bool DisconnectFromPCViewer()
        {
            this.communicationService.Disconnect();
            return true;
        }

        /// <summary>
        /// アプリケーション設定クラスを作成します。
        /// </summary>
        /// <returns>作成した AppConfiguration。</returns>
        protected override AppConfiguration CreateAppConfiguration()
        {
            return new AppConfigurationCommon();
        }

        /// <summary>
        /// プロジェクト設定クラスを作成します。
        /// </summary>
        /// <returns>作成した SoundProjectConfiguration。</returns>
        protected override SoundProjectConfiguration CreateProjectConfiguration()
        {
            return new SoundProjectConfigurationCommon();
        }

        /// <summary>
        /// カラムプリセット設定クラスを作成します。
        /// </summary>
        /// <returns>作成した PresetListColumnsConfiguration </returns>
        protected override PresetListColumnsConfiguration CreatePresetListColumnsConfiguration()
        {
            return new PresetListColumnsConfiguration();
        }

        /// <summary>
        /// ブックマーク設定クラスを作成します。
        /// </summary>
        /// <returns>作成した BookmarkConfiguration </returns>
        protected override BookmarkConfiguration CreateBookmarkConfiguration()
        {
            return new BookmarkConfiguration();
        }

        /// <summary>
        /// ドキュメントサービス特性を生成します。
        /// </summary>
        /// <returns>IDocumentServiceTraits を返します。</returns>
        protected override IDocumentServiceTraits CreateDocumentServiceTraits()
        {
            DocumentServiceTraits traits = new DocumentServiceTraits();

            // DocumentFactory
            traits.DocumentFactorys.Add(new SoundProjectDocumentFactory());
            traits.DocumentFactorys.Add(new SoundSetDocumentFactory());
            traits.DocumentFactorys.Add(new BankDocumentFactory());
            traits.DocumentFactorys.Add(new SoundProjectDocumentFactoryCafe());
            traits.DocumentFactorys.Add(new SoundSetDocumentFactoryCafe());
            traits.DocumentFactorys.Add(new BankDocumentFactoryCafe());
            traits.DocumentFactorys.Add(new SoundProjectDocumentFactoryCtr());
            traits.DocumentFactorys.Add(new SoundSetDocumentFactoryCtr());
            traits.DocumentFactorys.Add(new BankDocumentFactoryCtr());
            traits.DocumentFactorys.Add(new SoundProjectRvlDocumentFactory());
            traits.DocumentFactorys.Add(new SoundSetRvlDocumentFactory());
            traits.DocumentFactorys.Add(new BankRvlDocumentFactory());

            // DocumentReader
            traits.DocumentReaders.Add(new SoundProjectDocumentReader());
            traits.DocumentReaders.Add(new SoundSetDocumentReaderCommon(ApplicationBase.Instance.Traits.ComponentTraits.SoundSetComponentFactory));
            traits.DocumentReaders.Add(new BankDocumentReader());
            traits.DocumentReaders.Add(new SoundProjectDocumentReaderCafe());
            traits.DocumentReaders.Add(new SoundSetDocumentReaderCafe(ApplicationBase.Instance.Traits.ComponentTraits.SoundSetComponentFactory));
            traits.DocumentReaders.Add(new BankDocumentReaderCafe());
            traits.DocumentReaders.Add(new SoundProjectDocumentReaderCtr());
            traits.DocumentReaders.Add(new SoundSetDocumentReaderCtr(new SoundSetComponentFactoryCtr()));
            traits.DocumentReaders.Add(new BankDocumentReaderCtr());
            traits.DocumentReaders.Add(new SoundProjectRvlDocumentReader());
            traits.DocumentReaders.Add(new SoundSetRvlDocumentReader());
            traits.DocumentReaders.Add(new BankRvlDocumentReader());

            // DocumentWriter
            traits.DocumentWriters.Add(
                new SoundProjectDocumentWriter()
                {
                    ProductName = this.Traits.ProductName,
                    ProductVersion = this.Traits.ProductVersion
                });
            traits.DocumentWriters.Add(
                new SoundSetDocumentWriterCommon()
                {
                    ProductName = this.Traits.ProductName,
                    ProductVersion = this.Traits.ProductVersion
                });
            traits.DocumentWriters.Add(
                new BankDocumentWriter()
                {
                    ProductName = this.Traits.ProductName,
                    ProductVersion = this.Traits.ProductVersion
                });

            // DocumentConverter
            traits.DocumentConverters.Add(
                new SoundProjectCtrToSoundProjectDocumentConverter(
                    this.Traits.IntermediateOutputTraits)
                );
            traits.DocumentConverters.Add(
                new SoundSetCtrToSoundSetDocumentConverter(
                    this.Traits.IntermediateOutputTraits,
                    ApplicationBase.Instance.Traits.ComponentTraits.SoundSetComponentFactory)
                );
            traits.DocumentConverters.Add(
                new BankCtrToBankDocumentConverter(
                    this.Traits.IntermediateOutputTraits.BankDocumentTypeName,
                    this.Traits.IntermediateOutputTraits.BankFileExtension)
                );

            traits.DocumentConverters.Add(
                new SoundProjectRvlToSoundProjectDocumentConverter(
                    this.Traits.IntermediateOutputTraits)
                );
            traits.DocumentConverters.Add(
                new SoundSetRvlToSoundSetDocumentConverterCommon(
                    this.Traits.IntermediateOutputTraits,
                    ApplicationBase.Instance.Traits.ComponentTraits.SoundSetComponentFactory)
                );
            traits.DocumentConverters.Add(
                new BankRvlToBankDocumentConverter()
                );

            foreach (ISoundMakerPlugin plugin in SoundMakerPluginManager.Instance.Instances)
            {
                plugin.Initialize(ApplicationBase.Instance.Traits.ComponentTraits.SoundSetComponentFactory as SoundSetComponentFactory);
            }

            this.AddDocumentServiceTraits(traits);

            return traits;
        }

        /// <summary>
        /// ドキュメントインポートサービス特性を生成します。
        /// </summary>
        /// <returns>DocumentImportServiceTraits を返します。</returns>
        protected override DocumentImportServiceTraits CreateDocumentImportServiceTraits()
        {
            ObjectAggregateFactory<DocumentImportServiceTraits.ImporterKey, IDocumentImporter>
                importerFactory = new ObjectAggregateFactory<DocumentImportServiceTraits.ImporterKey, IDocumentImporter>();

            // Ctr -> Common
            importerFactory.Factories.Add(
                new DocumentImportServiceTraits.ImporterKey(
                    Platforms.Ctr.SoundProjectDocument,
                    this.Traits.IntermediateOutputTraits.SoundProjectDocumentTypeName
                    ),
                new ObjectFactoryDelegation<DocumentImportServiceTraits.ImporterKey, IDocumentImporter>(
                    key => new SoundDocumentImporterCtr(this.DocumentService, this.Traits.IntermediateOutputTraits)
                    )
                );

            importerFactory.Factories.Add(
                new DocumentImportServiceTraits.ImporterKey(
                    Platforms.Ctr.SoundSetDocument,
                    this.Traits.IntermediateOutputTraits.SoundSetDocumentTypeName
                    ),
                new ObjectFactoryDelegation<DocumentImportServiceTraits.ImporterKey, IDocumentImporter>(
                    key => new SoundSetDocumentImporterCtr(this.DocumentService, this.Traits.IntermediateOutputTraits)
                    )
                );

            importerFactory.Factories.Add(
                new DocumentImportServiceTraits.ImporterKey(
                    Platforms.Ctr.BankDocument,
                    this.Traits.IntermediateOutputTraits.BankDocumentTypeName
                    ),
                new ObjectFactoryDelegation<DocumentImportServiceTraits.ImporterKey, IDocumentImporter>(
                    key => new BankDocumentImporterCtr(this.DocumentService, this.Traits.IntermediateOutputTraits)
                    )
                );

            // Rvl -> Common
            importerFactory.Factories.Add(
                new DocumentImportServiceTraits.ImporterKey(
                    Platforms.Rvl.SoundProjectDocument,
                    this.Traits.IntermediateOutputTraits.SoundProjectDocumentTypeName
                    ),
                new ObjectFactoryDelegation<DocumentImportServiceTraits.ImporterKey, IDocumentImporter>(
                    key => new SoundRvlDocumentImporter(this.DocumentService, this.Traits.IntermediateOutputTraits)
                    )
                );

            importerFactory.Factories.Add(
                new DocumentImportServiceTraits.ImporterKey(
                    Platforms.Rvl.SoundSetDocument,
                    this.Traits.IntermediateOutputTraits.SoundSetDocumentTypeName
                    ),
                new ObjectFactoryDelegation<DocumentImportServiceTraits.ImporterKey, IDocumentImporter>(
                    key => new SoundSetRvlDocumentImporter(this.DocumentService, this.Traits.IntermediateOutputTraits)
                    )
                );

            importerFactory.Factories.Add(
                new DocumentImportServiceTraits.ImporterKey(
                    Platforms.Rvl.BankDocument,
                    this.Traits.IntermediateOutputTraits.BankDocumentTypeName
                    ),
                new ObjectFactoryDelegation<DocumentImportServiceTraits.ImporterKey, IDocumentImporter>(
                    key => new BankRvlDocumentImporter(this.DocumentService, this.Traits.IntermediateOutputTraits)
                    )
                );

            return new DocumentImportServiceTraits()
            {
                ImporterFactory = importerFactory,
            };
        }

        /// <summary>
        /// サービスのインスタンスを作成します。
        /// </summary>
        /// <param name="serviceType">作成するサービスの種類。</param>
        /// <returns>サービスのインスタンス。</returns>
        protected override object CreateService(Type serviceType)
        {
            if (serviceType == typeof(UIServiceBase))
            {
                return new FormsUIServiceCommon();
            }

            if (serviceType == typeof(SoundProjectService))
            {
                return new SoundProjectServiceCommon(
                    this.DocumentService,
                    this.Traits.IntermediateOutputTraits);
            }

            if (serviceType == typeof(ISoundProjectConvertService))
            {
                return new SoundProjectConvertService(
                    this.Traits.ConversionTraits,
                    new FormsBasedInvoker(null),
                    this.BankServices);
            }

            if (serviceType == typeof(ISoundProjectConvertDependencyService))
            {
                return new SoundProjectConvertDependencyService();
            }

            if (serviceType == typeof(RealtimeEditService))
            {
                return new SndEditService(new FormsBasedInvoker(null));
            }

            if (serviceType == typeof(CommunicationService))
            {
                return new CommunicationService(SoundMakerPluginManager.Instance.Instances.Select(p => p.CommunicationInfo));
            }

            if (serviceType == typeof(LoudnessService))
            {
                return new LoudnessService();
            }

            if (serviceType == typeof(StatisticsService))
            {
                return new StatisticsService();
            }

            return base.CreateService(serviceType);
        }

        /// <summary>
        /// コマンドの関連付けを初期化します。
        /// </summary>
        protected override void InitializeCommandBindings()
        {
            base.InitializeCommandBindings();

            // プレビュー関連コマンドハンドラ
            BindCommand(PreviewCommonCommands.RunTargetPlayer);
            BindCommand(PreviewCommonCommands.ShutdownTargetPlayer);
#if USE_PCSOUNDPLAYER
            BindCommand(PreviewCommonCommands.RunPCSoundPlayer);
#endif
            BindCommand(PreviewCommonCommands.RunPCAtkPlayer);
            BindCommand(PreviewCommonCommands.ConnectToTarget);
            BindCommand(PreviewCommonCommands.DisconnectFromTarget);
            BindCommand(PreviewCommands.PlayAll);
            BindCommand(PreviewCommands.PauseAll);
            BindCommand(PreviewCommands.StopAll);

            // ツール関連コマンドハンドラ
            BindCommand(ToolCommands.StopAllSounds);
            BindCommand(ToolCommands.ResetMidi);
        }

        /// <summary>
        /// コマンドバインディングを作成します。
        /// </summary>
        /// <param name="serviceType">関連付けるコマンド。</param>
        /// <returns>コマンドバインディングのインスタンス。</returns>
        protected override CommandBinding CreateCommandBinding(Command command)
        {
            // プレビュー関連コマンドハンドラ
            if (command == PreviewCommonCommands.RunTargetPlayer)
            {
                return new CommandBinding(this, new RunTargetPlayerHandler());
            }
            if (command == PreviewCommonCommands.ShutdownTargetPlayer)
            {
                return new CommandBinding(this, new ShutdownTargetPlayerHandler());
            }
#if USE_PCSOUNDPLAYER
            if (command == PreviewCommonCommands.RunPCSoundPlayer)
            {
                return new CommandBinding(this, PreviewCommonCommands.RunPCSoundPlayer.ID,
                                          QueryStatusRunPCSoundPlayer, ExecuteRunPCSoundPlayer);
            }
#endif
            if (command == PreviewCommonCommands.RunPCAtkPlayer)
            {
                return new CommandBinding(this, PreviewCommonCommands.RunPCAtkPlayer.ID,
                                          QueryStatusRunPCAtkPlayer, ExecuteRunPCAtkPlayer);
            }
            if (command == PreviewCommonCommands.ConnectToTarget)
            {
                return new CommandBinding(this, PreviewCommonCommands.ConnectToTarget.ID,
                                          QueryStatusConnectToTarget, ExecuteConnectToTarget);
            }
            if (command == PreviewCommonCommands.DisconnectFromTarget)
            {
                return new CommandBinding(this, PreviewCommonCommands.DisconnectFromTarget.ID,
                                          QueryStatusDisconnectFromTarget,
                                          ExecuteDisconnectFromTarget);
            }
            if (command == PreviewCommands.PlayAll)
            {
                return new CommandBinding(this, PreviewCommands.PlayAll.ID,
                                          QueryStatusPlayAll, ExecutePlayAll);
            }
            if (command == PreviewCommands.PauseAll)
            {
                return new CommandBinding(this, PreviewCommands.PauseAll.ID,
                                          QueryStatusPauseAll, ExecutePauseAll);
            }
            if (command == PreviewCommands.StopAll)
            {
                return new CommandBinding(this, PreviewCommands.StopAll.ID,
                                          QueryStatusStopAll, ExecuteStopAll);
            }

            // ツール関連コマンドハンドラ
            if (command == ToolCommands.StopAllSounds)
            {
                return new CommandBinding(this, ToolCommands.StopAllSounds.ID,
                                          QueryStatusStopAllSounds, ExecuteStopAllSounds);
            }
            if (command == ToolCommands.ResetMidi)
            {
                return new CommandBinding(this, ToolCommands.ResetMidi.ID,
                                          QueryStatusResetMidi, ExecuteResetMidi);
            }

            return base.CreateCommandBinding(command);
        }

        /// <summary>
        ///
        /// </summary>
        protected override bool ExecuteCommandInternal(Command command)
        {
            if (null != Form.ActiveForm)
            {
                Control activeControl = GetActiveControl(Form.ActiveForm.ActiveControl);

                if (activeControl != null &&
                    IsUsingIME(activeControl.Handle) == true)
                {

                    if (activeControl is TextBox ||
                        activeControl is ComboBox)
                    {
                        return false;
                    }
                }
            }

            return base.ExecuteCommandInternal(command);
        }

        /// <summary>
        /// アプリケーションを実行します。
        /// </summary>
        /// <param name="args">コマンドライン引数。</param>
        protected override void RunApplication(string[] args)
        {
            SplashWindow.SetOwner(this.MainWindow);

            this.SetHelpFilePath();

            ConvertService.Generator = string.Format("{0} {1}",
                this.Traits.ApplicationAssembly.GetTitle(),
                this.Traits.ApplicationAssembly.GetName().Version.ToString(3));
            ConvertService.BeginConvert += OnBeginConvert;
            ConvertService.EndConvert += OnEndConvert;

            AppConfiguration.Options.WaveOutDeviceNameUpdate += OnWaveOutDeviceNameUpdate;
            AppConfiguration.Options.MidiInputDeviceUpdate += OnMidiInputDeviceUpdate;

            this.communicationService.ViewerConnectionChanged += OnViewerConnectionChanged;
            this.communicationService.SndEditConnectionChanged += OnSndEditConnectionChanged;
            this.communicationService.SoundArchiveSending += OnSoundArchiveSending;

            try
            {
                this.CurrentSoundMakerPlugin.RuntimeGlobal_AXInit();

                StartupWaveOutput();

                try
                {
                    this.CurrentSoundMakerPlugin.RuntimeSoundSystem_InitSoundSystem(3, 4);
                    this.IsErrorAudioDevice = false;
                }
                catch
                {
                    // オーディオデバイスの初期化が失敗した場合は警告メッセージを表示します。
                    this.IsErrorAudioDevice = true;
                    this.ShowWarningMessageCanNotInitializeAudioDevice();
                }

                InitializeMidiInputDevices();
                StartupMidiInputDevices();

                base.RunApplication(args);
            }
            finally
            {
                ShutdownMidiInputDevices();

                this.CurrentSoundMakerPlugin.RuntimeSoundSystem_ShutdownSoundSystem();

                ShutdownWaveOutput();

                this.CurrentSoundMakerPlugin.RuntimeGlobal_AXQuit();
            }
        }

        /// <summary>
        /// プロジェクトが開かれると発生します。
        /// </summary>
        /// <param name="e"></param>
        protected override void OnProjectOpened(object sender, EventArgs e)
        {
            base.OnProjectOpened(sender, e);

            LoadPresetConfiguration(PresetConfigurationFilePath);

            ApplyProjectConfiguration();

            if (AppConfigurationCommon.IsHiddenFunctionEnabled() == true)
            {
                ProjectService.Project.ParameterValueChanged += OnParameterValueChanged;

                if (ProjectService.Project.EnabledConnectionTimeOut == true)
                {
                    this.communicationService.Timeout = ProjectService.Project.ConnectionTimeOut;
                }
            }

            this.LoudnessService.Enabled = true;

            if (this.AppConfiguration.Options.Application.Statistics.EnabledAutoMeasureLoudness == true)
            {
                // 平均ラウドネス値の計測を自動で行うように未計測のサウンドを全て登録します。
                var sounds = ProjectService.Sounds.Where(s => s.IntegratedLoudnessStatus == IntegratedLoudnessStatus.Unmeasured);
                this.LoudnessService.MeasureLowPriority(sounds);
            }

            this.ProjectService.Project.ParameterValueChanged += this.OnProjectParameterValueChanged;
        }

        /// <summary>
        /// プロジェクト設定が保存される前に発生します。
        /// </summary>
        /// <param name="e"></param>
        protected override void OnProjectConfigurationSaving(object sender, EventArgs e)
        {
            base.OnProjectConfigurationSaving(sender, e);

            ExtractProjectConfiguration();
        }

        /// <summary>
        /// アプリケーションを終了します。
        /// </summary>
        protected override bool ExitApplication()
        {
            this.communicationService.Disconnect();
            this.RealtimeEditService.Detach();

            return base.ExitApplication();
        }

        /// <summary>
        /// バンクサービスが追加されると発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">バンクサービスイベントデータ。</param>
        protected override void OnBankAdded(object sender, BankServiceEventArgs e)
        {
            base.OnBankAdded(sender, e);

            ChangeInterpolationType(e.BankService);

            e.BankService.Reloaded += ReloadedChangeInterpolationType;
        }

        /// <summary>
        /// バンクサービスが削除されると発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">バンクサービスイベントデータ。</param>
        protected override void OnBankRemoved(object sender, BankServiceEventArgs e)
        {
            base.OnBankRemoved(sender, e);

            e.BankService.Reloaded -= ReloadedChangeInterpolationType;
        }

        protected override bool TakeOverApplicationConfigurationFile()
        {
            if (base.TakeOverApplicationConfigurationFile() == true)
            {
                return true;
            }

            // Cafe のオプションファイルからコピー
            string oldFilePath = Path.Combine(Path.GetDirectoryName(ApplicationConfigurationDirectory), "NW4F_SoundMaker", "NW4F_SoundMaker.xml");
            this.CreateNewConfigFileByCopyOldConfigFile(oldFilePath, AppConfigurationFilePath,
                ((document) =>
                 {
                     if (document != null && document.DocumentElement != null)
                     {
                         XmlNode elem = document.DocumentElement.GetElementsByTagName("nwrso:OptionsCafe")[0];
                         if (elem != null)
                         {
                             XmlElement newElem = document.CreateElement("nwrso", "OptionsCommon", "NintendoWare.SoundMaker.OptionConfiguration");
                             newElem.AppendChild(elem.FirstChild);
                             document.DocumentElement.AppendChild(newElem);
                             document.DocumentElement.RemoveChild(elem);
                         }
                     }
                 }));

            // Cafe のリストカラム項目ファイルからコピー
            if (File.Exists(PresetListColumnsService.ConfigFilePath) == false)
            {
                oldFilePath = Path.Combine(Path.GetDirectoryName(ApplicationConfigurationDirectory), "NW4F_SoundMaker", Path.GetFileName(PresetListColumnsService.ConfigFilePath));
                this.CreateNewConfigFileByCopyOldConfigFile(oldFilePath, PresetListColumnsService.ConfigFilePath);
            }

            return true;
        }

        private void CreateNewConfigFileByCopyOldConfigFile(string oldFilePath, string newFilePath, Action<XmlDocument> action = null)
        {
            try
            {
                if (this.CopyOldVersionFile(null, oldFilePath, newFilePath) == false)
                {
                    return;
                }

                XmlDocument document = new XmlDocument();

                using (XmlReader reader = XmlTextReader.Create(newFilePath))
                {
                    document.Load(reader);
                }

                if (null == document.DocumentElement)
                {
                    return;
                }
                document.DocumentElement.SetAttribute("Platform", Platforms.Any.PlatformName);
                document.DocumentElement.SetAttribute("Version", "0.0.0.0");
                if (action != null)
                {
                    action(document);
                }

                using (XmlWriter writer = XmlTextWriter.Create(newFilePath))
                {
                    document.Save(writer);
                }
            }
            catch
            {
            }
        }

        /// <summary>
        ///
        /// </summary>
        private static IApplicationTraits CreateTraits()
        {
            ApplicationTraits traits = new ApplicationTraits()
            {
                ProductName = Application.ProductName,
                ProductVersion = Application.ProductVersion,
                ApplicationAssembly = Assembly.GetEntryAssembly(),
                ApplicationIcon = ImageResource.AppIcon,
                //PlatformImage = ImageResource.BitmapPlatformLogo,     // TODO : ★プラットフォームロゴが完成するまでコメントアウト
#if USE_PLUGIN_CONVERTER
                IntermediateOutputTraits = SoundMakerPluginManager.Instance.CurrentSoundMakerPlugin.CreateSoundIntermediateOutputTraits(),
                BinaryOutputTraits = SoundMakerPluginManager.Instance.CurrentSoundMakerPlugin.CreateSoundBinaryOutputTraits(),
#else
                IntermediateOutputTraits = new SoundIntermediateOutputTraits()
                {
                    SoundProjectDocumentTypeName = Platforms.Any.SoundProjectDocument,
                    SoundSetDocumentTypeName = Platforms.Any.SoundSetDocument,
                    BankDocumentTypeName = Platforms.Any.BankDocument,
                    SoundProjectFileExtension = SoundFileExtensions.SoundProjectFile,
                    SoundSetFileExtension = SoundFileExtensions.SoundSetFile,
                    BankFileExtension = SoundFileExtensions.BankFile,
                    BankIncludeFileExtension = SoundFileExtensions.BankIncludeFile,
                    TextSequenceSoundFileExtension = SoundFileExtensions.TextSequenceSoundFile,
                    SoundIDCppHeaderFileExtension = SoundFileExtensions.SoundIDCppHeaderFile,
                },
                BinaryOutputTraits = new SoundBinaryOutputTraits()
                {
                    BankBinaryFileExtension = SoundFileExtensions.BankBinaryFile,
                    SequenceSoundBinaryFileExtension = SoundFileExtensions.SequenceSoundBinaryFile,
                    SoundArchiveBinaryFileExtension = SoundFileExtensions.SoundArchiveBinaryFile,
                    StreamSoundBinaryFileExtension = SoundFileExtensions.StreamSoundBinaryFile,
                    StreamSoundPrefetchBinaryFileExtension = SoundFileExtensions.StreamSoundPrefetchBinaryFile,
                    WaveArchiveBinaryFileExtension = SoundFileExtensions.WaveArchiveBinaryFile,
                    WaveBinaryFileExtension = SoundFileExtensions.WaveBinaryFile,
                    WaveSoundBinaryFileExtension = SoundFileExtensions.WaveSoundBinaryFile,
                    WaveSoundBinary2FileExtension = SoundFileExtensions.WaveSoundBinary2File,
                    GroupBinaryFileExtension = SoundFileExtensions.GroupBinaryFile,
                },
                ListTraits = new ListTraitsCommon(),
                ComponentTraits = new SoundComponentTraitsCommon(),
#endif
            };

            SequenceSoundTextConverter sequenceSoundTextConverter =
                new SequenceSoundTextConverter(PathUtilityCommon.SequenceSoundConverterFilePath);

            SmfConverter smfConverter = new SmfConverter(PathUtilityCommon.SmfConverterFilePath);

#if USE_PLUGIN_CONVERTER
            traits.ConversionTraits = SoundMakerPluginManager.Instance.CurrentSoundMakerPlugin.CreateSoundProjectConversionTraits(traits.IntermediateOutputTraits, traits.BinaryOutputTraits, sequenceSoundTextConverter, smfConverter);
#else
            traits.ConversionTraits = new SoundProjectConversionTraits(true)
            {
                IntermediateOutputTraits = traits.IntermediateOutputTraits,
                BinaryOutputTraits = traits.BinaryOutputTraits,
                BinaryFileInfo = new BinaryFileInfo()
                {
                    BankSignature = "FBNK",
                    GroupSignature = "FGRP",
                    SequenceSoundSignature = "FSEQ",
                    SoundArchiveSignature = "FSAR",
                    StreamSoundSignature = "FSTM",
                    StreamSoundPrefetchSignature = "FSTP",
                    WaveArchiveSignature = "FWAR",
                    WaveSignature = "FWAV",
                    WaveSoundSignature = "FWSD",
                    WaveSound2Signature = "BAWSD   ",
                    BankVersion = new BinaryVersion(0, 1, 0, 0),
                    GroupVersion = new BinaryVersion(0, 1, 0, 0),
                    SequenceSoundVersion = new BinaryVersion(0, 1, 0, 0),
                    SoundArchiveVersion = new BinaryVersion(0, 2, 5, 0),
                    StreamSoundVersion = new BinaryVersion(0, 6, 3, 0),
                    StreamSoundPrefetchVersion = new BinaryVersion(0, 5, 2, 0),
                    WaveArchiveVersion = new BinaryVersion(0, 1, 0, 0),
                    WaveVersion = new BinaryVersion(0, 1, 2, 0),
                    WaveSoundVersion = new BinaryVersion(0, 1, 1, 0),
                    WaveSound2Version = new BinaryVersion(1, 0, 0, 0),
                },
                SequenceSoundTextConverterPath = sequenceSoundTextConverter.ConverterExePath,
                BuiltInWavePreprocessExePath = BuiltInWavePreprocessExePath,
                SmfConverterPath = smfConverter.ConverterExePath,
                SequenceSoundTextConverter = sequenceSoundTextConverter,
                SmfConverter = smfConverter,
                SequenceSoundBinaryReader = new SequenceSoundBinaryReader(),
                IsWaveSound2BinaryEnabled = AppConfigurationCommon.EnabledWaveSoundEdit,
            };
#endif
            return traits;
        }

        /// <summary>
        ///
        /// </summary>
        private Control GetActiveControl(Control control)
        {
            if (control == null) return null;

            foreach (Control childControl in control.Controls)
            {
                Control activeControl = GetActiveControl(childControl);
                if (activeControl != null)
                {
                    return activeControl;
                }
            }
            return control.Focused == true ? control : null;
        }

        /// <summary>
        /// IMEが使用中なのか調べます。
        /// </summary>
        private bool IsUsingIME(IntPtr handle)
        {
            const int GCS_COMPSTR = 0x0008;

            bool result = false;
            IntPtr hIMC = ImmGetContext(handle);
            int intLength = ImmGetCompositionString(hIMC, GCS_COMPSTR, null, 0);
            if (intLength > 0)
            {
                StringBuilder temp = new StringBuilder(intLength);
                ImmGetCompositionString(hIMC, GCS_COMPSTR, temp, intLength);
                string text = temp.ToString();

                if (text != String.Empty)
                {
                    byte[] tmp1 = System.Text.Encoding.Default.GetBytes(text.ToString());
                    byte[] tmp2 = new byte[intLength];
                    Array.Copy(tmp1, 0, tmp2, 0, intLength);
                    text = System.Text.Encoding.Default.GetString(tmp2);
                    if (text.Length > 0)
                    {
                        result = true;
                    }
                }
            }

            ImmReleaseContext(handle, hIMC);
            return result;
        }

        /// <summary>
        ///
        /// </summary>
        private void ApplyProjectConfiguration()
        {
            Preview.PreviewManager.Instance.PreviewPlayerManager.Settings.AutoWriteVariables =
                ProjectConfiguration.PreviewCommon.SequenceVariable.SequenceVariableAutoSync;

            Preview.PreviewManager.Instance.PreviewPlayerManager.Settings.VariablesSyncType =
                (Preview.SeqVariableSyncType)ProjectConfiguration.PreviewCommon.
                                             SequenceVariable.SequenceVariableAutoSyncInterval;

            var config = ProjectConfiguration.SoundProject.Connection;
            if (string.IsNullOrEmpty(config.Platform) == true)
            {
                this.MainWindow.SelectedPlatform = "NX";
                this.MainWindow.SelectedTargetName = "PCApplication";
            }
            else
            {
                this.MainWindow.SelectedPlatform = config.Platform;
                this.MainWindow.SelectedTargetName = config.ConnectionTarget;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void ExtractProjectConfiguration()
        {
            ProjectConfiguration.PreviewCommon.SequenceVariable.SequenceVariableAutoSync =
                Preview.PreviewManager.Instance.PreviewPlayerManager.Settings.AutoWriteVariables;

            ProjectConfiguration.PreviewCommon.SequenceVariable.SequenceVariableAutoSyncInterval =
                (int)Preview.PreviewManager.Instance.PreviewPlayerManager.Settings.VariablesSyncType;


            var config = ProjectConfiguration.SoundProject.Connection;
            config.Platform = this.MainWindow.SelectedPlatform;
            config.ConnectionTarget = this.MainWindow.SelectedTargetName;
        }

        /// <summary>
        ///
        /// </summary>
        private void InitializeMidiInputDevices()
        {
            XmlOptionsApplicationMidi midiXml = AppConfiguration.Options.Application.Midi;

            IDictionary<string, XmlMidiDevice> deviceXmls = midiXml.MidiSequencerInputDevices.CreateDictionary();
            IKeyedList<string, Midi.MidiInputDevice> devices = Midi.MidiDeviceManager.InputDevices;

            List<XmlMidiDevice> newDeviceXmls = new List<XmlMidiDevice>();

            if (midiXml.MidiSequencerInputDevices == null)
            {
                foreach (Midi.MidiInputDevice device in devices)
                {
                    newDeviceXmls.Add(
                        new XmlMidiDevice()
                        {
                            DeviceID = device.ID,
                            Name = device.Name,
                        }
                        );
                }
            }
            else
            {
                foreach (Midi.MidiInputDevice device in devices)
                {
                    if (deviceXmls.ContainsKey(device.Name))
                    {
                        newDeviceXmls.Add(
                            new XmlMidiDevice()
                            {
                                DeviceID = device.ID,
                                Name = device.Name,
                            }
                            );
                    }
                }
            }

            midiXml.MidiSequencerInputDevices = newDeviceXmls.ToArray();

            if (midiXml.MidiKeyboardInputDevice != null &&
                devices.ContainsKey(midiXml.MidiKeyboardInputDevice.Name))
            {
                midiXml.MidiKeyboardInputDevice.DeviceID =
                    devices.GetValue(midiXml.MidiKeyboardInputDevice.Name).ID;
            }
        }

        private void LoadPresetConfiguration(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            bool result = _presetConfiguration.Load(filePath);
            if (result) { return; }

            OnPresetConfigurationDefaultLoad(EventArgs.Empty);
        }

        private void SetHelpFilePath()
        {
            try
            {
                this.HelpURL = PathUtilityCommon.HelpURL;
                this.TroubleshootingHelpURL = PathUtilityCommon.TroubleshootingHelpURL;
                this.OptionHelpURL = PathUtilityCommon.OptionHelpURL;
                this.ProjectSettingsHelpURL = PathUtilityCommon.ProjectSettingsHelpURL;
            }
            catch
            {
                this.HelpURL = string.Empty;
                this.TroubleshootingHelpURL = string.Empty;
                this.OptionHelpURL = string.Empty;
                this.ProjectSettingsHelpURL = string.Empty;
            }
        }

        private void RunTargetPlayerInternal(string soundArchivePath)
        {
            try
            {
                if (this.runPcOrTargetPlayerDelegate != null)
                {
                    this.runPcOrTargetPlayerDelegate(new TargetPlayerRunArgument(ProjectService.ProjectDocument, soundArchivePath));
                }
            }
            catch
            {
            }
        }

        private void TransferSoundArchive(string path)
        {
            if (!this.communicationService.IsConnected) { return; }

            if (null == ProjectService.ProjectDocument)
            {
                throw new Exception("unexpected error");
            }

            // 相対パスを指定する。
            string filePath = PathEx.MakeRelative
                (path.GetFullPath(),
                  this.ContentDirectory.GetFullPath());
            this.communicationService.TransferSoundArchive(filePath);
        }

        private string CopySoundArchiveFile()
        {
            string returnPath = string.Empty;

            if (_isCopySoundArchiveFile == false) { return returnPath; }

            string tmpFolder;
#if USE_PCSOUNDPLAYER
            tmpFolder = this.InstallBinaryFilesDirectoryPath;
#else
            if (this.MainWindow.IsPlatformCafe() == true)
            {
                tmpFolder = this.InstallBinaryFilesDirectoryPath;
            }
            else
            {
                tmpFolder = Path.Combine(Path.GetTempPath(), "NintendoSDK_SoundMaker_Temporary\\Archives");
            }
#endif
            try
            {
                string srcPath = ConvertService.OutputDirectoryPath;
                string srcFile = Path.ChangeExtension(ConvertService.BinarySoundArchiveFilePath, "bfsar");
                string dstTmpFile = Path.Combine(tmpFolder, "temp.bfsar");
                string dstFile = Path.Combine(tmpFolder, Path.GetFileName(srcFile));

                if (File.Exists(tmpFolder) == true)
                {
                    File.Delete(tmpFolder);
                }

                Directory.CreateDirectory(tmpFolder);

                this.CopyFiles(srcPath, tmpFolder);
                if (File.Exists(dstFile) == true)
                {
                    File.Move(dstFile, dstTmpFile);
                }
                returnPath = dstTmpFile;
            }
            catch
            {
            }

            return returnPath;
        }

        private bool IsUseFile(string bcsar)
        {
            FileStream file = null;

            try
            {
                if (File.Exists(bcsar) == false)
                {
                    return false;
                }
                else
                {
                    file = File.OpenWrite(bcsar);
                }
            }
            catch
            {
                return true;
            }
            finally
            {
                if (file != null)
                {
                    file.Close();
                }
            }

            return false;
        }

        private void CreateLockFile(string latestCopy)
        {
            string dir = Path.GetDirectoryName(latestCopy);
            if (Directory.Exists(dir) == false)
            {
                Directory.CreateDirectory(dir);
            }

            using (FileStream file = File.OpenWrite(latestCopy))
            {
                DateTime now = DateTime.Now;
                string date = now.ToString("yyyy-MM-dd HH:mm:ss");
                file.Write(Encoding.ASCII.GetBytes(date), 0, Encoding.ASCII.GetByteCount(date));
            }
        }

        private void CopyFiles(string srcPath, string dstPath)
        {
            SynchronizeDirectory(srcPath, dstPath);
        }

        private string[] GetTargetFiles(string directory)
        {
            List<string> list = new List<string>();
            string[] filePaths = Directory.GetFiles(directory);

            foreach (string filePath in filePaths)
            {
                string extension = Path.GetExtension(filePath).ToLower();
                if (extension == BinaryStreamSoundExtension ||
                    extension == BinarySoundArchiveExtension ||
                    extension == BinaryAACExtension ||
                    extension == BinaryOpusExtension)
                {
                    list.Add(filePath);
                }
            }
            return list.ToArray();
        }

        ///
        private void SynchronizeDirectory(string srcDirectory, string dstDirectory)
        {
            string[] filePaths = GetTargetFiles(srcDirectory);
            foreach (string srcFilePath in filePaths)
            {
                string dstFilePath = Path.Combine(dstDirectory, Path.GetFileName(srcFilePath));

                if (File.Exists(dstFilePath) == false ||
                    File.GetLastWriteTime(dstFilePath) != File.GetLastWriteTime(srcFilePath))
                {
                    // ディレクトリが存在しないのか？
                    if (Directory.Exists(dstDirectory) == false)
                    {
                        Directory.CreateDirectory(dstDirectory);
                        File.SetAttributes(dstDirectory, File.GetAttributes(srcDirectory));
                    }

                    File.Copy(srcFilePath, dstFilePath, true);
                }
                else
                {
                }
            }

            //
            if (Directory.Exists(dstDirectory) == true)
            {
                DeleteNotExistFiles(srcDirectory, dstDirectory);
            }

            //
            string[] directories = Directory.GetDirectories(srcDirectory);
            foreach (string directory in directories)
            {
                SynchronizeDirectory(directory,
                                      Path.Combine(dstDirectory, Path.GetFileName(directory)));
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void DeleteNotExistFiles(string targetSrcDirectory, string targetDstDirectory)
        {
            //
            string[] dstFilePaths = GetTargetFiles(targetDstDirectory);
            foreach (string dstFilePath in dstFilePaths)
            {
                string srcFilePath = Path.Combine(targetSrcDirectory,
                                                   Path.GetFileName(dstFilePath));
                if (File.Exists(srcFilePath) == false)
                {
                    File.Delete(dstFilePath);
                }
            }

            //
            string[] dstDirectories = Directory.GetDirectories(targetDstDirectory);
            foreach (string dstDirectory in dstDirectories)
            {
                string srcDirectory = Path.Combine(targetSrcDirectory,
                                                    Path.GetFileName(dstDirectory));
                if (Directory.Exists(srcDirectory) == false)
                {
                    try
                    {
                        Directory.Delete(dstDirectory, true);
                    }
                    catch
                    {
                    }
                }
            }
        }

        private void InitLockFiles(string latestCopyA, string latestCopyB)
        {
            if (File.Exists(latestCopyA) == false && File.Exists(latestCopyB) == false)
            {
                this.CreateLockFile(latestCopyA);
                return;
            }


        }

        /// <summary>
        /// コンバートダイアログをモーダル表示します。
        /// </summary>
        private void ShowConvertDialog()
        {
            _communiatingDialog.Message = FrameworkResources.MessageResource.Message_SendingSoundArchive;
            _communiatingDialog.ShowDialog(this.MainWindow);
            _communiatingDialog = null;

            ApplicationBase.Instance.EnableCommandKeyProcess();
        }

        private void OnBeginConvert(object sender, SoundProjectConvertEventArgs e)
        {
            this.LoudnessService.Enabled = false; // ラウドネス計測を無効にします。
        }

        private new void OnEndConvert(object sender, SoundProjectConvertEventArgs e)
        {
            try
            {
                if (!e.IsPartsConvert && e.Succeeded)
                {
                    //
                    if (this.communicationService.IsConnected == true)
                    {
                        if (CommTool.CloseSarSoundPacket.Send() == false)
                        {
                            return;
                        }
                    }

                    string path = this.CopySoundArchiveFile();

                    if (string.IsNullOrEmpty(path))
                    {
                        return;
                    }

                    RunTargetPlayerInternal(path);
                    TransferSoundArchive(path);
                }
            }
            finally
            {
                this.runPcOrTargetPlayerDelegate = null;
                this.LoudnessService.Enabled = true; // ラウドネス計測を有効にします。
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnViewerConnectionChanged(object sender, NintendoWare.SoundMaker.Framework.Preview.Communications.ConnectionChangedEventArgs e)
        {
            if (e.IsError)
            {
                if (e.State == ConnectionState.Disconnecting)
                {
                    return;
                }

                Preview.MCS.Viewer.GetSoundInformationPacket.Received = true;
                Preview.MCS.Viewer.SetPreviewPlayerParameterPacket.Received = true;
                Preview.MCS.Viewer.SetEffectParametersPacket.Received = true;

                this.connectedToSoundPlayer = false;

                return;
            }

            //
            switch (e.State)
            {
                case ConnectionState.Connected:
                    this.connectedToSoundPlayer = true;
                    break;

                case ConnectionState.Disconnected:
                    this.connectedToSoundPlayer = false;
                    break;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnSndEditConnectionChanged(object sender, NintendoWare.SoundMaker.Framework.Preview.Communications.ConnectionChangedEventArgs e)
        {
            if (e.IsError == true)
            {
                CommSync.SndEditSyncPacket.QueryConnectToSoundPlayer = false;

                if (this.connectedToSoundPlayer == false &&
                    e.State == ConnectionState.Disconnected)
                {
                    UIService.ShowMessageBox(Resources.MessageResource.Message_FailedToConnectApp);
                }

                this.RealtimeEditService.Enabled = false;
                CommSync.SndEditSyncPacket.QueryConnectToSoundPlayer = false;
                return;
            }

            //
            switch (e.State)
            {
                case ConnectionState.Connected:
                    this.RealtimeEditService.Enabled = true;
                    this.RealtimeEditService.ConvertAllMonitoringTargets();
                    break;

                case ConnectionState.Disconnecting:


                case ConnectionState.Disconnected:
                    this.RealtimeEditService.Enabled = false;
                    CommSync.SndEditSyncPacket.QueryConnectToSoundPlayer = false;
                    break;
            }
        }

        /// <summary>
        /// サウンドアーカイブの転送が開始されると発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">空のイベントデータ。</param>
        private void OnSoundArchiveSending(object sender, EventArgs e)
        {
            _communiatingDialog = new CommunicatingDialog();

            ApplicationBase.Instance.DisableCommandKeyProcess();
            this.MainWindow.BeginInvoke(new MethodInvoker(ShowConvertDialog));
        }

        private object OnNoteOn(IntPtr seqPlayer, int bankNo, ref IRuntimeNoteOnInfo noteOnInfo, object userData)
        {
            // MIDI キーボード
            if (this.sequencePlayerForMidiKeyboard == seqPlayer)
            {
                return this.StartChannelFromSelectedBank(ref noteOnInfo, true);
            }

            // MIDI シーケンサ
            if (this.sequencePlayerForMidiSequencer.Contains(seqPlayer))
            {
                return this.StartChannelFromPreviewBank(bankNo, ref noteOnInfo);
            }

            return this.StartChannelFromSelectedBank(ref noteOnInfo, false);
        }

        private object StartChannelFromSelectedBank(ref IRuntimeNoteOnInfo noteOnInfo, bool linkSampleMapEnable)
        {
            if (this.MainWindow.ActivePage == null) { return null; }
            if (!(this.MainWindow.ActivePage.Panel is BankPanel)) { return null; }

            if (null == this._previewBankDocument || null == this._previewInstrument)
            {
                return null;
            }

            IPreviewSequenceChannel channel = null;

            try
            {
                channel = this.CurrentSoundMakerPlugin.CreatePreviewSequenceChannel(
                    this._previewBankDocument.Resource.Key, this._previewInstrument, ref noteOnInfo);

                channel.Start();
            }
            catch
            {
                channel = null;
                return null;
            }

            if (linkSampleMapEnable)
            {
                this.MainWindow.BeginInvoke(
                    new LinkSampleMapHandler(OnLinkSampleMap), new object[] { noteOnInfo });
            }

            return channel.Channel;
        }

        private object StartChannelFromPreviewBank(int bankNo, ref IRuntimeNoteOnInfo noteOnInfo)
        {
            if (this.PreviewBankPanel.ItemCount <= bankNo)
            {
                return null;
            }

            Preview.PreviewBank previewBank = this.PreviewBankPanel.GetItem((uint)bankNo).PreviewBank;

            if (previewBank == null || previewBank.Item == null)
            {
                return null;
            }

            Instrument instrument = previewBank.GetInstrument(noteOnInfo.PrgNo);

            if (instrument == null)
            {
                return null;
            }

            IPreviewSequenceChannel channel = this.CurrentSoundMakerPlugin.CreatePreviewSequenceChannel(
                previewBank.Item.FilePath,
                instrument,
                ref noteOnInfo);
            channel.Start();

            return channel.Channel;
        }

        private void OnLinkSampleMap(ref IRuntimeNoteOnInfo noteOnInfo)
        {
            if (null == this.MainWindow.ActivePage) { return; }
            if (!(this.MainWindow.ActivePage.Panel is BankPanel)) { return; }

            BankPanel bankPanel = this.MainWindow.ActivePage.Panel as BankPanel;
            if (null == bankPanel.BankDocument) { return; }

            bankPanel.SelectedKeyboardKey = noteOnInfo.Key;
            bankPanel.SelectedVelocity = noteOnInfo.Velocity;
        }

        private void OnWaveOutDeviceNameUpdate(object sender, EventArgs e)
        {
            StartupWaveOutput();
        }

        private void OnMidiInputDeviceUpdate(object sender, EventArgs e)
        {
            StartupMidiInputDevices();
        }


        /// <summary>
        ///
        /// </summary>
        private CommandStatus QueryStatusConnectToPCViewer(Command command)
        {
            CommandStatus status = CanConnectToPCViewer();

            if (status.IsSupported() == true &&
                status.IsEnabled() == true &&
                status.IsVisible() == true)
            {
                // 接続メニューのショートカットを有効にします。
                // 同じショートカットを複数のメニューで使えないので、それの回避策です。
                KeyStroke keyStroke = KeysEx.ToKeyStroke(KeysEx.ParseKey(command.ShortcutKeyText));
                KeyBindingManager keyBindingManager = CommandService.KeyBindings as KeyBindingManager;
                keyBindingManager.Add(KeyStroke.NullKeyStroke, keyStroke, command);
            }

            return status;
        }

        /// <summary>
        ///
        /// </summary>
        private bool ExecuteConnectToPCViewer(Command command)
        {
            ConnectToPCViewer();
            return true;
        }

        /// <summary>
        /// PC版Viewer から切断できるかどうか調べます。
        /// </summary>
        /// <param name="command">コマンド。</param>
        /// <returns>コマンドの状態。</returns>
        private CommandStatus QueryStatusDisconnectFromPCViewer(Command command)
        {
            CommandStatus status = CanDisconnectFromPCViewer();

            if (status.IsSupported() == true &&
                status.IsEnabled() == true &&
                status.IsVisible() == true)
            {
                KeyStroke keyStroke = KeysEx.ToKeyStroke(KeysEx.ParseKey(command.ShortcutKeyText));
                KeyBindingManager keyBindingManager = CommandService.KeyBindings as KeyBindingManager;
                keyBindingManager.Add(KeyStroke.NullKeyStroke, keyStroke, command);
            }

            return status;
        }

        /// <summary>
        /// PC版Viewer から切断します。
        /// </summary>
        /// <param name="command">コマンド。</param>
        /// <returns>コマンドを実行した場合は true、実行しなかったまたはキャンセルされた場合は false。</returns>
        private bool ExecuteDisconnectFromPCViewer(Command command)
        {
            return DisconnectFromPCViewer();
        }


        /// <summary>
        /// 接続できるかどうか調べます。
        /// </summary>
        /// <param name="command">コマンド。</param>
        /// <returns>コマンドの状態。</returns>
        private CommandStatus QueryStatusConnectToTarget(Command command)
        {
            CommandStatus status = CanConnectToTarget();

            if (status.IsSupported() == true &&
                status.IsEnabled() == true &&
                status.IsVisible() == true)
            {
                // 接続メニューのショートカットを有効にします。
                // 同じショートカットを複数のメニューで使えないので、それの回避策です。
                KeyStroke keyStroke = KeysEx.ToKeyStroke(KeysEx.ParseKey(command.ShortcutKeyText));
                KeyBindingManager keyBindingManager = CommandService.KeyBindings as KeyBindingManager;
                keyBindingManager.Add(KeyStroke.NullKeyStroke, keyStroke, command);
            }

            return status;
        }

        /// <summary>
        /// 接続します。
        /// </summary>
        /// <param name="command">コマンド。</param>
        /// <returns>コマンドを実行した場合は true、実行しなかったまたはキャンセルされた場合は false。</returns>
        private bool ExecuteConnectToTarget(Command command)
        {
            ConnectToTarget();
            return true;
        }

        /// <summary>
        /// 切断できるかどうか調べます。
        /// </summary>
        /// <param name="command">コマンド。</param>
        /// <returns>コマンドの状態。</returns>
        private CommandStatus QueryStatusDisconnectFromTarget(Command command)
        {
            CommandStatus status = CanDisconnectFromTarget();

            if (status.IsSupported() == true &&
                status.IsEnabled() == true &&
                status.IsVisible() == true)
            {
                KeyStroke keyStroke = KeysEx.ToKeyStroke(KeysEx.ParseKey(command.ShortcutKeyText));
                KeyBindingManager keyBindingManager = CommandService.KeyBindings as KeyBindingManager;
                keyBindingManager.Add(KeyStroke.NullKeyStroke, keyStroke, command);
            }

            return status;
        }

        /// <summary>
        /// 切断します。
        /// </summary>
        /// <param name="command">コマンド。</param>
        /// <returns>コマンドを実行した場合は true、実行しなかったまたはキャンセルされた場合は false。</returns>
        private bool ExecuteDisconnectFromTarget(Command command)
        {
            return DisconnectFromTarget();
        }

        /// <summary>
        ///
        /// </summary>
        private delegate bool ConnectDelegate();
        private bool Connect(ConnectDelegate func)
        {
            bool result = false;

            try
            {
                // 1秒間隔で2回、接続を試みるようにします。
                int retryCount = 2;

                while (retryCount > 0)
                {
                    result = func();
                    if (result == true)
                    {
                        break;
                    }
                    retryCount--;
                    Thread.Sleep(1000);
                }
            }
            catch
            {
            }

            return result;
        }

        /// <summary>
        /// SoundPlayer を起動できるかどうか調べます。
        /// </summary>
        /// <param name="command">コマンド。</param>
        /// <returns>コマンドの状態。</returns>
        private CommandStatus QueryStatusRunPCSoundPlayer(Command command)
        {
            return CanRunPCSoundPlayer();
        }

        /// <summary>
        /// SoundPlayer を起動します。
        /// </summary>
        /// <param name="command">コマンド。</param>
        /// <returns>コマンドを実行した場合は true、実行しなかったまたはキャンセルされた場合は false。</returns>
        private bool ExecuteRunPCSoundPlayer(Command command)
        {
            return RunPCSoundPlayer();
        }

        /// <summary>
        /// PC AtkPlayer を起動できるかどうか調べます。
        /// </summary>
        /// <param name="command">コマンド。</param>
        /// <returns>コマンドの状態。</returns>
        private CommandStatus QueryStatusRunPCAtkPlayer(Command command)
        {
            return CanRunPCAtkPlayer();
        }

        /// <summary>
        /// PC AtkPlayer を起動します。
        /// </summary>
        /// <param name="command">コマンド。</param>
        /// <returns>コマンドを実行した場合は true、実行しなかったまたはキャンセルされた場合は false。</returns>
        private bool ExecuteRunPCAtkPlayer(Command command)
        {
            return RunPCAtkPlayer();
        }

        /// <summary>
        /// SoundPlayer を停止できるかどうか調べます。
        /// </summary>
        /// <param name="index">コントロールに対応するプラグインの index</param>
        /// <returns>コマンドの状態。</returns>
        public CommandStatus CanShutdownTargetPlayer()
        {
            if (this.CurrentSoundMakerPlugin.TargetPlayer == null)
            {
                return CommandStatus.Supported;
            }

            if (null == ProjectService.ProjectDocument) { return CommandStatus.SupportedAndVisible; }

            if (this.communicationService.IsConnected == true ||
                this.communicationService.IsConnecting == true)
            {
                if (this.IsTargetPC == false && this.IsTargetSoundPlayer == false)
                {
                    return CommandStatus.SupportedAndVisible;
                }
            }

            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        /// <summary>
        /// SoundPlayer を停止します。
        /// </summary>
        /// <param name="index">コントロールに対応するプラグインの index</param>
        /// <returns>コマンドを実行した場合は true、実行しなかったまたはキャンセルされた場合は false。</returns>
        public bool ShutdownTargetPlayer()
        {
            this.CurrentSoundMakerPlugin.TargetPlayer.Shutdown();

            return true;
        }

        // <summary>
        // ラウドネスの自動計測の有効無効の値が変更されると呼び出されます。
        // </summary>
        public void ChangedEnabledAutoMeasureLoudness(bool enabledAutoMeasureLoudness)
        {
            if (ProjectService.Project == null)
            {
                return;
            }

            if (enabledAutoMeasureLoudness == true)
            {
                // 平均ラウドネス値の計測を自動で行うように未計測のサウンドを全て登録します。
                var sounds = ProjectService.Sounds.Where(s => s.IntegratedLoudnessStatus == IntegratedLoudnessStatus.Unmeasured);
                this.LoudnessService.MeasureLowPriority(sounds);
            }
            else
            {
                // ラウドネス計測を中止し計測待ちをクリアします。
                this.LoudnessService.Stop();
                this.LoudnessService.Clear();

                // 計測待ち状態をすべて未計測の状態にします。
                ProjectService.Sounds
                    .Where(s => s.IntegratedLoudnessStatus == IntegratedLoudnessStatus.WaitingMeasure)
                    .ForEach(s => s.IntegratedLoudnessStatus = IntegratedLoudnessStatus.Unmeasured);
            }
        }

        private CommandStatus QueryStatusPlayAll(Command command)
        {
            return PreviewPlayerPanel.CanPlayAll ?
                CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        private bool ExecutePlayAll(Command command)
        {
            PreviewPlayerPanel.PlayAll();
            _notePlayer.Stop();
            return true;
        }

        private CommandStatus QueryStatusPauseAll(Command command)
        {
            return PreviewPlayerPanel.CanPauseAll ?
                CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        private bool ExecutePauseAll(Command command)
        {
            PreviewPlayerPanel.PauseAll();
            _notePlayer.Stop();
            return true;
        }

        private CommandStatus QueryStatusStopAll(Command command)
        {
            return (PreviewPlayerPanel.CanStopAll || _notePlayer.IsPlaying) ?
                CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        private bool ExecuteStopAll(Command command)
        {
            StopAll();
            return true;
        }

        private CommandStatus QueryStatusStopAllSounds(Command command)
        {
            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        private bool ExecuteStopAllSounds(Command command)
        {
            this.LoudnessService.Stop();
            Preview.PreviewManager.Instance.PreviewPlayerManager.StopAll();
            _notePlayer.Stop();
            this.CurrentSoundMakerPlugin.RuntimeSoundSystem_StopAllVoices();
            return true;
        }

        private CommandStatus QueryStatusResetMidi(Command command)
        {
            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        private bool ExecuteResetMidi(Command command)
        {
            this._midiManager.Reset();
            return true;
        }

        private void ReloadedChangeInterpolationType(object sender, EventArgs e)
        {
            if (sender is BankService)
            {
                BankService bankService = sender as BankService;
                bankService.BankDocument.Dirty = false;
                ChangeInterpolationType(bankService);
            }
        }

        private void ChangeInterpolationType(BankService bankService)
        {
            if (bankService != null)
            {
                if (ChangeInterpolationTypeNoneToLinear(bankService.Bank) == true)
                {
                    bankService.BankDocument.Dirty = true;
                }
            }
        }

        private bool ChangeInterpolationTypeNoneToLinear(Component component)
        {
            bool result = false;

            if (component == null)
            {
                return result;
            }

            if (component is VelocityRegion)
            {
                VelocityRegion velocityRegion = component as VelocityRegion;
                if (velocityRegion.InterpolationType == InterpolationType.None)
                {
                    velocityRegion.InterpolationType = InterpolationType.Linear;
                    result = true;
                }
            }
            else
            {
                foreach (Component child in component.Children)
                {
                    result |= ChangeInterpolationTypeNoneToLinear(child);
                }
            }

            return result;
        }

        // <summary>
        // コンポーネントが追加されたら呼び出されます。
        // </summary>
        private void OnComponentsAdded(object sender, ComponentEventArgs e)
        {
            // ラウドネスの自動計測の対象に追加します。
            this.LoudnessService.AddAutoMeasure(e.Components);
        }

        // <summary>
        // コンポーネントが削除されたら呼び出されます。
        // </summary>
        private void OnComponentsRemoved(object sender, ComponentEventArgs e)
        {
            // ラウドネスの自動計測の対象から削除します。
            this.LoudnessService.RemoveAutoMeasure(e.Components);
        }

        // <summary>
        // プロジェクト、サウンドセットドキュメントの再読み込み前に発生します。
        // </summary>
        private void OnProjectReloading(object sender, ProjectReloadEventArgs e)
        {
            // ラウドネス計測を無効にします。
            this.LoudnessService.Enabled = false;
        }

        // <summary>
        // プロジェクトが閉じる前に発生します。
        // </summary>
        private void OnProjectClosing(object sender, EventArgs e)
        {
            this.LoudnessService.Enabled = false; // ラウドネス計測を無効にします。
            this.LoudnessService.Clear();         // ラウドネス計測待ちをクリアします。

            foreach (var document in this.ProjectService.SoundSetDocuments)
            {
                this.SaveStatistics(document); // 統計情報を保存します。（平均ラウドネス値）
            }

            this.ProjectService.Project.ParameterValueChanged -= this.OnProjectParameterValueChanged;
        }

        // <summary>
        // サウンドセットドキュメントが開くと呼び出されます。
        // </summary>
        private void OnSoundSetDocumentOpened(object sender, SoundDocumentResourceEventArgs e)
        {
            string filePath = this.StatisticsService.GetStatisticsFilePath(e.Resource.Key);
            var soundSet = (this.ProjectService.Documents[e.Resource.Key] as SoundSetDocument).SoundSet;

            // 統計情報を読み込みます。（平均ラウドネス値）
            this.StatisticsService.Load(filePath, soundSet, this.ProjectService.Project.SequenceMaxMeasureDuration);
        }

        // <summary>
        // ドキュメントが保存されると呼び出されます。
        // </summary>
        private void OnDocumentSaved(object sender, DocumentEventArgs e)
        {
            if (e.Document is SoundSetDocument == true)
            {
                // 統計情報を保存します。（平均ラウドネス値）
                this.SaveStatistics(e.Document as SoundSetDocument);
            }
        }

        private void SaveStatistics(SoundSetDocument document)
        {
            if (this.ProjectService.Project == null)
            {
                return;
            }

            Debug.Assert(document != null);

            string filePath = this.StatisticsService.GetStatisticsFilePath(document.Resource.Key);
            SoundSet soundSet = document.SoundSet;

            // 統計情報を保存します。（平均ラウドネス値）
            this.StatisticsService.Save(filePath, soundSet, this.ProjectService.Project.SequenceMaxMeasureDuration);
        }

        // <summary>
        // プロジェクト、サウンドセットドキュメントの再読み込みが完了すると発生します。
        // </summary>
        private void OnProjectReloaded(object sender, ProjectReloadEventArgs e)
        {
            // ラウドネス計測を有効にします。
            this.LoudnessService.Enabled = true;
        }

        // <summary>
        // ラウドネス計測を開始する直前に呼び出されます。
        // </summary>
        private void OnMeasuringLoudness(object sender, MeasureLoudnessEventArgs e)
        {
            this.ShutdownWaveOutput();
        }

        // <summary>
        // ラウドネス計測が終了したら呼び出されます。
        // </summary>
        private void OnMeasuredLoudness(object sender, MeasureLoudnessEventArgs e)
        {
            this.StartupWaveOutput();
        }

        /// <summary>
        ///
        /// </summary>
        public class PreviewNotePlayer
        {
            private int _playingKey = -1;

            public bool IsPlaying
            {
                get { return (0 <= _playingKey); }
            }

            private FormsApplicationCommon Application
            {
                get { return FormsApplicationCommon.Instance; }
            }

            public void Play(int key, int velocity)
            {
                Stop();

                _playingKey = key;

                Application.NoteOn(key, velocity);
            }

            public void Stop()
            {
                if (0 > _playingKey) { return; }

                Application.NoteOff(_playingKey);
                _playingKey = -1;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public event EventHandler ProgramChanged;

        private delegate void LinkSampleMapHandler(ref IRuntimeNoteOnInfo noteOnInfo);

        private void RunPCSoundPlayerCallback(TargetPlayerRunArgument args)
        {
            Preview.SoundPlayerWin.Start(ContentDirectory,
                                         AppConfiguration.OptionsCommon.Application.SoundPlayer.Port0,
                                         AppConfiguration.OptionsCommon.Application.SoundPlayer.Port1,
                                         AppConfiguration.OptionsCommon.Application.SoundPlayer.Port2,
                                         AppConfiguration.OptionsCommon.Application.SoundPlayer.Port3,
                                         AppConfiguration.OptionsCommon.Application.SoundPlayer.Port4,
                                         AppConfiguration.OptionsCommon.Application.SoundPlayer.Port5,
                                         args.ProjectDocument,
                                         PathUtilityCommon.SoundPlayerDirectory);
        }

        private void RunPCAtkPlayerCallback(TargetPlayerRunArgument args)
        {
            Preview.AtkPlayerWin.Start(PathUtilityCommon.PCAtkPlayerFilePath, args.SoundArchivePath);
        }

        /// <summary>
        /// Plugin 切り替え
        /// </summary>
        private void OnPlatformChanged(object sender, EventArgs e)
        {
            this.PlatformChange();
        }

        private void PlatformChange()
        {
            SoundProjectConvertService spcService = this.ConvertService as SoundProjectConvertService;
            spcService.SetCreateSoundProjectConverter(this.CurrentSoundMakerPlugin.GetCreateSoundProjectConverterDelegate());
            spcService.SetSoundIDCppHeaderExporter(this.CurrentSoundMakerPlugin.GetCreateSoundIDCppHeaderExporterDelegate());
            spcService.SetSoundArchiveBinaryMapExporter(this.CurrentSoundMakerPlugin.GetCreateSoundArchiveBinaryMapExporterDelegate());

            SoundProjectConvertDependencyService spcdService = this.ConvertDependencyService as SoundProjectConvertDependencyService;
            spcdService.SetCreateFileIDAggregateFactory(this.CurrentSoundMakerPlugin.GetCreateFileIDAggregateFactoryDelegate());
            spcdService.SetCreateFileID(this.CurrentSoundMakerPlugin.GetCreateFileIDDelegate());
            spcdService.SetCreateSoundArchiveBinaryXml(this.CurrentSoundMakerPlugin.GetCreateSoundArchiveBinaryXmlDelegate());

            WaveFileManager.Instance.SetCreateWaveFileInfo(this.CurrentSoundMakerPlugin.GetCreateWaveFileInfoDelegate());
        }

        private void AddDocumentServiceTraits(DocumentServiceTraits traits)
        {
            foreach (ISoundMakerPlugin plugin in SoundMakerPluginManager.Instance.Instances)
            {
                foreach (IDocumentFactory documentFactory in plugin.DocumentFactories)
                {
                    traits.DocumentFactorys.Add(documentFactory);
                }


                foreach (IDocumentReader documentReader in plugin.DocumentReaders)
                {
                    traits.DocumentReaders.Add(documentReader);
                }
            }
        }

        private void OnParameterValueChanged(object sender, ParameterEventArgs e)
        {
            switch (e.Key)
            {
                case ProjectParameterNames.Connection.EnabledConnectionTimeOut:
                    if ((bool)e.ParameterValue.Value == true)
                    {
                        this.communicationService.Timeout = ProjectService.Project.ConnectionTimeOut;
                    }
                    else
                    {
                        this.communicationService.Timeout = 5; // デフォルト値;
                    }

                    break;

                case ProjectParameterNames.Connection.ConnectionTimeOut:
                    if (ProjectService.Project.EnabledConnectionTimeOut == true)
                    {
                        this.communicationService.Timeout = (int)e.ParameterValue.Value;
                    }
                    break;
            }
        }

        private void OnProjectParameterValueChanged(object sender, ParameterEventArgs e)
        {
            // 平均ラウドネス値の計測の最大計測時間が変更された場合
            if (e.Key == ProjectParameterNames.Statistics.SequenceMaxMeasureDuration)
            {
                // 自動計測が有効の場合は、全てのシーケンスサウンドを再計測します。
                if (this.AppConfiguration.Options.Application.Statistics.EnabledAutoMeasureLoudness == true)
                {
                    // シーケンスサウンドが計測中かもしれないので計測を中止します。
                    this.LoudnessService.Stop();
                    this.LoudnessService.MeasureLowPriority(this.ProjectService.SequenceSounds);
                }
                else
                {
                    // 自動計測が無効な場合は、現時点では何もしません。
                }
            }
        }
    }
}
