﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Windows.Forms.CommandHandlers
{
    using System.Collections.Generic;
    using NintendoWare.SoundMaker.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.Windows.Forms.CommandHandlers;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// Common向けオプションを表示します。
    /// </summary>
    public class ShowOptionHandlerCommon : ShowOptionHandler
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="commandID">処理対象コマンドID。</param>
        public ShowOptionHandlerCommon(string commandID)
            : base(commandID)
        {
        }

        /// <summary>
        /// オプション設定を取得します。
        /// </summary>
        protected XmlOptionsCommon OptionsCommon
        {
            get { return FormsApplicationCommon.Instance.AppConfiguration.OptionsCommon; }
        }

        protected override OptionDialog CreateOptionDialog()
        {
            return new OptionDialogCommon();
        }

        protected override void LoadOptions(OptionDialog dialog)
        {
            Assertion.Argument.True(dialog is OptionDialogCommon);

            base.LoadOptions(dialog);

            OptionDialogCommon dialogCommon = dialog as OptionDialogCommon;

            dialogCommon.AudioOutputDeviceName = this.Options.Application.Sound.WaveOutDeviceName;

            if (this.Options.Application.Midi.MidiKeyboardInputDevice != null)
            {
                dialogCommon.MidiKeyboardInputDevices = new OptionDialogCommon.MidiDevice()
                {
                    DeviceID = this.Options.Application.Midi.MidiKeyboardInputDevice.DeviceID,
                    Name = this.Options.Application.Midi.MidiKeyboardInputDevice.Name,
                };
            }
            dialogCommon.midiSequencerInputDevices = this.GetSequencerMidiInputDevices();

            dialogCommon.SoundPlayerDvdRoot = this.OptionsCommon.Application.SoundPlayer.DvdRootPath;

            dialogCommon.SoundPlayerPort0 = this.OptionsCommon.Application.SoundPlayer.Port0;
            dialogCommon.SoundPlayerPort1 = this.OptionsCommon.Application.SoundPlayer.Port1;
            dialogCommon.SoundPlayerPort2 = this.OptionsCommon.Application.SoundPlayer.Port2;
            dialogCommon.SoundPlayerPort3 = this.OptionsCommon.Application.SoundPlayer.Port3;
            dialogCommon.SoundPlayerPort4 = this.OptionsCommon.Application.SoundPlayer.Port4;
            dialogCommon.SoundPlayerPort5 = this.OptionsCommon.Application.SoundPlayer.Port5;

            dialogCommon.EnabledAutoMeasureLoudness = this.Options.Application.Statistics.EnabledAutoMeasureLoudness;
        }

        protected override void ApplyOptions(OptionDialog dialog)
        {
            Assertion.Argument.True(dialog is OptionDialogCommon);

            if (dialog.EnabledAutoMeasureLoudness != this.Options.Application.Statistics.EnabledAutoMeasureLoudness)
            {
                FormsApplicationCommon.Instance.ChangedEnabledAutoMeasureLoudness(dialog.EnabledAutoMeasureLoudness);
            }

            base.ApplyOptions(dialog);

            OptionDialogCommon dialogCommon = dialog as OptionDialogCommon;

            this.Options.Application.Sound.WaveOutDeviceName = dialogCommon.AudioOutputDeviceName;

            if (dialogCommon.MidiKeyboardInputDevices == null)
            {
                this.Options.Application.Midi.MidiKeyboardInputDevice = null;
                if (dialogCommon.midiSequencerInputDevices != null)
                {
                    this.SetSequencerMidiInputDevices(dialogCommon.midiSequencerInputDevices);
                }
            }
            else
            {
                this.Options.Application.Midi.MidiKeyboardInputDevice = new XmlMidiDevice()
                {
                    DeviceID = dialogCommon.MidiKeyboardInputDevices.DeviceID,
                    Name = dialogCommon.MidiKeyboardInputDevices.Name,
                };
                this.SetSequencerMidiInputDevices(dialogCommon.midiSequencerInputDevices);
            }

            this.OptionsCommon.Application.SoundPlayer.DvdRootPath = dialogCommon.SoundPlayerDvdRoot;

            this.OptionsCommon.Application.SoundPlayer.Port0 = dialogCommon.SoundPlayerPort0;
            this.OptionsCommon.Application.SoundPlayer.Port1 = dialogCommon.SoundPlayerPort1;
            this.OptionsCommon.Application.SoundPlayer.Port2 = dialogCommon.SoundPlayerPort2;
            this.OptionsCommon.Application.SoundPlayer.Port3 = dialogCommon.SoundPlayerPort3;
            this.OptionsCommon.Application.SoundPlayer.Port4 = dialogCommon.SoundPlayerPort4;
            this.OptionsCommon.Application.SoundPlayer.Port5 = dialogCommon.SoundPlayerPort5;

            FormsApplicationCommon.Instance.StartupMidiInputDevices();
            FormsApplicationCommon.Instance.StartupWaveOutput();
        }

        private IEnumerable<OptionDialogCommon.MidiDevice> GetSequencerMidiInputDevices()
        {
            if (this.Options.Application.Midi.MidiSequencerInputDevices == null)
            {
                yield break;
            }

            int deviceID = 0;

            foreach (XmlMidiDevice device in this.Options.Application.Midi.MidiSequencerInputDevices)
            {
                yield return new OptionDialogCommon.MidiDevice()
                {
                    DeviceID = deviceID,
                    Name = device.Name,
                };
                deviceID++;
            }
        }

        private void SetSequencerMidiInputDevices(IEnumerable<OptionDialogCommon.MidiDevice> devices)
        {
            Ensure.Argument.NotNull(devices);

            List<XmlMidiDevice> xmlDevices = new List<XmlMidiDevice>();

            foreach (OptionDialogCommon.MidiDevice device in devices)
            {
                xmlDevices.Add(
                    new XmlMidiDevice()
                    {
                        DeviceID = device.DeviceID,
                        Name = device.Name,
                    });
            }

            this.Options.Application.Midi.MidiSequencerInputDevices = xmlDevices.ToArray();
        }
    }
}
