﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using NintendoWare.SoundFoundation.CommandHandlers;
using NintendoWare.SoundFoundation.Commands;
using NintendoWare.SoundFoundation.Projects;
using NintendoWare.SoundMaker.Framework.CommandHandlers;
using NintendoWare.SoundMaker.Framework.Commands;
using NintendoWare.SoundMaker.Framework.Windows.Forms;
using NintendoWare.SoundMaker.Preview;
using NintendoWare.SoundMaker.Resources;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Windows.Forms;

namespace NintendoWare.SoundMaker.Windows.Forms.CommandHandlers
{
    /// <summary>
    /// 平均ラウドネス値の計測コマンドを処理します。
    /// </summary>
    public class QueryAndMeasureIntegratedLoudnessHandler : CommandHandler
    {
        private Sound[] _targets = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="queryParameter">パラメータの問い合わせ先。</param>
        public QueryAndMeasureIntegratedLoudnessHandler(IQueryCommandParameter queryParameter)
            : base(queryParameter)
        {
        }

        /// <summary>
        /// コマンドハンドラが実行可能なコマンドIDを取得します。
        /// </summary>
        public override string TargetCommandID
        {
            get
            {
                return PreviewCommands.MeasureIntegratedLoudness.ID;
            }
        }

        /// <summary>
        /// コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <returns>コマンドの状態。</returns>
        protected override CommandStatus QueryStatusInternal(Command command, IQueryCommandParameter parameters)
        {
            if (GetTargetProjectService(parameters) == null)
            {
                return CommandStatus.SupportedAndVisible;
            }

            if (GetTargetSounds(parameters).Length == 0)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        protected override bool ExecuteInternal(Command command, IQueryCommandParameter parameters)
        {
            if (FormsApplicationCommon.Instance.IsErrorAudioDevice == true)
            {
                FormsApplicationCommon.Instance.ShowWarningMessageCanNotInitializeAudioDevice();
                return false;
            }

            _targets = GetTargetSounds(parameters);
            if (_targets.Length == 0)
            {
                return false;
            }

            FormsApplicationCommon.Instance.LoudnessService.MeasureHighPriority(_targets);

            return true;
        }

        /// <summary>
        /// 対象プロジェクトサービスを取得します。
        /// </summary>
        private SoundProjectService GetTargetProjectService(IQueryCommandParameter parameters)
        {
            if (parameters == null)
            {
                throw new ArgumentNullException("parameters");
            }

            return parameters.GetParameter(CommandParameterNames.TargetComponentService) as SoundProjectService;
        }

        /// <summary>
        /// 対象サウンドの一覧を取得します。
        /// </summary>
        private Sound[] GetTargetSounds(IQueryCommandParameter parameters)
        {
            if (parameters == null)
            {
                throw new ArgumentNullException("parameters");
            }

            return (from NintendoWare.SoundFoundation.Projects.Component component in
                    (parameters.GetParameter(CommandParameterNames.TargetComponents) as IEnumerable<NintendoWare.SoundFoundation.Projects.Component>)
                    where component is Sound
                    select component as Sound).ToArray();
        }
    }
}
