﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundMaker.Windows.Forms.CommandHandlers
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Text.RegularExpressions;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;

    public class OpenTextSequenceForWaveHandler : CommandHandler
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public OpenTextSequenceForWaveHandler(IQueryCommandParameter queryParameter)
            : base(queryParameter)
        {
        }

        /// <summary>
        /// コマンドハンドラが実行可能なコマンドIDを取得します。
        /// </summary>
        public override string TargetCommandID
        {
            get
            {
                return PreviewCommands.OpenTextSequenceForWave.ID;
            }
        }

        /// <summary>
        /// コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <returns>コマンドの状態。</returns>
        protected override CommandStatus QueryStatusInternal(Command command, IQueryCommandParameter parameters)
        {
            if (null == GetTargetProjectService(parameters)) { return CommandStatus.SupportedAndVisible; }

            var targetSounds = GetTargetSounds(parameters);

            if (1 != targetSounds.Length)
            {
                return CommandStatus.SupportedAndVisible;
            }

            // WaveSoundBase 以外が混ざっていたら実行しません。
            if (targetSounds.Any(target => !(target is WaveSoundBase)))
            {
                return CommandStatus.SupportedAndVisible;
            }

            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        protected override bool ExecuteInternal(Command command, IQueryCommandParameter parameters)
        {
            Sound[] targets = this.GetTargetSounds(parameters);
            if (1 != targets.Length) { return false; }

            var waveSound = targets[0] as WaveSoundBase;

            if (waveSound == null)
            {
                return false;
            }

            this.OpenTextSequenceFile(
                this.GetTextSequenceFilePathForWave(waveSound),
                waveSound.Name
                );

            return true;
        }

        private void OpenTextSequenceFile(string filePath, string startPosition)
        {
            if (string.IsNullOrEmpty(filePath))
            {
                return;
            }

            if (FormsApplication.Instance.AppConfiguration.Options.Application.General.EditorUseEditor == false)
            {
                Process.Start("notepad.exe", filePath);
            }
            else
            {
                string command = FormsApplication.Instance.AppConfiguration.Options.Application.General.EditorCommand;
                if (String.IsNullOrEmpty(command) == true)
                {
                    Process.Start(filePath);
                }
                else
                {
                    if (File.Exists(command) == false)
                    {
                        ApplicationBase.Instance.UIService.ShowMessageBox(
                            SoundMaker.Framework.Resources.MessageResource.Message_EditorNotFound,
                            SoundMaker.Framework.Resources.MessageResource.Message_UnknownError
                            );
                        return;
                    }
                    uint lineNum = GetLineNumberFromFile(filePath, startPosition);
                    string args = this.GetArgs(filePath, lineNum);
                    Process.Start(command, args);
                }
            }
        }

        private uint GetLineNumberFromFile(string filePath, string label)
        {
            uint lineNum = 1;

            try
            {
                if (File.Exists(filePath) == true)
                {
                    using (StreamReader reader = new StreamReader(filePath))
                    {
                        Regex regex =
                            new Regex("^(\\s*[a-zA-Z_][a-zA-Z0-9_]*\\s*:)*\\s*" + label + "\\s*:");
                        while (reader.EndOfStream == false)
                        {
                            string line = reader.ReadLine();
                            if (regex.IsMatch(line) == true)
                            {
                                return lineNum;
                            }
                            ++lineNum;
                        }

                        return 1;
                    }
                }
            }
            catch
            {
            }

            return 1;
        }

        private string GetArgs(string filePath, uint lineNum)
        {
            string args =
                FormsApplication.Instance.AppConfiguration.Options.Application.General.EditorArguments;
            if (args == null || args == string.Empty)
            {
                return string.Empty;
            }
            args = args.Replace("$(file)", filePath);
            args = args.Replace("$(line)", lineNum.ToString());

            return args;
        }

        /// <summary>
        /// 対象プロジェクトサービスを取得します。
        /// </summary>
        protected SoundProjectService GetTargetProjectService(IQueryCommandParameter parameters)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }
            return parameters.GetParameter(CommandParameterNames.TargetComponentService)
                   as SoundProjectService;
        }

        /// <summary>
        /// 対象サウンドの一覧を取得します。
        /// </summary>
        protected Sound[] GetTargetSounds(IQueryCommandParameter parameters)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }

            return (from Component component in
                        (parameters.GetParameter(CommandParameterNames.TargetComponents)
                          as IEnumerable<Component>)
                    where component is Sound
                    select component as Sound).ToArray();
        }

        private string GetTextSequenceFilePathForWave(WaveSoundBase sound)
        {
            string targetText = sound.Comment7.Trim(' ');

            if (string.IsNullOrEmpty(targetText))
            {
                return null;
            }

            if (targetText.StartsWith("//"))
            {
                return null;
            }

            var match = Regex.Match(targetText, "([^\\|]+) *\\| *(\\d) *\\| *(\\d+)( *\\| *([^\\|]+))?( *\\| *([^\\|]+))?");

            if (match.Groups.Count < 4)
            {
                return null;
            }

            try
            {
                return Path.Combine(
                    Path.GetDirectoryName(sound.FilePath),
                    match.Groups[1].Value.Trim(' ')
                    ).Replace('/', '\\');
            }
            catch
            {
                return null;
            }
        }
    }
}
