﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundMaker.Preview
{
    using System;
    using System.Collections.ObjectModel;
    using System.Diagnostics;
    using NintendoWare.SoundMaker.Framework.Preview.Communications;
    using NW4R.ProtocolSound;

    /// <summary>
    /// シーケンス変数の種類
    /// </summary>
    public enum SeqVariableType
    {
        Unknown = 0,                                                // 無効値
        Global = ToolSoundSeqVariableType.SeqVariableType_Global,   // グローバル変数
        Local = ToolSoundSeqVariableType.SeqVariableType_Local,     // ローカル変数
        Track = ToolSoundSeqVariableType.SeqVariableType_Track      // トラック変数
    }

    /// <summary>
    /// シーケンス変数コンテナ
    /// </summary>
    public class SeqVariableContainer
    {
        #region ** 定数

        public const uint InvalidTrackNo = uint.MaxValue;                             // トラック番号の無効値
        public const uint TrackCount = PreviewPlayerManager.TrackCount;           // トラック番号の最大値
        public const uint DefaultVariableCount = PreviewPlayerManager.SeqVariableCount;     // シーケンス変数のデフォルト数
        public const uint MaxVariableCount = 128;                                       // シーケンス変数の最大数（とりあえずの上限）

        #endregion

        // パラメータ
        private SeqVariableContainerID _id = SeqVariableContainerID.InvalidID;   // コンテナID
        private SeqVariableCollection _variables = new SeqVariableCollection();        // コレクション

        // 状態
        private bool _variableChangedEventLocked = false;   // イベント送信ロック
        private bool _variableChangedDirty = false;   // イベント送信ロック中の dirty フラグ


        #region ** コンストラクタ

        public SeqVariableContainer(SeqVariableContainerID id) : this(id, SeqVariable.InvalidValue, DefaultVariableCount) { }

        public SeqVariableContainer(SeqVariableContainerID id, int initValue) : this(id, initValue, DefaultVariableCount) { }

        public SeqVariableContainer(SeqVariableContainerID id, int initValue, uint variableCount)
        {
            if (null == id) { throw new ArgumentNullException(); }

            Initialize(id, variableCount, initValue);
        }

        #endregion

        #region ** プロパティ

        /// <summary>
        /// シーケンス変数の種類を取得します。
        /// </summary>
        public SeqVariableContainerID ID
        {
            get { return _id; }
        }

        public SeqVariableCollection Variables
        {
            get { return _variables; }
        }

        #endregion

        #region ** イベント

        public event EventHandler ValueChanged;

        #endregion

        #region ** メソッド

        public int GetWritableSize(bool withoutSeqVariables)
        {
            if (withoutSeqVariables) { return ToolSoundSeqVariableContainerHeader.StructSize; }
            return ToolSoundSeqVariableContainer.StructSize;
        }

        public void Read(ProtocolSoundReader reader)
        {
            ToolSoundSeqVariableContainer src = new ToolSoundSeqVariableContainer();
            src.Read(reader);

            // 取得したデータをチェックする
            Validate(src);

            // シーケンス変数を読み込む
            LockVariableChangedEvent();

            for (int i = 0; i < TrackCount; i++)
            {
                _variables[i].Value = src.variables[i];
            }

            // 変数の値に変更があった場合、ここでイベントが発行される
            UnlockVariableChangedEvent();
        }

        public void Write(ProtocolSoundWriter writer, bool withoutSeqVariables)
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }

            if (withoutSeqVariables)
            {
                ToStructWithoutVariables().Write(writer);
            }
            else
            {
                ToStruct().Write(writer);
            }
        }

        public void LockVariableChangedEvent()
        {
            _variableChangedEventLocked = true;
        }

        public void UnlockVariableChangedEvent()
        {
            if (!_variableChangedEventLocked) { return; }

            _variableChangedEventLocked = false;

            if (_variableChangedDirty)
            {
                DispatchValueChangedEvent();
            }
        }

        /// <summary>
        /// シーケンス変数セットを初期化します。
        /// </summary>
        /// <param name="type">変数の種類</param>
        /// <param name="trackNo">トラック番号</param>
        /// <param name="variableCount">保持する変数の数</param>
        private void Initialize(SeqVariableContainerID id, uint variableCount, int initValue)
        {
            Debug.Assert(MaxVariableCount >= variableCount, "unexpected error");

            // 変数コンテナを初期化する
            _variables.Clear();

            for (uint i = 0; i < variableCount; i++)
            {

                SeqVariable newVariable = new SeqVariable(i, initValue);
                newVariable.ValueChanged += OnVariableValueChanged;

                _variables.Add(newVariable);

            }

            // ID を初期化する
            _id = id;
        }

        private void Validate(ToolSoundSeqVariableContainer target)
        {
            // 取得するべきデータはないので、要求したデータかどうかをチェックする
            Debug.Assert((int)target.header.variableType == (int)_id.Type, "unexpected error");
            Debug.Assert(target.header.previewPlayerIndex == _id.PreviewPlayerIndex, "unexpected error");
            Debug.Assert(target.header.trackNo == _id.TrackNo, "unexpected error");
        }

        private ToolSoundSeqVariableContainerHeader ToStructWithoutVariables()
        {
            ToolSoundSeqVariableContainerHeader dest = new ToolSoundSeqVariableContainerHeader();

            dest.variableType = (ToolSoundSeqVariableType)_id.Type;
            dest.previewPlayerIndex = _id.PreviewPlayerIndex;
            dest.trackNo = _id.TrackNo;

            return dest;
        }

        private ToolSoundSeqVariableContainer ToStruct()
        {
            ToolSoundSeqVariableContainer dest = new ToolSoundSeqVariableContainer();

            dest.header = ToStructWithoutVariables();

            // 変数の値を出力する
            for (int i = 0; i < TrackCount; i++)
            {
                dest.variables[i] = _variables[i].Value;
            }

            return dest;
        }

        private void DispatchValueChangedEvent()
        {
            if (_variableChangedEventLocked) { return; }

            if (_variableChangedDirty && null != ValueChanged)
            {
                if (null == PreviewManager.Instance.InvokeControl) { return; }
                PreviewManager.Instance.InvokeControl.Invoke(ValueChanged, new object[] { this, new EventArgs() });
            }

            _variableChangedDirty = false;
        }

        private void OnVariableValueChanged(object sender, EventArgs e)
        {
            _variableChangedDirty = true;
            DispatchValueChangedEvent();
        }

        #endregion
    }

    /// <summary>
    /// シーケンス変数
    /// </summary>
    public class SeqVariable
    {
        #region ** 定数

        public const int DefaultValue = -1;                       // デフォルト値
        public const int InvalidValue = (int)short.MaxValue + 1;  // 無効な値
        public const uint InvalidVariableNo = uint.MaxValue;            // 無効な変数番号
        public const int MaxValue = short.MaxValue;           // Value の最大値
        public const int MinValue = short.MinValue;           // Value の最小値
        public const uint MaxVariableNo = uint.MaxValue;            // トラック番号の最大値

        #endregion

        private uint _no = InvalidVariableNo;  // 変数番号
        private int _value = InvalidValue;       // 変数の値
        private bool _checked = false;              // 有効チェック


        public SeqVariable(uint no) : this(no, InvalidValue) { }

        public SeqVariable(uint no, int value)
        {
            Debug.Assert(InvalidVariableNo != no, "unexpected error");

            _no = no;
            Value = value;
        }

        #region ** プロパティ

        /// <summary>
        /// シーケンス変数の番号を取得します。
        /// </summary>
        public uint No
        {
            get { return _no; }
        }

        /// <summary>
        /// シーケンス変数の値を取得または設定します。
        /// </summary>
        public int Value
        {
            get { return _value; }
            set
            {
                if (value == _value) { return; }

                if (value != InvalidValue)
                {
                    if (value > MaxValue) { throw new ArgumentOutOfRangeException(); }
                    if (value < MinValue) { throw new ArgumentOutOfRangeException(); }
                }

                _value = value;

                if (null != ValueChanged)
                {
                    ValueChanged(this, new EventArgs());
                }
            }
        }

        /// <summary>
        /// シーケンス変数が有効チェック値を取得します。
        /// </summary>
        public bool Checked
        {
            get { return _checked; }
            set
            {
                if (value == _checked) { return; }
                _checked = value;
            }
        }

        /// <summary>
        /// シーケンス変数が有効かどうかを取得します。
        /// </summary>
        public bool Enabled
        {
            get
            {
                if (!_checked) { return false; }
                return (InvalidValue != _value);
            }
        }

        #endregion

        #region ** イベント

        public event EventHandler ValueChanged;

        #endregion
    }

    /// <summary>
    /// シーケンス変数コンテナの識別子
    /// </summary>
    public class SeqVariableContainerID
    {
        #region ** 定数

        public static readonly SeqVariableContainerID InvalidID = new SeqVariableContainerID(); // 無効なID

        #endregion

        private SeqVariableType _type = SeqVariableType.Unknown;                  // 変数の種類
        private uint _previewPlayerIndex = PreviewPlayer.InvalidIndex;               // プレビュープレイヤーのインデックス
        private uint _trackNo = SeqVariableContainer.InvalidTrackNo;      // トラック番号

        #region ** コンストラクタ

        public SeqVariableContainerID(SeqVariableType type, uint previewPlayerIndex, uint trackNo)
        {
            Initialize(type, previewPlayerIndex, trackNo);
        }

        protected SeqVariableContainerID() { }

        #endregion

        #region ** プロパティ

        /// <summary>
        /// シーケンス変数の種類を取得します。
        /// </summary>
        public SeqVariableType Type
        {
            get { return _type; }
        }

        /// <summary>
        /// プレビュープレイヤーのインデックスを取得します。
        /// </summary>
        public uint PreviewPlayerIndex
        {
            get { return _previewPlayerIndex; }
        }

        /// <summary>
        /// トラック番号を取得します。
        /// </summary>
        public uint TrackNo
        {
            get { return _trackNo; }
        }

        public bool Valid
        {
            get
            {
                if (SeqVariableType.Unknown == _type) { return false; }
                if (PreviewPlayer.InvalidIndex == _previewPlayerIndex) { return false; }
                if (SeqVariableContainer.TrackCount < _trackNo) { return false; }

                return true;
            }
        }

        #endregion

        #region ** オペレータ

        public static bool operator ==(SeqVariableContainerID target1, SeqVariableContainerID target2)
        {
            if (null == (object)target1 && null != (object)target2) { return false; }

            if (null != (object)target1)
            {
                return target1.Equals(target2);
            }
            else
            {
                return target2.Equals(target1);
            }
        }

        public static bool operator !=(SeqVariableContainerID target1, SeqVariableContainerID target2)
        {
            return !(target1 == target2);
        }

        #endregion

        #region ** メソッド

        public override bool Equals(object obj)
        {
            if (null == obj || !(obj is SeqVariableContainerID)) { return false; }

            SeqVariableContainerID target = obj as SeqVariableContainerID;

            if (_type != target._type) { return false; }
            if (_previewPlayerIndex != target._previewPlayerIndex) { return false; }
            if (_trackNo != target._trackNo) { return false; }

            return true;
        }

        public override int GetHashCode()
        {
            return (int)_type ^ (int)_previewPlayerIndex ^ (int)_trackNo;
        }

        protected void Initialize(SeqVariableType type, uint playerIndex, uint trackNo)
        {
            if (SeqVariableType.Unknown == type) { throw new ArgumentOutOfRangeException(); }
            if (PreviewPlayer.InvalidIndex == playerIndex) { throw new ArgumentOutOfRangeException(); }
            if (SeqVariableContainer.TrackCount < trackNo) { throw new ArgumentOutOfRangeException(); }

            _type = type;
            _previewPlayerIndex = playerIndex;
            _trackNo = (SeqVariableType.Track == type) ? trackNo : 0;
        }

        #endregion
    }

    /// <summary>
    /// 編集用、現在値用の２つを保持するシーケンス変数セット
    /// </summary>
    public class SeqVariableContainerSet
    {
        private SeqVariableContainer _edit = null;
        private SeqVariableContainer _current = null;

        #region ** プロパティ

        public SeqVariableContainer Edit
        {
            get { return _edit; }
        }

        public SeqVariableContainer Current
        {
            get { return _current; }
        }

        #endregion

        #region ** メソッド

        protected void Initialize(SeqVariableContainer edit, SeqVariableContainer current)
        {
            if (null == edit) { throw new ArgumentNullException("edit"); }
            if (null == current) { throw new ArgumentNullException("current"); }

            _edit = edit;
            _current = current;
        }

        #endregion
    }

    #region ** 各シーケンス変数用に特殊化

    #region ** ContainerID の特殊化

    public class GlobalVariableContainerID : SeqVariableContainerID
    {
        public GlobalVariableContainerID()
        {
            Initialize(SeqVariableType.Global, 0, 0);
        }
    }

    public class LocalVariableContainerID : SeqVariableContainerID
    {
        public LocalVariableContainerID(PreviewPlayer player)
        {
            if (null == player) { throw new ArgumentNullException("player"); }
            Initialize(SeqVariableType.Local, player.Index, 0);
        }

        public LocalVariableContainerID(uint previewPlayerIndex)
        {
            Initialize(SeqVariableType.Local, previewPlayerIndex, 0);
        }
    }

    public class TrackVariableContainerID : SeqVariableContainerID
    {
        public TrackVariableContainerID(PreviewPlayer player, uint trackNo)
        {
            if (null == player) { throw new ArgumentNullException("player"); }
            Initialize(SeqVariableType.Track, player.Index, trackNo);
        }

        public TrackVariableContainerID(uint previewPlayerIndex, uint trackNo)
        {
            Initialize(SeqVariableType.Track, previewPlayerIndex, trackNo);
        }
    }

    #endregion

    #region ** ContainerSet の特殊化

    public class GlobalVariableContainerSet : SeqVariableContainerSet
    {
        public GlobalVariableContainerSet()
        {
            base.Initialize(new GlobalVariableContainer(), new GlobalVariableContainer());
        }
    }

    public class LocalVariableContainerSet : SeqVariableContainerSet
    {
        public LocalVariableContainerSet(PreviewPlayer player)
        {
            base.Initialize(new LocalVariableContainer(player), new LocalVariableContainer(player));
        }
    }

    public class TrackVariableContainerSet : SeqVariableContainerSet
    {
        public TrackVariableContainerSet(PreviewPlayer player, uint trackNo)
        {
            if (null == player) { throw new ArgumentNullException("player"); }

            base.Initialize(new TrackVariableContainer(player, trackNo), new TrackVariableContainer(player, trackNo));
        }
    }

    public class TrackVariableContainerSets
    {
        private TrackVariableContainerSetCollection _containers = new TrackVariableContainerSetCollection();

        public TrackVariableContainerSets(PreviewPlayer player)
        {
            if (null == player) { throw new ArgumentNullException("player"); }

            for (uint i = 0; i < PreviewPlayerManager.TrackCount; i++)
            {
                _containers.Add(new TrackVariableContainerSet(player, i));
            }
        }

        #region ** プロパティ

        public TrackVariableContainerSetCollection Items
        {
            get { return _containers; }
        }

        #endregion
    }

    #region ** コレクションクラス

    public class TrackVariableContainerSetCollection : Collection<TrackVariableContainerSet> { }

    #endregion

    #endregion

    #region ** Container の特殊化

    public class GlobalVariableContainer : SeqVariableContainer
    {
        public GlobalVariableContainer()
            : this(SeqVariable.InvalidValue) { }

        public GlobalVariableContainer(int initValue)
            : base(new GlobalVariableContainerID(), initValue) { }
    }

    public class LocalVariableContainer : SeqVariableContainer
    {
        public LocalVariableContainer(PreviewPlayer player)
            : this(player, SeqVariable.InvalidValue) { }

        public LocalVariableContainer(PreviewPlayer player, int initValue)
            : base(new LocalVariableContainerID(player), initValue) { }
    }

    public class TrackVariableContainer : SeqVariableContainer
    {
        public TrackVariableContainer(PreviewPlayer player, uint trackNo)
            : this(player, trackNo, SeqVariable.InvalidValue) { }

        public TrackVariableContainer(PreviewPlayer player, uint trackNo, int initValue)
            : base(new TrackVariableContainerID(player, trackNo), initValue) { }
    }

    #endregion

    #endregion

    #region ** コレクションクラス

    public class SeqVariableCollection : Collection<SeqVariable> { }

    public class SeqVariableContainerCollection : KeyedCollection<SeqVariableContainerID, SeqVariableContainer>
    {
        #region ** KeyedCollection の実装

        /// <summary>
        /// 派生クラスで実装された場合、指定した要素からキーを抽出します。
        /// </summary>
        /// <param name="item">キーの抽出元要素。</param>
        /// <returns>指定した要素のキー。</returns>
        protected override SeqVariableContainerID GetKeyForItem(SeqVariableContainer item)
        {
            return item.ID;
        }

        #endregion
    }

    #endregion
}
