﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundMaker.Preview.Configurations
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.Linq;
    using System.Text;
    using NintendoWare.SoundMaker.Configurations;
    using NintendoWare.SoundMaker.Configurations.Schemas;
    using NW4R.ProtocolSound;

    /// <summary>
    /// エフェクト設定を適用または抽出します。
    /// </summary>
    public class EffectDataApplier
    {
        private Dictionary<Type, ToolSoundEffectType> _dataTypes;
        private Dictionary<ToolSoundEffectType, Type> _xmlDataTypes;
        private Dictionary<ToolSoundEffectType, EffectConverter> _dataConverters;
        private EffectParameterCollection _parameterSets;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="parameterSets">エフェクトパラメータ。</param>
        public EffectDataApplier(EffectParameterCollection parameterSets)
        {
            if (null == parameterSets) { throw new ArgumentNullException("parameterSets"); }

            _parameterSets = parameterSets;

            _dataTypes = new Dictionary<Type, ToolSoundEffectType>();
            _dataTypes.Add(typeof(XmlNoEffect), ToolSoundEffectType.EffectType_NoEffect);
            _dataTypes.Add(typeof(XmlReverbHi), ToolSoundEffectType.EffectType_ReverbHi);
            _dataTypes.Add(typeof(XmlReverbStd), ToolSoundEffectType.EffectType_ReverbStd);
            _dataTypes.Add(typeof(XmlChorus), ToolSoundEffectType.EffectType_Chorus);
            _dataTypes.Add(typeof(XmlDelay), ToolSoundEffectType.EffectType_Delay);
            _dataTypes.Add(typeof(XmlMultiChannelReverb), ToolSoundEffectType.EffectType_MultiChannelReverb);
            _dataTypes.Add(typeof(XmlMultiChannelChorus), ToolSoundEffectType.EffectType_MultiChannelChorus);
            _dataTypes.Add(typeof(XmlMultiChannelDelay), ToolSoundEffectType.EffectType_MultiChannelDelay);

            _xmlDataTypes = new Dictionary<ToolSoundEffectType, Type>();
            _xmlDataTypes.Add(ToolSoundEffectType.EffectType_NoEffect, typeof(XmlNoEffect));
            _xmlDataTypes.Add(ToolSoundEffectType.EffectType_ReverbHi, typeof(XmlReverbHi));
            _xmlDataTypes.Add(ToolSoundEffectType.EffectType_ReverbStd, typeof(XmlReverbStd));
            _xmlDataTypes.Add(ToolSoundEffectType.EffectType_Chorus, typeof(XmlChorus));
            _xmlDataTypes.Add(ToolSoundEffectType.EffectType_Delay, typeof(XmlDelay));
            _xmlDataTypes.Add(ToolSoundEffectType.EffectType_MultiChannelReverb, typeof(XmlMultiChannelReverb));
            _xmlDataTypes.Add(ToolSoundEffectType.EffectType_MultiChannelChorus, typeof(XmlMultiChannelChorus));
            _xmlDataTypes.Add(ToolSoundEffectType.EffectType_MultiChannelDelay, typeof(XmlMultiChannelDelay));

            _dataConverters = new Dictionary<ToolSoundEffectType, EffectConverter>();
            _dataConverters.Add(ToolSoundEffectType.EffectType_NoEffect, new NoEffectConverter());
            _dataConverters.Add(ToolSoundEffectType.EffectType_ReverbHi, new ReverbHiConverter());
            _dataConverters.Add(ToolSoundEffectType.EffectType_ReverbStd, new ReverbStdConverter());
            _dataConverters.Add(ToolSoundEffectType.EffectType_Chorus, new ChorusConverter());
            _dataConverters.Add(ToolSoundEffectType.EffectType_Delay, new DelayConverter());
            _dataConverters.Add(ToolSoundEffectType.EffectType_MultiChannelReverb, new MultiChannelReverbConverter());
            _dataConverters.Add(ToolSoundEffectType.EffectType_MultiChannelChorus, new MultiChannelChorusConverter());
            _dataConverters.Add(ToolSoundEffectType.EffectType_MultiChannelDelay, new MultiChannelDelayConverter());
        }

        /// <summary>
        /// エフェクト設定を適用します。
        /// </summary>
        /// <param name="parameterSets">適用先のエフェクトパラメータ。</param>
        /// <param name="xmlData">エフェクト設定。</param>
        public static void Apply(EffectParameterCollection parameterSets, XmlEffectData xmlData)
        {
            new EffectDataApplier(parameterSets).Apply(xmlData);
        }

        /// <summary>
        /// エフェクト設定を抽出します。
        /// </summary>
        /// <param name="parameterSets">対象エフェクトパラメータ。</param>
        /// <param name="xmlData">抽出したデータの格納先。</param>
        public static void Extract(EffectParameterCollection parameterSets, XmlEffectData xmlData)
        {
            new EffectDataApplier(parameterSets).Extract(xmlData);
        }

        /// <summary>
        /// エフェクト設定を適用します。
        /// </summary>
        /// <param name="xmlData">エフェクト設定。</param>
        public void Apply(XmlEffectData xmlData)
        {
            if (null == xmlData) { throw new ArgumentNullException("xmlData"); }

            if (null == xmlData.Effects) { return; }

            foreach (XmlEffect xmlEffect in xmlData.Effects.Items)
            {

                if (!_dataTypes.ContainsKey(xmlEffect.GetType())) { continue; }

                ToolSoundEffectType effectType = _dataTypes[xmlEffect.GetType()];
                if (!_dataConverters.ContainsKey(effectType)) { continue; }

                EffectParameterSet dataSet = _parameterSets
                                                [EffectIDConverter.Convert(xmlEffect)];
                dataSet.Select(effectType);
                _dataConverters[effectType].Apply(xmlEffect, dataSet.Selected);

            }
        }

        /// <summary>
        /// エフェクト設定を抽出します。
        /// </summary>
        /// <param name="xmlData">抽出したデータの格納先。</param>
        public void Extract(XmlEffectData xmlData)
        {
            if (null == xmlData) { throw new ArgumentNullException("xmlData"); }

            List<XmlEffect> xmlEffects = new List<XmlEffect>();

            foreach (EffectParameterSet parameterSet in _parameterSets)
            {

                XmlEffect xmlEffect = Activator.CreateInstance(
                                        _xmlDataTypes[parameterSet.Selected.EffectType]) as XmlEffect;

                _dataConverters[parameterSet.Selected.EffectType].Extract(parameterSet.Selected, xmlEffect);

                xmlEffects.Add(xmlEffect);

            }

            xmlData.Effects.Items = xmlEffects.ToArray();
        }
    }
}
