﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Xml;
using NintendoWare.SoundFoundation;
using NintendoWare.SoundMaker.Configurations.Schemas;
using NintendoWare.SoundMaker.Framework.Configurations;

namespace NintendoWare.SoundMaker.Configurations
{
    /// <summary>
    /// Commonプリセットを管理します。
    /// </summary>
    public class PresetConfigurationCommon : PresetConfiguration
    {
        private const string Platform = Platforms.Any.PlatformName;
        private const string Version = "1.0.0.0";

        private const string ElementNameOfPresetConfiguration = "PresetConfigurationCommon";
        private const string ElementNameOfEffectPreset = "EffectCommon";

        private EffectDataCollection _effects = new EffectDataCollection();

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public PresetConfigurationCommon() : base(Platform, Version) { }

        /// <summary>
        /// エフェクトプリセットの一覧を取得します。
        /// </summary>
        public EffectDataCollection Effects
        {
            get { return _effects; }
        }

        protected override string RootElementName
        {
            get { return ElementNameOfPresetConfiguration; }
        }

        /// <summary>
        /// デフォルト設定を読み込みます。
        /// </summary>
        /// <param name="settings">XmlReader 設定。</param>
        /// <param name="forced">現在の設定を全て破棄する場合は true、読み込み済みの設定を残す場合は false。</param>
        protected override void LoadDefaultsInternal(XmlReaderSettings settings, bool forced)
        {
            if (forced)
            {
                _effects.Clear();
            }
        }

        /// <summary>
        /// プリセットアイテムデータを読み込みます。
        /// </summary>
        /// <param name="presetName">プリセット名。</param>
        /// <param name="presetItemName">プリセットアイテム名。</param>
        /// <param name="xml">PresetItem 要素。</param>
        protected override void ReadPresetItemData(string presetName, string presetItemName, string xml)
        {
            if (null == presetName) { throw new ArgumentNullException("presetName"); }
            if (null == presetItemName) { throw new ArgumentNullException("presetItemName"); }
            if (null == xml) { throw new ArgumentNullException("xml"); }

            switch (presetName)
            {
                case ElementNameOfEffectPreset:
                    ReadEffectPresetItem(presetItemName, xml);
                    return;
            }
        }

        /// <summary>
        /// 全てのプリセットを書き出します。
        /// </summary>
        /// <param name="writer">XmlWriter。</param>
        protected override void WritePresets(XmlWriter writer)
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }
            WriteEffectPreset(writer);
        }

        /// <summary>
        /// エフェクトプリセットアイテムを読み込みます。
        /// </summary>
        /// <param name="presetItemName">プリセットアイテム名。</param>
        /// <param name="xml">PresetItem 要素。</param>
        private void ReadEffectPresetItem(string presetItemName, string xml)
        {
            if (null == presetItemName) { throw new ArgumentNullException("presetItemName"); }
            if (null == xml) { throw new ArgumentNullException("xml"); }

            XmlEffectData data = ReadConfigurationPart<XmlEffectData>(xml);
            if (0 == data.Effects.Items.Length) { return; }

            data.Name = presetItemName;

            if (_effects.Contains(presetItemName))
            {
                _effects.Remove(presetItemName);
            }

            _effects.Add(data);
        }

        /// <summary>
        /// エフェクトプリセットを書き出します。
        /// </summary>
        /// <param name="writer">XmlWriter。</param>
        private void WriteEffectPreset(XmlWriter writer)
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }

            writer.WriteStartElement(ElementPrefixOfPresetConfiguration, ElementNameOfPreset, null);
            writer.WriteAttributeString(AttributeNameOfName, ElementNameOfEffectPreset);

            writer.WriteStartElement(ElementPrefixOfPresetConfiguration, ElementNameOfPresetItems, null);

            foreach (XmlEffectData data in _effects)
            {
                WriteEffectPresetItem(writer, data.Name, data);
            }

            writer.WriteEndElement();
            writer.WriteEndElement();
        }

        /// <summary>
        /// エフェクトプリセットアイテムを書き出します。
        /// </summary>
        /// <param name="writer">XmlWriter。</param>
        /// <param name="presetName">プリセット名。</param>
        /// <param name="data">エフェクトプリセットアイテムデータ。</param>
        private void WriteEffectPresetItem(XmlWriter writer, string presetName, XmlEffectData data)
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }
            if (null == presetName) { throw new ArgumentNullException("presetName"); }
            if (null == data) { throw new ArgumentNullException("data"); }

            writer.WriteStartElement(ElementPrefixOfPresetConfiguration, ElementNameOfPresetItem, null);
            writer.WriteAttributeString(AttributeNameOfName, presetName);

            WriteConfigurationPart<XmlEffectData>(writer, data);

            writer.WriteEndElement();
        }
    }
}
